/*
 * SceneCanvas - OpenGL canvas for VRwave's Scene
 *
 * created: mpichler, 19960923
 *
 * changed: kwagen, 19970507
 * changed: mpichler, 19970918
 * changed: apesen, 19970926
 *
 * $Id: SceneCanvas.java,v 1.35 1997/09/29 08:20:56 apesen Exp $
 */


package iicm.vrml.vrwave;

import iicm.ge3d.*;
import iicm.utils3d.Camera;
import iicm.utils3d.Hitpoint;
import iicm.utils3d.Vec3f;
import iicm.utils3d.Mat4f;
import iicm.utils3d.Ray;
import iicm.widgets.RadioMenuItem;
import iicm.vrml.pw.Anchor;
import iicm.vrml.pw.Node;
import iicm.vrml.pw.GroupNode;
import iicm.vrml.pw.MFString;
import iicm.vrml.pw.PointerSensor;
import iicm.vrml.pw.TimeSensor;
import iicm.vrml.pw.TouchSensor;
import iicm.vrml.pw.PlaneSensor;
import iicm.vrml.pw.CylinderSensor;
import iicm.vrml.pw.SphereSensor;
import iicm.vrml.pw.Transform;
import iicm.vrml.vrwave.pwdat.TransformData;

import java.util.*;
import java.awt.*;


/**
 * SceneCanvas - Canvas for the Scene
 * Copyright (c) 1996 IICM
 *
 * @author Michael Pichler
 * @version 0.3, changed: 24 Jan 97
 */


public class SceneCanvas extends OGLCanvas
{
  private Scene scene_;
  int ui_red = 0xff0000;
  int ui_grey = 0xbbbbbb;
  int ui_purple = 0x8000ff;

  private float winaspect_ = 1.33f;  // window aspect ratio
  // mouse location in relative coordinates
  // (0, 0) = (left, bottom), (1, 1) = (right, top)
  // press location
  private float downx_, downy_;  // mouseDown position (relative)
  private float downxc_, downyc_;  // mouseDown position (centered, unscaled)
  private float dragx_, dragy_;  // last mouseDrag position (relative)
  private float dragxc_, dragyc_;  // mouseDrag position (centered, unscaled)
  private boolean mousedown_ = false;  // true: mouse pressed, drags/up to be handled

  // fly to
  private VHitpoint poihitpt_ = new VHitpoint (Hitpoint.PICK_NORMAL);  // find normal vector on picking
  Vec3f poitarget_;  // point of interest (null if unset)
  Vec3f poinormal_;  // normal at POI
  boolean flytoactive_;  // during "flying" (hold mb)
  boolean redrawafterpick_;  // new POI shown

  // heads-up
  private final static int HU_NONE = -1;
  private final static int HU_EYES = 0;
  private final static int HU_BODY = 1;
  private final static int HU_PAN = 2;
  private final static int HU_NUMICONS = 3;
  private final static float hu_halfsize = 16;
  private final static int HUEY = 48;
  private final static int HUBY = 0;
  private final static int HUPY = -48;
  private final static int[] hu_iconpos = { HUEY, HUBY, HUPY };
  private int hu_icon_ = HU_NONE;  // active icon
  // heads-up gui lines
  private final static float[] hu_lines =
  { // eyes
    7.0f, 2, HUEY, 6, HUEY-4, 10, HUEY-4, 14.2f, HUEY, 10, HUEY+4, 6, HUEY+4, 2, HUEY,  // r
    7.0f, -1.8f, HUEY, -6, HUEY-4, -10, HUEY-4, -14, HUEY, -10, HUEY+4, -6, HUEY+4, -1.8f, HUEY,  // l
    4.0f, 8, HUEY-2, 9, HUEY+1, 6, HUEY+1, 8, HUEY-2,  // r
    4.0f, -8, HUEY-2, -9, HUEY+1, -6, HUEY+1, -8, HUEY-2,  // l
    // body
    6.0f, 8, HUBY+3, 4, HUBY+3, -1, HUBY+7, -1, HUBY-1, 3, HUBY-6, 3, HUBY-13,
    5.0f, -8, HUBY-1, -8, HUBY+3, -3, HUBY+7, -3, HUBY-8, -7, HUBY-13,
    4.0f, -2, HUBY+10, -1, HUBY+13, -4, HUBY+13, -2, HUBY+10,
    // pan
    2.0f, -12, HUPY, 12, HUPY, 2.0f, 0, HUPY-12, 0, HUPY+12,  // h, v
    3.0f, -8, HUPY+4, -12, HUPY, -8, HUPY-4,  // l
    3.0f, 8, HUPY+4, 12, HUPY, 8, HUPY-4,  // r
    3.0f, -4, HUPY-8, 0, HUPY-12, 4, HUPY-8,  // b
    3.0f, -4, HUPY+8, 0, HUPY+12, 4, HUPY+8,  // t
    0.0f
  }; // hu_lines

  // picking
  private Vector pointersensor_ = null;   // keeps last active sensors

  /**
   * constructor: window title, scene
   */

  public SceneCanvas (String title, Scene s)
  {
    super (title);
    scene_ = s;
    scene_.setCanvas (this);
    verbose = VRwave.verbose;
  }

  /**
   * reset state on loading a new scene
   */

  public void reset ()
  {
    poitarget_ = null;
    poinormal_ = null;
    flytoactive_ = false;
    pointersensor_ = null;
  }

  /**
   * convert x coordinate to fraction (0 = left, 1 = right)
   */

  public final float tofractX (float x)
  {
    return x / cwidth;
  }

  /**
   * convert y coordinate to fraction (0 = bottom, 1 = top)
   */

  public final float tofractY (float y)
  {
    return (cheight - y) / cheight;
  }

  /**
   * draw the scene (paint sounds too slow :-)
   */

  public void paint (Graphics gc)
  {
    // System.out.println ("SceneCanvas.paint");

    if (scene_ == null)  // no scene: just clear background
    {
      Dimension dim = size ();
      gc.setColor (getBackground ());
      gc.fillRect (0, 0, dim.width, dim.height);
      return;
    }

    super.paint (gc);  // create context on first draw

    if (!hasContext () || !setContext (scene_.curDrawingMode () != GE3D.ge3d_wireframe))  // shading flag
    {
      Dimension dim = size ();
      gc.setColor (getBackground());
      gc.fillRect (0, 0, dim.width, dim.height);
      gc.setFont (new Font ("TimesRoman", Font.BOLD, 36));
      FontMetrics metrics = gc.getFontMetrics ();
      String msg = "Don't Panic!";
      int x0 = dim.width / 2 - metrics.stringWidth (msg) / 2;
      int y0 = dim.height / 2 - metrics.getDescent () / 2 + metrics.getAscent () / 2;
      gc.setColor (Color.black);
      gc.drawString (msg, x0, y0);
      return;
    }

    // here cwidth, cheight defined
    winaspect_ = (float) cwidth / (float) cheight;
    scene_.setWinAspect (winaspect_);
    redrawafterpick_ = true;

    scene_.draw ();

    drawUI ();

    swapBuffers ();

    scene_.finishedDraw ();

    if (mousedown_ && hu_icon_ != HU_NONE)  // continuous motion for walk/headsup
    {
      // System.err.print ("repaint/walk ");
      holdDragWalk ();
      repaint ();  // schedule next redraw
    }
    else if (mousedown_ && flytoactive_)
    {
      // System.err.println ("repaint/fly to");
      holdFlyTo ();
      repaint ();  // schedule next redraw
    }
    else if (scene_.behavior ())
    {
      // JDK bug: repaint yields paint call even when window is iconified
      // not determinable with Component methods
// System.err.println ("repaint on behaviour");
// System.err.println ("component is valid, visible, showing, enabled: " +
// isValid () + ", " + isVisible () + ", " + isShowing () + ", " + isEnabled ());
// System.err.println ("GC: " + getGraphics ());

      repaint ();  // schedule next redraw
    }

  } // paint

  /**
   * draw mode dependend user interface atop 3D scene
   */

  void drawUI ()
  {
    switch (scene_.movemode_)
    {
      case Scene.WALK:
      {
        if (!mousedown_)
          return;

        GE3D.setDrawMode (GE3D.ge3d_wireframe);
        GE3D.simpleOrthoCamera (cwidth, cheight);
        GE3D.lineColorRGBi (ui_red);
        float[] cross = {
          2.0f, downxc_ - 15, downyc_, downxc_ + 15, downyc_,
          2.0f, downxc_, downyc_ - 15, downxc_, downyc_ + 15,
          0.0f
        };
        GE3D.drawPolyLines2D (cross);
      }
      break;

      case Scene.FLYTO:
      {
        if (poitarget_ == null || poinormal_ == null)
          return;

        // still in 3D coordinate system
        float[] p = poitarget_.value_;
        float[] n = poinormal_.value_;  // ass.: normalized
        GE3D.setDrawMode (GE3D.ge3d_wireframe);
        GE3D.lineColorRGBi (ui_purple);  // icon color
        GE3D.lineStyle ((short) 0x7777);  // ----
        GE3D.pushMatrix ();
        GE3D.translatefff (p[0], p[1], p[2]);

        if (Math.abs (n[2]) < 0.99999)
          GE3D.rotatef3f (GE3D.Z_AXIS, (float) Math.atan2 (n[1], n[0]));  // "x to y"
        // otherwise normal vector is parallel to z-axis
        GE3D.rotatef3f (GE3D.Y_AXIS, (float) Math.acos (n[2]));  // "z to x-y"
        GE3D.drawCircle (0.0f, 0.0f, 0.5f);
        GE3D.drawCircle (0.0f, 0.0f, 0.1f);
        float len = 0.3f;
        GE3D.drawLine2D (-len, 0, len, 0);
        GE3D.drawLine2D (0, -len, 0, len);
        GE3D.lineStyle ((short) -1);  // solid
        GE3D.popMatrix ();
      }
      break;

      case Scene.HEADSUP:
      {
        GE3D.setDrawMode (GE3D.ge3d_wireframe);
        GE3D.simpleOrthoCamera (cwidth, cheight);

        GE3D.lineColorRGBi (ui_grey);  // boxes
        for (int i = 0;  i < HU_NUMICONS;  i++)
          GE3D.drawRect2D (-hu_halfsize, hu_iconpos [i] - hu_halfsize,
            hu_halfsize, hu_iconpos [i] + hu_halfsize);

        GE3D.lineColorRGBi (ui_purple);  // icons
        GE3D.drawPolyLines2D (hu_lines);

        if (!mousedown_)
          return;
        GE3D.lineColorRGBi (ui_red);  // beam
        GE3D.drawLine2D (downxc_, downyc_, dragxc_, dragyc_);
      }
      break;
    }
  } // drawUI

  /**
   * request keyboard focus when mouse enters window
   */

  public boolean mouseEnter (Event e, int x, int y)
  {
    super.mouseEnter (e, x, y);
    requestFocus ();
    return false;
  }

  /**
   * mouse press (down). For 3 button mice Java generates a Meta
   * modifier for the right mouse button and an Alt event for the
   * middle mouse button. However, as many keyboards have only a
   * single Alt/Meta key (and Java may set the Alt modifier
   * additionally to Meta), we allow using Meta for the right mouse
   * button and Shift where otherwise the middle mouse button is used.
   */

  public boolean mouseDown (Event e, int x, int y)
  {
    if (VRwave.verbose)
    {
      printModifiers (e);
      System.out.println ("mouse pressed at " + x + ", " + y);
    }

    dragx_ = downx_ = tofractX (x);
    dragy_ = downy_ = tofractY (y);

    if ((e.clickCount == 2) ||  // anchor activation on double click or
      !(e.metaDown () || altDown (e) || e.shiftDown ()) &&  // click left
      (scene_.getInteraction () ^ e.controlDown ()))        // in interaction mode or with control
    {
      // System.out.println("CTRL-Click-Left (Picking)");

      VHitpoint hit = new VHitpoint (Hitpoint.PICK_NORMAL);  // normal for TouchSensor
      scene_.pick (downx_, downy_, hit, false, true);
      GroupNode parent = hit.hitparent_;

      Vector activesensors = getActiveSensors (hit);
      removeNewHits (activesensors);  // deletes sensors form pointersensor_ which are still hit
      double clicktime = scene_.currentTime ();

      if (pointersensor_ != null)  // deactivate old sensors
      {
        for (int j = 0;  j < pointersensor_.size ();  j++)
          ((PointerSensor) pointersensor_.elementAt (j)).mouseExit (clicktime);
      }     

      pointersensor_ = activesensors;

      if (activesensors != null)  // notify new sensors
      {
        for (int j = 0;  j < activesensors.size ();  j++)
        {
          Object psens = activesensors.elementAt (j);
          if (psens instanceof Anchor)
          {
            Anchor anchor = (Anchor) psens;
            MFString urls = anchor.url;
            // take first URL; cannot expect feedback when following an anchor fails
            if (urls.getValueCount () > 0)
            {
              String url = urls.getValueData ()[0];
              if (url.length () > 0)  // ignore empty URLs
                if (url.charAt (0) != '#')
                {
                  MFString parameters = anchor.parameter;
                  scene_.activateAnchor (url, parameters.getValueData (), parameters.getValueCount ());
                }
                else
                  System.err.println ("error: jumping to viewpoint " + url + " not implemented");
            }
          }
          else
            ((PointerSensor) psens).mouseDown (downx_, downy_, hit, clicktime);
        }
      }

      redrawafterpick_ = false;
      repaint ();

      mousedown_ = false;  // no further action on drag/up
      return true;  // handled
    } // anchors, sensors

    scene_.setInteraction (true);  // begin interaction
    mousedown_ = true;

    dragxc_ = downxc_ = x - cwidth / 2;
    dragyc_ = downyc_ = cheight / 2 - y;

    hu_icon_ = HU_NONE;
    switch (scene_.movemode_)
    {
      case Scene.WALK:
        iconOfWalk (e);  // set hu_icon_ according to modifiers
      break;

      case Scene.FLYTO:
        if (dragFlyTo (e, false))  // new poi or motion
          repaint ();
      break;

      case Scene.HEADSUP:
        if (Math.abs (downxc_) <= hu_halfsize)
        {
          for (int i = 0;  i < HU_NUMICONS;  i++)
            if (Math.abs (downyc_ - hu_iconpos [i]) <= hu_halfsize)
            { hu_icon_ = i;
              break;
            }
        }
        // System.out.println ("icon hit: " + hu_icon_);
      break;
    }

    return true;  // handled
  } // mouseDown

  /**
   * mouse drag (button pressed)
   */

  public boolean mouseDrag (Event e, int x, int y)
  {
    // System.out.println ("drag [" + x + ", " + y + "]");

    float lastx = dragx_;
    float lasty = dragy_;

    dragxc_ = x - cwidth / 2;
    dragyc_ = cheight / 2 - y;
    dragx_ = tofractX (x);
    dragy_ = tofractY (y);

    if (!(e.metaDown () || altDown (e) || e.shiftDown ()) &&  // interactive/ctrl drag left
      (scene_.getInteraction () ^ e.controlDown ()))
    {
      if (!redrawafterpick_)  // events during dragging are skipped 
        return false;         // if there was no repaint since last pick
      if (VRwave.verbose)
        System.out.println ("CTRL-Drag (Picking)");

      if (pointersensor_ != null)  // notify grabbing sensors
      {
        VHitpoint geomhit = null;  // TouchSensor needs to pick scene geometry
        Node geomhitnode = null;
        Vector geomhitsensors = null;
        VHitpoint senshit = null;  // drag Sensors need to pick Sensors
        Node senshitnode = null;
        Vector senshitsensors = null;

        double dragtime = scene_.currentTime ();

        for (int j = 0;  j < pointersensor_.size ();  j++)  // for all grabbing sensors
        {
          Object psensor = pointersensor_.elementAt (j);
          if (psensor instanceof TouchSensor)  // pick scene geometry
          {
            if (VRwave.verbose)
              System.out.println ("TouchSensor");
            if (geomhit == null)
            {
              geomhit = new VHitpoint (Hitpoint.PICK_NORMAL);  // normal for TouchSensor
              geomhitnode = scene_.pick (dragx_, dragy_, geomhit);  // might pick hit geometry only
              if (geomhitnode != null)  
                geomhitsensors = getActiveSensors (geomhit);  // pointersensor_ unchanged
            }
            // check if object hit is still one TouchSensor is observing
            boolean isover = (geomhitsensors != null && geomhitsensors.contains (psensor));
            ((PointerSensor) psensor).mouseDrag (dragx_, dragy_, isover ? geomhit : null, dragtime);
          }
          else if (psensor instanceof PlaneSensor)
          {
            if (VRwave.verbose)
              System.out.println ("PlaneSensor");
            if (senshit == null)
            {
              senshit = new VHitpoint (0);  // no normal for PlaneSensor
              senshitnode = scene_.pick (dragx_, dragy_, senshit, true, false);  // pick only dragsensors, keep not trf
              if (senshitnode != null)  
                senshitsensors = getActiveSensors (senshit);  // pointersensor_ unchanged
            }
            ((PlaneSensor) psensor).mouseDrag (dragx_, dragy_, senshit, dragtime);
          }
          else if (psensor instanceof CylinderSensor)
          {
            if (VRwave.verbose)
              System.out.println ("CylinderSensor");
            if (senshit == null)
            {
              senshit = new VHitpoint (0);  // no normal for CylinderSensor
              if (((CylinderSensor) psensor).dragMode_ == CylinderSensor.DISKMODE)
              {
                senshitnode = scene_.pick (dragx_, dragy_, senshit, true, false);  // pick only dragsensors, keep not trf
                if (senshitnode != null)  
                  senshitsensors = getActiveSensors (senshit);  // pointersensor_ unchanged
              }
              else
              {
                ((CylinderSensor) psensor).ray_ = scene_.getRay (dragx_, dragy_);
                ((CylinderSensor) psensor).mouseDrag (dragx_, dragy_, senshit, dragtime);
              }
            }
            ((CylinderSensor) psensor).mouseDrag (dragx_, dragy_, senshit, dragtime);
          }
          else if (psensor instanceof SphereSensor)  // pick virtual sphere 
          {
            if (VRwave.verbose)
              System.out.println ("SphereSensor");

            senshit = new VHitpoint (0);  // no normal for SphereSensor
            ((SphereSensor) psensor).ray_ = scene_.getRay (dragx_, dragy_);
            ((SphereSensor) psensor).mouseDrag (dragx_, dragy_, senshit, dragtime);
          }
        }
        redrawafterpick_ = false;
        repaint ();
      }
      else  // no grabbers: drag equivalent to move
      {
        mouseMove (e, x, y);
      }

    }  // interactiv-mode drag

    if (!mousedown_)
      return true;  // handled

    switch (scene_.movemode_)
    {
      case Scene.FLIP:
        dragFlip (e, dragx_ - lastx, dragy_ - lasty);
      break;

      case Scene.HEADSUP:
        if (hu_icon_ == HU_NONE)
          return true;
      // fall throgh
      case Scene.WALK:
        dragWalkHeadsUp (dragx_ - downx_, dragy_ - downy_);
      break;

      // case Scene.FLY:
      // break;

      case Scene.FLYTO:
        if (!dragFlyTo (e, true))
          return true;  // w/o repaint
      break;
    }

    repaint ();  // redraw

    return true;  // handled
  } // mouseDrag

  /**
   * mouse release (up)
   */

  public boolean mouseUp (Event e, int x, int y)
  {
    if (!(e.metaDown () || altDown (e) || e.shiftDown ()) &&  // interactive/ctrl release left
      (scene_.getInteraction () ^ e.controlDown ()))
      if (pointersensor_ != null)  // notify sensors
      {
        for (int j = 0;  j < pointersensor_.size ();  j++)
          ((PointerSensor) pointersensor_.elementAt (j)).mouseUp (scene_.currentTime ());
        redrawafterpick_ = false;
        repaint ();
      }     

    if (!mousedown_)
      return true;  // handled

    scene_.setInteraction (false);  // finish interaction
    mousedown_ = false;
    flytoactive_ = false;
    hu_icon_ = HU_NONE;
    if (scene_.interactionRelevant () || scene_.movemode_ == Scene.WALK || scene_.movemode_ == Scene.HEADSUP)
      repaint ();

    // printModifiers (e);
    // System.out.println ("mouse released at " + x + ", " + y);
    return true;  // handled
  }

  /**
   * mouse move
   */

  public boolean mouseMove (Event e, int x, int y)
  {
    if (!(e.metaDown () || altDown (e) || e.shiftDown ()) &&  // interactive/ctrl move left
      (scene_.getInteraction () ^ e.controlDown ()))
    {
      if (!redrawafterpick_)  // events during moveing are skipped 
        return false;         // if there was no repaint since last pick
      dragx_ = downx_ = tofractX (x);
      dragy_ = downy_ = tofractY (y);
      if (VRwave.verbose)
        System.out.println ("CTRL-Mouse-Move (Picking)");

      VHitpoint hit = new VHitpoint (Hitpoint.PICK_NORMAL);  // normal for TouchSensor
      scene_.pick (downx_, downy_, hit);

      Vector activesensors = getActiveSensors (hit);
      removeNewHits (activesensors);  // deletes sensors form pointersensor_ which are still hit
      double movetime = scene_.currentTime ();

      if (pointersensor_ != null)  // deactivate old sensors
      {
        for (int j = 0;  j < pointersensor_.size ();  j++)
        {
          Object node = pointersensor_.elementAt (j);
          if (node instanceof Anchor)
            scene_.clearStatusMessage ();
          else 
            ((PointerSensor) node).mouseExit (movetime);
        }
      }     

      pointersensor_ = activesensors;

      if (activesensors != null)  // notify new sensors
      {
        for (int j = 0;  j < activesensors.size ();  j++)
        {
          Object node = pointersensor_.elementAt (j);
          if (node instanceof Anchor)  // Anchor URL
          {
            MFString urls = ((Anchor) node).url;
            if (urls.getValueCount () > 0)
            {
              String description = ((Anchor) node).description.getValue ();
              String url = urls.getValueData ()[0];

              if (url.length () > 0)  // ignore empty URLs
                scene_.statusMessage (description + " [" + url + "]", 0);
              else
                scene_.statusMessage (description, 0);
            }
          }
          else
            ((PointerSensor) node).mouseMove (downx_, downy_, hit, movetime);
        }
      }
      redrawafterpick_ = false;
      repaint ();
    }
    return true;  // handled
  } // mouseMove

//   public boolean handleEvent (Event e)
//   {
//     // do we get this when modifiers are pressed/released? no.
//     // printModifiers (e);
// System.err.println ("got Event " + e);
//     return super.handleEvent (e);
//   }


  /**
   * key stroke (down)
   */

  public boolean keyDown (Event e, int key)
  {
    // compare SceneWindow::handleKeystroke
    if (VRwave.verbose)
    {
      printModifiers (e);
      System.out.println ("got key stroke " + key + " ");
    }

    boolean shift = e.shiftDown ();
    boolean ctrl = e.controlDown ();
    boolean meta = e.metaDown ();

    MenuItem[] menuitems = scene_.getMenus ();

    switch (key)
    {
      /* function keys */
      /* as function keys do not get through when embedded into another application,
       * we also allow control+digit (digit alone will be used for viewpoint jumps)
       */

      case '1':
        if (!ctrl)
          break;
      case Event.F1:
        scene_.showHelpfile ("index.html");
      break;

      case '3':
        if (!ctrl)
          break;
      case Event.F3:
        scene_.openFile ();
      break;

      case '4':
        if (!ctrl)
          break;
      case Event.F4:
        scene_.setNavigationMode (Scene.FLIP, Scene.UPDATE_ALL);
      break;

      case '5':
        if (!ctrl)
          break;
      case Event.F5:
        scene_.setNavigationMode (Scene.WALK, Scene.UPDATE_ALL);
      break;

      // F6: FLY not yet implemented

      case '7':
        if (!ctrl)
          break;
      case Event.F7:
        scene_.setNavigationMode (Scene.FLYTO, Scene.UPDATE_ALL);
      break;

      case '8':
        if (!ctrl)
          break;
      case Event.F8:
        scene_.setNavigationMode (Scene.HEADSUP, Scene.UPDATE_ALL);
      break;

      // F10 reserved for switching to menu (in frame version)

      /* arrow (cursor) keys */

      case Event.UP:
        cursorNavigation (e, 0, 1);
      break;
      case Event.DOWN:
        cursorNavigation (e, 0, -1);
      break;
      case Event.LEFT:
        cursorNavigation (e, -1, 0);
      break;
      case Event.RIGHT:
        cursorNavigation (e, 1, 0);
      break;

      /* ctrl-keys */

      case 2:   // ctrl-b
      case 16:  // ctrl-p
        scene_.toggleBehavior (Scene.UPDATE_ALL);
      break;

      case 6:  // ctrl-f
        scene_.drawingMode (GE3D.ge3d_flat_shading, shift);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [shift ? MenuDef.M_IFlatShading : MenuDef.M_FlatShading]).setState (true);
      break;

      case 8:  // ctrl-h
        scene_.drawingMode (GE3D.ge3d_hidden_line, shift);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [shift ? MenuDef.M_IHiddenLine : MenuDef.M_HiddenLine]).setState (true);
      break;

      case 9:  // ctrl-i, TAB
        scene_.toggleInteraction (Scene.UPDATE_ALL);
      break;

      case 12:  // ctrl-l
        scene_.toggleLineAntialiasing ();
        if (menuitems != null)
          ((CheckboxMenuItem) menuitems [MenuDef.M_AALines]).setState (scene_.getLineAntialiasing () != 0);
      break;

      case 13:  // ctrl-m
        scene_.setTextureMipmapping (5);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [MenuDef.M_AATexturesOn]).setState (true);
        scene_.redraw ();
      break;

      case 14:  // ctrl-n
        scene_.setTextureMipmapping (0);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [MenuDef.M_AATexturesOff]).setState (true);
        scene_.redraw ();
      break;

      // ctrl-p currently used as synonym for ctrl-b (play/pause behavior)

      case 19:  // ctrl-s
        scene_.drawingMode (GE3D.ge3d_smooth_shading, shift);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [shift ? MenuDef.M_ISmoothShad : MenuDef.M_SmoothShading]).setState (true);
      break;

      case 20:  // ctrl-t
        scene_.drawingMode (GE3D.ge3d_texturing, shift);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [shift ? MenuDef.M_ITexturing : MenuDef.M_Texturing]).setState (true);
      break;

      case 21:  // ctrl-u
        scene_.drawingMode (-1, true);  // interactive "same"
        if (menuitems != null)
          ((RadioMenuItem) menuitems [MenuDef.M_ITheSame]).setState (true);
      break;

      case 23:  // ctrl-w
        scene_.drawingMode (GE3D.ge3d_wireframe, shift);
        if (menuitems != null)
          ((RadioMenuItem) menuitems [shift ? MenuDef.M_IWireframe : MenuDef.M_Wireframe]).setState (true);
      break;

      case 24:  // ctrl-x
        System.exit (0);
      break;

      /* ordinary keys */

      case 'l':
        scene_.levelView ();
      break;

      case 'r':
        if (meta)
          scene_.reloadFile ();
        else
          scene_.resetView ();
      break;

      case 'u':
        scene_.untiltView ();
      break;
    }

    repaint ();

    return true;  // handled
  } // keyDown

  /*
   * key release (up)

  public boolean keyUp (Event e, int key)
  {
    // on key autorepeat event sequence is down-up, down-up, down-up, ...
    // printModifiers (e);
    // System.out.println ("got key release " + key);

    return true;  // handled
  } // keyUp
   */

  /**
   * map an event modifier to the appropriate heads up icon
   */

  private void iconOfWalk (Event e)
  {
    if (e.metaDown ())          // meta (or right mb): view (eyes)
      hu_icon_ = HU_EYES;
    else if (e.shiftDown () || altDown (e))  // shift (or middle mb): translate (pan)
      hu_icon_ = HU_PAN;
    else                        // no mods: walk/turn
      hu_icon_ = HU_BODY;
  }

  /**
   * navigation via arrow (cursor) keys.
   */

  private void cursorNavigation (Event e, float right, float up)
  {
    // scaling constants should be configurable
    // remember: mouse navigation done in fractions of window width/height
    boolean ctrl = e.controlDown ();

    switch (scene_.movemode_)
    {
      case Scene.FLIP:
      case Scene.WALK:
      case Scene.HEADSUP:
      {
        // simulate mouse move
        float accel = 0;
        if (scene_.movemode_ == Scene.FLIP)
          accel = ctrl ? 0.5f : 0.1f;
        else
          accel = ctrl ? 0.6f : 0.3f;
        float dx = right * accel;
        float dy = up * accel;

        if (scene_.movemode_ == Scene.FLIP)
          dragFlip (e, dx, dy);
        else
        {
          int oldicon = hu_icon_;
          iconOfWalk (e);  // set hu_icon_ according to modifiers
          dragWalkHeadsUp (dx, dy);
          hu_icon_ = oldicon;
        }
      }
      break;

      case Scene.FLYTO:
        // move towards/away from poi (if set)
        if (poitarget_ == null || poinormal_ == null || up == 0)
          return;

        fly2towards_ = (up > 0);
        fly2dotran_ = !ctrl;
        fly2dorot_ = ctrl || e.shiftDown ();
        holdFlyTo ();
      break;
    }

  } // cursorNavigation


  // now care for the navigation metaphors

  /**
   * drag in FLIP mode
   */

  private void dragFlip (Event e, float dx, float dy)
  {
    // System.out.println ("dragFlip. dx = " + dx + ", dy = " + dy + " - ");
    Camera cam = scene_.getCamera ();

    // compare HG3dInputHandler::drag_flipobj
    if (e.metaDown ())                  // meta (or right mb): zoom
    { // System.out.println ("zoom");
      cam.zoomOut (dy * 2.0f);
    }
    else if (e.shiftDown () || altDown (e))  // shift (or middle mb): rotate
    { // System.out.println ("rotate");
      cam.rotateXYcenter (-dx * 3.14f, dy * 1.57f, scene_.getCenter ());
    }
    else                                // no mods: translate
    { // System.out.println ("translate");
      cam.translateVP (-dx, -dy, winaspect_, 2.0f);
    }
    // redraw done by caller
  } // dragFlip

  /**
   * drag in WALK/HEADSUP mode
   */

  private void dragWalkHeadsUp (float dx, float dy)
  {
    // System.out.println ("dragWalkHeadsUp. dx = " + dx + ", dy = " + dy + " - ");
    // hu_icon_ set on press
    dw_x_ = dx;
    dw_y_ = dy;

    holdDragWalk ();

  } // dragWalkHeadsUp

  // just for data exchange between dragWalkHeadsUp and holdDragWalk
  private float dw_x_ = 0.0f;
  private float dw_y_ = 0.0f;

  private void holdDragWalk ()
  {
    // performs motion of dragWalk
    // repetitive callable w/o event

    float dx = dw_x_;
    float dy = dw_y_;
    float dx3 = dx * dx * dx;
    float dy3 = dy * dy * dy;
    Camera cam = scene_.getCamera ();

    // compare HG3dInputHandler::hu_hold_button
    switch (hu_icon_)
    {
      case HU_EYES:
        cam.rotateXYposition (- dx * 0.1963f - dx3 * 12.56f, dy * 0.0981f + dy3 * 6.28f);
      break;
      case HU_PAN:
        cam.translateVP (dx * 0.1f + dx3, dy * 0.1f + dy3, winaspect_, 2.0f);
      break;
      case HU_BODY:
        cam.rotateXYposition (- dx * 0.1963f - dx3 * 12.56f, 0.0f);
        cam.zoomOut (- dy * 0.02f - dy3 * 4.0f);
      break;
    }
    // redraw done by caller
  } // holdDragWalk

  /**
   * drag in FLYTO mode
   */

  private boolean dragFlyTo (Event e, boolean drag)
  {
    // returns true if redraw should be done (new poi was set or motion)

    // primary goal: same usage as VRweb for 3button mice
    // secondary goal: still usable with less buttoned mice
    boolean meta = e.metaDown ();
    boolean alt = altDown (e);
    boolean ctrl = e.controlDown ();
    boolean shift = e.shiftDown ();
    boolean poiinfo = false;

    if (shift && ctrl && meta)
    { // verbose info about point hit
      poiinfo = true;
      meta = alt = ctrl = shift = false;
    }      

    // do not change order of if branches
    if (meta)  // meta (or right mb): fly away
    {
      if (poitarget_ == null)
        return false;

      fly2towards_ = false;
      fly2dotran_ = !ctrl;
      fly2dorot_ = shift || ctrl;

      holdFlyTo ();
      return true;
    }

    if (shift || alt || ctrl)  // shift (or middle mb): fly towards
    {
      if (poitarget_ == null)
        return false;

      fly2towards_ = true;
      fly2dotran_ = !ctrl || shift;
      fly2dorot_ = ctrl || alt && shift;

      holdFlyTo ();
      return true;
    }

    // else no mods: set/drag point of interest
    if (drag && !redrawafterpick_)  // events during dragging are skipped 
      return false;                 // if there was no repaint since last pick

    { // ctrl+left needed above
      boolean hadpoi = (poitarget_ != null);
      flytoactive_ = false;  // no continuous redraw

      scene_.pick (dragx_, dragy_, poihitpt_);
      poitarget_ = poihitpt_.hitpoint_;  // null iff not hit (see Picker.pickScene ())
      poinormal_ = poihitpt_.normal_;

      if (poiinfo)
      {
        if (poitarget_ != null)
          System.out.println ("hit point " + poitarget_ + ", normal " + poinormal_);
        else
          System.out.println ("no hit");
      }

      boolean retval = hadpoi || poitarget_ != null;
      redrawafterpick_ = !retval;  // next pick postponed after redraw done
      return retval;
    }

  } // dragFlyTo

  // just for data exchange between dragFlyTo and holdFlyTo
  private boolean fly2towards_, fly2dotran_, fly2dorot_;
  static float fly2tran_ = 0.15f;
  static float fly2rot_ = 0.25f;

  private void holdFlyTo ()
  {
    if (poitarget_ == null)
    {
      System.err.println ("Internal Error. holdFlyTo should not be called when there is nothing to do");
      flytoactive_ = false;
      return;
    }

    if (mousedown_)
      flytoactive_ = true;  // expect to be called again when idle

    // perform motion of dragFlyTo
    // repetitive callable w/o event
    float ktran = fly2towards_ ? fly2tran_ : - fly2tran_;
    float krot = fly2towards_ ? fly2rot_ : - fly2rot_;
    Camera cam = scene_.getCamera ();

    if (fly2dorot_)
    {
      // rotate camera position around hit (towards -poinormal_)
      // System.err.println ("fly to: rotate " + krot);
      cam.approachNormal (poitarget_.value_, poinormal_.value_, krot);
    } // rotation

    if (fly2dotran_)
    {
      // move camera towards poitarget
      // System.err.println ("fly to: translate " + ktran);
      cam.approachPosition (poitarget_.value_, ktran, scene_.getNearClip ());
    } // translation

  } // holdFlyTo


  /**
   * get a list of hit PointerSensor nodes. pointersensor_ list unchanged.
   */

  private Vector getActiveSensors (VHitpoint hit)
  {
    if (hit.hitpoint_ == null)  // no hit
      return null;

    Vector hitpath = hit.hitpath_;
    Vector activesensors = new Vector ();

    for (int i = 0;  i < hitpath.size ();  i++)
    {
      GroupNode hitparent = (GroupNode) hitpath.elementAt (i);
      Enumeration enum = hitparent.getChildrenEnumerator ();
      Node sens = null;

      // search sensors among siblings of object hit
      while (enum.hasMoreElements ())
      {
        if ((sens = (Node) enum.nextElement ()) instanceof PointerSensor && !(sens instanceof Anchor))
        {
          activesensors.addElement (sens);
          if (VRwave.verbose)
            System.out.println ("Found Sensor: " + sens.nodeName() + " " + sens.objname);
        }
        // else
        //   System.out.println("Found: " + sens.nodeName() + " " + sens.objname );
      }

      // Anchor parent equivalent to sensor
      if (hitparent instanceof Anchor)
        activesensors.addElement (hitparent);

      // mpi: moved this before Transform block. apesen-TODO: check this ###
      if (activesensors.size () > 0)
        return activesensors;

      // if no sensor/anchor found, continue search for parent
      // take care of changing object coordinate system
      if (hitparent instanceof Transform) 
      {
        Transform trf = (Transform) hitparent;
        float[] trfmat = ((TransformData) trf.userdata).getMatrix ();
        float[] invmat = ((TransformData) trf.userdata).getInvMatrix ();

        if (VRwave.verbose)
          System.out.println ("back-calc: " + trf.nodeName ());
        hit.raystartobj_ = Mat4f.transformPoint3Mat44 (hit.raystartobj_.value_, trfmat);
        hit.raydirobj_ = Mat4f.transformVector3Mat44 (hit.raydirobj_.value_, trfmat);
        hit.normalobj_ = Mat4f.transformVector3Mat44transp (hit.normalobj_.value_, invmat);
        hit.normalobj_.normalize ();

// TODO also multiply hit_trfmat with invtrfmat from left

//System.out.println ("back-calc: " + trf.nodeName() );
//float [] mat = ((TransformData) trf.userdata).getMatrix ();
//System.out.println("["+mat[ 0]+","+mat[ 1]+","+mat[ 2]+","+mat[ 3]+"]");
//System.out.println("["+mat[ 4]+","+mat[ 5]+","+mat[ 6]+","+mat[ 7]+"]");
//System.out.println("["+mat[ 8]+","+mat[ 9]+","+mat[10]+","+mat[11]+"]");
//System.out.println("["+mat[12]+","+mat[13]+","+mat[14]+","+mat[15]+"]");
      }
      // note: also Billboard should be handled like Transform ###
    }

    return null;
  } // getActiveSensors

  /**
   * removes any elements of activesensors from pointersensor_
   */

  private void removeNewHits (Vector activesensors)
  {
    if (pointersensor_ == null || activesensors == null)
      return;

    int i = activesensors.size ();
    while (i-- > 0)
    {
      int j = pointersensor_.size ();
      while (j-- > 0)
      {
        if (activesensors.elementAt (i) == pointersensor_.elementAt (j))
          pointersensor_.removeElementAt (j);
      }
    }
  } // removeNewHits


  // missing Event.altDown ()

  static boolean altDown (Event e)
  {
    return (e.modifiers & Event.ALT_MASK) != 0;
  }

  // small helper for debugging

  static void printModifiers (Event e)
  {
    if (e.shiftDown ())
      System.out.print ("{shift}");
    if (e.controlDown ())
      System.out.print ("{control}");
    if (altDown (e))
      System.out.print ("{alt}");
    if (e.metaDown ())
      System.out.print ("{meta}");
  }

} // SceneCanvas
