 /*
 * A driver for the OMNIKEY PCMCIA smartcard reader CardMan 4040
 *
 * $Id: cm4040_cs.c.2.4.x 1064 2006-10-20 07:03:17Z mf $
 *                   
 * (c) 2000-2004 OMNIKEY
 *
 * All rights reserved
 */

#undef	PCMCIA_DEBUG
#undef	CHECK_VERSION

#include	<linux/config.h>
#include	<linux/version.h>

#ifdef CONFIG_MODVERSIONS
#define	MODVERSIONS
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#include       <config/modversions.h>
#else
#include	<linux/modversions.h>
#endif
#endif

#include	<linux/kernel.h>
#include	<linux/module.h>
#include	<linux/slab.h>
#include        <linux/init.h>
#include	<linux/fs.h>
#include        <linux/delay.h>
#include	<asm/uaccess.h>
#include	<asm/io.h>

#include	<pcmcia/version.h>
#include	<pcmcia/cs_types.h>
#include	<pcmcia/cs.h>
#include	<pcmcia/cistpl.h>
#include	<pcmcia/cisreg.h>
#include	<pcmcia/ciscode.h>
#include	<pcmcia/ds.h>
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#include	<pcmcia/bus_ops.h>
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
#define verify_area(type, addr, size) access_ok(type, addr, size)
#endif

#include	"cm4040_cs.h"

#undef	min
#define	min(a,b)	(a<b?a:b)



#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
static int io=0;
MODULE_PARM(io, "i");
#define DEBUG(n, args...) if (pc_debug >= (n)) printk(KERN_INFO args)
#else
#define DEBUG(n, args...)
#endif

static volatile char *version =
"OMNIKEY CardMan 4040 v2.0.0 www.omnikey.com";

#define	T_1SEC		(HZ)
#define T_10MSEC (HZ/100)
#define	CCID_DRIVER_BULK_DEFAULT_TIMEOUT  (150*HZ)
#define	CCID_DRIVER_ASYNC_POWERUP_TIMEOUT (35*HZ)
#define	CCID_DRIVER_MINIMUM_TIMEOUT (3*HZ)
#define POLL_RATE 1000
#define READ_WRITE_BUFFER_SIZE 512





static int cmx_open(struct inode *,struct file *);
static int cmx_close(struct inode *,struct file *);
static loff_t cmx_llseek(struct file *,loff_t,int);
static ssize_t cmx_read(struct file *,char *,size_t,loff_t *);
static int cmx_write(struct file *,const char *,size_t,loff_t *);
static int cmx_ioctl(struct inode *,struct file *,unsigned int,unsigned long);

static struct file_operations reader_fops = {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
  owner: THIS_MODULE,
#endif
	llseek:		cmx_llseek,
	read:		cmx_read,
	write:		cmx_write,
	ioctl:		cmx_ioctl,
	open:		cmx_open,
	release:	cmx_close,
	/* NULL...*/
};





static dev_link_t *reader_attach(void);	/*proto*/
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13)
static int cardman_event(event_t, int, event_callback_args_t *); /*proto*/
#endif
static void reader_detach(dev_link_t *link);	/*proto*/
static void reader_release(u_long arg);	/*proto*/

static int major;	



#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
static struct pcmcia_driver reader_driver =
{
  .owner          = THIS_MODULE,
  .drv            = 
	{
    .name   = "cm4040_cs",
  },
  .attach         = reader_attach,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,13)
  .event                   = cardman_event,
#endif
  .detach         = reader_detach,
};
#endif

typedef struct reader_dev_t
{
  dev_link_t	  link;		
  dev_node_t	  node;		
  wait_queue_head_t devq;	
                                
  unsigned int      fTimerExpired;
  struct timer_list timer;
  unsigned long     timeout;
  unsigned char     sBuf[READ_WRITE_BUFFER_SIZE];
  unsigned char     rBuf[READ_WRITE_BUFFER_SIZE];
  struct task_struct *owner;	
} reader_dev_t;

static dev_info_t dev_info = MODULE_NAME;
static dev_link_t *dev_table[CM_MAX_DEV] = { NULL, };



#ifndef PCMCIA_DEBUG
#define	xoutb	outb
#define	xinb	inb
#else
static inline void xoutb(unsigned char val,unsigned short port) {
  #if 1 
	DEBUG(6, "outb(val=%.2x,port=%.4x)\n",
		val, port);
	#endif
	outb(val,port);
}
static unsigned char xinb(unsigned short port) {
unsigned char val;
	val=inb(port);
#if 1
	DEBUG(6, "%.2x=inb(%.4x)\n",
		val,port);
#endif
	return val;
}
#endif
/*



static void Timer_Expired(unsigned long p)
{
  register reader_dev_t *dev = (reader_dev_t *) p;
  dev->fTimerExpired = 1;
}
*/



/*
static void Timer_Start(reader_dev_t *dev)
{
  init_timer(&dev->timer);
  dev->timer.expires = jiffies + dev->timeout;
  dev->timer.data = (u_long) dev;
  dev->timer.function=Timer_Expired;
  dev->fTimerExpired = 0;
  add_timer(&dev->timer);
}
*/




/*
static void Timer_Stop(reader_dev_t *dev)
{
  del_timer(&dev->timer);
}
*/


static inline void ipause(unsigned long amount) {
	set_current_state(TASK_INTERRUPTIBLE);
	schedule_timeout(amount);
}


static inline void upause(unsigned long amount) {
	set_current_state(TASK_UNINTERRUPTIBLE);
	schedule_timeout(amount);
}



static loff_t cmx_llseek(struct file * filp, loff_t off, int whence) {
	DEBUG(3,"cmx: cmx_llseek()\n");
	return -ESPIPE;
}




static inline int cmx_waitForBulkOutReady(reader_dev_t *dev)
{
	register int iobase = dev->link.io.BasePort1;
	register int j = 0;
        register int i;

	for (j=0;j<dev->timeout * 100;j++)
	{
          for (i=0;i<POLL_RATE;i++)
          {
	    if ((xinb(iobase + REG_OFFSET_BUFFER_STATUS) & BSR_BULK_OUT_FULL) == 0)
	    {
	      DEBUG(3,"BulkOut empty)\n");
	      return (1);
	    }
          }
	  ipause(T_10MSEC);
	}

	DEBUG(3,"BulkOut full)\n");

	return (0);
}




static inline int cmx_writeSync(unsigned char val,reader_dev_t *dev)
{
	register int iobase = dev->link.io.BasePort1;
	int rc;

	rc = cmx_waitForBulkOutReady(dev);
        if (rc != 1)
        {
	  return 0;
        }

	xoutb(val,iobase + REG_OFFSET_SYNC_CONTROL);
	rc = cmx_waitForBulkOutReady(dev);
        if (rc != 1)
        {
	  return 0;
        }
	return (1);
}




static inline int cmx_waitForBulkInReady(reader_dev_t *dev)
{
	register int iobase = dev->link.io.BasePort1;
	register int j = 0;
        register int i;

	for (j=0;j<dev->timeout * 100;j++)
	{
          for (i=0;i<POLL_RATE;i++)
          {
	    if ((xinb(iobase + REG_OFFSET_BUFFER_STATUS) & BSR_BULK_IN_FULL) == BSR_BULK_IN_FULL)
	    {
	      DEBUG(3,"BulkIn full)\n");
	      return (1);
	    }
          }
	  ipause(T_10MSEC);
	}

	DEBUG(3,"BulkIn empty)\n");
	return (0);
}




static ssize_t cmx_read(struct file *filp,char *buf,size_t count,loff_t *ppos)
{
register reader_dev_t *dev=(reader_dev_t *)filp->private_data;
register int iobase=dev->link.io.BasePort1;
unsigned long ulBytesToRead;
unsigned long i;
unsigned long ulMin;
int rc;
unsigned char uc;

	DEBUG(2, "-> cmx_read(%s,%d)\n",
		current->comm,current->pid);

	if (count==0)		
		return 0;

        if (count < 10)
	  return -EFAULT;

	if (filp->f_flags & O_NONBLOCK)
        { 
	  DEBUG(4, "cmx_read: filep->f_flags O_NONBLOCK set\n");
	  DEBUG(4, "cmx_read: <- cmx_read (failure)\n");
	  return -EAGAIN;
        }

	if ((dev->link.state & DEV_PRESENT)==0)	
		return -ENODEV;

	for (i=0; i<5; i++)
	{
          rc = cmx_waitForBulkInReady(dev);
          if (rc != 1)
          {
	    DEBUG(5,"cmx: cmx_waitForBulkInReady rc=%.2x\n",rc);
	    DEBUG(2, "<- cmx_read (failed)\n");
	    return -EIO;
          }
	  
	  dev->rBuf[i] = xinb(iobase + REG_OFFSET_BULK_IN);
	}

	ulBytesToRead = 5 + 
                        (0x000000FF&((char)dev->rBuf[1])) + 
                        (0x0000FF00&((char)dev->rBuf[2] << 8)) + 
                        (0x00FF0000&((char)dev->rBuf[3] << 16)) + 
                        (0xFF000000&((char)dev->rBuf[4] << 24));

        ulMin = (count < (ulBytesToRead+5))?count:(ulBytesToRead+5);

	for (i=0; i < (ulMin-5); i++)
	{
          rc = cmx_waitForBulkInReady(dev);
          if (rc != 1)
          {
	    DEBUG(5,"cmx: cmx_waitForBulkInReady rc=%.2x\n",rc);
	    DEBUG(2, "<- cmx_read (failed)\n");
	    return -EIO;
          }
	  
	  dev->rBuf[i+5] = xinb(iobase + REG_OFFSET_BULK_IN);
	}

	*ppos = ulMin;
        copy_to_user(buf, dev->rBuf, ulMin);

	
	rc = cmx_waitForBulkInReady(dev);
	if (rc != 1)
        {
	  DEBUG(5,"cmx: cmx_waitForBulkInReady rc=%.2x\n",rc);
	  DEBUG(2, "<- cmx_read (failed)\n");
	  return -EIO;
        }

	rc = cmx_writeSync(SCR_READER_TO_HOST_DONE ,dev);

	if (rc != 1)
	{
	  DEBUG(5,"cmx: cmx_writeSync c=%.2x\n",rc);
	  DEBUG(2, "<- cmx_read (failed)\n");
	  return -EIO;
	}



	uc = xinb(iobase + REG_OFFSET_BULK_IN);

	DEBUG(2,"<- cmx_read (successfully)\n");
	return ulMin;
}




static ssize_t cmx_write(struct file *filp,const char *buf,size_t count,
				loff_t *ppos)
{
register reader_dev_t *dev=(reader_dev_t *)filp->private_data;
register int iobase=dev->link.io.BasePort1;
ssize_t rc;
int i;
unsigned int uiBytesToWrite; 

	DEBUG(2, "-> cmx_write(%s,%d)\n",
		current->comm,current->pid);

	if (count==0)		
        {
	  DEBUG(2, "<- cmx_write nothing to do (successfully)\n");
	  return 0;
	}

	if (count < 5)
	{
	  DEBUG(2, "<- cmx_write buffersize=%d < 5\n",count);
	  return -EIO;
	}

	if (filp->f_flags & O_NONBLOCK)
        { 
	  DEBUG(4, "cmx_write: filep->f_flags O_NONBLOCK set\n");
	  DEBUG(4, "cmx_write: <- cmx_write (failure)\n");
	  return -EAGAIN;
        }

	if ((dev->link.state & DEV_PRESENT)==0)	
		return -ENODEV;

        uiBytesToWrite = count;
        copy_from_user(dev->sBuf, buf, uiBytesToWrite);

   switch (dev->sBuf[0])
      {
      case CMD_PC_TO_RDR_XFRBLOCK       :
      case CMD_PC_TO_RDR_SECURE         :  
      case CMD_PC_TO_RDR_TEST_SECURE    :  
      case CMD_PC_TO_RDR_OK_SECURE      :
          dev->timeout = CCID_DRIVER_BULK_DEFAULT_TIMEOUT;
         break;

      case CMD_PC_TO_RDR_ICCPOWERON     :  
         dev->timeout = CCID_DRIVER_ASYNC_POWERUP_TIMEOUT;
         break;

      case CMD_PC_TO_RDR_GETSLOTSTATUS  :  
      case CMD_PC_TO_RDR_ICCPOWEROFF    :  
      case CMD_PC_TO_RDR_GETPARAMETERS  :  
      case CMD_PC_TO_RDR_RESETPARAMETERS:  
      case CMD_PC_TO_RDR_SETPARAMETERS  :  
      case CMD_PC_TO_RDR_ESCAPE         :  
      case CMD_PC_TO_RDR_ICCCLOCK       :  
      default:
         dev->timeout = CCID_DRIVER_MINIMUM_TIMEOUT;
         break;
      }

	rc = cmx_writeSync(SCR_HOST_TO_READER_START ,dev);

	DEBUG(4, "start \n");

	for (i=0; i < uiBytesToWrite; i++)
	{
          rc = cmx_waitForBulkOutReady(dev);
          if (rc != 1)
          {
	    DEBUG(5,"cmx: cmx_waitForBulkOutReady rc=%.2x\n",rc);
	    DEBUG(2, "<- cmx_write (failed)\n");
	    return -EIO;
          }
	 
	  xoutb(dev->sBuf[i],iobase + REG_OFFSET_BULK_OUT);
	DEBUG(4, "%.2x ",dev->sBuf[i]);
	}
	DEBUG(4, "end\n");

	rc = cmx_writeSync(SCR_HOST_TO_READER_DONE ,dev);

	if (rc != 1)
	{
	  DEBUG(5,"cmx: cmx_writeSync c=%.2x\n",rc);
	  DEBUG(2, "<- cmx_write (failed)\n");
	  return -EIO;
	}

	DEBUG(2, "<- cmx_write (successfully)\n");
	return (count);
}




static int cmx_ioctl(struct inode *inode,struct file *filp,unsigned int cmd,
			unsigned long arg)
{
dev_link_t *link;
int rc,size;

	link=dev_table[MINOR(inode->i_rdev)];
	if (!(DEV_OK(link))){
	  DEBUG(4, "cmx: DEV_OK false\n");
		return -ENODEV;
	}
	if (_IOC_TYPE(cmd)!=CM_IOC_MAGIC) {
	  DEBUG(4,"cmx: ioctype mismatch\n");
	  return -EINVAL;
	}
	if (_IOC_NR(cmd)>CM_IOC_MAXNR) {
	  DEBUG(4,"cmx: iocnr mismatch\n");
	  return -EINVAL;
	} 
	size=_IOC_SIZE(cmd);
	rc=0;
	DEBUG(4,"cmx: iocdir=%.4x iocr=%.4x iocw=%.4x iocsize=%d cmd=%.4x\n",
		_IOC_DIR(cmd),_IOC_READ,_IOC_WRITE,size,cmd);

	if (_IOC_DIR(cmd)&_IOC_READ) {
		if (arg==0) rc=-EFAULT;
		else rc=verify_area(VERIFY_WRITE,(void *)arg,size);
	}
	if (_IOC_DIR(cmd)&_IOC_WRITE) {
		if (arg==0) rc=-EFAULT;
		else rc=verify_area(VERIFY_READ,(void *)arg,size);
	}

	return rc;
}




static int cmx_open (struct inode *inode, struct file *filp)
{
reader_dev_t *dev;
dev_link_t *link;
int i;

	DEBUG(2,"cmx: -> cmx_open(device=%d.%d process=%s,%d)\n",
		MAJOR(inode->i_rdev),MINOR(inode->i_rdev),
		current->comm,current->pid);

	i=MINOR(inode->i_rdev);
	if (i>=CM_MAX_DEV){
	  DEBUG(4, "cmx: MAX_DEV reached\n");
	  DEBUG(4, "cmx: <- cmx_open (failure)\n");
	  return -ENODEV;
	}
	link=dev_table[MINOR(inode->i_rdev)];
	if (link==NULL || !(DEV_OK(link))){
	  DEBUG(4, "cmx: link== NULL || DEV_OK false\n");
	  DEBUG(4, "cmx: <- cmx_open (failure)\n");
	  return -ENODEV;
	}
	if (link->open){
	  DEBUG(4, "cmx: DEVICE BUSY\n");
	  DEBUG(4, "cmx: <- cmx_open (failure)\n");
	  return -EBUSY;
	}

	dev=(reader_dev_t *)link->priv;
	filp->private_data=dev;

	if (filp->f_flags & O_NONBLOCK)
        { 
	  DEBUG(4, "cmx: filep->f_flags O_NONBLOCK set\n");
	  DEBUG(4, "cmx: <- cmx_open (failure)\n");
	  return -EAGAIN;
        }

	dev->owner=current;
	link->open=1;		

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_INC_USE_COUNT;
#endif

	DEBUG(2, "cmx: <- cmx_open (successfully)\n");
	return 0;
}




static int cmx_close(struct inode *inode,struct file *filp)
{
reader_dev_t *dev;
dev_link_t *link;
int i;

	DEBUG(2,"cmx: -> cmx_close(maj/min=%d.%d)\n",
		MAJOR(inode->i_rdev),MINOR(inode->i_rdev));

	i=MINOR(inode->i_rdev);
	if (i>=CM_MAX_DEV)
		return -ENODEV;
	link=dev_table[MINOR(inode->i_rdev)];
	if (link==NULL)
		return -ENODEV;

	dev=(reader_dev_t *)link->priv;


	link->open=0;		
	wake_up(&dev->devq);	

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_DEC_USE_COUNT;
#endif

	DEBUG(2, "cmx: <- cmx_close\n");
	return 0;
}







static void cmx_reader_release(dev_link_t *link) {
reader_dev_t *dev=(reader_dev_t *)link->priv;



 DEBUG(3, "cmx: -> cmx_reader_release\n"); 
 while(link->open) {
   DEBUG(3, KERN_INFO MODULE_NAME ": delaying release until "
	 "process '%s', pid %d has terminated\n",
	 dev->owner->comm,dev->owner->pid);
   
   
   

   wait_event(dev->devq,(link->open == 0));
 }
 DEBUG(3, "cmx: <- cmx_reader_release\n"); 
 return;
}



#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
static void cs_error(client_handle_t handle, int func, int ret) {
error_info_t err = { func, ret };

	CardServices(ReportError, handle, &err);
}
#endif





#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
#define	CS_CHECK(fn,args...) \
	((fail_rc=CardServices(fail_fn=(fn),args))!=CS_SUCCESS)
#endif

static void reader_config(dev_link_t *link,int devno) {
client_handle_t handle;
reader_dev_t *dev;
tuple_t tuple;
cisparse_t parse;
config_info_t conf;
u_char buf[64];
int fail_fn,fail_rc;
int rc;

	DEBUG(2,"cmx: -> reader_config\n");

	

	handle=link->handle;

	tuple.DesiredTuple=CISTPL_CONFIG;
	tuple.Attributes=0;
	tuple.TupleData=buf;
	tuple.TupleDataMax=sizeof(buf);
 	tuple.TupleOffset=0;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	if ((fail_rc=pcmcia_get_first_tuple(handle,&tuple))!=CS_SUCCESS){
	  fail_fn=GetFirstTuple;
		DEBUG(1, "pcmcia_get_first_tuple failed 0x%x\n",fail_rc);
	  goto cs_failed;
	}
	if ((fail_rc=pcmcia_get_tuple_data(handle,&tuple))!=CS_SUCCESS){
	  fail_fn=GetTupleData;
		DEBUG(1, "pcmcia_get_first_tuple failed 0x%x\n",fail_rc);
	  goto cs_failed;
	}
	if ((fail_rc=pcmcia_parse_tuple(handle,&tuple,&parse))!=CS_SUCCESS){
	  fail_fn=ParseTuple;
		DEBUG(1, "pcmcia_parse_tuple failed 0x%x\n",fail_rc);
	  goto cs_failed;
	}
	if ((fail_rc=pcmcia_get_configuration_info(handle,&conf))!=CS_SUCCESS){
	  fail_fn=GetConfigurationInfo;
		DEBUG(1, "pcmcia_get_configuration_info failed 0x%x\n",fail_rc);
	  goto cs_failed;
	}
#else
	if (CS_CHECK(GetFirstTuple,handle,&tuple))
		goto cs_failed;
	if (CS_CHECK(GetTupleData,handle,&tuple))
		goto cs_failed;
	if (CS_CHECK(ParseTuple,handle,&tuple,&parse))
		goto cs_failed;
	if (CS_CHECK(GetConfigurationInfo,handle,&conf))
		goto cs_failed;
#endif

	link->state|=DEV_CONFIG;
	link->conf.ConfigBase=parse.config.base;
	link->conf.Present=parse.config.rmask[0];
	link->conf.Vcc=conf.Vcc;
	DEBUG(2,"cmx: link->conf.Vcc=%d\n",link->conf.Vcc);

	link->io.BasePort2=0;
	link->io.NumPorts2=0;
	link->io.Attributes2=0;
	tuple.DesiredTuple=CISTPL_CFTABLE_ENTRY;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	for(rc=pcmcia_get_first_tuple(handle,&tuple);
		rc==CS_SUCCESS;
		rc=pcmcia_get_next_tuple(handle,&tuple)) 
#else
	for(rc=CardServices(GetFirstTuple,handle,&tuple);
		rc==CS_SUCCESS;
		rc=CardServices(GetNextTuple,handle,&tuple))
#endif
{
 	  DEBUG(2, "cmx: Examing CIS Tuple!\n");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
		rc=pcmcia_get_tuple_data(handle, &tuple);
#else
		rc=CardServices(GetTupleData,handle,&tuple);
#endif
		if(rc!=CS_SUCCESS)
			continue;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
		rc=pcmcia_parse_tuple(handle,&tuple,&parse);
#else
		rc=CardServices(ParseTuple,handle,&tuple,&parse);
#endif
		if(rc!=CS_SUCCESS)
			continue;

		DEBUG(2,"cmx: tupleIndex=%d\n",parse.cftable_entry.index);
		link->conf.ConfigIndex=parse.cftable_entry.index;

		
		if (parse.cftable_entry.io.nwin) {
			link->io.BasePort1=parse.cftable_entry.io.win[0].base;
			link->io.NumPorts1=parse.cftable_entry.io.win[0].len;
			link->io.Attributes1=IO_DATA_PATH_WIDTH_AUTO;
			if(!(parse.cftable_entry.io.flags & CISTPL_IO_8BIT))
				link->io.Attributes1=IO_DATA_PATH_WIDTH_16;
			if(!(parse.cftable_entry.io.flags & CISTPL_IO_16BIT))
				link->io.Attributes1=IO_DATA_PATH_WIDTH_8;
#ifdef PCMCIA_DEBUG
			if (io) link->io.BasePort1=io;
#endif
			link->io.IOAddrLines=parse.cftable_entry.io.flags
						& CISTPL_IO_LINES_MASK;
			DEBUG(2,"cmx: io.BasePort1=%.4x\n",link->io.BasePort1);
			DEBUG(2,"cmx: io.NumPorts1=%.4x\n",link->io.NumPorts1);
			DEBUG(2,"cmx: io.BasePort2=%.4x\n",link->io.BasePort2);
			DEBUG(2,"cmx: io.NumPorts2=%.4x\n",link->io.NumPorts2);
			DEBUG(2,"cmx: io.IOAddrLines=%.4x\n",link->io.IOAddrLines);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
			rc=pcmcia_request_io(handle,&link->io);
#else
			rc=CardServices(RequestIO,handle,&link->io);
#endif
			if (rc==CS_SUCCESS) {
				DEBUG(2,"cmx: RequestIO OK\n");
				break; 
			}
			else DEBUG(2,"cmx: RequestIO failed\n");
		}
	}
	if (rc!=CS_SUCCESS) {
		DEBUG(2,"cmx: Couldn't configure reader\n");
		goto cs_release;
	}

        link->conf.IntType=00000002;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	if ((fail_rc=pcmcia_request_configuration(handle,&link->conf))!=CS_SUCCESS)
	{
	  fail_fn=RequestConfiguration;
		DEBUG(1, "pcmcia_request_configuration failed 0x%x\n",fail_rc);
#else
	if (CS_CHECK(RequestConfiguration,handle,&link->conf))
	{
#endif
		goto cs_release;
	}

	DEBUG(2, "cmx: RequestConfiguration OK\n");

	dev=link->priv;
	sprintf(dev->node.dev_name,DEVICE_NAME "%d",devno);
	dev->node.major=major;
	dev->node.minor=devno;
	dev->node.next=NULL;
	link->dev=&dev->node;
	link->state&=~DEV_CONFIG_PENDING;

	DEBUG(2,"cmx: device " DEVICE_NAME "%d at 0x%.4x-0x%.4x\n",
		devno,
		link->io.BasePort1,
		link->io.BasePort1+link->io.NumPorts1);

	DEBUG(2, "cmx: <- reader_config (succ)\n");
	return;

cs_failed:
	cs_error(handle,fail_fn,fail_rc);
cs_release:
	reader_release((u_long)link);
	link->state&=~DEV_CONFIG_PENDING;
	DEBUG(2, "cmx: <- reader_config (failure)\n");
}





static int reader_event(event_t event, int priority,
			event_callback_args_t *args) {
dev_link_t *link;
reader_dev_t *dev;
int devno;

	DEBUG(3,"cmx: -> reader_event\n");
	link=args->client_data;
	dev=link->priv;
	for(devno=0;devno<CM_MAX_DEV;devno++)
		if (dev_table[devno]==link)
			break;
	if (devno==CM_MAX_DEV)
		return CS_BAD_ADAPTER;
	switch(event) {
	case CS_EVENT_CARD_INSERTION:
		DEBUG(5, "cmx: CS_EVENT_CARD_INSERTION\n");
		link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
		reader_config(link,devno);
		break;
	case CS_EVENT_CARD_REMOVAL:
		DEBUG(5, "cmx: CS_EVENT_CARD_REMOVAL\n");
		link->state &= ~DEV_PRESENT;
		
		break;
	case CS_EVENT_PM_SUSPEND:
		DEBUG(5, "cmx: CS_EVENT_PM_SUSPEND (fall-through to CS_EVENT_RESET_PHYSICAL)\n");
		link->state |= DEV_SUSPEND;
		
	case CS_EVENT_RESET_PHYSICAL:
		DEBUG(5, "cmx: CS_EVENT_RESET_PHYSICAL\n");
		if (link->state & DEV_CONFIG)
		{
		  DEBUG(5, "cmx: ReleaseConfiguration\n");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
		  pcmcia_release_configuration(link->handle);
#else
		  CardServices(ReleaseConfiguration, link->handle);
#endif
		}
		break;
	case CS_EVENT_PM_RESUME:
		DEBUG(5, "cmx: CS_EVENT_PM_RESUME (fall-through to CS_EVENT_CARD_RESET)\n");
		link->state &= ~DEV_SUSPEND;
		
	case CS_EVENT_CARD_RESET:
	        DEBUG(5, "cmx: CS_EVENT_CARD_RESET\n");
		if ((link->state & DEV_CONFIG)){
		  DEBUG(5, "cmx: RequestConfiguration\n");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
		  pcmcia_request_configuration(link->handle, &link->conf);
#else
		  CardServices(RequestConfiguration, link->handle, &link->conf);
#endif
		}
		break;
	default:
		DEBUG(5, "cmx: reader_event: unknown event %.2x\n",
			event);
		break;
	}
	DEBUG(3, "cmx: <- reader_event\n");
	return CS_SUCCESS;
}




static void reader_release(u_long arg) {
dev_link_t *link;
int rc;

	DEBUG(3,"cmx: -> reader_release\n");
	link=(dev_link_t *)arg;
	cmx_reader_release(link->priv); 
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	rc=pcmcia_release_configuration(link->handle);
#else
	rc=CardServices(ReleaseConfiguration,link->handle);
#endif
	if (rc!=CS_SUCCESS)
		DEBUG(5,"cmx: couldn't ReleaseConfiguration reasoncode=%.2x\n",rc);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	rc=pcmcia_release_io(link->handle,&link->io);
#else	
	rc=CardServices(ReleaseIO,link->handle,&link->io);
#endif
	if (rc!=CS_SUCCESS)
		DEBUG(5,"cmx: couldn't ReleaseIO reasoncode=%.2x\n",rc);
	DEBUG(3,"cmx: <- reader_release\n");
}





static dev_link_t *reader_attach(void) {
reader_dev_t *dev;
dev_link_t *link;
client_reg_t client_reg;
int i;

	DEBUG(3,"cmx: reader_attach\n");
	for(i=0;i<CM_MAX_DEV;i++)
		if (dev_table[i]==NULL)
			break;
	if (i==CM_MAX_DEV) {
		printk(KERN_NOTICE "cmx: all devices in use\n");
		return NULL;
	}
	
	DEBUG(5, "cmx: create reader device instance\n");
	dev=kmalloc(sizeof(reader_dev_t), GFP_KERNEL);
	if (dev==NULL) return NULL;



	memset(dev,0,sizeof(reader_dev_t));
	dev->timeout = CCID_DRIVER_MINIMUM_TIMEOUT;
	dev->fTimerExpired = 0;

	link=&dev->link;
	link->priv=dev;

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	link->release.function=&reader_release;
	link->release.data=(u_long)link;
#endif

	link->conf.IntType=INT_MEMORY_AND_IO;
	dev_table[i]=link;

	
	DEBUG(5, "cmx: Register with Card Services\n");
	client_reg.dev_info=&dev_info;
	client_reg.Attributes=INFO_IO_CLIENT | INFO_CARD_SHARE;
	client_reg.EventMask=
		CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
		CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
		CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler=&reader_event;
	client_reg.Version=0x0210;
	client_reg.event_callback_args.client_data=link;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	i=pcmcia_register_client(&link->handle,&client_reg);
#else
	i=CardServices(RegisterClient,&link->handle,&client_reg);
#endif
	if (i)
	{
		cs_error(link->handle,RegisterClient,i);
		reader_detach(link);
		return NULL;
	}
	init_waitqueue_head(&dev->devq);
	return link;
}





static void reader_detach_by_devno(int devno,dev_link_t *link) {
reader_dev_t *dev=link->priv;

	DEBUG(3,"cmx: -> detach_by_devno(devno=%d)\n",devno);
	if (link->state & DEV_CONFIG){
	  DEBUG(5, "cmx: device still configured (try to release it)\n");
	  reader_release((u_long)link);
	}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	  pcmcia_deregister_client(link->handle);
#else
	if (link->handle) CardServices(DeregisterClient,link->handle);
#endif
	dev_table[devno]=NULL;
	DEBUG(5,"cmx: freeing dev=%.8x\n",(unsigned int)dev);
	kfree(dev);
	DEBUG(3, "cmx: <- detach_by-devno\n");
	return;
}

static void reader_detach(dev_link_t *link) {
int i;
	DEBUG(3,"cmx: -> reader_detach(link=%.8x\n",
	 	(unsigned int)link);
	/* find device */
	for(i=0;i<CM_MAX_DEV;i++)
		if (dev_table[i]==link)
			break;
	if (i==CM_MAX_DEV) {
		printk(KERN_WARNING "cmx: detach for unkown device aborted\n");
		return;
	}
	reader_detach_by_devno(i,link);
	DEBUG(3, "cmx: <- reader_detach\n");
	return;
}





#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static int __init cmx_init(void) {
#else
int init_module(void) {
#endif
#ifdef CHECK_VERSION
servinfo_t serv;
#endif

	printk(KERN_INFO "%s\n",version);
#ifdef CHECK_VERSION
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	pcmcia_get_card_services_info(&serv);
#else
	CardServices(GetCardServicesInfo, &serv);
#endif
	if (serv.Revision != CS_RELEASE_CODE) {
		printk(KERN_NOTICE "cmx: Card Services release "
			"does not match.\n");
		return -1;
	}
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
  pcmcia_register_driver(&reader_driver);
#else
	register_pccard_driver(&dev_info, &reader_attach, &reader_detach);
#endif
	major=register_chrdev(0,DEVICE_NAME,&reader_fops);
	if (major<0) {
		printk(KERN_WARNING "cmx: could not get major number\n");
		return -1;
	}
	return 0;
}





#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
static void __exit cmx_exit(void) {
#else
void cleanup_module(void) {
#endif
int i;
	printk(KERN_INFO "cmx unloading\n");
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
  pcmcia_unregister_driver(&reader_driver);  
#else
	unregister_pccard_driver(&dev_info);
#endif
	for(i=0;i<CM_MAX_DEV;i++)
		if (dev_table[i]) reader_detach_by_devno(i,dev_table[i]);
	 unregister_chrdev(major,DEVICE_NAME);
};

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
module_init(cmx_init);
module_exit(cmx_exit);
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,10)
MODULE_LICENSE("Dual BSD/GPL");
#endif
