/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "swidget/STextView.h"

#define DEBUG_SPEED 0

/* I could debug speed of redrawing. Most time (98%) is taken in
  i += font.draw (c, p, dm, &g.array()[i], g.size()-i); */

#if DEBUG_SPEED
#ifndef USE_WINAPI
#include <sys/time.h>
#else
#include <winsock.h>
#include <time.h>
#endif

static struct timeval thatTime;

static void
timerStart()
{
  gettimeofday (&thatTime, 0);
}

static void
timerStop()
{
  struct timeval thisTime;
  gettimeofday (&thisTime, 0);
  if (thisTime.tv_usec < thatTime.tv_usec)
  {
    thisTime.tv_sec--;
    thisTime.tv_usec+=1000000;
  }
  thisTime.tv_sec -= thatTime.tv_sec;
  thisTime.tv_usec -= thatTime.tv_usec;
  int msec = (int) thisTime.tv_sec * 1000 + thisTime.tv_usec/1000;
  fprintf (stderr, "Elapsed time: %d msecs\n", msec);
}
#endif


static unsigned int sane_index (const SV_UINT& array, unsigned int index);

/**
 * The text data is mine. I'll delete it.
 */
STextView::STextView (void)
  : lrpen (SColor (0.0, 0.0, 0.0, 1.0)),
  rlpen (SColor (0.0, 0.0, 1.0, 1.0)),
  underlineColor("red")
{
  clipx = clipy = 0;
  clipw = cliph = 0;
  fontSize = 16.0;
  lineend = true;
  multiline = true;
  recalc ();
  textData.addTextDataListener (this);
}

/**
 * Make a text-view from external text
 * @param utf8 is the utf8 ancoded text
 */
STextView::STextView (const SString& utf8)
  : textData (utf8), lrpen (SColor (0.0, 0.0, 0.0, 1.0)),
  rlpen (SColor (0.0, 0.0, 1.0, 1.0)),
  underlineColor("red")
{
  clipx = clipy = 0;
  clipw = cliph = 0;
  fontSize = 16.0;
  lineend = true;
  multiline = true;
  recalc ();
  textData.addTextDataListener (this);
  textData.clearEvent();
}

STextView::~STextView ()
{
  clearExtents();
}

void
STextView::setClippingArea (int _x, int _y, 
          unsigned int _width, unsigned int _height)
{
  clipx = _x;
  clipy = _y;
  clipw = _width;
  cliph = _height;
}

/**
 * Set the font and recalculate sizes.
 * @param _font is the new font.
 */
void
STextView::setFont (const SString& _font, double _fontSize)
{
  if (_fontSize > 0.0) fontSize = _fontSize;
  font = SFont(_font, fontSize);
  setPen ();
  recalc ();
}

/**
 * Set the size of the font.
 * @param size is the size of the font.
 */
void
STextView::setFontSize (double newsize)
{
  font.setSize(newsize);
  setPen ();
  recalc ();
}

/**
 * Set the pen size according to font size.
 */
void
STextView::setPen ()
{
  double pensize = 1.0;
  double pointsize = font.getSize();
  if (pointsize <= 16)
  {
   pensize = 0.125;
  }
  else if (pointsize <=24)
  {
   pensize = 0.25;
  }
  else if (pointsize <=80)
  {
   pensize = 0.5;
  }
  else if (pointsize <=100)
  {
   pensize = 0.75;
  }
  lrpen.setLineWidth (pensize);
  rlpen.setLineWidth (pensize);
}

/**
 * Set the text alignment align is true if it is right aligned.
 */
void
STextView::setAlignment (bool align)
{
  alignment = align;
}
/**
 */
void
STextView::setMultiline (bool _multiline)
{
  multiline = _multiline;
  recalcBreaks ();
}
/**
 */
bool
STextView::isMultiline () const
{
  return multiline;
}
/**
 * Set the viewport. This is the location that casts to {0;0}
 * @param _viewPort is the new viewport.
 */
void
STextView::setViewPort (const SLocation& _viewPort)
{
  viewPort = _viewPort;
}

const SLocation&
STextView::getViewPort()
{
  return viewPort;
}

/**
 * Resize the component and redraw.
 */
void
STextView::resize (const SDimension& _dimension)
{
  SComponent::resize (_dimension);
//fprintf (stderr, "resized: %u %u (%u %u)\n", 
 // getSize().width, getSize().height, _dimension.width, _dimension.height);
  recalcBreaks ();
}

/**
 * Calcualte a line (y) and charatcer (x) index from a screen position.
 * @param l is the location on the canvas.
 */
STextIndex
STextView::getTextIndex (const SLocation& l)
{
  /* binary search linespan */
  unsigned int    top;
  unsigned int    bottom;
  unsigned int    mid;
  top = lineSpan.size();
  bottom = 0;
  int y = l.y - (location.y + viewPort.y) + 1;
  while (top > bottom)
  {
    mid = (top+bottom)/2;
    unsigned int sindex = lineHeight * sane_index (lineSpan, mid+1);
    if (y == (int) (sindex))
    {
      top = mid;
      break;
    }
    if (y < (int) (sindex))
    {
      top = mid;
      continue;
    }
    bottom = mid + 1;
  }
  unsigned int line = top;
  if (textData.size()<=line)
  {
    if (line > 0 && !textData.isProperLine (line-1))
    {
      return STextIndex (line-1, textData.size(line-1));
    }
  }

  SV_UINT brk; if (textData.size() > line) brk = breaks[line];

  unsigned int currExt=0;
  unsigned int i;
  /* Linear serach. I dont expect long lines. */
  unsigned int bl = true;
  unsigned int offset = lineHeight * sane_index (lineSpan, line);
  SLocation lm(location.x+viewPort.x, location.y+viewPort.y + (int) offset);

  /* optimization possible using brk vector  */
  bool endsline = false;
  for (i=0; i<textData.size(line); i++)
  {
     if (brk[currExt] == i && bl)
     {
       currExt++;
       lm.x = location.x+viewPort.x;
       lm.y = lineHeight * currExt + location.y+viewPort.y + (int) offset;
     }

     if (lm.y + (int) lineHeight  < l.y) continue; 
     bl = false;
     const SGlyph& g = textData.glyphAt (STextIndex (line, i));
     if (g.isLineEnd()) endsline = true;
      
     int nsize = (int) getExtent (line, i);
     /* ---x---1/2---x---- */
     if (lm.x  + nsize/2  > l.x) break;
     lm.x += nsize;
  }
  if (i==textData.size(line) && endsline)
  {
    return STextIndex (line+1, 0);
  }
  return STextIndex (line, i);
}

/**
 * Convert a line and index to a location on the screen
 * @param line is the line number. 
 * @param index is the index.
 */
SLocation
STextView::getTextLocation (const STextIndex& textIndex)
{
  unsigned int line = textIndex.line;
  unsigned int index = textIndex.index;
  unsigned int ll = sane_index (lineSpan, line);
  if (line >= textData.size())
  {
    return SLocation (location.x+viewPort.x, location.y+viewPort.y + lineHeight * ll);
  }
  //fprintf (stderr, "line=%u breaksize=%u\n", line, breaks.size());
  SV_UINT brk = (textData.size() > line) ?  breaks[line] : SV_UINT ();
  /* It may be second or third line. */
  unsigned int i = 0;
  int xoffset = 0;
  int yoffset = 0;
  for (i=0; i<brk.size()-1; i++)
  {
    if (index < brk[i]) break;
    yoffset++;
    xoffset = (int) (brk[i]==0) ? 0 : extents[line][brk[i]-1];
  }
  int si = (int) (index==0) ? 0 : extents[line][index-1];
  
  int iu = si - xoffset;

//fprintf (stderr, "sanme_index=%u xoffset=%u\n", sane_index (ext, index), xoffset);
  return SLocation (location.x+viewPort.x + iu, 
      location.y+viewPort.y + lineHeight * (ll+yoffset));
}

/**
 * This is coming from the SWindowListener
 * @param c is the canvas to draw on.
 * @param x is the upper lect corner.
 * @param y is the upper lect corner.
 * @param width is the width of this event.
 * @param height is the height of this event.
 */
void
STextView::redraw (SWindow *w, int x, int y, 
        unsigned int width, unsigned int height)
{


#if DEBUG_SPEED
  x = getLocation ().x;
  y = getLocation ().y;
  width = getSize().width;
  height = getSize().height;
  timerStart();
#endif

  SLocation lb (x, y);
  SLocation le (x+width, y+height);
  STextIndex index = getTextIndex (lb);
  SLocation l(location.x+viewPort.x, location.y+viewPort.y + (int) lineHeight 
             * (int) sane_index (lineSpan, index.line));
  //fprintf (stderr, "STextView::redraw %d %d %u %u\n", x, y, width, height);
  for (unsigned int i=index.line; i<textData.size(); i++)
  {
    /* clip the height, some glyphs are ot of bounding box  */
    unsigned int cs = drawGlyphLine (w, i, l, lb, le, true);
    l = SLocation (l.x, l.y + lineHeight * cs);
    if (l.y  > location.y + (int) size.height ) break;
    if (l.y  > y + (int)height) break;
  }

#if DEBUG_SPEED
  timerStop();
#endif
}
/**
 * This is coming from the SWindowListener
 * @param c is the canvas to draw on.
 * @param x is the upper lect corner.
 * @param y is the upper lect corner.
 * @param width is the width of this event.
 * @param height is the height of this event.
 */
void
STextView::redraw (SCanvas *c, int x, int y, 
        unsigned int width, unsigned int height)
{


#if DEBUG_SPEED
  x = getLocation ().x;
  y = getLocation ().y;
  width = getSize().width;
  height = getSize().height;
  timerStart();
#endif

  SLocation lb (x, y);
  SLocation le (x+width, y+height);
  STextIndex index = getTextIndex (lb);
  SLocation l(location.x+viewPort.x, location.y+viewPort.y + (int) lineHeight 
             * (int) sane_index (lineSpan, index.line));
  //fprintf (stderr, "STextView::redraw %d %d %u %u\n", x, y, width, height);
  for (unsigned int i=index.line; i<textData.size(); i++)
  {
    unsigned int cs = drawGlyphLine (c, i, l, lb, le, false);
    l = SLocation (l.x, l.y + lineHeight * cs);
    if (l.y  > location.y + (int) size.height ) break;
    if (l.y  > y + (int)height) break;
  }

#if DEBUG_SPEED
  timerStop();
#endif
}

/**
 * Draw a whole line of glyphs.
 * @param c is the canvas to draw to 
 * @param line is the line index to draw.
 * @param l is the beginning upper corner location
 * @param lb is the beginning exposure
 * @param le is the end exposure
 * @return the number of lines drawn.
 */
unsigned int 
STextView::drawGlyphLine (SCanvas* c, unsigned int line, 
  const SLocation& l, const SLocation& lb, const SLocation& le, bool iswindow)
{
  SV_UINT br; if (textData.size()  > line) br = breaks[line];
  unsigned int currExt = 0;
  SLocation lm = l;
  unsigned int ls = textData.size(line);
  unsigned int mycliph = 0;

  /**
   *  set clip to line so that we wont overflow... 
   */
  if (iswindow && clipw != 0 && cliph != 0)
  {
    int myclipy0 = (clipy > lm.y) ? clipy : lm.y;
    int myclipy1 = myclipy0 + lineHeight;
    if (myclipy1 > clipy + (int) cliph)
    {
      myclipy1 = clipy + (int) cliph;
    }
    mycliph = (myclipy1 > myclipy0) ? myclipy1-myclipy0 : 0;
    if (mycliph)
    {
      ((SWindow*)c)->setClippingArea (clipx, myclipy0, clipw, mycliph);
    }
  }
  for (unsigned int i=0; i<ls; i++)
  {
     if (br[currExt] == i)
     {
       currExt++;
       lm.x = l.x;
       lm.y = lineHeight * currExt + l.y;
       if (iswindow && clipw != 0 && cliph != 0)
       {
         int myclipy0 = (clipy > lm.y) ? clipy : lm.y;
         int myclipy1 = myclipy0 + lineHeight;
         if (myclipy1 > clipy + (int) cliph)
         {
           myclipy1 = clipy + (int) cliph;
         }
         mycliph = (myclipy1 > myclipy0) ? myclipy1-myclipy0 : 0;
         if (mycliph)
         {
            ((SWindow*)c)->setClippingArea (clipx, myclipy0, clipw, mycliph);
         }
       }
     }
     const SGlyph& g = textData.glyphAt (STextIndex (line, i));
     unsigned int e = getExtent (line, i);
     /* is it drawable ? */
     if (lm.x < le.x  && lb.x < lm.x + (int) e 
        && lm.y <  le.y && lb.y < lm.y + (int) lineHeight)
     {
       if (!iswindow || mycliph) drawGlyph (c, g, lm, e);
     }
     else
     {
       //drawGlyph (c, g, lm);
     }
     lm.x += e;
  }
  if (iswindow && clipw!=0 && cliph !=0)
  {
       ((SWindow*)c)->setClippingArea (clipx, clipy, clipw, cliph);
  }
  return currExt+1;
}

/**
 * Draw one signle glyph on the screen.
 * @param c is the canvas to draw to
 * @param g is the glyph.
 * @param l is the location of the glyph.
 * @return the length of the text drawn
 */
void
STextView::drawGlyph (SCanvas* c, const SGlyph& g, 
  const SLocation& l, unsigned int ext)
{
  if (!lineend && g.isLineEnd()) return;

  SS_Matrix2D dm;
  dm.y1 = -dm.y1; /* updown */
  dm.translate (0, font.ascent ());
  dm.translate ((double)l.x, (double)l.y);
  SPen p (lrpen);
  if (!g.lr)
  {
    p = rlpen;
  }
  if (g.selected)
  {
     SColor fg = p.getForeground();
     SColor bg = p.getBackground();
     p.setForeground (bg);
     p.setBackground (fg);
     c->bitfill (fg, l.x, l.y, ext, lineHeight);
  }

  font.draw (c, p, dm, g);

  /* FIXME: add underline */
  if (g.underlined)
  {
    unsigned int w = ext;
    unsigned int h = lineHeight/24+1;
    unsigned int base =  (lineAscent + h >= lineHeight) ?
         lineHeight -1 : lineAscent + h;
    /* construct a square */
    c->bitfill (underlineColor, l.x, l.y + (int) base - h, w, h);
  }
}

/**
 * This is called by the STextData
 */
void
STextView::textChanged (void* src, const STextDataEvent& event)
{
  STextIndex tb = event.start;
  /* get the bounds */
  STextIndex te = textData.getMaxTextIndex (event);
 // fprintf (stderr, "textChanged: from %u %u to %u %u (%u %u)\n", 
  //  tb.line, tb.index, te.line, te.index, event.remaining.line, event.remaining.index);
  if (textData.size()==0)
  {
//fprintf (stderr, "zero size data\n");
     recalc(); 
     SWindow* w = getWindow ();
     if (w)
     {
       w->redraw (true, location.x, location.y, size.width, size.height);
     }
     return;
  }
  unsigned int oldsize = lineSpan.size();
  unsigned int oldspan = sane_index (lineSpan, oldsize);

  /* change in text contents */
  if (!event.attribute)
  {
    /* only partial */
    if (multiline)
    {
      //fprintf (stderr, "PARTIAL\n");
      recalc (tb.line, te.line+1, (int)textData.size() - (int) lineSpan.size());
    }
    else
    {
      recalc();
    }
  }
  SWindow* w = getWindow();
  if (w == 0)
  {
    return;
  }
  unsigned int newsize = lineSpan.size();
  unsigned int newspan = sane_index (lineSpan, newsize);

  SLocation lb = getTextLocation (tb);
  /* till end */
  SLocation le = getTextLocation (
     STextIndex (te.line, textData.size(te.line)));
  //le.x = location.x+viewPort.x + (int)size.width;
  le.x = location.x + (int)size.width;

  if (event.attribute)
  {
    /* single */
    if (lb.y == le.y)
    {
//fprintf (stderr, "lb.x=%d le.x=%d (%u, %u)\n", lb.x, le.x, tb.index, te.index);
      if (le.x - lb.x > 0)
      {
        w->redraw (true, lb.x, lb.y, le.x - lb.x, (int)lineHeight + (le.y-lb.y));
      }
    }
    else // multi
    {
       w->redraw (true, location.x, lb.y, 
              size.width, (int)lineHeight + (le.y-lb.y));
    }
  }
  /* is this a one-line thingy? */
  else if (tb.line == te.line && oldsize == newsize && oldspan == newspan)
  {
     if (lb.y == le.y)
     {
       if (le.x - lb.x > 0)
       {
         w->redraw (true, lb.x, lb.y, le.x - lb.x, (int)lineHeight + (le.y-lb.y));
       }
     }
     else
     {
      w->redraw (true, location.x, lb.y, size.width, (int)lineHeight + (le.y-lb.y));
     }
  }
  else
  {
    if (lb.y < location.y + (int) size.height)
    {
      w->redraw (true, location.x, lb.y, 
          size.width, location.y + (int) size.height -lb.y);
    }
  }
}

/**
 * Caclulate the extent as one line.
 * @param line is the line to calculate.
 * @return an allocate array of line extents
 */
SS_UINT*
STextView::getExtent (unsigned int line)
{
  SS_UINT* ret = new SS_UINT[textData.size(line) + 1];
  unsigned int le=0;
  for (unsigned int i=0; i<textData.size(line); i++)
  {
     le  += getExtent (textData.glyphAt (STextIndex (line, i)));
     ret[i] = (SS_UINT) le;
//fprintf (stderr, "retex[%u]=%u\n", i, ret[i]);
  }
  /* udpate this. */
  return ret;
}

/**
 * Caclulate the extent as one line.
 * @param line is the line to calculate.
 * @return an allocate array of line extents
 */
SS_UINT*
STextView::getExtent (unsigned int line, SH_UINT* cache)
{
  SS_UINT* ret = new SS_UINT[textData.size(line) + 1];
  unsigned int le=0;
  unsigned int ce=0;
  for (unsigned int i=0; i<textData.size(line); i++)
  {
    const SGlyph& g = textData.glyphAt (STextIndex (line, i));
    ce = cache->get (g.charKey());
    if (ce ==0)
    {
      ce = getExtent (g);
      /* shaped glyphs extent can change. */
      if (g.getShapeArray()==0) cache->put (g.charKey(), ce);
    }
    le  += ce;
    ret[i] = (SS_UINT) le;
//fprintf (stderr, "retex[%u]=%u\n", i, ret[i]);
  }
  /* udpate this. */
  return ret;
}

/**
 * get the cached extent.
 */
unsigned int
STextView::getExtent (unsigned int line, unsigned int index) const
{
  unsigned int prev = (index==0) ? 0 : extents[line][index-1];
  return extents[line][index] - prev;
  //return getExtent (textData.glyphAt (STextIndex (line, index)));
}

/**
 * Return the extents of a single glyph.
 * @param g is the glyph to measure.
 */
unsigned int 
STextView::getExtent (const SGlyph& g)
{
  return (unsigned int) (0.5 + font.width (g));
}

/**
 * Break the stuff into lines. each index is a new line.
 * @param line is the line
 * @return indeces in extent - newline starters.
 */
SV_UINT
STextView::breakExtent (const unsigned int line) const
{
  SV_UINT u;
  if (!multiline)
  {
    u.append (textData.size(line));
    return SV_UINT(u);
  }
  unsigned int ms = 0;
  SS_UCS4* pext = (line < textData.size()) ? extents[line] : 0;
  for (unsigned int i=0; i<textData.size(line); i++)
  {
    unsigned int w = (unsigned int) pext[i] - ms;
    /* move it to the next line.*/
    if (i>0 && w >= size.width)
    {
       //if (!lineend || i+1 < ext.size())
       if (lineend || i+1<textData.size(line))
       {
         u.append (i);
         ms = (unsigned int) pext[i-1];
       }
       else if (i+1==textData.size(line)) 
       {
         const SGlyph& g = textData.glyphAt (STextIndex (line, i));
         if (!g.isLineEnd())
         {
           u.append (i);
           ms = (unsigned int) pext [i-1];
         }
       }
    }
  }
  u.append (textData.size(line));
  return SV_UINT(u);
}

/**
 * clear the cached extent buffer
 */
void
STextView::clearExtents()
{
  for (unsigned int i=0; i<extents.size(); i++)
  {
    delete extents[i];
  }
  extents.clear();
}

/**
 * Walk through the text and remake the linespan.
 * Recalculate the preferred sizes.
 */
void
STextView::recalc ()
{
  lineHeight = (unsigned int ) (font.ascent() + font.descent() + font.gap());
//fprintf (stderr, "lineHeight= %g + %g + %g = %u\n",
 //    font.ascent() , font.descent() , font.gap() , lineHeight);
  lineAscent = (unsigned int) font.ascent();
  clearExtents ();
  //fprintf (stderr, "recalc all\n");
  SH_UINT hint;
  for (unsigned int i=0; i<textData.size(); i++)
  {
    SS_UINT* e = getExtent (i, &hint);
    extents.append (e);
  }
  recalcBreaks ();
}

/**
 * recalculate partially. This is used for multi-line stuff
 * to make it more efficient.
 * @param from is the starting index.
 * @param until is the index before last
 * @paran addcount show how many lines were added. can be negative (removed)
 */
void
STextView::recalc (unsigned int from, unsigned int until, int addcount)
{
  //unsigned int longestline = 1;
  unsigned int sum = sane_index (lineSpan, from);
  int mycount=0;
  unsigned int i=0;
  SH_UINT cache;
  for (i=from; i<textData.size() && i<until; i++)
  {
    SS_UINT* e = getExtent (i, &cache);
    extents.insert (i, e);

    SV_UINT b = breakExtent (i);
    sum += b.size ();

    //fprintf (stderr, "inserting %u - size %u %u\n", i, b.size(), b[0]);
    lineSpan.insert (i, sum);
    breaks.insert (i, b);
    mycount++;
  }

  //fprintf (stderr, " recalc: added %d req: %d\n", mycount, addcount);
  unsigned int removesum = sum;
  for (int j=0; j<mycount-addcount; j++)
  {
    /* yes i ! */
    //fprintf (stderr, " removing %u\n", i);
    removesum = lineSpan[i];
    lineSpan.remove (i);
    breaks.remove (i);

    delete extents[i];
    extents.remove (i);
  }
  /* recalibrate the whole linespan array */
  if (removesum != sum)
  {
    //fprintf (stderr, " recalibrate %u -> %u\n", removesum, sum);
    while (i < textData.size())
    {
      unsigned int s = lineSpan[i];
      if (removesum > sum)
      {
         s -= removesum-sum;
      }
      else
      {
         s += sum-removesum;
      }
      lineSpan.replace (i, s);
      i++;
    }
  }
  preferredSize.height = (textData.size()==0) ? lineHeight
      : textData.size() *  lineHeight;
}

/**
 * economic recalc - rebuild only the 'breaks'
 */
void
STextView::recalcBreaks ()
{
  unsigned int sum = 0;
  unsigned int longestline = 1;
  lineSpan.clear();
  breaks.clear();
  for (unsigned int i=0; i<textData.size(); i++)
  {
    SV_UINT b = breakExtent (i);
    sum += b.size ();
    breaks.append (b);
    lineSpan.append (sum);
    /* NO ZERO SIZE LINES ARE ALLOWED BY DATAMODEL ! */
    //fprintf (stderr, "size %u\n", textData.size(i));
    unsigned int ls = (unsigned int) extents[i][textData.size(i)-1];
    if (ls > longestline) longestline = ls;
  }
  /* Recalculate preferred size */
  preferredSize.width = longestline;
  preferredSize.height = (textData.size()==0) ? lineHeight
      : textData.size() *  lineHeight;
}

/**
 * Set the background.
 * @param bg is the new background
 */
void
STextView::setBackground (const SColor& bg)
{
  lrpen.setBackground (bg);
  rlpen.setBackground (bg);
}

/**
 * Set the foreground.
 * @param fg is the new foreground
 */
void
STextView::setForeground (const SColor& rlfg, const SColor& lrfg)
{
  lrpen.setForeground (rlfg);
  rlpen.setForeground (lrfg);
}
const SColor&
STextView::getBackground ()
{
  return lrpen.getBackground();
}

const SColor&
STextView::getForeground (bool lr)
{
  return (lr) ? lrpen.getForeground() : rlpen.getForeground();
}

/**
 * If show <- newline characters
 * @param _lineend is true if lineend is shown.
 */
void
STextView::setLineEndMark (bool _lineend)
{
  lineend = _lineend;
  recalc();
  SWindow* w = getWindow();
  if (w == 0)
  {
    return;
  }
  if (!w->isVisible()) return;
  w->redraw (true, location.x, location.y, size.width, size.height);
}

/**
 * Is new line shown?
 * @return true if newline characters are shown.
 */
bool
STextView::getLineEndMark () const
{
  return lineend;
}

/**
 * calculate the height of the document.
 */
unsigned int 
STextView::getDocumentHeight() const
{
  if (textData.size() == 0)
  {
    return lineHeight;
  }
  unsigned int fheight = lineSpan[textData.size()-1];
  if (textData.isProperLine (textData.size()-1))
  {
    fheight += 1;
  }
  return (fheight * lineHeight);
}

/**
 * return the 'sane index'.
 * That is, at index 0 it should be 0
 * at index at array->size() is should be the last element.
 */
static unsigned int
sane_index (const SV_UINT& array, unsigned int index)
{
  if (index == 0) return  0;
  return array[index-1];
}

void
STextView::setUnderlineColor (const SColor& c)
{
  underlineColor = c;
}
