/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <swidget/SCaret.h>

/**
 * Create a caret.
 * A caret can have two directions : < or >
 */
SCaret::SCaret (void)
  : lrpen (SColor (0.0, 0.0, 0.0, 1.0)), 
  rlpen (SColor (0.0, 0.0, 1.0, 1.0))
{
  lr = '>';
  showCaret = true;
  skipBlink = false;
  state = true;
  lrpen.setBackground (SColor(1.0, 1.0, 1.0, 1.0));
  rlpen.setBackground (SColor(1.0, 1.0, 1.0, 1.0));
  timer = 0;
}

SCaret::~SCaret ()
{
  if (timer) delete timer;
}

/**
 * Set the direction of this cursor.
 * @param lr is true if we are an lr cursor.
 */
void
SCaret::setDirection (char _lr)
{
  lr = _lr;
  redraw ();
}

char
SCaret::getDirection() const
{
  return lr;
}

/**
 * Set the background.
 * @param bg is the new background
 */
void
SCaret::setBackground (const SColor& bg)
{
  lrpen.setBackground (bg);
  rlpen.setBackground (bg);
}

/**
 * Set the foreground.
 * @param fg is the new foreground
 */
void
SCaret::setForeground (const SColor& rlfg, const SColor& lrfg)
{
  lrpen.setForeground (rlfg);
  rlpen.setForeground (lrfg);
}

/**
 * The caret usually takes two glyp positions.
 * The middle is the insertion point
 */
void
SCaret::redraw (SCanvas* canvas)
{
  if (isSaneSize())
  {
    redraw (canvas, location.x, location.y, size.width, size.height);
  }
}

/**
 * The caret usually takes two glyp positions.
 * The middle is the insertion point
 * caret is draw in the middle.
 */
void
SCaret::redraw (SCanvas* canvas, int x, int y, unsigned int width, unsigned int height)
{
  if (!isOn()) return;
  if (state == false) return;
  if (!isSaneSize()) return;

  double middleX = (double)getLocation().x;
  double middleY = (double)getLocation().y + (double)getSize().height/2;
  double dx = (double)getSize().width/3;
  double dy = (double)getSize().height/3;
  char a[64];
  sprintf (a, "caret=%u %u %c;", 
         getSize().width, getSize().height, lr);
  if (lr == '>')
  {
    /* should return false */
    if (!canvas->newpath (middleX, middleY - dy +1, a))
    {
      canvas->moveto (middleX, middleY - dy +1);
      canvas->lineto (middleX + dx -1, middleY);
      canvas->lineto (middleX, middleY + dy -1);
      canvas->closepath ();
    }
    canvas->fill (lrpen);
  }
  else
  {
    /* should return false */
    if (!canvas->newpath (middleX, middleY - dy +1, a))
    {
      canvas->moveto (middleX, middleY - dy +1);
      canvas->lineto (middleX - dx +1, middleY);
      canvas->lineto (middleX, middleY + dy -1);
      canvas->closepath ();
    }
    canvas->fill (rlpen);
  }
}

/**
 * Show or hide the caret
 */
void
SCaret::on (bool _on)
{
  if (_on != showCaret) redraw ();
  showCaret = _on;
  if (state != _on) redraw ();
  state = _on;
  skipBlink = true;
}

/**
 * Request a redraw from parent if possible.
 * The redraw will happen async, later.
 */
void
SCaret::redraw ()
{
  SWindow* w = getWindow();
  if (w != 0)
  {
    if (isSaneSize())
    {
      w->redraw (true, location.x - (int) size.width/2, location.y,
         size.width,  size.height);
    }
  }
}

/**
 * Show if caret is shown
 */
bool
SCaret::isOn () const
{
  return showCaret;
}

/**
 * Show if it is animating.
 */
bool
SCaret::isAnimating() const
{
  return timer!=0;
}

/**
 * The timer event
 */
bool
SCaret::timeout (const SEventSource* s)
{
  if (timer == 0)
  {
    fprintf (stderr, "strange. timeout..\n");
    return false;
  }
  if (!showCaret)
  {
    return true;
  }
  if (skipBlink)
  {
    skipBlink = false;
    return true;
  }
  state = !state;
  redraw ();
  return true;
}

/**
 * Resize the component
 * @param d is the new size
 */
void 
SCaret::resize(const SDimension& d)
{
  redraw ();
  SComponent::resize (d);
  redraw ();
}

/**
 * Move the component
 * @param l is the new location
 */
void 
SCaret::move(const SLocation& l)
{
  redraw ();
  SComponent::move (l);
  redraw ();
}

/**
 * Do bliking animation of 
 * @param _animate is true. 
 * stop animation otherwise.
 */
void
SCaret::animate (bool _animate)
{
  if ((timer!=0)==_animate) return;
  if (timer) delete timer;
  timer = 0;
  if (_animate)
  {
    timer = STimer::newTimer(500, this);
    skipBlink = false;
  }
  state = showCaret;
  redraw ();
}

/**
 * return true if it has a possible size.
 */
bool
SCaret::isSaneSize()
{
  SLocation up = getLocation() + getSize();
  /* out */
  if (up.x < 0 || up.y < 0) return false;
  /* out */
  if (up.x > 8000 || up.y > 8000) return false;
  return true;
}
