/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "stoolkit/SCluster.h"
#include "stoolkit/SProperties.h"
#include "stoolkit/SUniMap.h"

static unsigned int 
getIndicCluster (unsigned int scriptcode, 
  const SV_UCS4& unicode, 
  unsigned int index, SV_UCS4* ret, int* finished);

static SUniMap* clusters;
static SUniMap* indic;
static SProperties*  ligatureUnics;
static SProperties*  ligatureClust;

static SProperties*  ligatureCache;
static SS_UCS4  counters[SD_SCRIPT_MAX];

static SS_UCS4 nextLigature (unsigned int script, 
   const SS_UCS4* unicode, unsigned int length);

static void initLigatures();

SString yuditClusterError;

/**
 * Try to form a cluster - an abstract glyphs that can 
 * be broken apart once made. It can be rendered by 
 * a font that contains glyphs and ligatureUnics by subdividing
 * the cluster. The cluster is in memory order - 
 * vowels are place on the appropriate side.
 * @param ucs4 is the input vector.
 * @param i is the index in this vector - next character.
 * @param finished is set to 0 if more data is needed
 *  this parameter can be null.
 * @return the new index in ucs4.
 */
unsigned int
getCluster (const SV_UCS4& ucs4, 
   unsigned int index, SV_UCS4* retchar, int *finished)
{
  if (finished) *finished = -1;
  initLigatures();
  int scriptcode = getUnicodeScript (ucs4[index]);
  if (scriptcode < 0) return index;
  unsigned int ret = index;
  yuditClusterError.clear();
  switch (scriptcode)
  {
  case SD_DEVANAGARI: 
  case SD_BENGALI: 
  case SD_GURMUKHI: 
  case SD_GUJARATI: 
  case SD_ORIYA: 
  case SD_KANNADA:
  case SD_MALAYALAM:
  case SD_SINHALA:
  case SD_TELUGU:
    if (!indic->isOK()) break;;
    ret = getIndicCluster (
         (unsigned int)scriptcode, ucs4, index, retchar, finished);
    break;
  case SD_TAMIL:
    if (!clusters->isOK()) break;
    ret = clusters->lift (ucs4, index, true, retchar);
    break;
  }
  if (finished ==0 && yuditClusterError.size())
  {
    // If you want to debug thinsg uncoment this.
    //fprintf (stderr, "SCluster.cpp:%*.*s\n", SSARGS(yuditClusterError));
  }
  return ret;
}

/**
 * Get cluster for North Indian Devanagari-like scripts
 * The cluster is rendered and treated together. It has
 * a unicode and a seperated memory representation.
 * Memory representation is only used for fallback rendering.
 * A cluster is
 * a) Consonant
 * b) Consonant + Halant
 * c) Consonant + Halant + ZWJ
 * d) Consonant + Nukta + Halant
 * e) Consonant + Nukta + Halant + ZWJ
 * f) Independent Vowel
 * g) Independent Vowel + Vowel
 * h) [b|c|d|e]*
 * i) [b|c|d|e]* a
 * j) [b|c|d|e]* Vowel
 * k) [a-i] ending with Modifier
 * l) [a-i] ending with ZWNJ
 * @param scriptcode is one of the scripts (Hard-Coded)
 * @return index if nothing was lifted off vector, return
 * the number of unicode characters + index otherwise.
 * append the output cluster to ret, last element is ligature
 * code - if any.
 * @param finished is set to 1 if exact match happens
 *                           0 is not yet finished
 *                          -1 if illegal sequence start.
 * It also sets yuditClusterError to an appripriate string.
 */
static unsigned int 
getIndicCluster (unsigned int scriptcode, 
  const SV_UCS4& unicode, unsigned int index, SV_UCS4* ret, int* finished)
{
  unsigned int usize = unicode.size();
  unsigned int i;
  if (finished) *finished = 1;
  char prevchartype = -1;
  SS_UCS4 nextLig = 0;
  for (i=index;i<usize; i++)
  { 
   SS_UCS4 next = unicode[i];
   char chartype = (char) indic->encode (next);
//fprintf (stderr, "getIndicCluster=%u %d\n", next, chartype);
   unsigned int sc = getUnicodeScript (next);
   if (sc!=scriptcode && chartype != SD_INDIC_ZWNJ && chartype != SD_INDIC_ZWJ)
   {
     if (ret->size()==0)
     {
       /* can not start with it */
       if (finished) *finished=-1;
     }
     nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
     if (nextLig) ret->append (nextLig);
     return i; 
   }
   switch (chartype)
   {
   case SD_INDIC_INDEP_VOWEL:
     ret->append (next);
     if (i+1 < usize)
     { 
       SS_UCS4 n = unicode[i+1];
       char ct = (char) indic->encode (n);
       if (ct != SD_INDIC_BOTTOM_VOWEL
            && ct != SD_INDIC_TOP_VOWEL 
            && ct != SD_INDIC_LEFT_VOWEL
            && ct != SD_INDIC_LEFT_RIGHT_VOWEL
            && ct != SD_INDIC_RIGHT_VOWEL
            && ct != SD_INDIC_MODIFIER
            && ct != SD_INDIC_HALANT)
       {
          if (ret->size()==1) return index;
          nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index+1);
          if (nextLig) ret->append (nextLig);
          return i+1;
       }
     }
     break;
   case SD_INDIC_LEFT_VOWEL:
     if (ret->size()==0)
     {
       /* can not start with it */
       if (finished) *finished=-1;
       yuditClusterError = "Cluster should not start with dependent vowel.";
       return index; 
     }
     if (prevchartype != SD_INDIC_CONSONANT 
           && prevchartype != SD_INDIC_HALANT
           && prevchartype != SD_INDIC_NUKTA
           && prevchartype != SD_INDIC_INDEP_VOWEL)
     {
       yuditClusterError = "Dependent vowel should be preceded by consonant, nukta, halant or independent vowel.";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     if (scriptcode == SD_MALAYALAM)
	ret->insert (ret->size()-1, next);
     else ret->insert (0, next);
     if (i+1 < usize)
     { 
       SS_UCS4 n = unicode[i+1];
       char ct = (char) indic->encode (n);
       if (ct != SD_INDIC_MODIFIER)
       {
         nextLig = nextLigature (scriptcode, &unicode.array()[index],i-index+1);
         if (nextLig) ret->append (nextLig);
         return i+1;
       }
     }
     break;
   case SD_INDIC_LEFT_RIGHT_VOWEL:
     if (ret->size()==0)
     {
       /* can not start with it */
       if (finished) *finished=-1;
       yuditClusterError = "Cluster should not start with dependent vowel.";
       return index; 
     }
     if (prevchartype != SD_INDIC_CONSONANT 
           && prevchartype != SD_INDIC_HALANT
           && prevchartype != SD_INDIC_NUKTA
           && prevchartype != SD_INDIC_INDEP_VOWEL)
     {
       yuditClusterError = "Dependent vowel should be preceded by consonant, nukta, halant or independent vowel.";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     /* this will be the fallback rendering */
     {
       SS_UCS4 l = getLRVowelLeft (next);
       SS_UCS4 r = getLRVowelRight (next);
       if (l && r)
       {
         if (scriptcode == SD_MALAYALAM)
	    ret->insert (ret->size()-1, l);
	 else ret->insert (0, l);
         ret->append (r);
       }
       else
       {
         ret->append (next);
       }
     }
     if (i+1 < usize)
     { 
       SS_UCS4 n = unicode[i+1];
       char ct = (char) indic->encode (n);
       if (ct != SD_INDIC_MODIFIER)
       {
         nextLig = nextLigature (scriptcode, &unicode.array()[index],i-index+1);
         if (nextLig) ret->append (nextLig);
         return i+1;
       }
     }
     break;
   case SD_INDIC_MODIFIER:
     if (ret->size()==0)
     {
       /* can not start with it */
       yuditClusterError = "Cluster should not start with a modifier.";
       if (finished) *finished=-1;
       return index; 
     }
     if (     prevchartype != SD_INDIC_INDEP_VOWEL 
           && prevchartype != SD_INDIC_TOP_VOWEL
           && prevchartype != SD_INDIC_BOTTOM_VOWEL
           && prevchartype != SD_INDIC_LEFT_VOWEL
           && prevchartype != SD_INDIC_LEFT_RIGHT_VOWEL
           && prevchartype != SD_INDIC_RIGHT_VOWEL
           && prevchartype != SD_INDIC_CONSONANT
	   && prevchartype != SD_INDIC_NUKTA)
     {
       nextLig = nextLigature (scriptcode, 
               &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     ret->append (next);
     nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index+1);
     if (nextLig) ret->append (nextLig);
     return i +1;

   case SD_INDIC_SIGN:
     if (ret->size()==0)
     {
       /* can start with it */
       // if (finished) *finished=-1;
       return index; 
     }
     ret->append (next);
     nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index+1);
     if (nextLig) ret->append (nextLig);
     return i+1;

   case SD_INDIC_RIGHT_VOWEL:
   case SD_INDIC_TOP_VOWEL:
   case SD_INDIC_BOTTOM_VOWEL:
     if (ret->size()==0)
     {
       /* can not start with it */
       yuditClusterError = "Cluster should not start with dependent vowel.";
       if (finished) *finished=-1;
       return index; 
     }
     if (prevchartype != SD_INDIC_CONSONANT 
           && prevchartype != SD_INDIC_HALANT
           && prevchartype != SD_INDIC_NUKTA
	   && prevchartype != SD_INDIC_INDEP_VOWEL)
     {
        yuditClusterError = "Dependent vowel should be preceded by consonant, nukta, halant or independent vowel.";
       nextLig = nextLigature (scriptcode, 
               &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     ret->append (next);
     if (i+1 < usize)
     { 
       SS_UCS4 n = unicode[i+1];
       char ct = (char) indic->encode (n);
       if (ct != SD_INDIC_MODIFIER)
       {
         nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index+1);
         if (nextLig) ret->append (nextLig);
         return i +1;
       }
     }
     break;
   case SD_INDIC_CONSONANT:
     if (ret->size() > 0 && prevchartype != SD_INDIC_HALANT 
           && prevchartype != SD_INDIC_ZWJ)
     {
       yuditClusterError = "Consonant should be preceded by halant or nukta or ZWJ";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     ret->append (next);
     if (i+1 < usize)
     { 
       SS_UCS4 n = unicode[i+1];
       char ct = (char) indic->encode (n);
       if (ct != SD_INDIC_HALANT 
            && ct != SD_INDIC_NUKTA
            && ct != SD_INDIC_MODIFIER
            && ct != SD_INDIC_BOTTOM_VOWEL
            && ct != SD_INDIC_TOP_VOWEL 
            && ct != SD_INDIC_LEFT_VOWEL
            && ct != SD_INDIC_LEFT_RIGHT_VOWEL
            && ct != SD_INDIC_RIGHT_VOWEL)
       {
          if (ret->size()==1) return index;
          nextLig = nextLigature (scriptcode,&unicode.array()[index],i-index+1);
          if (nextLig) ret->append (nextLig);
          return i+1;
       }
     }
     break;
   case SD_INDIC_ZWNJ:
     if (ret->size()==0)
     {
       /* can not start with it */
       yuditClusterError = "Cluster can not start with a ZWNJ.";
       if (finished) *finished=-1;
       return index; 
     }
     if (prevchartype != SD_INDIC_HALANT) 
     {
       yuditClusterError = "ZWNJ should be preceded by a halant.";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index+1);
     if (nextLig) ret->append (nextLig);
     return i+1;
   case SD_INDIC_NUKTA:
     if (ret->size()==0)
     {
       /* can not start with it */
       yuditClusterError = "Cluster can not start with a nukta.";
       if (finished) *finished=-1;
       return index; 
     }
     if (prevchartype != SD_INDIC_CONSONANT)
     {
       yuditClusterError = "Nukta should be preceded by a consonant.";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     ret->append (next);
     if (i+1 < usize)
     { 
       SS_UCS4 n = unicode[i+1];
       char ct = (char) indic->encode (n);
       if (ct != SD_INDIC_HALANT 
            && ct != SD_INDIC_MODIFIER
            && ct != SD_INDIC_BOTTOM_VOWEL
            && ct != SD_INDIC_TOP_VOWEL 
            && ct != SD_INDIC_LEFT_VOWEL
            && ct != SD_INDIC_LEFT_RIGHT_VOWEL
            && ct != SD_INDIC_RIGHT_VOWEL)
       {
          if (ret->size()==1) return index;
          nextLig = nextLigature (scriptcode,&unicode.array()[index],i-index+1);
          if (nextLig) ret->append (nextLig);
          return i+1;
       }
     }
     break;
   case SD_INDIC_ZWJ:
     if (ret->size()==0)
     {
       /* can not start with it */
       yuditClusterError = "Cluster can not start with a ZWJ.";
       if (finished) *finished=-1;
       return index; 
     }
     if (prevchartype != SD_INDIC_HALANT) 
     {
       yuditClusterError = "ZWJ should be preceded by a halant.";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     ret->append (next);
     break;
   case SD_INDIC_HALANT:
     if (ret->size()==0)
     {
       /* can not start with it */
       yuditClusterError = "Cluster can not start with a halant.";
       if (finished) *finished=-1;
       return index; 
     }
     if (prevchartype != SD_INDIC_INDEP_VOWEL 
           && prevchartype != SD_INDIC_CONSONANT
           && prevchartype != SD_INDIC_NUKTA)
     {
       yuditClusterError = "Halant should be preceded by an independent vowel, a consonant or nukta.";
       nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
       if (nextLig) ret->append (nextLig);
       return i;
     }
     ret->append (next);
     break;
   default:
     if (ret->size()==0)
     {
       if (finished) *finished=1;
       return index; 
     }
     nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
     if (nextLig) ret->append (nextLig);
     return i;
    break;
   }
   prevchartype = chartype;
  }
  if (finished) *finished = 0;
  if (ret->size()>1)
  {
     nextLig = nextLigature (scriptcode, &unicode.array()[index], i-index);
     if (nextLig) ret->append (nextLig);
     return i;
  }
  ret->clear();
  return index;
}


/**
 * Generate a next ligature number if it still does not exist
 */
static SS_UCS4 nextLigature (unsigned int script, 
   const SS_UCS4* unicode, unsigned int length)
{
  initLigatures ();
  if (length<2) return 0;

  SString key = SString((char*)unicode, sizeof (SS_UCS4) * length);
  const SString* cac = ligatureCache->get (key);
  SS_UCS4 liga;
  if (cac && cac->size()==sizeof (SS_UCS4))
  {
    liga = *(SS_UCS4*) (cac->array());
    return liga;
  }
  liga = counters[script];
  /* check overflow */
  if ((liga & 0xffff) == 0xffff) return 0;
  liga++;
  counters[script] = liga;
  /* FIXME: check overflow */
  SString vle = SString((char*)&liga, sizeof (SS_UCS4));
  ligatureCache->put (key, vle);
  //fprintf (stderr, "New Ligature[%d]=%X\n", script, liga);
  return liga;
}

int
getUnicodeScript (SS_UCS4 comp)
{
  if (comp < 0x0900 ) return -1; 
  if (comp < 0x0980) return SD_DEVANAGARI;
  if (comp < 0x0A00) return SD_BENGALI;
  if (comp < 0x0A80) return SD_GURMUKHI;
  if (comp < 0x0B00) return SD_GUJARATI;
  if (comp < 0x0B80) return SD_ORIYA;
  if (comp < 0x0C00) return SD_TAMIL;
  if (comp < 0x0C80) return SD_TELUGU;
  if (comp < 0x0D00) return SD_KANNADA;
  if (comp < 0x0D80) return SD_MALAYALAM;
  if (comp < 0x0E00) return SD_SINHALA;
  return -1;
}
/**
 * return true if this is covered
 */
bool
isCoveredScipt (SS_UCS4 comp, int sc)
{
  switch (sc)
  {
  case SD_YUDIT: return false;
  case SD_DEVANAGARI: return (comp>=0x0900 && comp<0x0980);
  case SD_BENGALI: return (comp>=0x0980 && comp<0x0a00);
  case SD_GURMUKHI: return (comp>=0x0a00 && comp<0x0a80);
  case SD_GUJARATI: return (comp>=0x0a80 && comp<0x0b00);
  case SD_ORIYA: return (comp>=0x0b00 && comp<0x0b80);
  case SD_TAMIL: return (comp>=0x0b80 && comp<0x0c00);
  case SD_TELUGU: return (comp>=0x0c00 && comp<0x0c80);
  case SD_KANNADA: return (comp>=0x0c80 && comp<0x0d00);
  case SD_MALAYALAM: return (comp>=0x0d00 && comp<0x0d80);
  case SD_SINHALA: return (comp>=0x0d80 && comp<0x0e00);
  }
  return false;
}

/**
 * Put ligature away to remember
 */
void
putLigatureUnicode (SS_UCS4 ligature, const SS_UCS4* buffer, unsigned int bufsize)
{
  if (ligature <=  0x80000000 || ligature >= 0xA0000000) return;
  initLigatures();
  SString key ((char*)& ligature, sizeof (SS_UCS4));
  const SString* ret = ligatureUnics->get (key);
  if (ret) return; /* already there */
  ligatureUnics->put (key, SString((char*)buffer, bufsize * sizeof (SS_UCS4)));
}

static void
initLigatures()
{
  if (ligatureUnics == 0)
  {
    clusters = new SUniMap("cluster");
    CHECK_NEW (clusters);
    indic = new SUniMap("indic");
    CHECK_NEW (indic);
    ligatureUnics = new SProperties(); 
    CHECK_NEW (ligatureUnics);
    ligatureClust = new SProperties();
    CHECK_NEW (ligatureClust);
    ligatureCache = new SProperties();
    CHECK_NEW (ligatureCache);
    for (unsigned int i=0; i<SD_SCRIPT_MAX; i++)
    {
      counters[i] = 0x80000000 + (0x10000 * i);
    }
  }
}

/**
 * Put ligature away to remember
 */
void
putLigatureCluster (SS_UCS4 ligature, const SS_UCS4* buffer, unsigned int bufsize)
{
  if (ligature <=  0x80000000 || ligature >= 0xA0000000) return;
  initLigatures ();
  SString key ((char*)& ligature, sizeof (SS_UCS4));
  const SString* ret = ligatureClust->get (key);
  if (ret) return; /* already there */
  ligatureClust->put (key, SString((char*)buffer, bufsize * sizeof (SS_UCS4)));
}

unsigned int
getLigatureUnicode (SS_UCS4 ligature, SS_UCS4* buffer)
{
  if (ligatureUnics == 0) return 0;
  const SString* ret = ligatureUnics->get (
      SString((char*) &ligature, sizeof (SS_UCS4)));
  if (ret==0) return 0;
  if (buffer==0) return ret->size()/sizeof (SS_UCS4);
  memcpy (buffer, ret->array(), ret->size());
  return ret->size()/sizeof (SS_UCS4);
}

unsigned int
getLigatureCluster (SS_UCS4 ligature, SS_UCS4* buffer)
{
  if (ligatureClust == 0) return 0;
  const SString* ret = ligatureClust->get (
      SString((char*) &ligature, sizeof (SS_UCS4)));
  if (ret==0) return 0;
  if (buffer==0) return ret->size()/sizeof (SS_UCS4);
  memcpy (buffer, ret->array(), ret->size());
  return ret->size()/sizeof (SS_UCS4);
}

int 
getLigatureScriptCode (SS_UCS4 comp)
{
  if (comp < 0x80000000) return -1;
  SS_UCS4 en = comp & 0x7fff0000;
  en = en >> 16;
  return (int) en;
}

/* get script name or null */
char*
getLigatureScript (SS_UCS4 comp)
{
  if (comp <= 0x80000000 || comp >= 0xA0000000) return 0;
  SS_UCS4 en = comp & 0x7fff0000;
  en = en >> 16;
  /* I modified this to return Script name as in MS Opentype spec.*/
  switch (en)
  {
  case SD_YUDIT: return "yudit";
  case SD_DEVANAGARI: return "deva";
  case SD_BENGALI: return "beng";
  case SD_GURMUKHI: return "guru";
  case SD_GUJARATI: return "gujr";
  case SD_ORIYA: return "orya";
  case SD_TAMIL: return "taml";
  case SD_TELUGU: return "telu";
  case SD_KANNADA: return "knda";
  case SD_MALAYALAM: return "mlym";
  case SD_SINHALA: return "sinh"; /* I could not find this in opentype doc*/
  }
  return 0;
}

/**
 Shall we do Opentype language system TAGS?
 ASM Assamese
 BEN Bengali
 GUJ Gujarati
 HIN Hindi
 KOK Konkani
 KAN Kannada
 MAL Malayalam (Old)
 MLR Malayalam (Reformed)
 MAR Marathi
 NEP Nepali
 ORI Oriya
 PAN Gurumukhi (Punjabi)
 SAN Sanskrit
 TAM Tamil
 TEL Telugu
 */

bool
isLigature (SS_UCS4 _comp)
{
  return (_comp > 0x80000000 && _comp < 0xA0000000);
}

/**
 * FIXME: add halants 
 */
SS_UCS4
getHalant (int index)
{
  switch (index)
  {
  case SD_DEVANAGARI: 
    return 0x094D;
  case SD_BENGALI: 
    return 0x09CD;
  case SD_GURMUKHI: 
    return 0x0A4D;
  case SD_GUJARATI: 
    return 0x0ACD;
  case SD_ORIYA: 
    return 0x0B4D;
  case SD_TELUGU:
    return 0x0C4D;
  case SD_KANNADA:
    return 0x0CCD;
  case SD_MALAYALAM:
    return 0x0D4D;
  case SD_SINHALA:
    return 0x0DCD;
  default:
    return 0;
  }
  return 0;
}

int getCharType (SS_UCS4 unchar)
{
  initLigatures();
  char echartype = (char) indic->encode (unchar);
  return (int) echartype;
}

/**
 * get left part of LR vowel
 */
SS_UCS4
getLRVowelLeft (SS_UCS4 u)
{
  switch (u)
  {
  case 0x09CB:
  case 0x09CC:
    return 0x09c7;
  case 0x0b4b:
  case 0x0b4c:
    return 0x0b47;
  case 0x0d4b:
    return 0x0d47;
  case 0x0d4a:
  case 0x0d4c:
    return 0x0d46;
  default:
    break;
  }
  return 0;
}
/**
 * get right part of LR vowel
 */
SS_UCS4
getLRVowelRight (SS_UCS4 u)
{
  switch (u)
  {
  case 0x09CB:
    return 0x09be;
  case 0x09CC:
    return 0x09d7;
  case 0x0b4b:
    return 0x0b3e;
  case 0x0b4c:
    return 0x0b57;
  case 0x0d4a:
  case 0x0d4b:
    return 0x0d3e;
  case 0x0d4c:
    return 0x0d57;
  default:
    break;
  }
  return 0;
}
