/*
 * oglx.c -- $Id$
 * GL window management for X11, using GLX
 *
 * Copyright (c) 1998.  See accompanying LEGAL file for details.
 */

#include "config.h"
#include "pstdlib.h"
#include "playx.h"
#include "playgl.h"

#include <GL/glx.h>

/* during debug, generate as many Expose events as possible */
#ifndef P_DEBUG
# define BACKING_STORE WhenMapped
#else
# define BACKING_STORE NotUseful
#endif

static p_glwin *gl_winlist = 0;

struct p_glwin {
  p_glwin* next;
  p_scr *s;
  p_win *parent;
  int offscreen;
  Drawable d;
  GLXContext glx;
  XVisualInfo *v;
  int error_base, event_base;
};

static int pgl_attrib[] = { GLX_DOUBLEBUFFER,
			    GLX_RGBA,
			    GLX_RED_SIZE, 1,
			    GLX_GREEN_SIZE, 1,
			    GLX_BLUE_SIZE, 1,
			    GLX_DEPTH_SIZE, 1,
			    None };

p_glwin *
p_glcreate(p_win *parent, int width, int height, int x, int y)
{
  p_scr *s = parent->s;
  Display *dpy = s->xdpy->dpy;
  XVisualInfo *v;
  Colormap cmap;
  int error_base, event_base;
  p_glwin *w, *cousin = gl_winlist;

  while (cousin && cousin->s!=s) cousin = cousin->next;

  if (!cousin) {
    if (!glXQueryExtension(dpy, &error_base, &event_base))
      return 0;

    v = glXChooseVisual(dpy, s->scr_num, pgl_attrib);
    if (!v) return 0;

  } else {
    error_base = cousin->error_base;
    event_base = cousin->event_base;
    v = cousin->v;
  }

  w = p_malloc(sizeof(p_glwin));
  if (!w) return 0;

  w->s = s;
  w->parent = parent;
  w->error_base = error_base;
  w->event_base = event_base;
  w->v = v;

  /* go ahead and install rgb-model 5x9x5 palette GLX can use */
  if (s->vclass==PseudoColor) x_rgb_palette(parent);
  cmap = parent->parent? parent->parent->cmap : parent->cmap;
  if (cmap==None)
    cmap = DefaultColormap(dpy, s->scr_num);

  w->offscreen = (parent->parent!=0);
  if (!w->offscreen) {
    XSetWindowAttributes attr;
    attr.background_pixel = 0;
    attr.border_pixel = 0;
    attr.colormap = cmap;
    attr.backing_store = BACKING_STORE;
    /* let all events propagate to parent window */

    w->d = XCreateWindow( dpy, parent->d, x, y, width, height,
			 0, v->depth, InputOutput, v->visual,
			 CWBackPixel | CWBorderPixel | CWColormap |
			 CWBackingStore, &attr );
    if (w->d) XMapWindow(dpy, w->d);
  } else {
#ifndef GLX_MESA_pixmap_colormap
    w->d = glXCreateGLXPixmap(dpy, v, parent->d);
#else
    w->d = glXCreateGLXPixmapMESA(dpy, v, parent->d, cmap);
#endif
  }

  w->glx = (w->d!=None)?
    glXCreateContext(dpy, v, cousin?cousin->glx:0, True) : 0;
  if (!w->glx) {
    p_free(w);
    return 0;
  }

  w->next = gl_winlist;
  gl_winlist = w;
  return w;
}

void
p_gldestroy(p_glwin *w)
{
  if (w) {
    p_glwin *list = gl_winlist;
    if (list==w) {
      gl_winlist = w->next;
    } else while (list) {
      if (list->next==w) {
	list->next = w->next;
	break;
      }
    }
    glXDestroyContext(w->s->xdpy->dpy, w->glx);
    if (w->offscreen) glXDestroyGLXPixmap(w->s->xdpy->dpy, w->d);
    else XDestroyWindow(w->s->xdpy->dpy, w->d);
  }
}

void
p_glswap(p_glwin *w)
{
  glXSwapBuffers(w->s->xdpy->dpy, w->d);
  p_flush(w->parent);
}

void
p_glcurrent(p_glwin *w)
{
  glXMakeCurrent(w->s->xdpy->dpy, w->d, w->glx);
}
