/* This is for emacs: -*-Mode: C++;-*- */
#if !defined(__INC_UCXX_SCRIPT_H)
#define __INC_UCXX_SCRIPT_H

#include <string>
#include <list>
#include <vector>
#include <stdexcept>

#include <sigc++/signal_system.h>
#include <sigc++/tunnel.h>

#include <uc++/any.h>

namespace uC
{

namespace Script
{

using namespace std;

class Object;
class ObjectFactory;

//
// Exception classes
//
class Exception : public runtime_error
{
  public:
    Exception(const string& s) : runtime_error(s) { }
};

//
// Helper class
//
class ObjectContainer
{
  public:
    typedef vector<Object *>::size_type size_type;

    ObjectContainer() { }

    size_type size() const { return vec_.size(); }
    Object& operator[](size_type n) { return *vec_[n];  }
    const Object& operator[](size_type n) const { return *vec_[n];  }
    void push_back(Object& obj) { vec_.push_back(&obj); }
  private:
    vector<Object *> vec_;
};

class ClassObject : public SigC::Object
{
  public:
    ClassObject(ObjectFactory& fact, bool doref = true)
        : factory_(fact), scriptref_(doref) { }
    
    virtual SigC::Object& instantiate(const ObjectContainer& args) = 0;
    virtual void associate(SigC::Object& instance, Script::Object& obj);
    
    // instance attribute accessors
    virtual Script::Object& get_attribute(SigC::Object& instance,
                                          const string& name);
    virtual void set_attribute(SigC::Object& instance,
                               const string& name,
                               Script::Object& value);
    
    ObjectFactory& factory() { return factory_; }
    bool script_referenced() const { return scriptref_; }
  private:
    ObjectFactory& factory_;
    bool scriptref_;
};

typedef SigC::Slot1<Object&, const ObjectContainer&> Slot;

//
// The following interfaces may be implemented by an object.
//
class Callable
{
  public:
    virtual Object& call(const ObjectContainer& params) = 0;
};

class Namespace
{
  public:
    virtual Object& member(const string& name) = 0;
    virtual void insert(const string& name, Object& object) = 0;
};

class Instance
{
  public:
    virtual SigC::Object& object() = 0;
    virtual Object& get_attribute(const string& name) = 0;
    virtual void set_attribute(const string& name, Object& value) = 0;
};

class ObjectFactory;

//
// Opaque object
//
class Object
{
  public:
    Object(ObjectFactory& factory)
        : factory_(factory), refcnt_(1) { }
    virtual ~Object();
    
    void reference() {
      refcnt_++;
    }
    void unreference();

    ObjectFactory& factory() { return factory_; }

    // If a specific interface is not implemented, an exception is thrown
    virtual Callable& callable_interface() = 0;
    virtual Namespace& namespace_interface() = 0;
    virtual Instance& instance_interface() = 0;

    // Extraction of basic types
    virtual Any value() const = 0;
  private:
    ObjectFactory& factory_;
    unsigned refcnt_;
};

typedef ObjectContainer Signature;

//
// All objects are created and destroyed here
//
class ObjectFactory
{
    friend class Object;
  public:
    virtual ~ObjectFactory();
    
    virtual Object& create_value(const Any& v) = 0;
    virtual Object& create_method(const Slot& slot,
                                  const Signature& s = Signature()) = 0;
    virtual Object& create_class(const ClassObject& c,
                                 const Signature& supers = Signature()) = 0;
    virtual Object& create_namespace() = 0;

    virtual Object& wrap_instance(const Object& klass, SigC::Object& obj) = 0;
    
    virtual Object& null_object() = 0;
  private:
    virtual void destroy(Object& object) = 0;
};


class Language : public SigC::Object
{
  public:
    Language(Namespace& ns, 
	     ObjectFactory& fact,
	     SigC::Tunnel *t) : ns_(&ns), fact_(&fact), tunnel_(t) { }
    Language(const Language& l) {
      ns_ = l.ns_;
      fact_ = l.fact_;
      tunnel_ = l.tunnel_;
    }
    Language& operator=(const Language& l) {
      ns_ = l.ns_;
      fact_ = l.fact_;
      tunnel_ = l.tunnel_;
      
      return *this;
    }
    Namespace& root_namespace() { return *ns_; }
    ObjectFactory& factory() { return *fact_; }
    
    SigC::Tunnel *tunnel() { return tunnel_; }
  private:
    Namespace *ns_;
    ObjectFactory *fact_;
    SigC::Tunnel *tunnel_;
};

} // Script

} // uC

#endif
