#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <sigc++/signal_system.h>

#include "sigc++/thread_tunnel.h"

namespace SigC
{

static void null_func()
{
}

list<Tunnel *> *ThreadTunnel::tunnels_ = 0;

void ThreadTunnel::sync_tunnels()
{
  if (tunnels_)
  {
    while (!tunnels_->empty())
    {
      // synchronize
      pack(slot(null_func))->tunnel(tunnels_->front(), true);
      tunnels_->pop_front();
    }
    delete tunnels_;
    tunnels_ = 0;
  }
}

void ThreadTunnel::InputHandler::callback(Dispatcher *disp, Event)
{
  Packet packet;

  while (1)
  {
#ifdef SIGC_PTHREADS
    tunnel_->mutex_.lock();
#endif
    if (tunnel_->pending_ <= 0)
      break;

    tunnel_->receive_packet(packet);
    tunnel_->pending_--;

#if SIGC_PTHREADS
    if (!packet.sync)
    {
      tunnel_->mutex_.unlock();
      packet.callback->invoke();
      delete packet.callback;
    }
    else
    {      
      packet.callback->invoke();
      tunnel_->mutex_.unlock();
      tunnel_->cb_finished_.signal();
    }
#else
    packet.callback->invoke();
#endif    
  }
#if SIGC_PTHREADS
  // This is non-obvious: take a look at the loop exit ;-)
  tunnel_->mutex_.unlock();
#endif
}

ThreadTunnel::ThreadTunnel(Dispatcher *disp, bool mt)
{
  pending_ = 0;
  disp_ = disp;
  multi_threaded_ = mt;
  
  if (pipe(pipe_) != 0)
    throw FatalError(errno);

  input_cb_ = new InputHandler(this);
  
  disp_->add_input_handler(input_cb_, pipe_[0]);
  
  if (!tunnels_)
  {
    tunnels_ = new list<Tunnel *>;
    atexit(sync_tunnels);
  }
  tunnels_->push_back(this);
}

ThreadTunnel::~ThreadTunnel()
{
  // Synchronize, so all previous requests get done
  pack(slot(null_func))->tunnel(this, true);
  
  disp_->remove(input_cb_, Dispatcher::Read);
  
  for (list<Tunnel *>::iterator it = tunnels_->begin(); 
       it != tunnels_->end(); ++it)
    if ((*it) == this)
      tunnels_->erase(it);
  
  delete input_cb_;

  close(pipe_[0]);
  close(pipe_[1]);
}

void ThreadTunnel::send(Callback *cb, bool sync)
{
#ifdef SIGC_PTHREADS
  mutex_.lock();
  if (sync && !multi_threaded_)
    // We have the dispatcher in the same thread
    cb->invoke();
  else
  {
    Packet packet;

    packet.callback = cb;
    packet.sync = sync;

    send_packet(packet);
    pending_++;
    if (sync)
      while (pending_ >= 1)
        cb_finished_.wait(mutex_);
  }
  mutex_.unlock();
#else
  if (sync)
    cb->invoke();
  
  send_packet(packet);
#endif
}

void ThreadTunnel::receive_packet(Packet& packet)
{
  int n;

  for (unsigned read_bytes = 0; read_bytes < sizeof(Packet); read_bytes += n)
  {
    if ((n = read(pipe_[0], ((char *)&packet) + read_bytes,
                  sizeof(Packet) - read_bytes)) == -1)
      throw FatalError(errno);
  }
}

void ThreadTunnel::send_packet(const Packet& packet)
{
  int n;
  
  for (unsigned written_bytes = 0; written_bytes < sizeof(Packet);
       written_bytes += n)
  {
    if ((n = write(pipe_[1], ((char *)&packet) + written_bytes,
                   sizeof(Packet) - written_bytes)) == -1)
      throw FatalError(errno);
  }
}

ThreadTunnel::FatalError::FatalError(int e) : runtime_error("")
{
#if defined(SIGC_PTHREADS)
#if defined(HAVE_STRERROR_R)
  char buf[128];
  what_ = strerror_r(e, buf, sizeof(buf));
#elif defined(HAVE___STRERROR_R)
  char buf[128];
  what_ = __strerror_r(e, buf, sizeof(buf));
#else
  // We don't have a thread-safe strerror
  mutex_.lock();
  what_ = strerror(e); 
  mutex_.unlock();
#endif
#else
  what_ = strerror(e);
#endif // SIGC_PTHREADS
}

}
