
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: odk.c,v 1.149 2006/02/07 08:43:58 mschwerin Exp $
 *
 */

#include "config.h"

#include <libgen.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <xine.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#include <pthread.h>

#include "environment.h"
#include "i18n.h"
#include "heap.h"
#include "lang.h"
#include "joystick.h"
#include "lirc.h"
#include "logger.h"
#include "oxine.h"
#include "scheduler.h"

#include "odk.h"
#include "odk_private.h"

/* 
 * ***************************************************************************
 * Description:     This is where window plugins are to be added. The plugin
 *                  descriptions have to be defined in the plugin's header
 *                  file which may only be included in odk.c.
 * ***************************************************************************
 */
#include "odk_plugin.h"
#include "x11.h"
#define WINDOW_PLUGINS { \
    x11_plugin_desc, \
	NULL }

/* Thank's to glib project */
#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))

/* 
 * ***************************************************************************
 * Description:     Stuff needed for seeking.
 * ***************************************************************************
 */
static pthread_t seek_thread;

struct seek_arg_t {
    odk_t *odk;
    int how;
};


/* 
 * ***************************************************************************
 * Name:            odk_get_pos_length
 * Access:          public
 *
 * Description:     Returns the position and the length of the current stream.
 * ***************************************************************************
 */
int
odk_get_pos_length (odk_t * odk, int *pos, int *time, int *length)
{
    int t = 0;
    int ret = 0;
    xine_stream_t *stream = odk->win->stream;

    if (stream && (xine_get_status (stream) == XINE_STATUS_PLAY)) {
        while (((ret = xine_get_pos_length (stream, pos, time, length)) == 0)
               && (++t < 10))
            usleep (100000);    /* wait before trying again */
    }
    return ret;
}


/* 
 * ***************************************************************************
 * Name:            seek_relative_thread
 * Access:          private
 *
 * Description:     Thread to seek in the stream relative to the current
 *                  position.
 * ***************************************************************************
 */
static void *
seek_relative_thread (void *seek_arg_p)
{
    struct seek_arg_t *seek_arg = (struct seek_arg_t *) seek_arg_p;

    odk_t *odk = seek_arg->odk;
    int how = (int) seek_arg->how;

    pthread_detach (pthread_self ());

    int pos_time;
    int length;
    if (odk_get_pos_length (odk, NULL, &pos_time, &length)) {
        pos_time /= 1000;

        int final = pos_time + how;
        xine_play (odk->win->stream, 0,
                   CLAMP (final * 1000, 0, (length - 2000)));
    }

    ho_free (seek_arg);
    odk->is_seeking_in_stream = 0;

    pthread_exit (NULL);
    return NULL;
}


/* 
 * ***************************************************************************
 * Name:            odk_seek
 * Access:          private
 *
 * Description:     Seek in the currently playing stream.
 * ***************************************************************************
 */
static void
odk_seek (odk_t * odk, int how)
{
    if (odk->current_mode == ODK_MODE_NULL)
        return;
    if (odk->current_mode == ODK_MODE_LOGO)
        return;
    if (odk->is_seeking_in_stream)
        return;
    if (!xine_get_stream_info (odk->win->stream, XINE_STREAM_INFO_SEEKABLE))
        return;

    struct seek_arg_t *seek_arg = ho_new (struct seek_arg_t);
    seek_arg->odk = odk;
    seek_arg->how = how;

    odk->is_seeking_in_stream = 1;

    if (pthread_create (&seek_thread, NULL,
                        seek_relative_thread, seek_arg) != 0) {
        ho_free (seek_arg);
        odk->is_seeking_in_stream = 0;
    }
}


int
odk_get_audio_lang (odk_t * odk, int channel, char *lang)
{
    return xine_get_audio_lang (odk->win->stream, channel, lang);
}


int
odk_get_spu_lang (odk_t * odk, int channel, char *lang)
{
    return xine_get_spu_lang (odk->win->stream, channel, lang);
}


/* 
 * ***************************************************************************
 * Name:            odk_eject
 * Access:          public
 *
 * Description:     Eject the current stream.
 * ***************************************************************************
 */
int
odk_eject (odk_t * odk)
{
    return xine_eject (odk->win->stream);
}


/* 
 * ***************************************************************************
 * Name:            odk_get_current_mrl
 * Access:          public
 *
 * Description:     Returns the MRL of the currently playing stream.
 * ***************************************************************************
 */
char *
odk_get_current_mrl (odk_t * odk)
{
    return odk->current_mrl;
}


/* 
 * ***************************************************************************
 * Name:            odk_current_is_logo_mode
 * Access:          public
 *
 * Description:     Returns true if we're currently in logo mode.
 * ***************************************************************************
 */
int
odk_current_is_logo_mode (odk_t * odk)
{
    return (odk->current_mode == ODK_MODE_LOGO);
}


/* 
 * ***************************************************************************
 * Name:            odk_current_is_normal_mode
 * Access:          public
 *
 * Description:     Returns true if we're currently in normal playback mode.
 * ***************************************************************************
 */
int
odk_current_is_normal_mode (odk_t * odk)
{
    return (odk->current_mode == ODK_MODE_NORMAL);
}


/* 
 * ***************************************************************************
 * Name:            odk_is_vcd
 * Access:          public
 *
 * Description:     Returns true if the current stream is a VCD.
 * ***************************************************************************
 */
int
odk_current_is_vcd (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->win->stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "vcd") == 0));
}


/* 
 * ***************************************************************************
 * Name:            odk_is_cdda
 * Access:          public
 *
 * Description:     Returns true if the current stream is an Audio CD.
 * ***************************************************************************
 */
int
odk_current_is_cdda (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->win->stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "cdda") == 0));
}


/* 
 * ***************************************************************************
 * Name:            odk_is_dvd
 * Access:          public
 *
 * Description:     Returns true if the current stream is a DVD.
 * ***************************************************************************
 */
int
odk_current_is_dvd (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->win->stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "dvd") == 0));
}


/* 
 * ***************************************************************************
 * Name:            odk_is_dvb
 * Access:          public
 *
 * Description:     Returns true if the current stream is DVB.
 * ***************************************************************************
 */
int
odk_current_is_dvb (odk_t * odk)
{
    const char *input = xine_get_meta_info (odk->win->stream,
                                            XINE_META_INFO_INPUT_PLUGIN);

    return (input && (strcasecmp (input, "dvb") == 0));
}


/* 
 * ***************************************************************************
 * Name:            odk_current_has_video
 * Access:          public
 *
 * Description:     Has the current stream got video?
 * ***************************************************************************
 */
int
odk_current_has_video (odk_t * odk)
{
    int has_video = (odk->novideo_post != NULL);

    has_video |= xine_get_stream_info (odk->win->stream,
                                       XINE_STREAM_INFO_HAS_VIDEO);
    has_video |= xine_get_stream_info (odk->animation_stream,
                                       XINE_STREAM_INFO_HAS_VIDEO);

    return has_video;
}


/* 
 * ***************************************************************************
 * Name:            odk_current_has_chapters
 * Access:          public
 *
 * Description:     Has the current stream got chapters?
 * ***************************************************************************
 */
int
odk_current_has_chapters (odk_t * odk)
{
    return xine_get_stream_info (odk->win->stream,
                                 XINE_STREAM_INFO_HAS_CHAPTERS);
}


/* 
 * ***************************************************************************
 * Name:            odk_supports_video_driver
 * Access:          public
 *
 * Description:     Is the video driver supported by odk?
 * ***************************************************************************
 */
int
odk_supports_video_driver (const char *video_driver)
{
    int i = 0;
    window_plugin_desc_t *plugins[] = WINDOW_PLUGINS;
    window_plugin_desc_t *plugin_p = plugins[i];

    while (plugins[i]) {
        while (plugin_p->driver) {
            if (strcasecmp (plugin_p->driver, video_driver) == 0) {
                return TRUE;
            }
            plugin_p++;
        }

        plugin_p = plugins[++i];
    }

    return FALSE;
}


/* 
 * ***************************************************************************
 * Name:            odk_get_meta_info
 * Access:          public
 *
 * Description:     Returns stream meta information.
 *                  The returned string has to be freed using ho_free.
 * ***************************************************************************
 */
char *
odk_get_meta_info (odk_t * odk, meta_info_t type)
{
    return meta_info_get_from_stream (odk->win->stream, type);
}


/* 
 * ***************************************************************************
 * Name:            odk_get_stream_info
 * Access:          public
 *
 * Description:     Returns stream information.
 * ***************************************************************************
 */
uint32_t
odk_get_stream_info (odk_t * odk, odk_stream_info_t info)
{
    return xine_get_stream_info (odk->win->stream, info);
}


/* 
 * ***************************************************************************
 * Name:            odk_get_stream_param_name
 * Access:          public
 *
 * Description:     Returns name of the stream parameter as a string.
 * ***************************************************************************
 */
char *
odk_get_stream_param_name (odk_t * odk, odk_stream_param_t param)
{
    char *param_name = NULL;

    switch (param) {
        case ODK_PARAM_AUDIO_CHANNEL:
            param_name = ho_strdup (_("Audio Channel"));
            break;
        case ODK_PARAM_AUDIO_OFFSET:
            param_name = ho_strdup (_("Audio-Video Offset"));
            break;
        case ODK_PARAM_SPU_CHANNEL:
            param_name = ho_strdup (_("Subtitle Channel"));
            break;
        case ODK_PARAM_SPU_OFFSET:
            param_name = ho_strdup (_("Subtitle Offset"));
            break;
        case ODK_PARAM_VO_DEINTERLACE:
            param_name = ho_strdup (_("Deinterlace"));
            break;
        case ODK_PARAM_VO_ASPECT_RATIO:
            param_name = ho_strdup (_("Aspect Ratio"));
            break;
        case ODK_PARAM_VO_HUE:
            param_name = ho_strdup (_("Hue"));
            break;
        case ODK_PARAM_VO_SATURATION:
            param_name = ho_strdup (_("Saturation"));
            break;
        case ODK_PARAM_VO_CONTRAST:
            param_name = ho_strdup (_("Contrast"));
            break;
        case ODK_PARAM_VO_BRIGHTNESS:
            param_name = ho_strdup (_("Brightness"));
            break;
        case ODK_PARAM_VO_ZOOM_Y:
        case ODK_PARAM_VO_ZOOM_X:
            param_name = ho_strdup (_("Zoom"));
            break;
        case ODK_PARAM_AUDIO_MUTE:
        case ODK_PARAM_AUDIO_VOLUME:
            param_name = ho_strdup (_("Volume"));
            break;
        default:
            break;
    }

    if (!param_name) {
        fatal (_("Unknown parameter %d!"), param);
        abort ();
    }

    return param_name;
}


/* 
 * ***************************************************************************
 * Name:            odk_get_param_as_string
 * Access:          public
 *
 * Description:     Returns stream parameter as a string.
 * ***************************************************************************
 */
char *
odk_get_stream_param_as_string (odk_t * odk, odk_stream_param_t param)
{
    int param_value = odk_get_stream_param (odk, param);

    char tmp[128];
    char *param_string = NULL;

    switch (param) {
        case ODK_PARAM_SPEED:
            switch (param_value) {
                case ODK_SPEED_NORMAL:
                    param_string = ho_strdup (_("Playing"));
                    break;
                case ODK_SPEED_PAUSE:
                    param_string = ho_strdup (_("Paused"));
                    break;
                case ODK_SPEED_SLOW_4:
                    param_string = ho_strdup (_("Extra Slow Motion"));
                    break;
                case ODK_SPEED_SLOW_2:
                    param_string = ho_strdup (_("Slow Motion"));
                    break;
                case ODK_SPEED_FAST_2:
                    param_string = ho_strdup (_("Fast Forward"));
                    break;
                case ODK_SPEED_FAST_4:
                    param_string = ho_strdup (_("Extra Fast Forward"));
                    break;
            }
            break;
        case ODK_PARAM_AUDIO_CHANNEL:
            if (param_value == AUDIO_CHANNEL_OFF)
                param_string = ho_strdup (_("none"));
            else if (param_value == AUDIO_CHANNEL_AUTO)
                param_string = ho_strdup (_("auto"));
            else {
                char lang[256];
                if (odk_get_audio_lang (odk, param_value, lang)) {
                    snprintf (tmp, 127, "%d - %s", param_value,
                              get_language_from_iso639_1 (lang));
                } else {
                    snprintf (tmp, 127, "%d", param_value);
                }
                param_string = ho_strdup (tmp);
            }
            break;
        case ODK_PARAM_SPU_CHANNEL:
            if (param_value == SPU_CHANNEL_OFF)
                param_string = ho_strdup (_("none"));
            else if (param_value == SPU_CHANNEL_AUTO)
                param_string = ho_strdup (_("auto"));
            else {
                char lang[256];
                if (odk_get_spu_lang (odk, param_value, lang)) {
                    snprintf (tmp, 127, "%d - %s", param_value,
                              get_language_from_iso639_1 (lang));
                } else {
                    snprintf (tmp, 127, "%d", param_value);
                }
                param_string = ho_strdup (tmp);
            }
            break;
        case ODK_PARAM_VO_DEINTERLACE:
            if (param_value)
                param_string = ho_strdup (_("enabled"));
            else
                param_string = ho_strdup (_("disabled"));
            break;
        case ODK_PARAM_VO_ASPECT_RATIO:
            switch (param_value) {
                case XINE_VO_ASPECT_AUTO:
                    param_string = ho_strdup (_("auto"));
                    break;
                case XINE_VO_ASPECT_SQUARE:
                    param_string = ho_strdup (_("square"));
                    break;
                case XINE_VO_ASPECT_4_3:
                    param_string = ho_strdup ("4:3");
                    break;
                case XINE_VO_ASPECT_ANAMORPHIC:
                    param_string = ho_strdup ("anamorphic");
                    break;
                case XINE_VO_ASPECT_DVB:
                    param_string = ho_strdup ("DVB");
                    break;
                default:
                    break;
            }
            break;
        case ODK_PARAM_VO_ZOOM_X:
        case ODK_PARAM_VO_ZOOM_Y:
            {
                int zoom_x = odk_get_stream_param (odk, ODK_PARAM_VO_ZOOM_X);
                int zoom_y = odk_get_stream_param (odk, ODK_PARAM_VO_ZOOM_Y);

                if (zoom_x != zoom_y)
                    snprintf (tmp, 127, "x=%d %% y=%d %%", zoom_x, zoom_y);
                else
                    snprintf (tmp, 127, "%d %%", zoom_x);

                param_string = ho_strdup (tmp);
            }
            break;
        case ODK_PARAM_VO_HUE:
        case ODK_PARAM_VO_SATURATION:
        case ODK_PARAM_VO_CONTRAST:
        case ODK_PARAM_VO_BRIGHTNESS:
            snprintf (tmp, 127, "%d %%", param_value / 655);
            param_string = ho_strdup (tmp);
            break;
        case ODK_PARAM_AUDIO_OFFSET:
        case ODK_PARAM_SPU_OFFSET:
            snprintf (tmp, 127, "%d ms", param_value / 9);
            param_string = ho_strdup (tmp);
            break;
        case ODK_PARAM_AUDIO_MUTE:
        case ODK_PARAM_AUDIO_VOLUME:
            {
                int vol = odk_get_stream_param (odk, ODK_PARAM_AUDIO_VOLUME);
                int mute = odk_get_stream_param (odk, ODK_PARAM_AUDIO_MUTE);

                if (mute)
                    snprintf (tmp, 127, "%d %% (%s)", vol, _("muted"));
                else
                    snprintf (tmp, 127, "%d %%", vol);

                param_string = ho_strdup (tmp);
            }
            break;
        default:
            break;
    }

    if (!param_string) {
        fatal (_("Unknown parameter %d!"), param);
        abort ();
    }

    return param_string;
}


/* 
 * ***************************************************************************
 * Name:            odk_get_param
 * Access:          public
 *
 * Description:     Returns stream parameter
 * ***************************************************************************
 */
int
odk_get_stream_param (odk_t * odk, odk_stream_param_t param)
{
    if (param == ODK_PARAM_POSITION) {
        int pos_time;
        int length;
        int pos = 1;
        if (!odk_get_pos_length (odk, NULL, &pos_time, &length)) {
            return -1;
        }
        if (length)
            pos = (int) (((double) pos_time / (double) length) *
                         (double) 100);
        return pos;
    }

    return xine_get_param (odk->win->stream, param);
}


/* 
 * ***************************************************************************
 * Name:            odk_set_stream_param
 * Access:          public
 *
 * Description:     Sets stream parameter
 * ***************************************************************************
 */
int
odk_set_stream_param (odk_t * odk, odk_stream_param_t param, int value)
{
    if (param == ODK_PARAM_POSITION) {
        int cur_time;
        int length_time;

        if (!odk_get_pos_length (odk, NULL, &cur_time, &length_time)) {
            return 0;
        }

        int how = ((length_time * value / 100) - cur_time) / 1000;

        odk_seek (odk, how);

        return value;
    }

    /* We neveer want to turn subtitles off, as this affects our OSD. */
    if (param == ODK_PARAM_SPU_CHANNEL && value == SPU_CHANNEL_OFF) {
        value = SPU_CHANNEL_AUTO;
    }

    xine_set_param (odk->win->stream, param, value);

    return xine_get_param (odk->win->stream, param);
}


/* 
 * ***************************************************************************
 * Name:            odk_change_stream_param
 * Access:          public
 *
 * Description:     Changes stream parameter
 * ***************************************************************************
 */
int
odk_change_stream_param (odk_t * odk,
                         odk_stream_param_t param, int how, int min, int max)
{
    if (param == ODK_PARAM_POSITION) {
        odk_seek (odk, how);

        return 0;
    }

    int old_value = xine_get_param (odk->win->stream, param);
    int new_value = old_value + how;

    if ((param == ODK_PARAM_VO_ASPECT_RATIO)
        || (param == ODK_PARAM_VO_DEINTERLACE)
        || (param == ODK_PARAM_AUDIO_MUTE)
        || (param == ODK_PARAM_AUDIO_CHANNEL)
        || (param == ODK_PARAM_SPU_CHANNEL)) {
        if (new_value > max)
            new_value = min;
        if (new_value < min)
            new_value = max;
    }

    if (new_value > max)
        new_value = max;
    if (new_value < min)
        new_value = min;

    odk_set_stream_param (odk, param, new_value);

    return xine_get_param (odk->win->stream, param);
}


/* 
 * ***************************************************************************
 * Name:            stream_stop
 * Access:          private
 *
 * Description:     Stops the current stream. Disposes the novideo background
 *                  stream. Releases and disposes the novideo post plugin.
 *                  Releases strings.
 * ***************************************************************************
 */
static void
stream_stop (odk_t * odk)
{
    if (odk->image_finish_job) {
        cancel_job (odk->image_finish_job);
        odk->image_finish_job = 0;
    }
    if (odk->novideo_post) {
        odk_post_audio_unwire (odk);
        odk->novideo_post = NULL;
    }
    if (odk->current_mrl) {
        ho_free (odk->current_mrl);
        odk->current_mrl = NULL;
    }
    if (odk->current_sub_mrl) {
        ho_free (odk->current_sub_mrl);
        odk->current_sub_mrl = NULL;
    }

    playlist_clear (odk->current_alternatives);

    odk->is_seeking_in_stream = 0;
    odk->current_mode = ODK_MODE_NULL;

    xine_close (odk->subtitle_stream);
    xine_close (odk->animation_stream);
    xine_close (odk->background_stream);
    xine_close (odk->win->stream);

    /* This is not really necessary (I still want this checked) */
#ifdef DEBUG
    assert (!odk->image_finish_job);
    assert (!odk->novideo_post);
    assert (!odk->current_mrl);
    assert (!odk->current_sub_mrl);
#endif
}


/* 
 * ***************************************************************************
 * Name:            odk_stop
 * Access:          public
 *
 * Description:     Stops the current stream. Disposes the novideo background
 *                  stream. Releases and disposes the novideo post plugin.
 *                  Releases strings. Finally an event is sent telling the
 *                  GUI, that the stream was stopped.
 * ***************************************************************************
 */
void
odk_stop (odk_t * odk)
{
    stream_stop (odk);

    oxine_event_t ev;
    ev.type = OXINE_EVENT_PLAYBACK_STOPPED;
    odk_oxine_event_send (odk, &ev);
}


/* 
 * ***************************************************************************
 * Name:            get_suffix
 * Access:          private
 *
 * Description:     Returns the suffix of a MRL
 * ***************************************************************************
 */
static char *
get_suffix (const char *mrl)
{
    char *suffix = NULL;

    if (mrl) {
        suffix = rindex (mrl, '.');

        if (suffix) {
            suffix += 1;
        }
    }

    return suffix;
}


/* 
 * ***************************************************************************
 * Name:            is_image
 * Access:          private
 *
 * Description:     Is the file (MRL) an image.
 * ***************************************************************************
 */
int
is_image (const char *mrl)
{
    if (mrl) {
        char *suffix = get_suffix (mrl);
        if (suffix) {
            if ((strcasecmp (suffix, "gif") == 0) ||
                (strcasecmp (suffix, "jpg") == 0) ||
                (strcasecmp (suffix, "jpeg") == 0) ||
                (strcasecmp (suffix, "png") == 0)) {
                return 1;
            }
        }
    }

    return 0;
}

#ifdef HAVE_IMAGE_TOOLS

#define CONVERT_COMMAND "%s %s \
    | "BIN_PAMSCALE" -xyfit %d %d \
    | "BIN_PNMPAD" -black -width %d -height %d \
    | "BIN_PPMTOY4M" -F25:1 -S 420mpeg2 \
    | "BIN_MPEG2ENC" -b 3000 -o %s"

/* 
 * ***************************************************************************
 * Name:            get_convert_command
 * Access:          private
 *
 * Description:     Returns a command to convert MRL to a video.
 * ***************************************************************************
 */
static char *
get_convert_command (const char *mrl)
{
    char *suffix = get_suffix (mrl);

    if (suffix != NULL) {
        if (strcasecmp (suffix, "gif") == 0) {
            return BIN_GIFTOPNM;
        }

        else if ((strcasecmp (suffix, "jpg") == 0) ||
                 (strcasecmp (suffix, "jpeg") == 0)) {
            return BIN_JPGTOPNM;
        }

        else if (strcasecmp (suffix, "png") == 0) {
            return BIN_PNGTOPNM;
        }
    }

    return NULL;
}
#endif

static void
log_xine_error_code (xine_stream_t * stream,
                     const char *err_msg, const char *mrl)
{
    int ret = xine_get_error (stream);
    char *xine_msg = _("Unknown error");
    switch (ret) {
        case XINE_ERROR_NO_INPUT_PLUGIN:
            xine_msg = _("No input plugin");
            break;
        case XINE_ERROR_NO_DEMUX_PLUGIN:
            xine_msg = _("Could not find demuxer plugin");
            break;
        case XINE_ERROR_DEMUX_FAILED:
            xine_msg = _("Demuxer plugin failed");
            break;
        case XINE_ERROR_MALFORMED_MRL:
            xine_msg = _("Malformed MRL");
            break;
        default:
            break;
    }
    error (err_msg, mrl, xine_msg);
}


/* 
 * ***************************************************************************
 * Name:            mutexed_play
 * Access:          private
 *
 * Description:     Plays the stream MRL.
 * ***************************************************************************
 */
static int
mutexed_play (odk_t * odk, const char *mrl, const char *sub_mrl, int mode)
{
    int new_mode;
    char mrla[1024];

    assert (odk);
    assert (mrl);
    assert (mode != ODK_MODE_NULL);

    /* We convert images to a video stream first. */
    if (is_image (mrl)) {
#ifdef HAVE_IMAGE_TOOLS
        /* Filename of the converted image (cache) */
        snprintf (mrla, 1023, "%s/%s.mpg", get_dir_oxine_cache (), mrl);

        if (access (mrla, R_OK) != 0 || file_is_newer (mrl, mrla)) {
            char *cmdc = get_convert_command (mrl);
            if (cmdc) {
                /* Create the cache directory. */
                char *dirn = ho_strdup (mrla);
                mkdir_recursive (dirname (dirn), 0700);
                ho_free (dirn);

                int width;
                int height;
                odk_get_output_size (odk, &width, &height);

                /* Create convert command */
                char cmds[1024];
                char *inp = filename_escape_for_shell (mrl);
                char *out = filename_escape_for_shell (mrla);
                snprintf (cmds, 1023, CONVERT_COMMAND, cmdc, inp, width,
                          height, width, height, out);
                ho_free (inp);
                ho_free (out);

                /* Execute command */
                debug ("executing %s", cmds);
                if (system (cmds) != 0) {
                    error (_("Could not convert image!"));
                    return 0;
                }
            }
        }

        new_mode = mode;
#else
        warn (_("I cannot play image files."));
        return 0;
#endif
    } else {
        strcpy (mrla, mrl);
        new_mode = mode;
    }

    char *original_mrl = NULL;
    char *original_sub_mrl = NULL;
    if (mrl)
        original_mrl = ho_strdup (mrl);
    if (sub_mrl)
        original_sub_mrl = ho_strdup (sub_mrl);

    char *escaped_mrl = NULL;
    char *escaped_sub_mrl = NULL;
    if (mrla)
        escaped_mrl = filename_escape_to_uri (mrla);
    if (sub_mrl)
        escaped_sub_mrl = filename_escape_to_uri (sub_mrl);
#ifdef DEBUG
    debug ("mrl '%s'", escaped_mrl);
    debug ("sub '%s'", escaped_sub_mrl);
#endif

    /* Stop any stream that might be running. */
    stream_stop (odk);

    /* We try to open the stream */
    if (!xine_open (odk->win->stream, escaped_mrl)) {
        log_xine_error_code (odk->win->stream,
                             _("Could not open '%s': %s!"), original_mrl);
        if (escaped_mrl)
            ho_free (escaped_mrl);
        if (escaped_sub_mrl)
            ho_free (escaped_sub_mrl);
        if (original_mrl)
            ho_free (original_mrl);
        if (original_sub_mrl)
            ho_free (original_sub_mrl);
        return 0;
    }

    int has_video = odk_get_stream_info (odk, ODK_STREAM_INFO_HAS_VIDEO);

    /* Reset zoom */
    odk_set_stream_param (odk, ODK_PARAM_VO_ZOOM_X, 100);
    odk_set_stream_param (odk, ODK_PARAM_VO_ZOOM_Y, 100);

    /* Reset audio-video and subtitle offset. */
    odk_set_stream_param (odk, ODK_PARAM_AUDIO_OFFSET, 0);
    odk_set_stream_param (odk, ODK_PARAM_SPU_OFFSET, 0);

    /* Reset audio channel to auto if selected channel does not exist, or the
     * stream has no video. */
    int audio_channel = odk_get_stream_param (odk, ODK_PARAM_AUDIO_CHANNEL);
    int audio_channel_max =
        odk_get_stream_info (odk, ODK_STREAM_INFO_MAX_AUDIO_CHANNEL);
    if ((audio_channel > audio_channel_max)
        || (!has_video && (audio_channel == AUDIO_CHANNEL_OFF)))
        odk_set_stream_param (odk, ODK_PARAM_AUDIO_CHANNEL,
                              AUDIO_CHANNEL_AUTO);

    /* Reset spu channel to auto if selected channel does not exist. */
    int spu_channel = odk_get_stream_param (odk, ODK_PARAM_SPU_CHANNEL);
    int spu_channel_max =
        odk_get_stream_info (odk, ODK_STREAM_INFO_MAX_SPU_CHANNEL);
    if (spu_channel > spu_channel_max)
        odk_set_stream_param (odk, ODK_PARAM_SPU_CHANNEL, SPU_CHANNEL_AUTO);

    /* Set background for streams without video. */
    if (!has_video) {
        /* Show standard background */
        if (odk->visual_anim_style == 0) {
            xine_open (odk->background_stream, odk->background_stream_mrl);
            xine_play (odk->background_stream, 0, 0);
        }

        /* Show a post plugin. */
        else if (odk->visual_anim_style == 1) {
            odk_post_audio_rewire (odk);
        }

        /* Show a stream animation. */
        else if (odk->visual_anim_style == 2) {
            xine_open (odk->animation_stream, odk->animation_stream_mrl);
            xine_play (odk->animation_stream, 0, 0);
        }
    }

    if (escaped_sub_mrl) {
        if (xine_open (odk->subtitle_stream, escaped_sub_mrl)) {
            xine_stream_master_slave (odk->win->stream,
                                      odk->subtitle_stream,
                                      XINE_MASTER_SLAVE_PLAY |
                                      XINE_MASTER_SLAVE_STOP);
        } else {
            log_xine_error_code (odk->subtitle_stream,
                                 _("Could not open '%s': %s!"),
                                 original_sub_mrl);
        }
    }

    /* We try to play the stream. */
    if (!xine_play (odk->win->stream, 0, 0)) {
        log_xine_error_code (odk->subtitle_stream,
                             _("Could not play '%s': %s!"), original_mrl);
        if (escaped_mrl)
            ho_free (escaped_mrl);
        if (escaped_sub_mrl)
            ho_free (escaped_sub_mrl);
        if (original_mrl)
            ho_free (original_mrl);
        if (original_sub_mrl)
            ho_free (original_sub_mrl);
        return 0;
    }

    /* Now we successfully opened and started the file we can set mode etc. */
    odk->current_mode = new_mode;
    odk->current_mrl = original_mrl;
    odk->current_sub_mrl = original_sub_mrl;

    /* We send an event to tell the frontend that playback has started. */
    oxine_event_t ev;
    ev.type = OXINE_EVENT_PLAYBACK_STARTED;
    odk_oxine_event_send (odk, &ev);

    /* Adapt OSD size to new stream. */
    odk_osd_adapt_size (odk);

    if (escaped_mrl)
        ho_free (escaped_mrl);
    if (escaped_sub_mrl)
        ho_free (escaped_sub_mrl);

    return 1;
}


/* 
 * ***************************************************************************
 * Name:            odk_play
 * Access:          public
 *
 * Description:     Plays the stream MRL.
 * ***************************************************************************
 */
int
odk_play (odk_t * odk, const char *mrl, const char *sub_mrl, int mode)
{
    /* We dont want more than one thread to enter here. This is important when
     * converting images, as this takes quite a lot of time and we dont want
     * another play request to get in there while we are still converting. */
#ifdef DEBUG_THREADS
    debug ("[thread: %d] waiting for lock on play_mutex",
           (int) pthread_self ());
#endif
    /* Acquire a lock on the play mutex. */
    pthread_mutex_lock (&odk->play_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] aquired lock on play_mutex", (int) pthread_self ());
#endif

    int result = mutexed_play (odk, mrl, sub_mrl, mode);

    /* This is necessary as xine-lib sometimes resets the locale. I guess,
     * that this must be a bug in xine-lib. */
#ifdef HAVE_SETLOCALE
    if (!setlocale (LC_ALL, ""))
        error ("The current locale is not supported by the C library!");
#endif

    /* Release the lock on the play mutex. */
    pthread_mutex_unlock (&odk->play_mutex);
#ifdef DEBUG_THREADS
    debug ("[thread: %d] released lock on play_mutex", (int) pthread_self ());
#endif

    return result;
}


/* 
 * ***************************************************************************
 * Name:            odk_show_window
 * Access:          public
 *
 * Description:     Shows the output window.
 * ***************************************************************************
 */
void
odk_show_window (odk_t * odk, int fullscreen)
{
    odk->win->show (odk->win);
    odk->win->fullscreen (odk->win, fullscreen);
    odk_osd_adapt_size (odk);
}


/* 
 * ***************************************************************************
 * Name:            odk_hide_window
 * Access:          public
 *
 * Description:     Hides the output window.
 * ***************************************************************************
 */
void
odk_hide_window (odk_t * odk)
{
    odk->win->hide (odk->win);
}


/* 
 * ***************************************************************************
 * Name:            odk_set_fullscreen
 * Access:          public
 *
 * Description:     Switches to an from fullscreen mode.
 * ***************************************************************************
 */
void
odk_set_fullscreen (odk_t * odk, int fullscreen)
{
    odk->win->fullscreen (odk->win, fullscreen);
    odk_osd_adapt_size (odk);
}


/* 
 * ***************************************************************************
 * Name:            odk_is_fullscreen
 * Access:          public
 *
 * Description:     Is the window currently in fullscreen mode?
 * ***************************************************************************
 */
int
odk_is_fullscreen (odk_t * odk)
{
    return odk->win->is_fullscreen (odk->win);
}


/* 
 * ***************************************************************************
 * Name:            odk_run
 * Access:          public
 *
 * Description:     Starts the event loop.
 * ***************************************************************************
 */
void
odk_run (odk_t * odk)
{
    odk->win->event_loop (odk->win);
}


/* 
 * ***************************************************************************
 * Name:            odk_exit
 * Access:          public
 *
 * Description:     Stops the event loop.
 * ***************************************************************************
 */
void
odk_exit (odk_t * odk)
{
    odk->win->stop_event_loop (odk->win);
}


/* 
 * ***************************************************************************
 * Name:            odk_create_window
 * Access:          private
 *
 * Description:     Create output window.
 * ***************************************************************************
 */
static odk_window_t *
odk_create_window (xine_t * xine, const char *video_driver)
{
    int i = 0;
    odk_window_t *window = NULL;
    window_plugin_desc_t *plugins[] = WINDOW_PLUGINS;
    window_plugin_desc_t *plugin_p = plugins[i];

    while (!window && plugins[i]) {
        /* Try to find our preferred video driver */
        while (plugin_p->driver
               && strcasecmp (plugin_p->driver, video_driver)) {
            plugin_p++;
        }

        /* Try to init window plugin. */
        if (plugin_p->driver) {
            debug ("Driver '%s' found in odk_plugin '%s'.", video_driver,
                   plugins[i]->name);
            window = plugin_p->constructor (xine, plugin_p->driver);
        } else {
            debug ("Driver '%s' not found in odk_plugin '%s'.",
                   video_driver, plugins[i]->name);
        }

        plugin_p = plugins[++i];
    }

    return window;
}


/* 
 * ***************************************************************************
 * Name:            odk_config_register
 * Access:          private
 *
 * Description:     Registers all config values used by this module.
 * ***************************************************************************
 */
static void
odk_config_register (odk_t * odk)
{
    char *visual_anims[] = {
        "None",
        "Post Plugin",
        "Stream Animation",
        NULL
    };

    odk->image_change_delay =
        xine_config_register_num (odk->xine, "gui.image_change_delay", 3,
                                  _("delay (sec) between image change in "
                                    "the slideshow"),
                                  _("delay (sec) between image change in "
                                    "the slideshow"), 10, NULL, NULL);
    odk->visual_anim_style =
        xine_config_register_enum (odk->xine, "gui.visual_anim", 0,
                                   visual_anims, _("Visual animation style"),
                                   _("Display some video animations "
                                     "when current stream is audio "
                                     "only (e.g.: mp3)."), 10, NULL, NULL);
    odk->background_stream_mrl =
        xine_config_register_string (odk->xine, "gui.background_stream_mrl",
                                     OXINE_DATADIR "/playingmenu.mpg",
                                     _("MRL of the audio-only background"),
                                     _("The MRL that will be displayed "
                                       "if 'gui.visual_anim' is set to "
                                       "'None'"), 10, NULL, NULL);
    odk->animation_stream_mrl =
        xine_config_register_string (odk->xine, "gui.animation_stream_mrl",
                                     OXINE_DATADIR "/animation.mpg",
                                     _("MRL of the audio-only video "
                                       "animation"),
                                     _("The MRL that will be displayed "
                                       "if 'gui.visual_anim' is set to "
                                       "'Stream Animation'"), 10, NULL, NULL);
    odk->use_unscaled_osd =
        xine_config_register_bool (odk->xine, "gui.osd_use_unscaled", 1,
                                   _("Use unscaled OSD"),
                                   _("Use unscaled (full screen resolution) "
                                     "OSD if possible"), 10, NULL, NULL);
}

/* 
 * ***************************************************************************
 * Name:            odk_init
 * Access:          public
 *
 * Description:     Initializes drawable, xine stream and OSD.
 * ***************************************************************************
 */
odk_t *
odk_init (xine_t * xine)
{
    odk_t *odk = ho_new (odk_t);
    odk->xine = xine;

    odk->current_mrl = NULL;
    odk->current_sub_mrl = NULL;
    odk->current_title = NULL;
    odk->current_alternatives = playlist_new ();

#ifdef DEBUG_THREADS
    pthread_mutexattr_init (&odk->play_mutex_attr);
    pthread_mutexattr_settype (&odk->play_mutex_attr,
                               PTHREAD_MUTEX_ERRORCHECK_NP);
    pthread_mutex_init (&odk->play_mutex, &odk->play_mutex_attr);
#else
    pthread_mutex_init (&odk->play_mutex, NULL);
#endif

    /* Register configuration values. */
    odk_config_register (odk);

    /* Open video port and create output window. */
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (xine, "video.driver", &centry);
    const char *video_driver = centry.str_value;

    odk->win = odk_create_window (xine, video_driver);
    if (!odk->win || !odk->win->video_port) {
        warn (_("Failed to create window using video driver '%s'!"),
              video_driver);
        odk->win = odk_create_window (xine, "auto");
    }
    if (!odk->win || !odk->win->video_port) {
        error (_("Failed to create window using default driver."));
        return NULL;
    }

    /* Open audio port. */
    xine_config_lookup_entry (xine, "audio.driver", &centry);
    const char *audio_driver = centry.str_value;

    odk->win->audio_port = xine_open_audio_driver (xine, audio_driver, NULL);
    if (!odk->win->audio_port) {
        warn (_("Failed to open audio port using audio driver '%s'!"),
              audio_driver);
        odk->win->audio_port = xine_open_audio_driver (xine, NULL, NULL);
    }
    if (!odk->win->audio_port) {
        error (_("Failed to open audio port using default audio driver."));
        return NULL;
    }

    /* Create streams. */
    xine_audio_port_t *audio_port = odk->win->audio_port;
    xine_video_port_t *video_port = odk->win->video_port;

    odk->win->stream = xine_stream_new (xine, audio_port, video_port);

    odk->background_stream = xine_stream_new (xine, NULL, video_port);
    xine_set_param (odk->background_stream, ODK_PARAM_AUDIO_CHANNEL,
                    AUDIO_CHANNEL_OFF);

    odk->animation_stream = xine_stream_new (xine, NULL, video_port);
    xine_set_param (odk->animation_stream, ODK_PARAM_AUDIO_CHANNEL,
                    AUDIO_CHANNEL_OFF);

    odk->subtitle_stream = xine_stream_new (xine, NULL, video_port);
    xine_set_param (odk->subtitle_stream, ODK_PARAM_AUDIO_CHANNEL,
                    AUDIO_CHANNEL_OFF);

    /* Initialize OSD stuff. */
    odk_osd_init (odk);

    /* Initialize event handling. */
    odk_event_init (odk);

    /* Initialize post plugin stuff. */
    odk_post_init (odk);

    return odk;
}


/* 
 * ***************************************************************************
 * Name:            odk_free
 * Access:          public
 *
 * Description:     Free stuff.
 * ***************************************************************************
 */
void
odk_free (odk_t * odk)
{
    /* Free event handling. */
    odk_event_free (odk);

    /* Free OSD stuff. */
    odk_osd_free (odk);

    /* Dispose streams */
    xine_dispose (odk->win->stream);
    xine_dispose (odk->subtitle_stream);
    xine_dispose (odk->background_stream);
    xine_dispose (odk->animation_stream);

    xine_close_audio_driver (odk->xine, odk->win->audio_port);

    pthread_mutex_destroy (&odk->play_mutex);
#ifdef DEBUG_THREADS
    pthread_mutexattr_destroy (&odk->play_mutex_attr);
#endif

    odk->win->dispose (odk->win);

    if (odk->current_mrl)
        ho_free (odk->current_mrl);
    if (odk->current_sub_mrl)
        ho_free (odk->current_sub_mrl);
    if (odk->current_title)
        ho_free (odk->current_title);
    if (odk->current_alternatives)
        playlist_free (odk->current_alternatives);

    ho_free (odk);
}
