
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gui_utils.c,v 1.43 2006/01/18 16:02:49 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "environment.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "lang.h"
#include "main_menu.h"
#include "otk.h"
#include "oxine.h"
#include "playing_menu.h"
#include "playlist_menu.h"
#include "playlist_m3u.h"
#include "scheduler.h"

void
set_status_string (void *oxine_p, otk_widget_t * widget)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (odk_get_stream_param (oxine->odk, ODK_PARAM_SPEED) == ODK_SPEED_PAUSE) {
        otk_label_set_text (widget, _("Paused"));
    } else {
        otk_label_set_text (widget, _("Playing"));
    }
}


void
set_time_string (void *oxine_p, otk_widget_t * widget)
{
    char time_str[640];

    time_t current = time (NULL);
    struct tm *brokentime = localtime (&current);

    strftime (time_str, 639, "%X", brokentime);

    otk_label_set_text (widget, time_str);
}


void
clean_otk_stuff (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    otk_clear (oxine->otk);
    odk_osd_hide_image (oxine->odk);

    oxine->user_interface_is_visible = 0;
    oxine->control_menu_is_visible = 0;
    oxine->repaint_menu = clean_otk_stuff;

    if (oxine->otk_clear_job) {
        cancel_job (oxine->otk_clear_job);
        oxine->otk_clear_job = 0;
    }

    if (odk_current_is_dvd (oxine->odk))
        odk_forward_events_to_xine (oxine->odk, 1);
    else
        odk_forward_events_to_xine (oxine->odk, 0);
}

#define STREAM_INFO_POS_X (20)
#define STREAM_INFO_POS_Y (20)
#define STREAM_INFO_ALIGN (OTK_ALIGN_LEFT | OTK_ALIGN_TOP)


void
show_stream_parameter (oxine_t * oxine, odk_stream_param_t param)
{
    int control_menu_was_visible = oxine->control_menu_is_visible;

    if (oxine->user_interface_is_visible)
        return;
    if (oxine->otk_clear_job)
        cancel_job (oxine->otk_clear_job);

    clean_otk_stuff (oxine);
    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);

    if (param == ODK_PARAM_POSITION) {
        control_menu_was_visible = 1;
    }

    else {
        char msg[256];
        char *param_str = odk_get_stream_param_as_string (oxine->odk, param);

        if (param == ODK_PARAM_SPEED) {
            snprintf (msg, 256, "%s", param_str);
        }

        else {
            char *param_name = odk_get_stream_param_name (oxine->odk, param);
            snprintf (msg, 256, "%s: %s", param_name, param_str);
            ho_free (param_name);
        }

        ho_free (param_str);

        otk_label_new (oxine->otk, STREAM_INFO_POS_X, STREAM_INFO_POS_Y,
                       STREAM_INFO_ALIGN, msg);
    }

    otk_draw (oxine->otk);

    if (control_menu_was_visible)
        show_control_menu_cb (oxine);

    oxine->otk_clear_job =
        schedule_job (6000, (void (*)(void *)) clean_otk_stuff, oxine);
}


void
show_message (oxine_t * oxine, char *message)
{
    clean_otk_stuff (oxine);
    if (!odk_current_is_normal_mode (oxine->odk)) {
        show_background (oxine, OXINE_DATADIR "/mainmenu.mpg");
    }

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_border_new (oxine->otk, 100, 200, 600, 200);
    show_clock (oxine);

    otk_widget_t *l;
    l = otk_label_new (oxine->otk, 400, 300,
                       OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER, message);
    otk_label_set_max_width (l, 560);

    oxine->user_interface_is_visible = 1;
    otk_draw (oxine->otk);
}


void
show_message_ok (oxine_t * oxine, char *message, otk_button_cb_t ok_cb)
{
    clean_otk_stuff (oxine);
    if (!odk_current_is_normal_mode (oxine->odk)) {
        show_background (oxine, OXINE_DATADIR "/mainmenu.mpg");
    }

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_border_new (oxine->otk, 100, 200, 600, 200);
    show_clock (oxine);

    otk_widget_t *l;
    l = otk_label_new (oxine->otk, 400, 280,
                       OTK_ALIGN_CENTER | OTK_ALIGN_BOTTOM, message);
    otk_label_set_max_width (l, 560);

    otk_widget_t *b;
    b = otk_button_new (oxine->otk, 320, 320, 160, 40, "OK", ok_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    otk_widget_set_focus (b);

    oxine->user_interface_is_visible = 1;
    otk_draw (oxine->otk);
}


void
show_message_ok_cancel (oxine_t * oxine, char *message, otk_button_cb_t ok_cb,
                        otk_button_cb_t cancel_cb)
{
    clean_otk_stuff (oxine);
    if (!odk_current_is_normal_mode (oxine->odk)) {
        show_background (oxine, OXINE_DATADIR "/mainmenu.mpg");
    }

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_border_new (oxine->otk, 100, 200, 600, 200);
    show_clock (oxine);

    otk_widget_t *l;
    l = otk_label_new (oxine->otk, 400, 280,
                       OTK_ALIGN_CENTER | OTK_ALIGN_BOTTOM, message);
    otk_label_set_max_width (l, 560);

    otk_widget_t *b;
    b = otk_button_new (oxine->otk, 230, 320, 160, 40, "OK", ok_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    otk_widget_set_focus (b);

    b = otk_button_new (oxine->otk, 410, 320, 160, 40,
                        _("Cancel"), cancel_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

    oxine->user_interface_is_visible = 1;
    otk_draw (oxine->otk);
}


static void
playitem_play (oxine_t * oxine, playitem_t * playitem,
               otk_button_cb_t error_cb)
{
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "gui.playlist.current_track",
                              &centry);

    if (!playitem) {
        assert (oxine->playback_ended_menu);
        oxine->playback_ended_menu (oxine);

        /* Do not restart the playlist. */
        centry.num_value = -1;
        xine_config_update_entry (oxine->xine, &centry);
        xine_config_save (oxine->xine, get_file_config ());

        return;
    }

    if ((strncasecmp (playitem->mrl, "http://", 7) == 0)
        || (strncasecmp (playitem->mrl, "rtsp://", 7) == 0)
        || (strncasecmp (playitem->mrl, "dvd://", 6) == 0)
        || (strncasecmp (playitem->mrl, "vcd://", 6) == 0)
        || (strncasecmp (playitem->mrl, "cdda:/", 6) == 0))
        please_wait (oxine);
    else
        clean_otk_stuff (oxine);

    centry.num_value = playlist_current_pos (oxine->playlist);
    xine_config_update_entry (oxine->xine, &centry);
    xine_config_save (oxine->xine, get_file_config ());

    if (!odk_play (oxine->odk, playitem->mrl,
                   playitem->subtitle_mrl, ODK_MODE_NORMAL)) {
        could_not_play (oxine, error_cb, oxine->playback_ended_menu);
    }
}


void
playlist_play_first (oxine_t * oxine)
{
    playitem_t *playitem = playlist_get_first (oxine->playlist);

    if (playitem) {
        // save the playlist
        m3u_save (oxine->playlist, get_file_lastplaylist ());
    }

    playitem_play (oxine, playitem, (otk_button_cb_t) playlist_play_next);
}


void
playlist_play_next (oxine_t * oxine)
{
    playitem_t *playitem = playlist_get_next (oxine->playlist);

    playitem_play (oxine, playitem, (otk_button_cb_t) playlist_play_next);
}


void
playlist_play_prev (oxine_t * oxine)
{
    playitem_t *playitem = playlist_get_prev (oxine->playlist);

    playitem_play (oxine, playitem, (otk_button_cb_t) playlist_play_prev);
}


void
playlist_play_item (oxine_t * oxine, playitem_t * playitem)
{
    playlist_set_current (oxine->playlist, playitem);

    playitem_play (oxine, playitem, (otk_button_cb_t) playlist_play_next);
}
