
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist.h,v 1.14 2005/12/14 12:54:09 mschwerin Exp $
 *
 */

#ifndef HAVE_FILELIST_H
#define HAVE_FILELIST_H

#include "list.h"

typedef enum {
    FILE_TYPE_UNKNOWN,
    FILE_TYPE_UPLINK,
    FILE_TYPE_REGULAR,
    FILE_TYPE_DIRECTORY,
    FILE_TYPE_MOUNTPOINT,
    FILE_TYPE_MEDIAMARKS,
    FILE_TYPE_M3U,
    FILE_TYPE_AUTODISC
} fileitem_type_t;

typedef enum {
    ALLOW_FILES_MUSIC,
    ALLOW_FILES_VIDEO,
    ALLOW_FILES_IMAGE,
    ALLOW_FILES_SUBTITLE,
    ALLOW_FILES_ANY
} fileitem_allowed_t;

typedef struct filelist_s filelist_t;
typedef struct fileitem_s fileitem_t;

struct fileitem_s {
    // The title of this item.
    char *title;

    // The mrl of this item.
    char *mrl;

    // A description of this fileitem.
    char *description;

    // A thumbnail of this fileitem.
    char *thumbnail_mrl;

    // The type of this fileitem.
    fileitem_type_t type;

    // The list of items below this item.
    filelist_t *sublist;

    // The number of times this item has already been listened to.
    // This is only used for the favorites.
    int listened_count;
};

struct filelist_s {
    l_list_t *list;

    // The mrl of the parent of items in this list.
    char *mrl;

    // The type of files allowed in this list.
    fileitem_allowed_t allowed_filetypes;

    // This is used to save the top position of list when displaying it.
    int top_position;
    // This is used to save the current position in the list when displaying it.
    int cur_position;
};

/* 
 * ***************************************************************************
 * Name:            filelist_new
 * Access:          public
 *
 * Description:     Creates a new filelist. Only files of the allowed type
 *                  will be added when expanding a directory.
 * ***************************************************************************
 */
filelist_t *filelist_new (fileitem_allowed_t allowed_filetypes,
                          const char *mrl);


/* 
 * ***************************************************************************
 * Name:            filelist_free
 * Access:          public
 *
 * Description:     Removes all fileitems and frees the list.
 * ***************************************************************************
 */
void filelist_free (filelist_t * filelist);


/* 
 * ***************************************************************************
 * Name:            filelist_clear
 * Access:          public
 *
 * Description:     Removes all fileitems from the list.
 * ***************************************************************************
 */
void filelist_clear (filelist_t * filelist);


/* 
 * ***************************************************************************
 * Name:            filelist_sort
 * Access:          public
 *
 * Description:     Sorts the list. swap_cb is a callback function, that is
 *                  used to determin if two entries must be swapped.
 * ***************************************************************************
 */
void filelist_sort (filelist_t * filelist, l_swap_cb_t swap_cb);


/* 
 * ***************************************************************************
 * Name:            filelist_add
 * Access:          public
 *
 * Description:     Adds a new entry to the list.
 * ***************************************************************************
 */
fileitem_t *filelist_add (filelist_t * filelist, const char *title,
                          const char *mrl, fileitem_type_t type);


/* 
 * ***************************************************************************
 * Name:            filelist_remove
 * Access:          public
 *
 * Description:     Removes the fileitem from the list.
 * ***************************************************************************
 */
void filelist_remove (filelist_t * filelist, fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            filelist_expand
 * Access:          public
 *
 * Description:     Depending on the type of the fileitem it is expanded.
 *                  Expanding means that any directory, playlist or
 *                  mediamarks-file is read and added to the sublist of the
 *                  fileitem.
 * ***************************************************************************
 */
void filelist_expand (filelist_t * parent, fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            filelist_first
 * Access:          public
 *
 * Description:     Returns the first fileitem.
 * ***************************************************************************
 */
fileitem_t *filelist_first (filelist_t * filelist);


/* 
 * ***************************************************************************
 * Name:            filelist_next
 * Access:          public
 *
 * Description:     Returns the next fileitem.
 * ***************************************************************************
 */
fileitem_t *filelist_next (filelist_t * filelist, fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            filelist_contains
 * Access:          public
 *
 * Description:     Determines, if the list contains an entry with the given
 *                  MRL.
 * ***************************************************************************
 */
int filelist_contains (filelist_t * filelist, const char *mrl);


/* 
 * ***************************************************************************
 * Name:            create_title
 * Access:          public
 *
 * Description:     Creates a title from the MRL. The returned string must be
 *                  freed when not used any more.
 * ***************************************************************************
 */
char *create_title (const char *mrl);
#endif
