/* -*- c++ -*-
*
* Copyright (C) 2004, 2005, 2006 Mekensleep
*
* Mekensleep
* 24 rue vieille du temple
* 75004 Paris
*       licensing@mekensleep.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301, USA.
*
* Authors:
*  Loic Dachary <loic@gnu.org>
*  Cedric Pinson <cpinson@freesheep.org>
*
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef USE_NPROFILE
#include <nprofile/profile.h>
#else  // USE_NPROFILE
#define NPROFILE_SAMPLE(a)
#endif // USE_NPROFILE

#include <osgCal/Model>

#include <osgDB/Registry>
#include <osgDB/FileUtils>
#include <osgDB/FileNameUtils>

#define LINE_BUFFER_SIZE 4096

bool readObject(const std::string& fileName, osgCal::CoreModel& coreModel) {
  // Read file and update model
  CalCoreModel* calCoreModel  = coreModel.getCalCoreModel();
  osg::notify(osg::NOTICE) << "Loading cal3d character from " << fileName.c_str() << std::endl;

  float scale = 1.0f;
  bool bScale = false;

  FILE* f = fopen(fileName.c_str(), "r");
  if(!f)
    {
      osg::notify(osg::WARN) << "Opening failure of cal cfg file " << fileName << std::endl;
      return false;
    }

  // Extract path from fileName
  std::string dir = osgDB::getFilePath( fileName );
  if(dir == "") dir = ".";

  char buffer[LINE_BUFFER_SIZE];

  while (fgets(buffer, LINE_BUFFER_SIZE,f))
    {
      // Ignore comments or empty lines
      if (*buffer == '#' || *buffer == 0)
        continue;

      char* equal = strchr(buffer, '=');
      if (equal)
        {
          // Terminates first token
          *equal++ = 0;
          // Removes ending newline (CR & LF)
          {
            int last = strlen(equal) - 1;
            if (equal[last] == '\n') equal[last] = 0;
            if (last > 0 && equal[last-1] == '\r') equal[last-1] = 0;
          }
          std::string fullpath = dir + "/" + std::string(equal);

          if (!strcmp(buffer, "scale")) 
            { 
              bScale  = true;
              scale = atof( equal );
              continue;
            }

          osg::notify(osg::NOTICE) << "    subloading " << buffer << ": " << fullpath.c_str() << std::endl;

          if (!strcmp(buffer, "skeleton")) 
            {
              if(!calCoreModel->loadCoreSkeleton(fullpath))
                {
                  osg::notify(osg::WARN) << "Loading cal CFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
                }         
            } 
          else if (!strcmp(buffer, "animation"))
            {
              if(calCoreModel->loadCoreAnimation(fullpath) < 0)
                {
                  osg::notify(osg::WARN) << "Loading cal CFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
                }
            }
          else if (!strcmp(buffer, "mesh"))
            {
              if(calCoreModel->loadCoreMesh(fullpath) < 0)
                {
                  osg::notify(osg::WARN) << "Loading cal CFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
                }
            }
          else if (!strcmp(buffer, "material"))  
            {
              int materialId = calCoreModel->loadCoreMaterial(fullpath);

              if(materialId < 0) 
                {
                  osg::notify(osg::WARN) << "Loading cal CFG error: " << CalError::getLastErrorDescription().c_str() << std::endl;
                } 
              else 
                {
                  calCoreModel->createCoreMaterialThread(materialId); 
                  calCoreModel->setCoreMaterialId(materialId, 0, materialId);


                  CalCoreMaterial* material = calCoreModel->getCoreMaterial(materialId);
                  // the texture file path is relative to the CRF data directory
                  for(std::vector<CalCoreMaterial::Map>::iterator i = material->getVectorMap().begin();
                      i != material->getVectorMap().end();
                      i++) 
                    {
                      i->strFilename = dir + "/" + i->strFilename;
                    }

                }
            }
        }
    }

  // scaling must be done after everything has been created
  if( bScale )
    {
      osg::notify(osg::NOTICE) << "scaling model by " << scale << " factor" << std::endl;
      calCoreModel->scale( scale );
    }
  fclose(f);

  return true;  
}

bool writeObject(const std::string& fileName, const osgCal::CoreModel& coreModel) {
  return true;
}

//
// OSG interface to read/write from/to a file.
//
class ReaderWriterCFG : public osgDB::ReaderWriter {
public:
  virtual const char* className() { return "osg Cal cfg object reader"; }

  virtual bool acceptsExtension(const std::string& extension) const
  {
    return osgDB::equalCaseInsensitive(extension, "cfg");
  }

  virtual ReadResult readObject(const std::string& file, const osgDB::ReaderWriter::Options* options) const
  {
    std::string ext = osgDB::getLowerCaseFileExtension(file);
    if (!acceptsExtension(ext)) return ReadResult::FILE_NOT_HANDLED;

    osg::notify(osg::NOTICE)<<"file="<<file<<std::endl;

    std::string fileName = osgDB::findDataFile( file, options );

    osg::notify(osg::NOTICE)<<"fileName="<<fileName<<std::endl;

    if (fileName.empty()) return ReadResult::FILE_NOT_FOUND;

    osg::notify(osg::INFO)<<   "ReaderWriterCFG::readObject( "<<fileName.c_str()<<" )\n";

    osg::ref_ptr<osgCal::CoreModel> coreModel = new osgCal::CoreModel;
    if(!::readObject(fileName, *coreModel))
      return ReadResult("failed to load correctly " + fileName);
    else
      return ReadResult(coreModel.get());
  }
  virtual WriteResult writeObject(const osg::Object& object, const std::string& fileName, const Options* =NULL)
  {
    if(!::writeObject( fileName, static_cast<const osgCal::CoreModel&>(object) ))
      return WriteResult("failed to save coreModel in file " + fileName);
    else
      return WriteResult(WriteResult::FILE_SAVED);
  }
};

osgDB::RegisterReaderWriterProxy<ReaderWriterCFG> static_readerWriter_CFG_Proxy;
