//                                               -*- C++ -*-
/**
 *  @file  LogNormalFactory.cxx
 *  @brief Factory for LogNormal distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: LogNormalFactory.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <cmath>
#include "LogNormalFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS




/* Default constructor */
LogNormalFactory::LogNormalFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
LogNormalFactory * LogNormalFactory::clone() const
{
  return new LogNormalFactory(*this);
}


/* Here is the interface that all derived class must implement */

LogNormal * LogNormalFactory::build(const NumericalSample & sample) const
{
  const UnsignedLong size(sample.getSize());
  if (size == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a LogNormal distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a LogNormal distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  /* Maximum likelihood estimate:
     S_0 = \sum_i (X_i - \gamma)^{-1}
     S_1 = \sum_i \log(X_i - \gamma)
     S_2 = \sum_i \log^2(X_i - \gamma)
     S_3 = \sum_i \log(X_i - \gamma) / (X_i - \gamma)
     S_4 = \sum_i i = N
     \mu = S_1 / S_4
     \sigma^2 = S_2 / S_4 - \mu^2
     \gamma = S_0 (\sigma^2 - \mu) + S_3
  */
  const NumericalScalar std(sample.computeStandardDeviationPerComponent()[0]);
  const NumericalScalar quantileEpsilon(ResourceMap::GetAsNumericalScalar("DistributionImplementation-DefaultQuantileEpsilon"));
  NumericalScalar step(quantileEpsilon * std);
  const NumericalScalar xMin(sample.getMin()[0]);
  NumericalScalar right(xMin - quantileEpsilon);
  NumericalScalar constraintRight(ComputeConstraint(ComputeMaximumLikelihoodSums(right, sample)));
  NumericalScalar left(right - step);
  NumericalScalar constraintLeft(ComputeConstraint(ComputeMaximumLikelihoodSums(left, sample)));
  // First, the bracketing interval. We should find a change of sign within [Xmin-sigma, Xmin], else use another estimator
  while ((constraintLeft < 0.0) == (constraintRight < 0.0) && (left > xMin - std))
    {
      right = left;
      constraintRight = constraintLeft;
      left -= step;
      constraintLeft = ComputeConstraint(ComputeMaximumLikelihoodSums(left, sample));
      step *= 2.0;
    }
  // If we were not able to bracket gamma, use the old estimator
  if (left <= xMin - std)
    {
      LOGWARN(OSS() << "Warning! Unable to bracket the location parameter gamma. Using old estimate based on the minimum of data.");
      const NumericalScalar gamma(xMin - std * quantileEpsilon);
      NumericalScalar mu(0.0);
      NumericalScalar sigma(0.0);
      for (UnsignedLong i = 0; i < size; ++i)
        {
          const NumericalScalar x(log(sample[i][0] - gamma));
          const NumericalScalar delta(x - mu);
          const NumericalScalar factor(1.0 / (i + 1.0));
          sigma = i * sigma * factor + (1.0 - factor) * delta * delta * factor;
          mu += delta * factor;
        }
      return LogNormal(mu, sigma, gamma).clone();
    }
  // Second, the bisection
  while (fabs(right - left) > (1.0 + fabs(right + left)) * quantileEpsilon)
    {
      const NumericalScalar guess(0.5 * (left + right));
      const NumericalScalar constraintGuess(ComputeConstraint(ComputeMaximumLikelihoodSums(guess, sample)));
      if ((constraintGuess < 0.0) == (constraintLeft < 0.0))
        {
          left = guess;
          constraintLeft = constraintGuess;
        }
      else
        {
          right = guess;
          constraintRight = constraintGuess;
        }
    }
  // Third, the final estimates
  const NumericalScalar gamma(0.5 * (left + right));
  const NumericalPoint sums(ComputeMaximumLikelihoodSums(gamma, sample));
  const NumericalScalar mu(sums[1] / sums[4]);
  const NumericalScalar sigma(sqrt(sums[2] / sums[4] - mu * mu));
  return LogNormal(mu, sigma, gamma).clone();
}

LogNormal * LogNormalFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    LogNormal distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a LogNormal distribution from the given parameters";
    }
}

LogNormal * LogNormalFactory::build() const
{
  return LogNormal().clone();
}

/* Compute the different sums needed by the likelihood maximization */
NumericalPoint LogNormalFactory::ComputeMaximumLikelihoodSums(const NumericalScalar gamma,
                                                              const NumericalSample & sample)
{
  const UnsignedLong size(sample.getSize());
  NumericalPoint sums(5, 0.0);
  for (UnsignedLong i = 0; i < size; ++i)
    {
      const NumericalScalar delta(sample[i][0] - gamma);
      const NumericalScalar logDelta(log(delta));
      const NumericalScalar inverseDelta(1.0 / delta);
      sums[0] += inverseDelta;
      sums[1] += logDelta;
      sums[2] += logDelta * logDelta;
      sums[3] += logDelta * inverseDelta;
    }
  sums[4] = size;
  return sums;
}

/* Compute the maximum likelihood constraint for the shift parameter \gamma */
NumericalScalar LogNormalFactory::ComputeConstraint(const NumericalPoint & sums)
{
  return sums[0] * (sums[2] - sums[1] * (1.0 + sums[1] / sums[4])) + sums[4] * sums[3];
}

END_NAMESPACE_OPENTURNS
