//                                               -*- C++ -*-
/**
 *  @file  Wilks.cxx
 *  @brief Wilks is a generic view of Wilks methods for computing
 * probabilities and related quantities by sampling and estimation
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: Wilks.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <cmath>
#include "Wilks.hxx"
#include "Exception.hxx"
#include "NumericalSample.hxx"
#include "DistFunc.hxx"

BEGIN_NAMESPACE_OPENTURNS




/*
 * @class Wilks
 */

/* Constructor */
Wilks::Wilks(const RandomVector & vector)
  : vector_()
{
  // Check if the given vector is 1D as no theory has been made so far (2011) to define a quantile in higher dimension
  if (vector.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: the given vector must be 1D.";
  vector_ = vector;
}

/* Sample size computation */
UnsignedLong Wilks::ComputeSampleSize(const NumericalScalar quantileLevel,
                                      const NumericalScalar confidenceLevel,
                                      const UnsignedLong marginIndex)
{
  if ((quantileLevel <= 0.0) || (quantileLevel >= 1.0)) throw InvalidArgumentException(HERE) << "Error: quantile level must be in ]0, 1[, here quantile level=" << quantileLevel;
  if ((confidenceLevel <= 0.0) || (confidenceLevel >= 1.0)) throw InvalidArgumentException(HERE) << "Error: confidence level must be in ]0, 1[, here confidence level=" << confidenceLevel;
  // Initial guess based on asymptotic normality of the empirical quantile
  NumericalScalar x(DistFunc::qNormal(confidenceLevel));
  NumericalScalar x2(x * x);
  UnsignedLong n((UnsignedLong)(ceil((quantileLevel * x2 + 2.0 * marginIndex + sqrt(quantileLevel * x2 * (quantileLevel * x2 + 4.0 * marginIndex))) / (2.0 * (1.0 - quantileLevel)))));
  // Initialize cQuantileLevel to cQuantileLevel[i] = Binomial(n, n - marginIndex + i) * quantileLevel ^ (n - marginIndex + i) * (1 - quantileLevel) ^ (marginIndex - i)
  // For i = marginIndex, cQuantileLevel[i] = quantileLevel ^ n
  NumericalPoint cQuantileLevel(marginIndex + 1, pow(quantileLevel, n));
  // Descending recursion n! / (m - i)! / (n - m + i)! = n! / (m - (i - 1))! / (n - m + (i - 1))! * (m - (i - 1)) / (n - m + i)
  // -> Binomial(n, n - m + i - 1) = Binomial(n, n - m + i) * (n - m + i) / (m - i)
  for (UnsignedLong i = marginIndex; i > 0; --i) cQuantileLevel[i - 1] = cQuantileLevel[i] * (1.0 - quantileLevel) / quantileLevel * (n - marginIndex + i) / (marginIndex - i + 1.0);
  NumericalScalar cumulQuantileLevel;
  do
    {
      cumulQuantileLevel = 0.0;
      ++n;
      // Update the terms in the sum
      // Binomial(n + 1, n + 1 - m + j) * quantileLevel ^ (n + 1 - m + j) * (1 - quantileLevel) ^ (m - j)= Binomial(n, n - m + j) * quantileLevel ^ (n - m + j) * (1 - quantileLevel) ^ (m - j) * quantileLevel * (n + 1) / (n + 1 - m + j)
      for (UnsignedLong j = 0; j <= marginIndex; ++j)
        {
          cQuantileLevel[j] *= (n * quantileLevel) / (n - marginIndex + j);
          cumulQuantileLevel += cQuantileLevel[j];
        }
    }
  while (cumulQuantileLevel > 1.0 - confidenceLevel);
  return n;
}

/* Estimate an upper bound of the quantile of the random vector for the given quantile level and confidence level, using the marginIndex upper statistics */
NumericalPoint Wilks::computeQuantileBound(const NumericalScalar quantileLevel,
                                           const NumericalScalar confidenceLevel,
                                           const UnsignedLong marginIndex) const
{
  // Compute the needed sample size
  const UnsignedLong size(ComputeSampleSize(quantileLevel, confidenceLevel, marginIndex));
  // Generate a sorted sample of the needed size
  const NumericalSample sample(vector_.getNumericalSample(size).sort());
  // The upper bound is the marginIndex upper statistics
  return sample[size - 1 - marginIndex];
}

END_NAMESPACE_OPENTURNS
