//                                               -*- C++ -*-
/**
 *  @file  NatafIndependentCopulaGradient.cxx
 *  @brief Class for the Nataf transformation evaluation for independent
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NatafIndependentCopulaGradient.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "NatafIndependentCopulaGradient.hxx"
#include "DistFunc.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class NatafIndependentCopulaGradient
       *
       * This class offers an interface for the Nataf function for independent copula
       */

      typedef Distribution::DistFunc DistFunc;

      CLASSNAMEINIT(NatafIndependentCopulaGradient);

      /* Parameter constructor */
      NatafIndependentCopulaGradient::NatafIndependentCopulaGradient(const UnsignedLong dimension):
	NumericalMathGradientImplementation(),
	dimension_(dimension)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      NatafIndependentCopulaGradient * NatafIndependentCopulaGradient::clone() const
      {
	return new NatafIndependentCopulaGradient(*this);
      }

      /* String converter */
      String NatafIndependentCopulaGradient::__repr__() const
      {
	OSS oss;
	oss << "class=" << NatafIndependentCopulaGradient::GetClassName()
            << " dimension=" << dimension_;
                   
	return oss;
      }

      /*
       * Evaluation
       * The Nataf transform T reads:
       * Ti(xi) = Q(xi), where Q = Phi^{-1} and Phi is the CDF of the standard normal distribution
       * We have:
       * Jij = dTi/dxj = Q'(xi) if i = j
       *               = 0 else
       */
      NatafIndependentCopulaGradient::Matrix NatafIndependentCopulaGradient::gradient(const NumericalPoint & in) const
	/* throw(InvalidArgumentException, InternalException) */
      {
	Matrix result(dimension_, dimension_);
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    NumericalScalar x(in[i]);
	    if ((x < 0.0) || (x > 1.0)) throw InvalidArgumentException(HERE) << "Error: cannot evaluate the NatafIndependentCopulaGradient if all the components are not in [0, 1], here in=" << in;
	    // q = Normal(0,1).computeQuantile(x)
	    NumericalScalar q(DistFunc::qNormal(x));
	    // 2.506628274631000502415765 = sqrt(2*Pi)
	    result(i, i) = 2.506628274631000502415765 * exp(0.5 * q * q);
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong NatafIndependentCopulaGradient::getInputDimension() const
	/* throw(InternalException) */
      {
	return dimension_;
      }

      /* Accessor for output point dimension */
      UnsignedLong NatafIndependentCopulaGradient::getOutputDimension() const
	/* throw(InternalException) */
      {
	return dimension_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
