//                                               -*- C++ -*-
/**
 *  @file  DrawableImplementation.hxx
 *  @brief Abstract top-level class for all drawables
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: DrawableImplementation.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_DRAWABLEIMPLEMENTATION_HXX
#define OPENTURNS_DRAWABLEIMPLEMENTATION_HXX

#include <map>
#include "Description.hxx"
#include "Collection.hxx"
#include "PersistentObject.hxx"
#include "NumericalSample.hxx"
#include "Exception.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Graph {

      /**
       * @class DrawableImplementation
       *
       * The class describes the a drawable graphic
       * Instances of this class can actually be plotted
       */  

      class DrawableImplementation :
	public Common::PersistentObject
      {

	CLASSNAME;

      public:

	typedef Type::NumericalPoint               NumericalPoint;
	typedef Type::Description                  Description;
	typedef NumericalPoint                     BoundingBox;
	typedef Common::InvalidArgumentException   InvalidArgumentException;
	typedef Common::StorageManager             StorageManager;
	typedef Common::PersistentObject           PersistentObject;
	typedef Stat::NumericalSample              NumericalSample;

	/** default graphic paramaters */
	static String NoSpecifiedLabel;
	
	static String DefaultColor;

	static String DefaultSurfaceColor;
	
	static String DefaultFillStyle;
	
	static String DefaultPointStyle;
	
	static String DefaultLineStyle;

	static String DefaultPattern;

	static String AlternativePattern;

	static const UnsignedLong DefaultLineWidth;

	static const UnsignedLong DataThreshold;

	/** Default Constructor with legend label */
	DrawableImplementation(const NumericalSample & data,
			       const String & legend = NoSpecifiedLabel)
	  /* throw(InvalidArgumentException) */;
       
	/** String converter */
	virtual String __repr__() const;

	/** Comparison operator */
	Bool operator == (const DrawableImplementation & other) const;

	/** Code for point style accessor */
	virtual UnsignedLong getPointCode(const String & key) const;

	/** Check validity of data */
	virtual Bool isValidData(const NumericalSample & data) const;

	/** Check validity of color */
	virtual Bool isValidColor(const String & key) const;

	/** Check validity of line style */
	virtual Bool isValidLineStyle(const String & key) const;

	/** Check validity of point style */
	virtual Bool isValidPointStyle(const String & key) const;

	/** Check validity of fill style */
	virtual Bool isValidFillStyle(const String & key) const;
	
	/** Check validity of pattern parameter */
	virtual Bool isValidPattern(const String & pattern) const;

	/** Legend name accessor */
	virtual String getLegendName() const;
	virtual void setLegendName(const String & legendName);

	/** Data accessor */
	virtual NumericalSample getData() const;

	/** Line style accessor */
	virtual String getLineStyle() const;
	virtual void setLineStyle(const String & lineStyle)
	  /* throw(InvalidArgumentException) */;

	/** Color accessor */
	virtual String getColor() const;
	virtual void setColor(const String & color)
	  /* throw(InvalidArgumentException) */;
	
	/** Point style accessor */
	virtual String getPointStyle() const;
	virtual void setPointStyle(const String & pointStyle)
	  /* throw(InvalidArgumentException) */;

	/** Filling style (for surfaces) accessor */
	virtual String getFillStyle() const;
	virtual void setFillStyle(const String & fillStyle)
	  /* throw(InvalidArgumentException) */;
	
	/** Line width accessor */
	virtual UnsignedLong getLineWidth() const;
	virtual void setLineWidth(const UnsignedLong lineWidth)
	  /* throw(InvalidArgumentException) */;
	
	/** Accessor for pattern */
	virtual String getPattern() const;
	virtual void setPattern(const String style) /* throw(InvalidArgumentException) */;

	/** Accessor for center */
	virtual NumericalPoint getCenter() const;
	virtual void setCenter(const NumericalPoint & center);

	/** Accessor for radius */
	virtual NumericalScalar getRadius() const;
	virtual void setRadius(const NumericalScalar radius);
	
	/** Accessor for labels */
	virtual Description getLabels() const;
	virtual void setLabels(const Description & labels);

	/** Accessor for color palette */
	virtual Description getPalette() const;
	virtual void setPalette(const Description & palette)
	  /* throw(InvalidArgumentException) */;

	/** Accessor for origin */
	virtual NumericalScalar getOrigin() const;
	virtual void setOrigin(const NumericalScalar origin);

	/** Accessor for first coordinate */
	virtual NumericalSample getX() const;
	virtual void setX(const NumericalSample & x);

	/** Accessor for second coordinate */
	virtual NumericalSample getY() const;
	virtual void setY(const NumericalSample & y);
	
	/** Accessor for levels */
	virtual NumericalPoint getLevels() const;
	virtual void setLevels(const NumericalPoint & levels);

	/** Accessor for drawLabels */
	virtual Bool getDrawLabels() const;
	virtual void setDrawLabels(const Bool & drawLabels);

	/** Bounding box accessor */
	virtual BoundingBox getBoundingBox() const;
	
	/** R command generating method, for plotting through R */
	virtual String draw() const;

	/** Clean all the temporary data created by draw() method */
	virtual void clean() const;

	/** Clone method */
	virtual DrawableImplementation * clone() const;

	/** Give the colors name */
	static Description GetValidColors();
	
	/** Give the line style names */
	static Description GetValidLineStyles();
	
	/** Give the fill style names */
	static Description GetValidFillStyles();
	
	/** Give the point style names */
	static Description GetValidPointStyles();
	
 	/** Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

 	/** Method load() stores the object through the StorageManager */
	void load(StorageManager::Advocate & adv);

      protected:
	DrawableImplementation() {};
        friend class Base::Common::Factory<DrawableImplementation>;

	static Bool IsFirstInitialization;
	
	/** A map matching keys with R codes for point symbols */
	static std::map<String, UnsignedLong> SymbolCodes;

	/** Valid colors */
	static Description ValidColors;

	/** Valid line styles */
	static Description ValidLineStyles;

	/** Valid fill styles */
	static Description ValidFillStyles;

	/** Bounding box size */
	static const UnsignedLong BoundingBoxSize;

	/** Data accessor */
	virtual void setData(const NumericalSample & data)
	  /* throw(InvalidArgumentException) */;

	/** The name of the drawable, to be displayed in the legend */
	String legendName_;
	
	/** The data to be plotted */
	NumericalSample data_;

	/** Color of the curve */
	String color_;

	/** Fill style */
	String fillStyle_;

	/** Line style of the curve */
	String lineStyle_;

	/** Line style of the curve */
	String pointStyle_;

	/** Line width of the curve */
	UnsignedLong lineWidth_;

	/** Initialize Valid parameter list */
	static void InitializeValidParameterList();

      private:
	/** Temporary file name for the exchange with R */
	mutable String dataFileName_;

      }; /* class DrawableImplementation */

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */

#endif /* OPENTURNS_DRAWABLEIMPLEMENTATION_HXX */
