//                                               -*- C++ -*-
/**
 * @file  NumericalMathEvaluationImplementation.hxx
 * @brief Abstract top-level class for all numerical math function implementations
 *
 * (C) Copyright 2005-2010 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 */

#ifndef OPENTURNS_NUMERICALMATHEVALUATIONIMPLEMENTATION_HXX
#define OPENTURNS_NUMERICALMATHEVALUATIONIMPLEMENTATION_HXX

#include "PersistentObject.hxx"
#include "NumericalPoint.hxx"
#include "NumericalPointWithDescription.hxx"
#include "NumericalSample.hxx"
#include "Exception.hxx"
#include "Description.hxx"
#include "Matrix.hxx"
#include "Collection.hxx"
#include "Pointer.hxx"
#include "Cache.hxx"

namespace OpenTURNS {

  namespace Base {

    namespace Func {


      /**
       * @class NumericalMathEvaluationImplementation
       *
       * This class offers an abstract interface for the implementation
       * of an real numerical mathematical function into the platform.
       */
      class NumericalMathEvaluationImplementation 
	: public Common::PersistentObject
      {
	CLASSNAME;
      public:

	typedef Common::Pointer<NumericalMathEvaluationImplementation> Implementation;
	typedef Type::Description                                      Description;
	typedef Type::NumericalPoint                                   NumericalPoint;
	typedef Type::Matrix                                           Matrix;
	typedef Common::InvalidArgumentException                       InvalidArgumentException;
	typedef Common::InternalException                              InternalException;
	typedef Stat::NumericalSample                                  NumericalSample;
	typedef Common::StorageManager                                 StorageManager;
	typedef Type::PersistentCollection<NumericalScalar>            CacheKeyType;
	typedef Type::PersistentCollection<NumericalScalar>            CacheValueType;
	typedef Type::Cache<CacheKeyType,CacheValueType>               CacheType;
	typedef Common::Pointer<CacheType>                             CacheImplementation;
	typedef Type::NumericalPointWithDescription                    NumericalPointWithDescription;

	/** Default constructor */
	NumericalMathEvaluationImplementation();

	/** Virtual constructor */
	virtual NumericalMathEvaluationImplementation * clone() const;

	/** Comparison operator */
	Bool operator ==(const NumericalMathEvaluationImplementation & other) const;

	/** String converter */
	virtual String __repr__() const;

	/** Description Accessor, i.e. the names of the input and output parameters */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Input description Accessor, i.e. the names of the input parameters */
	Description getInputDescription() const;

	/** Output description Accessor, i.e. the names of the Output parameters */
	Description getOutputDescription() const;

	/** Enable or disable the internal cache */
	void enableCache() const;
	void disableCache() const;
	Bool isCacheEnabled() const;

	/* Here is the interface that all derived class must implement */
	
	/** Test for actual implementation */
	virtual Bool isActualImplementation() const;

	/** Operator () */
	virtual NumericalPoint operator() (const NumericalPoint & in) const
	  /* throw(InvalidArgumentException, InternalException) */;

	/** Operator () on a sample, not pure virtual because a generic implementation is given */
	virtual NumericalSample operator() (const NumericalSample & inSample) const
	  /* throw(InvalidArgumentException, InternalException) */;

	/** Accessor for input point dimension */
	virtual UnsignedLong getInputDimension() const
	  /* throw(InternalException) */;

	/** Accessor for output point dimension */
	virtual UnsignedLong getOutputDimension() const
	  /* throw(InternalException) */;

	/** Gradient according to the marginal parameters */
	virtual Matrix parametersGradient(const NumericalPoint & in) const;

	/** Parameters value and description accessor */
	virtual NumericalPointWithDescription getParameters() const;
	virtual void setParameters(const NumericalPointWithDescription & parameters);

	/** Get the number of calls to operator() */
	UnsignedLong getCallsNumber() const;

 	/** Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(StorageManager::Advocate & adv);


      protected:

	/** Number of calls since the construction */
	mutable UnsignedLong callsNumber_;

	/* A cache to store already computed points */
	mutable CacheImplementation p_cache_;

      private:

 	/** The description of all the components */
	Description description_;

	/** The value and description of all the parameters */
	NumericalPointWithDescription parameters_;

      }; /* class NumericalMathEvaluationImplementation */


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_NUMERICALMATHEVALUATIONIMPLEMENTATION_HXX */
