//                                               -*- C++ -*-
/**
 * @file  NumericalMathEvaluationImplementation.cxx
 * @brief Abstract top-level class for all distributions
 *
 * (C) Copyright 2005-2010 EDF
 *
 * Permission to copy, use, modify, sell and distribute this software
 * is granted provided this copyright notice appears in all copies.
 * This software is provided "as is" without express or implied
 * warranty, and with no claim as to its suitability for any purpose.
 *
 *
 * \author $LastChangedBy: dutka $
 * \date   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 */

#include "NumericalMathEvaluationImplementation.hxx"
#include "Exception.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Base {


    namespace Type {

      typedef Func::NumericalMathEvaluationImplementation::CacheType NumericalMathEvaluationImplementationCache;
      TEMPLATE_CLASSNAMEINIT(NumericalMathEvaluationImplementationCache);

      static Common::Factory<NumericalMathEvaluationImplementationCache> RegisteredFactory_Cache("NumericalMathEvaluationImplementationCache");

      /* These methods are implemented here for the needs of Cache */
      /* We should be careful because they may interfere with other definitions placed elsewhere */
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<UnsignedLong>);
      static Common::Factory<PersistentCollection<UnsignedLong> > RegisteredFactory_alt1("PersistentCollection<UnsignedLong>");
      TEMPLATE_CLASSNAMEINIT(PersistentCollection<PersistentCollection<NumericalScalar> >);
      static Common::Factory<PersistentCollection<PersistentCollection<NumericalScalar> > > RegisteredFactory_alt2("PersistentCollection<PersistentCollection<NumericalScalar> >");

    } /* namespace Type */



    namespace Func {

      using Common::NotYetImplementedException;

      CLASSNAMEINIT(NumericalMathEvaluationImplementation);

      static Common::Factory<NumericalMathEvaluationImplementation> RegisteredFactory("NumericalMathEvaluationImplementation");

      /* Default constructor */
      NumericalMathEvaluationImplementation::NumericalMathEvaluationImplementation()
	: PersistentObject(),
	  callsNumber_(0),
	  p_cache_(new CacheType),
	  description_(0),
	  parameters_(0)
      {
	// We disable the cache by default
	p_cache_->disable();
      }

      /* Virtual constructor */
      NumericalMathEvaluationImplementation * NumericalMathEvaluationImplementation::clone() const
      {
	return new NumericalMathEvaluationImplementation(*this);
      }


      /* Comparison operator */
      Bool NumericalMathEvaluationImplementation::operator ==(const NumericalMathEvaluationImplementation & other) const
      {
	return true;
      }
  
      /* String converter */
      String NumericalMathEvaluationImplementation::__repr__() const {
	OSS oss;
	oss << "class=" << NumericalMathEvaluationImplementation::GetClassName()
	    << " name=" << getName()
	    << " description=" << description_
	    << " parameters=" << parameters_;
	return oss;
      }
  
      /* Description Accessor */
      void NumericalMathEvaluationImplementation::setDescription(const Description & description)
      {
	if (description.getSize() != getInputDimension() + getOutputDimension()) throw InvalidArgumentException(HERE) << "Error: the description must have a size of input dimension + output dimension, here size=" << description.getSize() << ", input dimension=" << getInputDimension() << ", output dimension=" << getOutputDimension();
	description_ = description;
      }


      /* Description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getDescription() const
      {
	return description_;
      }

      /* Input description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getInputDescription() const
      {
	Description inputDescription(0);
	// Check if the description has been set
	if (description_.getSize() > 0) for (UnsignedLong i = 0; i < getInputDimension(); i++) inputDescription.add(description_[i]);
	else for (UnsignedLong i = 0; i < getInputDimension(); i++) inputDescription.add((OSS() << "x" << i));
	return inputDescription;
      }

      /* Output description Accessor */
      NumericalMathEvaluationImplementation::Description NumericalMathEvaluationImplementation::getOutputDescription() const
      {
	Description outputDescription(0);
	// Check if the description has been set
	if (description_.getSize() > 0) for (UnsignedLong i = getInputDimension(); i < description_.getSize(); i++) outputDescription.add(description_[i]);
	else for (UnsignedLong i = 0; i < getOutputDimension(); i++) outputDescription.add((OSS() << "y" << i));
	return outputDescription;
      }

      /* Test for actual implementation */
      Bool NumericalMathEvaluationImplementation::isActualImplementation() const
      {
	return true;
      }

      /* Here is the interface that all derived class must implement */
	
      /* Operator () */
      NumericalMathEvaluationImplementation::NumericalSample NumericalMathEvaluationImplementation::operator() (const NumericalSample & inSample) const
	/* throw(InvalidArgumentException, InternalException) */
      {
	UnsignedLong size(inSample.getSize());
	NumericalSample outSample(size, NumericalPoint(this->getOutputDimension()));
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    outSample[i] = operator()(inSample[i]);
	  }
	return outSample;
      }


      /* Enable or disable the internal cache */
      void NumericalMathEvaluationImplementation::enableCache() const {
	p_cache_->enable();
      }

      void NumericalMathEvaluationImplementation::disableCache() const {
	p_cache_->disable();
      }

      Bool NumericalMathEvaluationImplementation::isCacheEnabled() const {
	return p_cache_->isEnabled();
      }


      /* Gradient according to the marginal parameters */
      NumericalMathEvaluationImplementation::Matrix NumericalMathEvaluationImplementation::parametersGradient(const NumericalPoint & in) const
      {
	return Matrix(parameters_.getDimension(), getOutputDimension());
      }

      /* Parameters value and description accessor */
      NumericalMathEvaluationImplementation::NumericalPointWithDescription NumericalMathEvaluationImplementation::getParameters() const
      {
	return parameters_;
      }

      void NumericalMathEvaluationImplementation::setParameters(const NumericalPointWithDescription & parameters)
      {
	parameters_ = parameters;
      }

      /* Operator () */
      NumericalMathEvaluationImplementation::NumericalPoint NumericalMathEvaluationImplementation::operator() (const NumericalPoint & in) const
	/* throw(InvalidArgumentException, InternalException) */
      {
	throw NotYetImplementedException(HERE);
      }

      /* Accessor for input point dimension */
      UnsignedLong NumericalMathEvaluationImplementation::getInputDimension() const
	/* throw(InternalException) */
      {
	throw NotYetImplementedException(HERE);
      }

      /* Accessor for output point dimension */
      UnsignedLong NumericalMathEvaluationImplementation::getOutputDimension() const
	/* throw(InternalException) */
      {
	throw NotYetImplementedException(HERE);
      }

      /* Get the number of calls to operator() */
      UnsignedLong NumericalMathEvaluationImplementation::getCallsNumber() const
      {
	return callsNumber_;
      }

      /* Method save() stores the object through the StorageManager */
      void NumericalMathEvaluationImplementation::save(StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.saveAttribute( "callsNumber_", callsNumber_ );
	adv.saveAttribute( "cache_", *p_cache_ );
	adv.saveAttribute( "description_", description_ );
	adv.saveAttribute( "parameters_", parameters_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void NumericalMathEvaluationImplementation::load(StorageManager::Advocate & adv)
      {
	Common::TypedInterfaceObject<CacheType> cache;
	PersistentObject::load(adv);
	adv.loadAttribute( "callsNumber_", callsNumber_ );
	adv.loadAttribute( "cache_", cache );
	p_cache_ = cache.getImplementation();
	adv.loadAttribute( "description_", description_ );
	adv.loadAttribute( "parameters_", parameters_ );
      }

    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
