/*
** Copyright (C) 2000 Idan Shoham <idan@m-tech.ab.ca>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <iostream>
#include <string>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include "project.h"
#include "projectFile.h"
#include "vacation.h"

/// Returns 1 if matches, 0 if doesn't 
int Project::do_match(const char *str, const char *regexp, int case_sig)
{
    const char * p;

    for ( p = regexp ; *p && *str ;  )
    {
	switch ( *p )
	{
	    case '?':
		str++;
		p++;
		break;
  
	    case '*':
		/* Look for a character matching 
	           the one after the '*' */
		p++;
		if (!*p)
		    return 1; /* Automatic match */
  
		while(*str)
		{
		    while ( *str && (case_sig ? (*p != *str) : (toupper(*p)!=toupper(*str))))
			str++;
		    if (do_match(str, p, case_sig))
			return 1;
		    if (!*str)
			return 0;
		    else
			str++;
		}
  
		return 0;
  
	    default:
		if (case_sig)
		{
		    if (*str != *p)
			return 0;
		}
		else
		{
		    if (toupper(*str) != toupper(*p))
			return 0;
		}
  
		str++, p++;
		break;
	}
    }

    if ( !*p && !*str )
	return 1;

    if ( ( !*p && str[0] == '.' ) && ( str[1] == 0 ) )
	return(1);
  
    if (!*str && *p == '?')
    {
	while (*p == '?')
	    p++;
	return !*p;
    }

    if (!*str && (*p == '*' && p[1] == '\0'))
	return 1;

    return 0;
}



TASK *Project::FindTask(const char *id)
{
    for ( TPLCI pt = mTaskList.begin() ;
	  pt != mTaskList.end() ;  
	  pt++ )
    {
	if ( (*pt) == NULL )
	    Error("Empty list entry found in FindTask()");
	if ( (*pt)->id() == NULL )
	    Error("Task with no id found in FindTask()");
	if ( strcasecmp((*pt)->id(),id) == 0 )
	    return *pt;
    }

    return NULL;
}

MILESTONE *Project::FindMilestone(char *id)
{
    for ( MPLCI pm = mMilestoneList.begin() ; pm != mMilestoneList.end() ;  pm++ )
    {
	if ( (*pm)->id() == NULL )
	    Error("Milestone with no id found in FindMilestone()");
	if ( strcasecmp((*pm)->id(),id) == 0 )
	    return *pm;
    }

    return NULL;
}

void Project::DaysDone(char *taskid, int days)
{
    TASK *task;

    if ( ( task = FindTask(taskid) ) == NULL )
	Error("Invalid task id %s", taskid);

    task->setDaysDone( days );
}


void ProjectFile::AddTaskGraph( char *d1name, char *d2name, char *fname )
{
    int d1, d2;

    d1 = mProject.FindDay(d1name);
    if ( d1 == -1 )
	Error("Invalid day name %s", d1name);

    d2 = mProject.FindDay(d2name);
    if ( d2 == -1 )
	Error("Invalid day name %s", d2name);

    mReporter.AddTaskGraph(d1, d2, fname);
}


void Reporter::AddTaskGraph( int d1, int d2, char *fname )
{
    string fn = fname;
    TASKGRAPH * tg = new TASKGRAPH( d1, d2, fn );
    mTaskGraphs.push_back(tg);
}


void Reporter::AddNetworkDiagram(int c1, int c2, char *fname)
{
    if ( c1 < 0 )
	Error("Invalid first column '%d'", c1 );
    if( c2 < c1 )
	Error("Invalid second column '%d'", c2 );

    string fn = fname;
    TASKNET * tn = new TASKNET( c1, c2, fn );
    mTaskNets.push_back(tn);
}


void Project::AddNetworkX(char *taskid, char *xstr)
{
    TASK *task;

    if ( ( task = FindTask(taskid) ) == NULL )
	Error("Invalid task id %s", taskid);

    task->nx = atoi(xstr);
}




void Project::AddNetworkY(char *taskid, char *ystr)
{
    TASK *task;

    if ( ( task = FindTask(taskid) ) == NULL )
	Error("Invalid task id %s", taskid);

    task->ny = atoi(ystr);
}


void Project::AddStartNetworkY(char *ystr)
{
    pc_start_ny = atoi(ystr);
}


void Project::AddFinishNetworkY(char *ystr)
{
    pc_finish_ny = atoi(ystr);
}


RESOURCE *Project::FindResource(const char *id)
{
    for ( RPLCI rl = mResourceList.begin() ; rl !=  mResourceList.end() ; rl++ )
    {
	if ( (*rl)->id() == NULL )
	    Error("Resource ID == NULL in FindResource");
	if( strcasecmp((*rl)->id(),id)==0 )
	    return *rl;
    }

    return NULL;
}


void Project::AddResource(char *id, char *name)
{

    if ( FindResource(id) != NULL )
	Error("Duplicate resource definition for %s", id);

    RESOURCE * r = new RESOURCE( id, name );
    
    mResourceList.push_back(r);
}


void Project::SetEfficiency(char *id, double e)
{
    RESOURCE *r;

    if ( ( r = FindResource(id) ) == NULL )
	Error("Invalid resource id %s", id);
     r->setEfficiency(e);
}


void Project::AddGroup(char *id, char **members, int Nmembers)
{
    if ( FindResource(id) != NULL )
	Error("There is already a resource called %s", id);
    AddResource(id, "Resource group");
    RESOURCE * r = FindResource(id);
    r->is_group = 1;
    for ( int i = 0; i < Nmembers; ++i )
    {
	RESOURCE * m = FindResource(members[i]);
	if ( m == NULL )
	    Error("Invalid group member %s", members[i]);
	r->contains.push_back( m );
	m->belongs_to.push_back( r );
    }
}


void Project::AddTask(TASK *task)
{
    if ( FindTask(task->id()) != NULL )
	Error("Duplicate task %s", task->id());
    if ( task->duration() <= 0 )
	Error("Invalid duration '%d'", task->duration());

    mTaskList.push_back(task);
}

void Project::AddTask(char *id, char *name, int duration)
{
    if ( id == NULL )
	Error("NULL task id");

    TASK * t = new TASK(id, name, duration);
    AddTask(t);
}

void Project::Blockify(char *id)
{
    TASK *t;

    if ( ( t = FindTask(id) ) == NULL )
	Error("Invalid task %s", id);

    t->Blockify();
}


void Project::AddDescription(char *id, char *desc)
{
    TASK *t;
    t = FindTask(id);
    if( t==NULL )
	Error("Invalid task id %s",id);
    t->setDesc(desc);
}


void Project::SetCompletion(char *task, char *complete)
{
    TASK * t = FindTask(task);
    if ( t == NULL )
	Error("Invalid task id %s",task);
    double c = atof(complete);
    t->setPercentComplete(c);
}


void ProjectFile::AddCandidates(char *taskid, char **resources, int Nresources)
{
    for ( int i = 0; i < Nresources; ++i )
    {
	mProject.AddCandidate(taskid, resources[i]);
    }
}

void Project::AddCandidate(const char *taskid, const char *resid)
{
    TASK *task;
    RESOURCE *res;

    if ( ( task = FindTask(taskid) ) == NULL )
	Error("Invalid task id %s", taskid);

    if ( (res = FindResource(resid) ) == NULL )
	Error("Invalid resource id %s", resid);

    task->addCandidate(res);
}

void Project::TaskNote(char *taskid, char *text)
{
    TASK *t;
    t = FindTask(taskid);
    if( t==NULL )
	Error("Invalid task ID '%s'", taskid);
    t->AddNote(text);
}


void Project::ResourceNote(char *resid, char *text)
{
    RESOURCE * r = FindResource(resid);
    if( r==NULL )
	Error("Invalid resource ID '%s'", resid);
    r->AddNote(text);
}
 

void Project::AddDependencies(char *taskid, char **tasks, int Ntasks)
{
    TASK *task = FindTask(taskid);

    if ( task == NULL )
	Error("Invalid task id %s", taskid);

    for ( int i = 0; i < Ntasks; ++i )
    {
	TASK *reqtask = FindTask(tasks[i]);
	
	if ( reqtask == NULL )
	    Error("Invalid task id %s", tasks[i]);

	// add a link from this task to the prior task
	task->addDepends(reqtask);

	// add a link from the prior task to this task
	reqtask->addFollows(task);
    }
}


void ProjectFile::checkComplete()
{
    mProject.checkComplete();
    mReporter.checkComplete();
}


void Reporter::checkComplete()
{
    return;
}

void Project::SetDateFormat( char * format )
{
    if ( date_format != NULL )
	Error("Duplicate date format specifier");
    date_format = strdup(format);
}


void Project::checkComplete()
{
    if ( date_format == NULL )
	Error("No date format specified");
    if ( strcasecmp(date_format,"raw")==0 )
	;
    else if( strcasecmp(date_format,"count")==0 )
	;
    else if( strcasecmp(date_format,"calendar")==0 )
	;
    else if( strcasecmp(date_format,"iso")==0 )
	;
    else
	Error("Date_format must be raw/count/calendar/iso");

    // any tasks with no resources?
    for ( TPLCI pt = mTaskList.begin() ;
	  pt != mTaskList.end() ;  
	  pt++ )
    {
	if ( (*pt)->numCandidates() == 0 )
	{
	    Error("Task %s has no resources",(*pt)->id());
	}
    }

    // did we remember to put in a start time?
    if ( startTime == 0 )
    {
	Error("You must specify a start time");
    }
}


void Project::StartTask(char *taskid, char *dayname, int type)
{
   TASK * t = FindTask(taskid);
   if ( t == NULL )
       Error("Can't find task %s", taskid);

   int d = FindDay(dayname);
   if ( d == -1 )
       Error("Can't find day %s", dayname);

   StartTask(t, d, type);
}


void Project::FinishTask(char *taskid, char *dayname, int type)
{
    TASK * t = FindTask(taskid);
    if ( t == NULL )
	Error("Can't find task %s", taskid);

    int d = FindDay(dayname);
    if( d == -1 )
	Error("Can't find day %s", dayname);

    FinishTask(t, d, type);
}

void ProjectFile::WorkBlock(const char *taskid, const char *resid,
			    char *day1, char *day2, TimeBlock::Type type)
{
    int d1 = mProject.FindDay(day1);
    if ( d1 == -1 )
	Error("Can't find day %s", day1);

    int d2 = mProject.FindDay(day2);
    if ( d2 == -1 )
	Error("Can't find day %s", day2);

    mProject.WorkBlock(taskid, resid, d1, d2, type);
}


void Project::WorkBlock(const char *taskid, const char *resid,
			int d1, int d2, TimeBlock::Type type)
{
    TASK * t = FindTask(taskid);
    if ( t == NULL )
	Error("Can't find task %s", taskid);

    RESOURCE * r = FindResource(resid);
    if ( r == NULL )
	Error("Can't find resource %s", resid);

    if ( d2 < d1 )
	Error("Finish date %d before start date %d", d1, d2);


    WorkBlock( t, r, d1, d2, type);
}

/// Create a vacation
void ProjectFile::Vacation(char *resid, char *d1name, char *d2name)
{
    int d1, d2;

    d1 = mProject.FindDay(d1name);
    if ( d1 == -1 )
	Error("Can't find day %s", d1name);
    if ( d2name == NULL )
	d2 = d1;
    else
    {
	d2 = mProject.FindDay(d2name);
	if ( d2 == -1 )
	  Error("Can't find day %s", d2name);
    }

    mProject.Vacation(resid, d1, d2);
}

// Create a vacation
void Project::Vacation(char *resid, int d1, int d2)
{
    int found_resource = 0;

    if ( d2 < d1 )
	Error("Invalid vacation.  Start after finish?");

    for ( RPLCI rl = mResourceList.begin() ; rl != mResourceList.end() ; rl++ )
    {
	RESOURCE * r = *rl;
	if ( r->is_group )
	    continue;
	if ( do_match(r->id(), resid, 0) == 0 )
	    continue;

	found_resource = 1;
  
	// printf("Vacation: %s - %d - %d\n",resid,d1,d2);
	VACATION *vacation = new VACATION(1);
	AddTask(vacation);
	AddCandidate(vacation->id(), r->id());
	WorkBlock(vacation->id(), r->id(), d1, d2, TimeBlock::MANUAL_SCHEDULE);
	// printf("Added vacation %s\n",taskid);
    }

    if ( found_resource == 0 )
	Error("Could not find resources matching %s", resid);
}

void Project::AddMilestone(char *id, char *name)
{
    if ( FindMilestone(id) != NULL )
	Error("Duplicate milestone %s", id);

    MILESTONE *m = new MILESTONE(id, name);
    
    mMilestoneList.push_back( m );
}


void Project::AddAfter(char *msid, char **tasks, int Ntasks)
{
    MILESTONE *milestone = FindMilestone(msid);
    if ( milestone == NULL )
	Error("Invalid milestone id %s", msid);

    for ( int i = 0; i < Ntasks; ++i )
    {
	TASK *reqtask = FindTask(tasks[i]);
	if ( reqtask == NULL )
	    Error("Invalid task id %s", tasks[i]);

	milestone->addDepends(reqtask);
    }
}


void Project::SetResourceRate(char *resid, char *camount)
{
    RESOURCE *r;

    r = FindResource(resid);
    if ( r == NULL )
	Error("Can't find resource %s", resid);

    r->setRate(atof(camount));
}


void Project::AddTaskItem(char *taskid, char *camount, char *desc)
{
    TASK * t = FindTask(taskid);
    if ( t == NULL )
	Error("Can't find task %s", taskid);

    ITEM * item = new ITEM(desc, atof(camount));
    t->addItem(item);
}


void Project::SetProjectRate(char *camount)
{
    if ( RESOURCE::defaultRate() != -1.0 )
	Error("Duplicate definition of project rate");

    RESOURCE::setDefaultRate(atof(camount));
}


void Project::AddProjectItem(char *camount, char *desc)
{
    ITEM *item = new ITEM(desc, atof(camount));

    mItems.push_back(item);
}


void Project::SetFinishDate(char *dayname)
{
    int d;

    d = FindDay(dayname);
    if( d == -1 )
	Error("Can't find day %s", dayname);

    finishDay = d;
}


void Project::SetStartTime(const char *year,
			   const char *month,
			   const char *mday) 
{
   struct tm t;
   t.tm_year = atoi(year)-1900;
   t.tm_mon = atoi(month)-1;
   t.tm_mday = atoi(mday);
   t.tm_sec = 0;
   t.tm_min = 0;
   t.tm_hour = 0;
   t.tm_wday = 0;
   t.tm_yday = 0;

   // Convert "struct tm" to time_t
   time_t startTime = mktime(&t); 

   SetStartTime(startTime);
}

void Project::SetStartTime( time_t when )
{
    startTime = when;
}


void ProjectFile::Load(const char *name)
{
    FILE *f;
    char buf[BUFLEN], **words;
    int Nwords;
    int lineno = 0;
    int invalid_lines = 0;

    f = fopen(name, "r");
    if ( f == NULL )
	Error("Cannot open data file %s", name);

    while ( buf == fgets(buf,BUFLEN,f) )
    {
      try
      {
	++lineno;
	Debug("Parsing %s:%d", name, lineno);
	Nwords = ParseLine( buf, &words );
	if ( Nwords==0 || words[0]==NULL || *words[0]==0 || *words[0]=='#' )
	    continue;

	// printf("Parsing: %s (%d)\n",words[0],Nwords);
	// for( i=0; i<Nwords; ++i )
	//   printf("... %d [%s]\n",i, words[i]);

	if ( ( strcasecmp(words[0],"include") == 0 ) && ( Nwords == 2 ) )
	{
	    int old_lineno = lineno;
	    lineno = 1;
	    cout << "Loading " << words[1] << endl;
	    Load(words[1]);
	    lineno = old_lineno;
	}
	else if ( ( strcasecmp(words[0],"resource") == 0 ) && ( Nwords == 3 ) )
	    mProject.AddResource( words[1], words[2] );
	else if ( ( strcasecmp(words[0],"resource_note") == 0 ) && ( Nwords == 3 ) )
	    mProject.ResourceNote(words[1], words[2]);
	else if ( ( strcasecmp(words[0],"efficiency") == 0 ) && ( Nwords == 3 ) )
	    mProject.SetEfficiency(words[1],atof(words[2]));
	else if ( ( strcasecmp(words[0],"task") == 0 ) && ( Nwords == 4 ) )
	    mProject.AddTask(words[1], words[2], atoi(words[3]));
	else if ( ( strcasecmp(words[0],"milestone") == 0 ) && ( Nwords == 3 ) )
	    mProject.AddMilestone(words[1],words[2]);
	else if ( ( strcasecmp(words[0],"task_note") == 0 ) && ( Nwords == 3 ) )
	    mProject.TaskNote(words[1],words[2]);
	else if ( ( strcasecmp(words[0],"block") == 0 ) && ( Nwords == 2 ) )
	    mProject.Blockify(words[1]);
	else if ( ( strcasecmp(words[0],"describe") == 0 ) && ( Nwords == 3 ) )
	    mProject.AddDescription(words[1], words[2]);
	else if ( ( strcasecmp(words[0],"group") == 0 ) && ( Nwords>=3 ) )
	    mProject.AddGroup(words[1],words+2,Nwords-2);
	else if ( ( strcasecmp(words[0],"candidate") == 0 ) && ( Nwords>=3 ) )
	    AddCandidates(words[1], words+2, Nwords-2);
	else if ( ( strcasecmp(words[0],"depends") == 0 ) && ( Nwords>=3 ) )
	    mProject.AddDependencies(words[1],words+2,Nwords-2);
	else if ( ( strcasecmp(words[0],"after") == 0 ) && ( Nwords>=3 ) )
	    mProject.AddAfter(words[1],words+2,Nwords-2);
	else if ( ( strcasecmp(words[0],"complete") == 0 ) && ( Nwords == 3 ) )
	    mProject.SetCompletion(words[1],words[2]);
	else if ( ( strcasecmp(words[0],"complete") == 0 ) && ( Nwords == 2 ) )
	    mProject.SetCompletion(words[1],"100");
	else if ( ( strcasecmp(words[0],"start") == 0 ) && ( Nwords == 3 ) )
	    mProject.StartTask(words[1],words[2],TNORM);
	else if ( ( strcasecmp(words[0],"astart") == 0 ) && ( Nwords == 3 ) )
	    mProject.StartTask(words[1],words[2],TACTUAL);
	else if ( ( strcasecmp(words[0],"bstart") == 0 ) && ( Nwords == 3 ) )
	    mProject.StartTask(words[1],words[2],TBASE);
	else if ( ( strcasecmp(words[0],"finish") == 0 ) && ( Nwords == 3 ) )
	    mProject.FinishTask(words[1],words[2],TNORM);
	else if ( ( strcasecmp(words[0],"afinish") == 0 ) && ( Nwords == 3 ) )
	    mProject.FinishTask(words[1],words[2],TACTUAL);
	else if ( ( strcasecmp(words[0],"bfinish") == 0 ) && ( Nwords == 3 ) )
	    mProject.FinishTask(words[1],words[2],TBASE);
	else if ( ( strcasecmp(words[0],"future") == 0 ) && ( Nwords == 5 ) )
	{
	    WorkBlock(words[1], words[2], words[3], words[4], 
		      TimeBlock::MANUAL_SCHEDULE);
	}
	else if ( ( strcasecmp(words[0],"past") == 0 ) && ( Nwords == 5 ) )
	{
	    WorkBlock(words[1], words[2], words[3], words[4], 
		      TimeBlock::WORK_DONE);
	}
	else if ( ( strcasecmp(words[0],"done") == 0 ) && ( Nwords == 3 ) )
	    mProject.DaysDone(words[1], atoi(words[2]));
	else if ( ( strcasecmp(words[0],"vacation") == 0 ) && ( Nwords == 3 ) )
	    Vacation(words[1], words[2], words[2]);
	else if ( ( strcasecmp(words[0],"vacation") == 0 ) && ( Nwords == 4 ) )
	    Vacation(words[1], words[2], words[3]);
	else if ( ( strcasecmp(words[0],"rate") == 0 ) && ( Nwords == 3 ) )
	    mProject.SetResourceRate(words[1],words[2]);
	else if ( ( strcasecmp(words[0],"item") == 0 ) && ( Nwords == 4 ) )
	    mProject.AddTaskItem(words[1],words[2],words[3]);
	else if ( ( strcasecmp(words[0],"prate") == 0 ) && ( Nwords == 2 ) )
	    mProject.SetProjectRate(words[1]);
	else if ( ( strcasecmp(words[0],"pitem") == 0 ) && ( Nwords == 3 ) )
	    mProject.AddProjectItem(words[1], words[2]);
	else if ( ( strcasecmp(words[0],"startdate") == 0 ) && ( Nwords == 4 ) )
	    mProject.SetStartTime(words[1], words[2], words[3]);
	else if ( ( strcasecmp(words[0], "finishdate") == 0 ) && ( Nwords == 2 ) )
	    mProject.SetFinishDate(words[1]);
	else if ( ( strcasecmp(words[0],"textreport") == 0 ) && ( Nwords == 2 ) )
	    mReporter.text_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"xmlreport") == 0 ) && ( Nwords == 2 ) )
	    mReporter.xml_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"texreport") == 0 ) && ( Nwords == 2 ) )
	    mReporter.tex_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"htmlreport") == 0 ) && ( Nwords == 2 ) )
	    mReporter.html_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0], "show_dependencies") == 0 )
		  && ( Nwords == 1 ) )
	    mReporter.dependencies = 1;
	else if ( ( strcasecmp(words[0], "show_vacations") == 0 )
		  && ( Nwords == 1 ) )
	    mReporter.vacations = 1;
	else if ( ( strcasecmp(words[0], "show_resource_notes") == 0 )
		  && ( Nwords == 1 ) )
	    mReporter.resource_notes = 1;
	else if ( ( strcasecmp(words[0],"show_task_notes") == 0 ) && ( Nwords == 1 ) )
	    mReporter.task_notes = 1;
	else if ( ( strcasecmp(words[0],"show_task_ids") == 0 ) && ( Nwords == 1 ) )
	    mReporter.task_ids = 1;
	else if ( ( strcasecmp(words[0],"show_milestone_ids") == 0 ) && ( Nwords == 1 ) )
	    mReporter.milestone_ids = 1;
	else if ( ( strcasecmp(words[0],"weekly_tex") == 0 ) && ( Nwords == 2 ) )
	    mReporter.weekly_tex_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"weekly_html") == 0 ) && ( Nwords == 2 ) )
	    mReporter.weekly_html_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"weekly_txt") == 0 ) && ( Nwords == 2 ) )
	    mReporter.weekly_txt_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"slippage_tex") == 0 ) && ( Nwords == 2 ) )
	    mReporter.slippage_tex_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"slippage_html") == 0 ) && ( Nwords == 2 ) )
	    mReporter.slippage_html_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"slippage_txt") == 0 ) && ( Nwords == 2 ) )
	    mReporter.slippage_txt_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"monthly_tex") == 0 ) && ( Nwords == 2 ) )
	    mReporter.monthly_tex_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"monthly_html") == 0 ) && ( Nwords == 2 ) )
	    mReporter.monthly_html_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"monthly_txt") == 0 ) && ( Nwords == 2 ) )
	    mReporter.monthly_txt_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"printtaskdays") == 0 ) && ( Nwords == 1 ) )
	    mReporter.print_task_days = 1;
	else if ( ( strcasecmp(words[0],"utilgraph") == 0 ) && ( Nwords == 2 ) )
	    mReporter.util_graph = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"hardschedule") == 0 ) && ( Nwords == 2 ) )
	    mReporter.hard_schedule = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"taskgraph") == 0 ) && ( Nwords == 4 ) )
	    AddTaskGraph(words[1], words[2], words[3]);
	else if ( ( strcasecmp(words[0],"network") == 0 ) && ( Nwords == 4 ) )
	    mReporter.AddNetworkDiagram(atoi(words[1]), atoi(words[2]), 
					words[3]);
	else if ( ( strcasecmp(words[0],"netx") == 0 ) && ( Nwords == 3 ) )
	    mProject.AddNetworkX(words[1], words[2]);
	else if ( ( strcasecmp(words[0],"nety") == 0 ) && ( Nwords == 3 ) )
	    mProject.AddNetworkY(words[1], words[2]);
	else if ( ( strcasecmp(words[0],"start_nety") == 0 ) && ( Nwords == 2 ) )
	    mProject.AddStartNetworkY(words[1]);
	else if ( ( strcasecmp(words[0],"finish_nety") == 0 ) && ( Nwords == 2 ) )
	    mProject.AddFinishNetworkY(words[1]);
	else if ( ( strcasecmp(words[0],"cost_html")  ==  0 ) && ( Nwords == 2 ) )
	    mReporter.html_cost_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"cost_tex")  ==  0 ) && ( Nwords == 2 ) )
	     mReporter.tex_cost_report = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"dateformat") == 0 ) && ( Nwords == 2 ) )
	    mProject.SetDateFormat(words[1]);
	else if ( ( strcasecmp(words[0],"tg_tasklabel") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_tasklabel = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_left") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_left = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_width") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_width = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_top") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_top = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_height") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_height = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_space") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_space = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_gray") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_gray = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_lightgray") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_lightgray = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_white") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_white = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_fontname1") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_fontname1 = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"tg_fontsize1") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_fontsize1 = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_fontname2") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_fontname2 = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"tg_fontsize2") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_fontsize2 = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_fontname3") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_fontname3 = strdup(words[1]);
	else if ( ( strcasecmp(words[0],"tg_fontsize3")  ==  0 ) && ( Nwords  ==  2 ) )
	     mReporter.tg_fontsize3 = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_textup") ==  0 ) && ( Nwords == 2 ) )
	     mReporter.tg_textup = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_mlgray") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_mlgray = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_xborder") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_xborder = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_yborder") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_yborder = atoi(words[1]);
	else if ( ( strcasecmp(words[0],"tg_nodays") == 0 ) && ( Nwords == 1 ) )
	     mReporter.tg_nodays = 1;
	else if ( ( strcasecmp(words[0],"tg_daysofmonth") == 0 ) && ( Nwords == 1 ) )
	    mReporter.tg_daysofmonth = 1;
	else if ( ( strcasecmp(words[0],"tg_sortbyresource") == 0 ) && ( Nwords == 1 ) )
	    mReporter.tg_sortbyresource = true;
	else if ( ( strcasecmp(words[0],"tg_showpast") == 0 ) && ( Nwords == 1 ) )
	    mReporter.tg_showpast = true;
	else if ( ( strcasecmp(words[0],"tg_pastgray") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_pastgray = atof(words[1]);
	else if ( ( strcasecmp(words[0],"tg_showvacation") == 0 ) && ( Nwords == 1 ) )
	    mReporter.tg_showvacation = true;
	else if ( ( strcasecmp(words[0],"tg_vacationgray") == 0 ) && ( Nwords == 2 ) )
	     mReporter.tg_vacationgray = atof(words[1]);
	else if ( ( strcasecmp(words[0], "pc_width") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_width = atoi(words[1]);
	else if ( ( strcasecmp(words[0], "pc_height") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_height = atoi(words[1]);
	else if ( ( strcasecmp(words[0], "pc_space") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_space = atoi(words[1]);
	else if ( ( strcasecmp(words[0], "pc_textin") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_textin = atoi(words[1]);
	else if ( ( strcasecmp(words[0], "pc_textup") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_textup = atoi(words[1]);
	else if ( ( strcasecmp(words[0], "pc_fontname1") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_fontname1 = strdup(words[1]);
	else if ( ( strcasecmp(words[0], "pc_fontsize1") == 0 ) && ( Nwords == 2 ) )
	    mReporter.pc_fontsize1 = atoi(words[1]);
	else
	{
	    cerr << name << ":" << lineno << ": warning: Invalid Line `" 
		 << buf << "'\n";
	    ++invalid_lines;
	}
    }
    catch ( ProjectException & e )
    {
	cerr << name << ":" << lineno << ": " << e.what() << endl;
	++invalid_lines;
    }
    catch ( ProjectFileException & e )
    {
	cerr << name << ":" << lineno << ": " << e.what() << endl;
	++invalid_lines;
    }

    }
    
    

    fclose(f);

    if ( invalid_lines )
	Error("There were %d errors parsing %s", invalid_lines, name);
}

void Project::PrintFile()
{
    printf("# Resources:\n");
    for ( RPLCI rl = mResourceList.begin() ; rl != mResourceList.end() ; rl++ )
	printf("resource %s \"%s\"\n", (*rl)->id(), (*rl)->name());

    printf("\n# Tasks:\n");
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	printf("task %s \"%s\" %d\n",(*pt)->id(),(*pt)->name(),(*pt)->duration());

    printf("\n# Candidates:\n");
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	for ( RPLCI rl = (*pt)->begin_cando(); rl != (*pt)->end_cando(); rl++ )
	    printf("candidate %s %s\n",(*pt)->id(), (*rl)->id());

    printf("\n# Dependencies:\n");
    for ( TPLCI pt = mTaskList.begin() ; pt != mTaskList.end() ; pt++ )
	for ( TPLCI tpl = (*pt)->begin_depends(); tpl != (*pt)->end_depends(); tpl++ )
	    printf("depends %s %s\n", (*pt)->id(), (*tpl)->id());
}


