/*************************************************************************
 *
 *  $RCSfile: parser.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: hr $ $Date: 2003/03/27 11:07:53 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>

#ifndef _SV_SVAPP_HXX //autogen
#include <vcl/svapp.hxx>
#endif

#ifndef _SV_MSGBOX_HXX //autogen
#include <vcl/msgbox.hxx>
#endif

#include "fields.hxx"
#include "parser.hxx"
#include "scanner.hxx"

SiParser::SiParser(SvStream &aStream)
{
	m_bHaveApp		= TRUE;
	m_bHaveVcl		= TRUE;
	m_pScanner		= new SiScanner(aStream);
	m_bSyntaxError	= FALSE;
}

SiParser::~SiParser()
{
	delete m_pScanner;
}

int SiParser::GetLine() const
{
	return m_pScanner->GetLine();
}

void SiParser::SyntaxError(SiLexem const& aWrongLexem, ExpectedLexem anExpectedLexem)
{
	ByteString ErrorType;
	ByteString Message;

	// lexikalischer Fehler ?
	if (aWrongLexem.GetType() == LT_ERR)
	{
		ErrorType = "lexical error, ";

		switch (aWrongLexem.GetError())
		{
			case LE_NOTPRINTABLE:
				 Message  = "nonprintable character in string constant behind: " ;
				 Message += "'";
				 Message += aWrongLexem.GetValueAsString();
				 Message += "'";
				 break;

			case LE_NOTBEGIN:
				 Message  = "character is not a legal begin of a lexem: ";
				 Message += "'";
				 Message += aWrongLexem.GetValueAsString();
				 Message += "'";
				 break;

			default:
				 Message = "unknown";
				 break;
		}
	}
	else
	{
		ErrorType = "syntax error at token ";
		Message  = "'";
		Message += aWrongLexem.GetValueAsString();
		Message += "' ";

		switch (anExpectedLexem)
		{
			case EXP_DECLARATOR:  Message += "expected declarator; i.e. File ..."; break;
			case EXP_END:         Message += "expected keyword <End>"; break;
			case EXP_POPEN:       Message += "expected '('"; break;
			case EXP_PCLOSE:      Message += "expected ')'"; break;
			case EXP_IDENTIFIER:  Message += "expected identifier"; break;
			case EXP_ASSIGN:      Message += "expected '='"; break;
			case EXP_SINGLEVALUE: Message += "expected constant string (i.e. \"foo\") or identifier (i.e. FOO) or integer"; break;
			case EXP_VALUE:       Message += "expected constant string (i.e. \"foo\") or identifier (i.e. FOO) or integer or list"; break;
			case EXP_INTEGER:     Message += "expected integer (i.e. 49)"; break;
			case EXP_SEMICOLON:   Message += "expected ';'"; break;
			default:              Message += "unknown error"; break;
		}
	}

	m_bSyntaxError = TRUE;
	DisplayError(m_pScanner->GetLine(), ErrorType,Message);
}

void SiParser::DisplayError(int nLine, ByteString const& ErrorType, ByteString const& Message) const
{
	if( m_bHaveApp && m_bHaveVcl )
	{
		ByteString aMsg( "line " );
		aMsg += ByteString::CreateFromInt32(nLine);
		aMsg += ": ";
		aMsg += ErrorType;
		aMsg += " ";
		aMsg += Message;

		ErrorBox( NULL, WB_OK | WB_SYSMODAL, UniString::CreateFromAscii(aMsg.GetBuffer()) ).Execute();
	}
	else
	{
		ByteString aMsg( ErrorType );
		aMsg += ": ";
		aMsg += m_pScanner->GetFilename();
		aMsg += ": line ";
		aMsg += ByteString::CreateFromInt32(nLine);
		aMsg += ", ";
		aMsg += Message;

		fprintf( stdout, "%s\n", aMsg.GetBuffer() );
	}
}

void SiParser::ResetError()
{
	m_bSyntaxError = FALSE;
}

BOOL SiParser::IsDeclarator(SiLexem const& aLexem) const
// returns
//      TRUE:  Lexem.Type ist ein Identifier und Identifier in ("File", "Module", ...)
//      FALSE: sonst
{
	SiLexemType aType = aLexem.GetType();

	// Performance: haeufige Objekte stehen weiter oben ...
	return
		   aType == LT_FILE
		|| aType == LT_PROFILEITEM
		|| aType == LT_REGISTRYITEM
		|| aType == LT_MODULE
		|| aType == LT_DIRECTORY
		|| aType == LT_FOLDER
		|| aType == LT_FOLDERITEM
		|| aType == LT_DATACARRIER
		|| aType == LT_PROFILE
		|| aType == LT_MODULELIST
		|| aType == LT_MODULESET
		|| aType == LT_OS2_CLASS
		|| aType == LT_OS2_CREATOR
		|| aType == LT_OS2_TEMPLATE
		|| aType == LT_SHORTCUT
		|| aType == LT_INSTALLATION
		|| aType == LT_REGISTRYAREA
		|| aType == LT_PROCEDURE
		|| aType == LT_CUSTOM
		|| aType == LT_SLIDE
		|| aType == LT_HELPTEXT
		|| aType == LT_SCPACTION
		|| aType == LT_STARREGISTRY
		|| aType == LT_STARREGISTRY_ITEM
		|| aType == LT_CONFIGURATION_ITEM
		;
}


BOOL SiParser::ParseScript()
// Ein Script ist eine Liste von Deklarationen. Es kann leer sein.
// "File ... End" "Component ... End" usw.
//
// pre:  wir stehen am Anfang des gesamten Lexem-Stroms
//       oder hinter einem "End"
// post: wir stehen am Ende des Stromes
{
	while( IsDeclarator(m_pScanner->GetCurrentLexem()) )
	{
		if( !ParseDeclaration() )
			return FALSE;
		if( m_bHaveApp && m_bHaveVcl )
			Application::Reschedule();
	}

	SiLexem Lex = m_pScanner->GetCurrentLexem();

	// Eof darf kommen. Wenn es das nicht ist, ist das aktuelle
	// Lexem kein Deklarator.
	//
	if (Lex.GetType() != LT_EOF)
	{
		SyntaxError(Lex, EXP_DECLARATOR);
		return FALSE;
	}

	m_pScanner->ReadNextLexem();
	return TRUE;
}


void SiParser::Recover()
// Nach einem Fehler wird eine Stelle im Lexemstrom gesucht, ab der das Parsen
// wieder Sinn macht. Deshalb wird nach dem Schlsselwort "End" gesucht
//
// pre:  -
// post: wird sind im Lexemstrom hinter End oder auf Eof
{
	SiLexem Lex;

	// Lexem suchen
	//
	do
	{
		Lex = m_pScanner->ReadNextLexem();
	}
	while (Lex.GetType() != LT_EOF
	   &&  Lex.GetType() != LT_END);

	// hinter das Lexem gehen
	//
	m_pScanner->ReadNextLexem();
}


BOOL SiParser::Continue()
// Diese Methode kann aufgerufen werden, wenn das Parsen wegen eines Fehlers
// abgebrochen worden ist, und wieder anlaufen soll (z.B. um alle Fehlermeldungen)
// anzuzeigen.
//
// pre:  Recover wurde vorher aufgerufen und Pre(ParseScript)
// post: siehe ParseScript()
{
	return ParseScript();
}


BOOL SiParser::ParseDeclaration()
// Parst eine Deklaration: "Declarator ID property* End"
//
// pre:  wir stehen im Lexem-Strom am Scriptanfang oder hinter einer Deklaration
// post: wir stehen im Lexem-Strom hinter einer Deklaration
{
	SiLexem Lex = m_pScanner->GetCurrentLexem();

	// .Declarator ID...
	//
	// (Declarator = Component, File, Directory, ..)
	//
	if (!IsDeclarator(Lex))
	{
		SyntaxError(Lex, EXP_DECLARATOR);
		return FALSE;
	}

	SiLexemType aDeclarator = Lex.GetType(); // fr OnDeclarator merken
	Lex = m_pScanner->ReadNextLexem();    // Declarator wurde verarbeitet


	// Declaration .ID ...
	//
	if (Lex.GetType() != LT_ID)
	{
		SyntaxError(Lex, EXP_IDENTIFIER);
		return FALSE;
	}

	OnDeclaration(aDeclarator,Lex.GetValueAsIdentifier());
	Lex = m_pScanner->ReadNextLexem(); // Identifier wurde verarbeitet

	// Declaration ID .property = value ...
	//
	if (!ParsePropertyList())
		return FALSE;

	// Declaration ID p = v; p = v; .End
	//
	if (m_pScanner->GetCurrentLexem().GetType() != LT_END)
	{
		SyntaxError(m_pScanner->GetCurrentLexem(), EXP_END);
		return FALSE;
	}

	OnDeclarationEnd();
	m_pScanner->ReadNextLexem(); // "End" wurde verabeitet
	return TRUE;
}


BOOL SiParser::ParsePropertyList()
// Parst eine Property-Liste (kann leer sein)
//
// pre:  wir stehen im Lexemstrom hinter einer Deklaration
//       "File(F1) . "
// post: wir stehen hinter dem letzten Value (return TRUE)
//       "File(F1) p = v . End"
{
	// .property1 = value1;
	// property2 = value2;
	//
	while (m_pScanner->GetCurrentLexem().GetType() == LT_ID)
	{
		if (!ParseProperty())
			return FALSE;
	}

	// kein Identifier mehr: wahrscheinlich stehen wir vor dem "End"
	// Das prft aber der Aufrufer.
	return TRUE;
}


BOOL SiParser::ParseProperty()
{
	SiLexem aPropertyName = m_pScanner->GetCurrentLexem();

	if (aPropertyName.GetType() != LT_ID)
	{
		SyntaxError(aPropertyName, EXP_IDENTIFIER);
		return FALSE;
	}

	SiLexem Lex		  = m_pScanner->ReadNextLexem(); // Name wurde verarbeitet
	int		nLanguage = USHRT_MAX; // Default

	// optionale Sprachangabe vorhanden?
	//
	// propertname .(01) = value;
	//
	if (Lex.GetType() == LT_POPEN)
	{
		Lex = m_pScanner->ReadNextLexem(); // ( wurde verarbeitet

		// ... (.01) ...
		if (Lex.GetType() != LT_ICONST)
		{
			SyntaxError(Lex, EXP_INTEGER);
			return FALSE;
		}

		nLanguage = (int) Lex.GetValueAsInt32();
		Lex = m_pScanner->ReadNextLexem(); // 01 wurde verarbeitet

		// (01.)
		if (Lex.GetType() != LT_PCLOSE)
		{
			SyntaxError(Lex, EXP_PCLOSE);
			return FALSE;
		}

		Lex = m_pScanner->ReadNextLexem(); // ) wurde verarbeitet
	}

	OnProperty(aPropertyName.GetValueAsString(), nLanguage);


	// name .= value;
	//
	if (Lex.GetType() != LT_ASSIGN)
	{
		SyntaxError(Lex, EXP_ASSIGN);
		return FALSE;
	}

	Lex = m_pScanner->ReadNextLexem();

	// name = .value;
	// (value = 1 | ID | "foo" | (F1, 1, "foo")
	//
	switch (Lex.GetType())
	{
		case LT_ICONST:
			if( aPropertyName.GetValueAsString().CompareIgnoreCaseToAscii(PROPERTY_CRC) == COMPARE_EQUAL )
				OnSingleValue( Lex.GetValueAsUInt32() );
			else
				OnSingleValue( Lex.GetValueAsInt32() );

			Lex = m_pScanner->ReadNextLexem();
			break;

		case LT_SCONST:
			OnSingleValue(Lex.GetValueAsString());
			Lex = m_pScanner->ReadNextLexem();
			break;

		case LT_ID:
			// ToDo: Unterscheidung ByteString/ID
			OnSingleValue(Lex.GetValueAsIdentifier());
			Lex = m_pScanner->ReadNextLexem();
			break;

		case LT_POPEN:
			if (!ParseValueList())
				return FALSE;
			// post: wir stehen hinter der schlieenden Klammer
			Lex = m_pScanner->GetCurrentLexem();
			break;

		default:
			SyntaxError(Lex, EXP_VALUE);
			return FALSE;
	}

	// name = value .;
	//
	if (Lex.GetType() != LT_SEMICOLON)
	{
		SyntaxError(Lex, EXP_SEMICOLON);
		return FALSE;
	}

	m_pScanner->ReadNextLexem();
	return TRUE;
}

BOOL SiParser::ParseValueList()
// Parst eine Liste von Werten
// Eine Liste hat folgende Form: (v1, v2, v3, ... vn)
//
// pre:  wir stehen im Lexemstrom auf der ffnenden Klammer
// post: wir stehen hinter der schlieenden Klammer
//       fr jeden Wert wurde die virtuelle Methode OnListeValue() gerufen
{
	if (m_pScanner->GetCurrentLexem().GetType() != LT_POPEN)
	{
		SyntaxError(m_pScanner->GetCurrentLexem(), EXP_POPEN);
		return FALSE;
	}

	SiLexem Lex;

	// Werte parsen: a, b, ...
	//
	do
	{
		Lex = m_pScanner->ReadNextLexem(); // '(' oder ',' berlesen

		switch (Lex.GetType())
		{
			case LT_ICONST:
			{
				OnListValue( Lex.GetValueAsInt32() );
				Lex = m_pScanner->ReadNextLexem(); // Wert wurde verarbeitet
				break;
			}

			case LT_SCONST:
			{
				OnListValue(Lex.GetValueAsString());
				Lex = m_pScanner->ReadNextLexem(); // Wert wurde verarbeitet
				break;
			}

			case LT_ID:
			{
				OnListValue(Lex.GetValueAsIdentifier());
				Lex = m_pScanner->ReadNextLexem(); // Wert wurde verarbeitet
				break;
			}

			case LT_PCLOSE:
				break; // Leere Liste: nichts tun, Schleife terminiert

			default:
			{
				SyntaxError(Lex,EXP_SINGLEVALUE);
				return FALSE;
			}
		}
		// post: wir stehen hinter dem Wert, mit einem Komma wird iteriert
	}
	while (Lex.GetType() == LT_COMMA);

	// Wir mssen jetzt hinter dem letzten Wert
	// und vor der schlieenden Klammer stehen

	if (Lex.GetType() != LT_PCLOSE)
	{
		SyntaxError(Lex, EXP_PCLOSE);
		return FALSE;
	}

	m_pScanner->ReadNextLexem(); // schlieende Klammer berlesen
	return TRUE;
}

