/*
 * Copyright (c) 2000, Amnon BARAK (amnon@cs.huji.ac.il). All rights reserved.
 *
 *       MOSIX MFS mtune.c,v 1.0.0.0 2000/07/02 
 *
 * Permission to use, copy and distribute this software is hereby granted 
 * under the terms of version 2 or any later version of the GNU General Public
 * License, as published by the Free Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED IN ITS "AS IS" CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/*
 * Author(s): Shlomo Matichin, Amnon Shiloh.
 */


/* MTUNE.C
 * this program creates the mfscosts.h file to be #included by the kernel in order
 * to estimate communication overheads in MOSIX's MFS.
 *
 * All results are given in units of 1/1000000 of a second and include both
 * local and remote overheads.
 *
 * We distinguish between two kinds of penalties for communication overhead:
 *
 *	C: cost (system time) at client
 *	S: cost (system time) at server
 *
 * The local communication overheads are measured for system calls with:
 *      1. system calls:
 *              MFS_COST_CONN_{S/C}
 *      2. input:
 *              MFS_COST_INKB_{S/C}
 *      3. output:
 *              MFS_COST_OUTKB_{S/C}
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <signal.h>
#include <a.out.h>
#include <fcntl.h>
#include <sys/time.h>
#include <setjmp.h>
#include <sys/mman.h>
#include <math.h>
#include <sys/vfs.h>

/*
 * CONSTANTS
 */
#define KILOBYTE (1024)
#define MEGABYTE (KILOBYTE*KILOBYTE)
#define MAXSMP 8
#define MIN_REPS 1000
#define MAX_REPS 100000
#define DEF_REPS 10000
#define STD_SPD 10000
#define	STD_TIME 7714000000LL
#define START SIGUSR1
#define END SIGUSR2
/* out file name */
#define HFILE "mfscosts.h"
#define MFILE "mfs.costs"
/* maximum length of temp files path string */
#define MAXTEMPPATHLENGTH 100
/* defualt temp file size */
#define MIN_READS_BETWEEN_SEEKS 128
#define MAX_READS_BETWEEN_SEEKS 1024
#define DEF_TEMPFILE_NAME "/MfStUnEtEmP"
#define DEF_TEMPFILE_PATH "/tmp"
/* default size for data to deliver when reading or writing */
#define DEF_BLOCK_SIZE (8*1024)
/*
 * VARIABLES
 */
/* run arguments */
int ok = 0;
int needy = 0;
int assist = 0;
int debug = 0;
int reps = DEF_REPS;

/* byte sizes for final computation */
int sizeofpackdata, sizeof2packdata;

/* run variables */
int this, other;
int nhere, nthere;
double speed1, speed2;
int ips1, ips2;
int f1[MAXSMP], f2[MAXSMP];
long freememthis,freememother;

/* multi process variables */
int startpipe[2];
char buf[132768];
volatile int nsig;

/* the counter for the tight loop */
volatile long long counter;
/* the jump buffer for measure to handle the interrupt */
jmp_buf env;

/* real temp files for the check */
char localtempfile[MAXTEMPPATHLENGTH] = DEF_TEMPFILE_PATH;
char remotetempfile[MAXTEMPPATHLENGTH] = DEF_TEMPFILE_PATH;

int local_tmp=0, remote_tmp=0;
long tmp_size = 0;
int tmp_readn, tmp_readn2;

/* results */
double client_A, client_B, client_C, client_D, client_E;
double server_A, server_B, server_C, server_D, server_E;

char *comment;
/*
 * FUCNTIONS
 */
/* the assembler test loop for cpu speed*/
extern int testloop();
/* signal handlers*/
int note(), jmp(), endit();
int cpuspeed();
int count();

/* force mosix to migrate proccess */
int migrate(int to);
/* same with prompting retries */
void get_to(int where);
/* find out where */
int whereami();

/* initializators*/
void getsmps();
void getspeeds();
int howmany(int where);

/* user input */
int YES(int eol);
int readnum();
void rest_user();

/* mine or reweritten: */
void setup();
void initfiles();
void create_temp_files(int minsize, int maxsize);
void close_temp_files();
void measure(int fd, int ipsx);
void test(int empty, int syscall, int readpack, int read2pack, int writepack, int write2pack, 
	  double *lout, double *rout);
void testall();
void bring_to_cache();
void summerize();
void fdefine(FILE *fh, FILE *fm, char *st, int val);
void run(int fd, int empty, int syscall, int readpack, int read2pack, int writepack, int write2pack);

/* error handeling */
#define NOT_MOSIX 1
#define NO_SUID 2
#define NO_SINGLE 3
#define NETWORK_DOWN 4
#define REPS_PARAMETER_ERROR 5
#define LOCAL_TEMP_FILE_ERROR 6
#define REMOTE_TEMP_FILE_ERROR 7
#define NOT_ENOUGH_MEMORY 9
#define TEMPFILE_SIZE_TOOSMALL 10
void error_msg(int errornum)
{
	switch (errornum){
	case NOT_MOSIX:
		fprintf(stderr,"mtune ignored: This is NOT a MOSIX system!\n");
		break;
	case NO_SUID:
		fprintf(stderr,"Not Super-Useer\n");
		break;
	case NO_SINGLE:
		fprintf(stderr,"Bring both nodes to single-user mode and come back.\n");
		break;
	case NETWORK_DOWN:
		fprintf(stderr,
			"Unable to reach other node. Check network and that you run prep_tune on both.\n");
		break;
	case REPS_PARAMETER_ERROR:
		fprintf(stderr,"Invalid -n parameter: should be between %d and %d.\n",
			MIN_REPS, MAX_REPS);
		break;
	case LOCAL_TEMP_FILE_ERROR:
		fprintf(stderr,"Error opening local temp file: %s",localtempfile);
		break;
	case REMOTE_TEMP_FILE_ERROR:
		fprintf(stderr,"Error opening remote temp file: %s",remotetempfile);
		break;
	case NOT_ENOUGH_MEMORY:
		fprintf(stderr,"Aborting due lack of memory.\n");
		break;
	case TEMPFILE_SIZE_TOOSMALL:
		fprintf(stderr,"Temp file size specified is too small!\n");
		break;
	}
	exit(1);
}

int main(int argc, char *argv[])
{
	while(argc > 1 && argv[1][0] == '-')
	{
		switch(argv[1][1])
		{
		case 'o':
			ok = 1;
			break;
		case 'a':
			ok = 1;
			assist = 1;
			break;
		case 'c':
			if(argc > 2)
			{
				comment = argv[2];
				argc--;
				argv++;
			}
			break;
		case 'n':
			if(argv[1][2])
				reps = atoi(&argv[1][2]);
			else if(argc > 2)
			{
				argc--;
				argv++;
				reps = atoi(argv[1]);
			}
			if(reps < MIN_REPS || reps > MAX_REPS)
				error_msg(REPS_PARAMETER_ERROR);
			break;
		case 't':
			if(argv[1][2])
				strcpy(localtempfile,&argv[1][2]);
			else if(argc > 2)
			{
				argc--;
				argv++;
				strcpy(localtempfile,argv[1]);
			}
			break;
		case 'T':
			if(argv[1][2])
				strcpy(remotetempfile,&argv[1][2]);
			else if(argc > 2)
			{
				argc--;
				argv++;
				strcpy(remotetempfile,argv[1]);
			}
			break;	
		}
		argc--;
		argv++;
	}
	if(argc > 1)
		other = atoi(argv[1]);
	if(getenv("VERBOSE")) debug++;
	setup();
	getsmps();
	initfiles();
	getspeeds();
	rest_user();
	bring_to_cache();
	testall();
	close_temp_files();
	summerize();
	return 0;
}

void rest_user()
{
	if(!ok)
		printf("Thank you, that is all.\n\n");
}

void setup()
{
	int lfd = open("/proc/self/lock", O_WRONLY);
	FILE *f;
	char filename[MAXTEMPPATHLENGTH];
	register int i;

	/* make the buffer full of not zero to prevent optimization */
	memset(buf,0XFF,sizeof(buf));
	printf("MOSIX MFS tuning program: \n\n");
	if(getuid()!=0)
		error_msg(NO_SUID);
	if(lfd == -1)
		error_msg(NOT_MOSIX);
	write(lfd, "1\n", 2);
	close(lfd);
	get_to(0);
	this = whereami();
	while (other <= 0)
	{
		printf("Enter second node number: ");
		fflush(stdout);
		other = readnum();
	}
	if(!ok)
	{
		printf("Are both nodes in single-user mode, network enabled and in quiet-mode? ");
		if(!YES(1))
		{
			printf("Not being in single-user mode might produce inaccurate results. Continue? ");
			if(!YES(0))
				error_msg(NO_SINGLE);
		}
	}
	for(i = 1 ; i < NSIG ; i++)
		if(i != SIGKILL && i != SIGCHLD && i != SIGSTOP 
		   && i != SIGTSTP && i != SIGCONT)
			signal(i, (sig_t)endit);
	/* set the packet data size */
	sizeofpackdata = DEF_BLOCK_SIZE;
	sizeof2packdata = 2*sizeofpackdata;
	/* check amount of memory on the two nodes */
	sprintf(filename,"/proc/mosix/nodes/%d/mem",this);
	if((f = fopen(filename,"r")) == NULL)
		error_msg(NOT_MOSIX);
	fscanf(f,"%ld",&freememthis);
	if(debug) printf("Free memory on this node: %ld.\n",freememthis);
	fclose(f);
	sprintf(filename,"/proc/mosix/nodes/%d/mem",other);
	if((f = fopen(filename,"r")) == NULL)
		error_msg(NOT_MOSIX);
	fscanf(f,"%ld",&freememother);
	if(freememother<0) 
		error_msg(NETWORK_DOWN);
	if(debug) printf("Free memory on other node: %ld.\n",freememother);
	fclose(f);
	return;
}

/*
 * Find processor speeds using both general loop and tight loop.
 */

void getspeeds()
{
	printf("Taking processor speeds:\n");
	get_to(0);
	speed1 = cpuspeed();
	get_to(other);
	speed2 = cpuspeed();
	get_to(0);
	if(assist) return;
	if(speed1 < speed2 * 0.99 || speed2 < speed1 * 0.99)
	{
		printf("\nThe two processors are of varying speeds:\n");
		printf("This makes accurate measurement slightly more complex\n");
		printf("and will require some manual assistance.  OK [Y/n]? ");
		needy = YES(1);
		if(!needy)
			printf("\nOK, I will continue automatically, but the results may not be as accurate!\n\n");
	}
	else
	{
		printf("\nThe two processors look similar: if their hardware is exactly the same,\n");
		printf("then tuning can proceed completely automatically.\n");
		printf("If, however, any of the following differs between the two computers:\n\n");
		printf("  1) the processor itself.\n");
		printf("  2) the type or speed of the memory.\n");
		printf("  3) the number of memory wait-states (if any).\n");
		printf("  4) the networking hardware.\n");
		printf("  5) one computer uses a kernel with SMP configured and the other does not.\n");
		printf("\nthen some manual assistance is required to produce more accurate results.\n");
		printf("\nAre the two computers identical [Y/n]? ");
		needy = !YES(1);
	}
	return;
}

void getsmps()
{
	nhere = howmany(this);
	nthere = howmany(other);
}

void initfiles()
{
	int i;
	long minfreemem = freememthis > freememother ? freememother : freememthis;
	int minfilesize, maxfilesize;
	/* the next macro defines how memory is checked to be big enough to contain all 
	   the temp file in the cache. 128KB is necceseray in order to cantain the cpu mesuring
	   tasks and prevent paging out the cache */
#define CHECKMEM(freemem,bytes) (((freemem) - 131072) < (bytes))
	
	for(i=0 ; i<nhere ; i++)
	{
		if((f1[i] = creat("/tmp/JuNkFiLe1",0)) == -1)
		{
		fail:
			perror("Failed to open scratch files");
			endit(1);
		}
		unlink("/tmp/JuNkFiLe1");
	}
	for(i=0 ; i<nthere ; i++)
	{
		if((f2[i] = creat("/tmp/JuNkFiLe2",0)) == -1)
			goto fail;
		unlink("/tmp/JuNkFiLe2");
	}
	if(pipe(startpipe))
		goto fail;
	fcntl(startpipe[0], F_SETFL, O_NONBLOCK);
	/* opening local&remote temp files */
	/* calculate the minimum file size for a lseek to accour maximum once in 128 reads */
	minfilesize = MIN_READS_BETWEEN_SEEKS;
	maxfilesize = MAX_READS_BETWEEN_SEEKS;
	if(CHECKMEM(minfreemem,maxfilesize * sizeof2packdata))
		/* amount of free memory, plus and extra, in round chuncks of 2 packets */
		maxfilesize = (minfreemem - 128*KILOBYTE) / sizeof2packdata;
	if(maxfilesize < minfilesize) 
	{
		fprintf(stderr,"Not enoguh memory available for temp file caching in %s host!\n",
			minfreemem==freememthis ? "local" : "remote");
		endit(1);
	}
	create_temp_files(minfilesize, maxfilesize);
}

void create_temp_files(int minsize, int maxsize)
{
	char temp_buf[16];
	int i;
	struct statfs stat;

	/* create local temp file */
	sprintf(temp_buf,"/mfs/home");
	sprintf(buf,"%s%s%s",temp_buf,localtempfile,DEF_TEMPFILE_NAME);
	while(local_tmp==0)
		if((local_tmp = open(buf, O_RDWR | O_CREAT)) == -1)
		{
			printf("Error trying to create local temp file!\n");
		tryagain_local:
			printf("Please enter where (directory) I can create a large temporary file\n(or press <Enter> to abort)\n");
			printf("%s/",temp_buf);
			if((fgets(buf+strlen(temp_buf)+1,MAXTEMPPATHLENGTH,stdin)==NULL) || 
			   (strlen(buf+strlen(temp_buf)+1)==0))
				endit(1);
			strcpy(buf+strlen(buf),DEF_TEMPFILE_NAME);
		}
	statfs(buf+strlen(temp_buf),&stat);
	if(debug) printf("statfs(\"%s\"): BSIZE=%d, BFREE=%qd\n",
		buf+strlen(temp_buf), stat.f_bsize, (long long)stat.f_bfree);
	unlink(buf);
	if((stat.f_bsize*(long long)stat.f_bfree) < (minsize*sizeof2packdata))
	{
		printf("Not enough space for local temp file! minimum size: %d Bytes\n",
		       minsize*sizeof2packdata);
		close(local_tmp);
		goto tryagain_local;
	}
	if((stat.f_bsize*(long long)stat.f_bfree) < (maxsize*sizeof2packdata))
		maxsize = stat.f_bsize*(long long)stat.f_bfree / sizeof2packdata;
	for(i=0 ; i<maxsize ; i++)
		if(write(local_tmp,buf,sizeof2packdata)<sizeof2packdata)
		{
			perror("Local temp file");
			endit(1);
		}
	lseek(local_tmp, 0, SEEK_SET);
	/* copy the finale tempfile path back to it's string */
	strcpy(localtempfile,buf + strlen(temp_buf));
	localtempfile[strlen(localtempfile)-strlen(DEF_TEMPFILE_NAME)]='\0';
			
	/* create remote temp file */
	sprintf(temp_buf,"%s%d","/mfs/",other);
	sprintf(buf,"%s%s%s",temp_buf,remotetempfile,DEF_TEMPFILE_NAME);
	while(remote_tmp==0)
		if((remote_tmp = open(buf ,O_RDWR | O_CREAT)) == -1)
		{
			printf("Error trying to create remote temp file!\n");
		tryagain_remote:
			printf("Please enter where  (directory) I can create a large temporary file\n(or press <Enter> to abort)\n");
			printf("%s",temp_buf);
			if((fgets(buf+strlen(temp_buf)+1,MAXTEMPPATHLENGTH,stdin)==0) || 
			   (strlen(buf+strlen(temp_buf)+1)==0))
				endit(1);
			strcpy(buf+strlen(buf),DEF_TEMPFILE_NAME);
		}
	unlink(buf);
	for(i=0 ; i<minsize ; i++)
		if(write(remote_tmp,buf,sizeof2packdata)<sizeof2packdata)
		{
			printf("Error: minimum file size for this test is %d Bytes\n",
			       minsize * sizeof2packdata);
			printf("Writing halted at %d!\n", (i-1)*sizeof2packdata);
			close(remote_tmp);
			goto tryagain_remote;
		}
	for(; i<maxsize ; i++)
		if(write(remote_tmp,buf,sizeof2packdata)<sizeof2packdata)
			break;
	lseek(remote_tmp, 0, SEEK_SET);
	strcpy(remotetempfile,buf+strlen(temp_buf));
	remotetempfile[strlen(remotetempfile)-strlen(DEF_TEMPFILE_NAME)]='\0';
	tmp_readn = i;
	tmp_readn2 = 2*tmp_readn;
	fsync(local_tmp);
	fsync(remote_tmp);
	return;
}

void close_temp_files()
{
	close(local_tmp);
	close(remote_tmp);
}

void bring_to_cache()
{
	int i;
	
	lseek(local_tmp, 0, SEEK_SET);
	lseek(remote_tmp, 0, SEEK_SET);
	for(i=0 ; i<tmp_readn ; i++)
	{
		read(local_tmp, buf, sizeof2packdata);
		read(remote_tmp, buf, sizeof2packdata);
	}
	lseek(local_tmp, 0, SEEK_SET);
	lseek(remote_tmp, 0, SEEK_SET);
}
		
void testall()
{
	test( 1, 0, 0, 0, 0, 0, NULL, NULL);  /* empty phase - measure speed */
	test( 0, 1, 0, 0, 0, 0, &client_A, &server_A); /* measure A */

	test( 0, 0, 1, 0, 0, 0, &client_B, &server_B); /* measure B */
	lseek(remote_tmp, 0, SEEK_SET);
	lseek(local_tmp, 0, SEEK_SET);
// the next test is at no use to us as for now
//	test( 0, 0, 0, 1, 0, 0, &client_C, &server_C); 

	test( 0, 0, 0, 0, 1, 0, &client_D, &server_D); /* measure D */
	lseek(local_tmp, 0, SEEK_SET);
	lseek(remote_tmp, 0, SEEK_SET);
	test( 0, 0, 0, 0, 0, 1, &client_E, &server_E); /* measure E */  
	/* normilize the results to P400 */
	client_A = client_A * speed1 / (reps*STD_SPD);
	client_B = client_B * speed1 / (reps*STD_SPD);
	client_D = client_D * speed1 / (reps*STD_SPD);
	client_E = client_E * speed1 / (reps*STD_SPD);
	server_A = server_A * speed2 / (reps*STD_SPD);
	server_B = server_B * speed2 / (reps*STD_SPD);
	server_D = server_D * speed2 / (reps*STD_SPD);
	server_E = server_E * speed2 / (reps*STD_SPD);
}

void test(int empty, int syscall, int readpack, int read2pack, int writepack, int write2pack, 
	  double *lout, double *rout)
{
	int lref = 0;
	int local, remote;
	int s1[MAXSMP], s2[MAXSMP];
	register int i;

	signal(START, (sig_t)note);
	if(empty)
		printf("INITIAL CALIBRATION:\n");
	else if(syscall)
		printf("SIMPLE SYSTEM CALL:\n");
	else if(readpack)
		printf("READING %d BYTES:\n",sizeofpackdata);
	else if(read2pack)
		printf("READING %d BYTES:\n",sizeof2packdata);
	else if(writepack)
		printf("WRITING %d BYTES:\n",sizeofpackdata);
	else if(write2pack)
		printf("WRITING %d BYTES:\n",sizeof2packdata);
	else printf("NO TEST!\n");
	fflush(stdout);
        /* start of test code */
	/* empty is an empty test, to see how the cpus mesure with all the interrupts and calls going */
        /* on even if nothing is actually done (the overhead of the testing itself)*/
	if(empty)
		goto remote_test;
	/*LOCAL TEST */
	get_to(0);
	nsig = 0;
	for(i=0 ; i<nhere ; i++)
		switch(s1[i] = fork())
		{
		case -1:
			for(i-- ; i>=0 ; i--)
				kill(s1[i], SIGKILL);
			perror("Fork");
			endit(1);
		case 0:
			measure(f1[i], ips1);   /* start mesuring */
		}
	while(nsig < nhere)
		sleep(1);
	sleep(3);
	run(local_tmp, empty, syscall, readpack, 
					read2pack, writepack, write2pack);
	for(i=0 ; i<nhere ; i++)
		kill(s1[i], END);
	while(wait(0) > 0) /* wait returns error when no children left */
		;
	for(i=0 ; i<nhere ; i++)
		lref += lseek(f1[i], (off_t)0, 1);
	/* REMOTE TEST */
 remote_test:
	nsig=0;
	while(migrate(other))
	{
		printf("Cannot migrate to %d", other);
		printf(". Retry? ");
		if(!YES(1)) 
			endit(1);
	}
	for(i=0 ; i<nthere ; i++)
		switch(s2[i] = fork())
		{
		case -1:
			perror("Fork");
			for(i-- ; i>=0 ; i--)
				kill(s2[i], SIGKILL);
			endit(1);
		case 0:
			measure(f2[i], ips2);
		}
	get_to(0);
	for(i=0 ; i<nhere ; i++)
		switch(s1[i] = fork())
		{
		case -1:
			for(i-- ; i>=0 ; i--)
				kill(s1[i], SIGKILL);
			for(i=0 ; i<nthere ; i++)
				kill(s2[i], SIGKILL);
			perror("Fork");
			endit(1);
		case 0:
			measure(f1[i], ips1);
		}
	while(nsig < nhere + nthere)
		sleep(1);
	sleep(2);
	run(remote_tmp, empty, syscall, readpack, 
					read2pack, writepack, write2pack);
	for(i=0 ; i<nhere ; i++)
		kill(s1[i], END);
	for(i=0 ; i<nthere ; i++)
		kill(s2[i], END);
	while(wait(0) >= 0)
		;
	local = remote = 0;
	for(i=0 ; i<nhere ; i++)
		local += lseek(f1[i], 0, SEEK_CUR);
	for(i=0 ; i<nthere ; i++)
		remote += lseek(f2[i], 0, SEEK_CUR);
	if(empty)   /* if empty - then the variable returned by measure with ips1 imply performance */
	{           /* not loss of performance - use it to determine ips */
		ips1 = local / nhere;
		printf("%s of node %d count %d loops/second\n",
		       nhere > 1 ? "Processors" : "The processor", this, ips1);
	        ips2 = remote / nthere;
		printf("%s of node %d count %d loops/second\n",
		       nthere > 1 ? "Processors" : "The processor", other, ips2);
		return;  /* exit - results are not returned! */
	}
	/* report back with result */
	printf("\tC = %.2f seconds (%.2f locally), S = %.2f seconds.\n",
	       (local-lref)/1000000.0, lref/1000000.0, remote/1000000.0);
	*lout = local - lref;
	*rout = remote;
}

void run(int fd, int empty, int syscall, int readpack, int read2pack, int writepack, int write2pack)
{
	register int i,j,r;
 
	if(syscall)   /* preform syscalls */
		for(i=0 ; i<reps ; i++)
			read(fd,buf,0);
 
	else if(read2pack)  /* read 2 packets in one call */
	{
		j=tmp_readn;
		for(i=0 ; i<reps ; i++)
		{
			r = read(fd, buf, sizeof2packdata);
			if(r != sizeof2packdata)
			{
				perror("Read Error!");
			}
			if(!(--j))
			{
				j=tmp_readn;
				lseek(fd, 0, SEEK_SET);
			}
		}
	}
	else if(readpack)
	{
		j=tmp_readn2;
		for(i=0 ; i<reps ; i++)
		{
			r = read(fd, buf, sizeofpackdata);
			if(r != sizeofpackdata)
			{
				perror("read error");
				printf("Read %d bytes instead of %d, offset=%d, i=%d/%d,j=%d/%d\n", r, sizeofpackdata, (int)lseek(fd, 0, SEEK_CUR), i, reps, j, tmp_readn2);
			}
			if(!(--j))
			{
				j=tmp_readn2;
				lseek(fd, 0, SEEK_SET);
			}
		}
	}
	else if(write2pack)
	{
		j=tmp_readn;
		for(i=0 ; i<reps ; i++)
		{
			r = write(fd, buf, sizeof2packdata);
			if(r != sizeof2packdata)
			{
				perror("Write Error!");
			}
			if(!(--j))
			{
				j=tmp_readn;
				lseek(fd, 0, SEEK_SET);
			}
		}
	}
	else if(writepack)
	{
		j=tmp_readn2;
		for(i=0 ; i<reps ; i++)
		{
			r = write(fd, buf, sizeofpackdata);
			if(r != sizeofpackdata)
			{
				perror("write error");
				printf("Write %d bytes instead of %d, offset=%d, i=%d/%d,j=%d/%d\n", r, sizeofpackdata, (int)lseek(fd, 0, SEEK_CUR), i, reps, j, tmp_readn2);
			}
			if(!(--j))
			{
				j=tmp_readn2;
				lseek(fd, 0, SEEK_SET);
			}
		}
	}
	else sleep(10); 
	return;
}

void summerize()
{
	double c_syscost, c_inkbcost, c_outkbcost;
	double s_syscost, s_inkbcost, s_outkbcost;
	char now[100];
	FILE *date;
	char *c;
	FILE *fh, *fm;

	if(needy)
	{
		printf("\nYou manual assistance is now required:\n\n");
		printf("Please type the exact following command on the other node's (%d) console:\n\n", 
		       other);
		printf("\tmtune -a");
		if(reps!=DEF_REPS)
			printf(" -n%d",reps);
		if(strcmp(localtempfile,DEF_TEMPFILE_PATH))
			printf(" -T%s",localtempfile);
		if(strcmp(remotetempfile,DEF_TEMPFILE_PATH))
			printf(" -t%s",remotetempfile);
		printf(" %d\n\n",this);
		printf("Please wait for the results, then type those results (3 numbers) here: ");
		fflush(stdout);
		while(1)
		{
			fgets(now, sizeof(now), stdin);
			now[sizeof(now)-1] = '\0';
			for(c=now ; *c ; c++)
				if(*c==',' || *c==';') *c=' ';
			if(sscanf(now,"%lf %lf %lf", &s_syscost, &s_inkbcost, &s_outkbcost) == 3)
				break;
			printf("\nPlease type the results (3 numbers) from the other node's (%d) console\n",
			       other);
			printf("(separated by spaces) here: ");
			fflush(stdout);
		}
		printf("Thank you, test continues:\n\n");
	}
	
	if((date = popen("/bin/date","r")))
	{
		if(fgets(now, 100, date))
		{
			now[99] = '\0';
			now[strlen(now)-1] = '\0';
		}
		else
			now[0] = '\0';
		pclose(date);
	}

	c_syscost = client_A;
	c_inkbcost = ((client_B - client_A) / sizeofpackdata) * KILOBYTE;
	c_outkbcost = ((client_E - client_D) / sizeofpackdata) * KILOBYTE;
	if(!needy)
	{
		s_syscost = server_A;
		s_inkbcost = ((server_B - server_A) / sizeofpackdata) * KILOBYTE;
		s_outkbcost = ((server_E - server_D) / sizeofpackdata) * KILOBYTE;
	}
	if(debug)
	{
		printf("Client costs: sys: %.4f inKB: %.4f outKB: %.4f\n",
		       c_syscost,c_inkbcost,c_outkbcost);
		printf("Server costs: sys: %.4f inKB: %.4f outKB: %.4f\n",
		       s_syscost,s_inkbcost,s_outkbcost);
	}
	if ((fh = fopen(assist ? "/dev/null" : HFILE, "w")) == NULL) /* an assisting test should not write */
	    {
		    perror(HFILE);
		    endit(1);
	    }
	if((fm = fopen(assist ? "/dev/null" : MFILE, "w")) == NULL)
	{
		perror(MFILE);
		endit(1);
	}	
	fprintf(fh, "/* mfscosts.h -- MOSIX */\n");
	fprintf(fh, "/* cost of MFS operations (microseconds) */\n\n");
	fprintf(fh, "/* measuring date: %s */\n", now);
	if(comment)
	{
		for(c = comment ; *c ; c++)
		if(*c == '*' && *(c+1) == '/')
			*c = '/';
		fprintf(fh, "/* %s */\n", comment);
	}
	
	fdefine(fh, fm, "MFS_COST_CONN_S", (int)s_syscost);
	fdefine(fh, fm, "MFS_COST_CONN_C", (int)c_syscost);
	fdefine(fh, fm, "", 0);
	fdefine(fh, fm, "MFS_COST_INKB_S", (int)s_inkbcost);
	fdefine(fh, fm, "MFS_COST_INKB_C", (int)c_inkbcost);
	fdefine(fh, fm, "", 0);
	fdefine(fh, fm, "MFS_COST_OUTKB_S", (int)s_outkbcost);
	fdefine(fh, fm, "MFS_COST_OUTKB_C", (int)c_outkbcost);
	fclose(fh);
	fprintf(fm, "\n");
	fclose(fm);
	if(assist) 
	{
		printf("\nSubtest Complete:\n");
		printf("Please copy the exact following list of 3 numbers to the primary test:\n\n");
		printf("\t %d %d %d\n", (int)s_syscost, (int)s_inkbcost, (int)s_outkbcost);
		exit(0);
	}
	
	printf("End Of Test");
	if(!ok)
		printf(": The measured results are in %s",HFILE);
	printf("\n"); 
	exit(0);
}

void fdefine(FILE *fh, FILE *fm, char *st, int val)
{
	if(strlen(st)==0) { fprintf(fh,"\n"); return; }
	fprintf(fh,"#define %s %d\n", st, val<0 ? 0 : val);
	fprintf(fm,"%d ",val<0 ? 0 : val);
}

/* amnons functions */
void get_to(int where)
{
	while(migrate(where))
	{
		printf("Cannot migrate to %d. Retry ? ", where);
		if(!YES(1))
			endit(1);
	}
}

int YES(int eol)
{
	char buf[1024];
	char *fgets();

	buf[0] = 'n';
	fgets(buf, 1024, stdin);
	return((buf[0] == 'y') || (eol && ((buf[0] == '\n') || (buf[0] == '\0'))));
}

int readnum()
{
	char buf[1024];
	char *fgets();
	int n = 0;

	if(!fgets(buf, 1024, stdin))
		endit(1);
	sscanf(buf, "%d", &n);
	return(n);
}

int endit(val)
{
	exit(val);
}

int migrate(int to)
{
	int mfd = open("/proc/self/migrate", 1);
	char t[15];
	int result;
	
	sprintf(t, "%d\n", to);
	result = write(mfd, t, strlen(t));
	close(mfd);
	return(result < 0);
}

int whereami()
{
	FILE *f = fopen("/proc/self/where", "r");
	int w;

	if(fscanf(f, "%d", &w) != 1)
		error_msg(NOT_MOSIX);
	fclose(f);
	if(w == 0)
	{
		f = fopen("/proc/mosix/admin/mospe", "r");
		if(fscanf(f, "%d", &w) != 1)
			error_msg(NOT_MOSIX);
		fclose(f);
	}
	return(w);
}

int howmany(int where)
{
	char fn[50];
	FILE *f;
	int n;

	sprintf(fn, "/proc/mosix/nodes/%d/cpus", where);
	if(!(f = fopen(fn, "r")) || fscanf(f, "%d", &n) != 1)
		error_msg(NOT_MOSIX);
	fclose(f);
	if(n > MAXSMP)
	{
		fprintf(stderr, "Sorry, node %d is an %d-processor SMP,\n", where, n);
		fprintf(stderr, "which is more than the maximum of %d allowed by this program.\n", MAXSMP);
		exit(1);
	}
	if(n <= 0)
	{
		errno = -n;
		sprintf(fn, "Sorry, node %d is inaccessible", where);
		perror(fn);
		exit(1);
	}
	if(n > 1)
		printf("Node %d has %d processors\n", where, n);
	return(n);
}
  
int cpuspeed()
{
	struct timeval t0, t1, t2;
	int loc = whereami();
	int speed;

	register int usec;

	sleep(1);
	gettimeofday(&t0, 0);
	gettimeofday(&t1, 0);
	testloop();
	gettimeofday(&t2, 0);
	usec = (t2.tv_sec-t1.tv_sec)*1000000+t2.tv_usec-t1.tv_usec;
	usec -= (t1.tv_sec-t0.tv_sec)*1000000+t1.tv_usec-t0.tv_usec;
	speed = STD_TIME / usec;
	if(speed == STD_SPD)
		printf("Processor %d's speed is like a Pentium-III at 1GHz\n", loc);
	else
		printf("Processor %d's speed is %d/%d of a Pentium-III at 1GHz\n",
			loc, speed, STD_SPD);
	return(speed);
}

int note(int i)
{
	char x[2*MAXSMP];
	int n;

	signal(i, (sig_t)note);
	n = read(startpipe[0], x, sizeof(x));
	if(n > 0)
		nsig += n;
	return (0);
}

void measure(int fd, int ipsx)
{
	struct timeval t, t2;
	long usec;
	long sk;

	if(setjmp(env))
	{
		gettimeofday(&t2, 0);
		usec = (t2.tv_sec - t.tv_sec) * 1000000 + t2.tv_usec-t.tv_usec;
		if(ipsx)
			/*sk = ((long long)ipsx) * usec / 1000000 - counter;*/
			sk = usec - counter * 1000000LL / ipsx;
		else
			sk = counter * 1000000 / usec;
		if(sk < 0)
			sk = 0;
		lseek(fd, sk, 0);
		exit(0);
	}
	sleep(3); /* allow time for other SMP forks and parent migration */
	signal(END, (sig_t)jmp);
	write(startpipe[1], "x", 1);
	kill(getppid(), START);
	gettimeofday(&t, 0);
	count();
}

int count()
{
	int j = 3, k = 4;

	for(counter = 0 ;; counter++)
	{
		j += k;
		k += j;
		if(j|k)
			k++;
		;
	}
	return(k);
}

int jmp()
{
	longjmp(env, 1);
}
