# application-nsdr.tcl --
#
#       Primary tcl file for nsdr.  Defines defaults and assembles GUI.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Application ErrorWindow NsdrUI UserApplication \
    ScopeZone Program ProgramSource ProgramSource/SAP \
    ProgramSource/SAP/MeGa ProgramSource/Proxy \
    AddressAllocator/SAP ProgramAnnouncer

# Application object for the Nsdr session directory tool.
# Takes care of random stuff not handled by the other application
# components.
Class NsdrApplication -superclass Application

#
NsdrApplication public init argv {

    $self next nsdr

    global env
    if {![info exists env(HOME)]} {
        new ErrorWindow {Your HOME environment variable must be set.}
        exit 1
    }

    label .label
    set defaultLabelFont [.label cget -font]
    destroy .label

    set o [$self options]
    $self init_args $o
    $self init_resources $o
    set argv [$o parse_args $argv]

	# Source the user's hook file if it exists. The function
	# user_hook, which may be defined in this file, will be
	# called at the end of init.
	if {[$o get_option userhookFile] != ""} {
		if {[file isfile [$o get_option userhookFile]] && \
			[file readable [$o get_option userhookFile]]} {
			source [$o get_option userhookFile]
		} else {
			puts stderr "Unable to source \"[$o get_option userhookFile]\". Not a file or not readable."
		}
	}

    if {$argv!=""} {
	set dst [split $argv /]
	set n [llength $dst]
	if { $n < 2 } {
	    $self fatal "must specify both address and port in the form addr/port ($s)"
	}
        set addr [lindex $dst 0]
        set port [lindex $dst 1]
        set firstchar [string index $addr 0]
        if [string match \[a-zA-Z\] $firstchar] {
	    set s [gethostbyname $addr]
	    if { $s == "" } {
		$self fatal "cannot lookup host name: $addr"
	    }
	    set addr $s
        }
	if { ![string match \[0-9\]* $port] || $port >= 65536 } {
	    $self fatal "illegal port '$port'"
	}
	set octets [split $addr .]
	set n [llength $octets]
	if { $n != 4 } {
	    puts "invalid address (not IP4)"
	    $self fatal "invalid address (not IP4)"
	}
	set first 1
	foreach octet $octets {
	    if {$first == 1} {
		if {$octet <224 || $octet >239} {
		    puts "not in class D multicast"
		    $self fatal "not in class D multicast"
		}
		set first 0
	    } else {
		if {$octet <0 || $octet>255} {
		    puts "invalid addr: not 8-byte"
		    $self fatal "invalid addr: not 8-byte"
		}
	    }
	}
	$o add_default SAPaddress $addr
	$o add_default SAPport $port
    }

    # nsdr takes a while to start up so we'll give the user some feedback.
    wm withdraw .
    toplevel .startup
    wm title .startup [winfo name .]
    label .startup.label -text {Starting up...} -font $defaultLabelFont
    pack .startup.label -fill both -expand true
    update

    $self init_local
    # $self init_confbus

    UserApplication init_apps
    $self init_ui
    $self init_sap
    $self init_proxy
    $self init_mega

    $self instvar ui_
    if {[llength [$ui_ set sources_]] == 0} {
	puts stderr "No program sources..."
	exit 1
    }

    $self user_hook

    destroy .startup
    wm deiconify .
}

NsdrApplication instproc init_args o {
	$o register_option -a extraAppFiles
	$o register_option -u userhookFile
    	$o register_option -usemega megaAddrs
	$o register_option -sloc serviceLocation
        $o register_option -sapAdr SAPaddress
        $o register_option -sapPrt SAPport
	$o register_boolean_option -simple simpleInterface
	foreach a "SAP Mega Scuba Proxy" {
	    	$o register_boolean_option -use$a use$a
		$o register_boolean_option -no$a use$a 0
	}
}

NsdrApplication instproc init_resources o {
    $o load_preferences "nsdr"

    # defaults
    global env
    $o add_default cachedir [file join $env(HOME) .mash nsdr-cache]
    $o add_default cacheWriteInterval 300
    $o add_default useSAP 1
    $o add_default SAPaddress 224.2.127.254
    $o add_default SAPport 9875

    $o add_default sapZones 224.2.128.0/17,239.255.0.0/16
    $o add_default sapTTL 255

    $o add_default useProxy 0
    $o add_default proxyLocation quimby.cs.berkeley.edu/12000
    $o add_default useProxyAnnouncer 0

    $o add_default simpleInterface 0
    $o add_default alarmLead 300
    $o add_default webbrowser xm
    $o add_default unifiedVic 0

    set filename [file join $env(HOME) .mash nsdr-apps.tcl]
    if {![file exists $filename]} {
        set filename {}
    }
    $o add_default appFiles $filename

    # mega/scuba defaults
    $o add_default useMega 0
    $o add_default megaBW 128000
    $o add_default megaStartupBW 64000
    $o add_default megaMaxBW 128000
    $o add_default asCtrl 224.4.5.24/50000/31
    $o add_default asCtrlBW 20000
    $o add_default serviceLocation static:sdgw

    # font stuff (largely stolen from vic)
    $o add_default foundry adobe
    set foundry [$o get_option foundry]
    set helv10 [$self search_font $foundry helvetica medium 10 r]
    set helv10b [$self search_font $foundry helvetica bold 10 r]
    set helv10o [$self search_font $foundry helvetica bold 10 o]
    set helv12b [$self search_font $foundry helvetica bold 12 r]
    set times14 [$self search_font $foundry times medium 14 r]

    $o add_default medfont $helv12b
    $o add_default smallfont $helv10b
    $o add_default entryFont $helv10
    $o add_default helpFont $times14

    option add *Font [$self get_option medfont] startupFile
    option add *Button.Font [$self get_option smallfont] startupFile
    option add *Checkbutton.Font [$self get_option smallfont] startupFile
    option add *Label.Font [$self get_option smallfont] startupFile
    option add *Listbox.Font [$self get_option smallfont] startupFile
    option add *Menu.Font [$self get_option smallfont] startupFile
    option add *Menubutton.Font [$self get_option smallfont] startupFile
    option add *Message.Font [$self get_option smallfont] startupFile
    option add *Entry.Font [$self get_option entryFont] startupFile

    option add *Entry.Background white
    option add *padX 2
    option add *padY 2

    set name [$self get_option nsdrName]
    set phone [$self get_option nsdrPhone]
    if {$name == "" || $phone == ""} {
	$self run_resource_dialog $name $phone
    }
}

NsdrApplication private run_resource_dialog {name phone} {
    set w .form
    frame $w

    frame $w.msg -relief ridge
    label $w.msg.label -font [$self get_option medfont] \
	-wraplength 4i -justify left \
	-text "Please specify values for the following resources. \
These strings will identify you and give your phone number in \
outgoing session announcements that you create.  The values you \
enter will be saved in ~/.mash/prefs so you will not have to \
re-enter them." -relief ridge
    pack $w.msg.label -padx 6 -pady 6
    pack $w.msg -side top

    foreach i {name phone} {
	frame $w.$i -bd 2
	entry $w.$i.entry -relief sunken
	label $w.$i.label -width 10 -anchor e
	pack $w.$i.label -side left
	pack $w.$i.entry -side left -fill x -expand 1 -padx 8
    }
    if {"[$self get_option rtpName]" != ""} {
	$w.name.entry insert end [$self get_option rtpName]
    }
    $w.name.label config -text Name:
    $w.phone.label config -text Phone:
    pack $w.msg -pady 10
    pack $w.name $w.phone -side top -fill x

    frame $w.buttons
    button $w.buttons.accept -text Accept -command "set dialogDone 1"
    button $w.buttons.dismiss -text Quit -command "set dialogDone -1"
    pack $w.buttons.accept $w.buttons.dismiss -side left \
	-expand 1 -padx 20 -pady 10
    pack $w.buttons
    pack $w -padx 10

    global dialogDone
    while { 1 } {
	set dialogDone 0
	focus $w.name.entry
	tkwait variable dialogDone
	if {$dialogDone < 0} {
	    exit 0
	}
	set name [string trim [$w.name.entry get]]
	if { [string length $name] <= 3 } {
	    new ErrorWindow "please enter a reasonable name"
	    continue
	}
	set phone [string trim [$w.phone.entry get]]
	#FIXME
	if { [string length $phone] <= 10} {
	    new ErrorWindow "please enter a complete phone number"
	    continue
	}
	break
    }

    $self add_option nsdrName $name
    $self add_option nsdrPhone $phone

    global env
	set mash [file join $env(HOME) .mash]
    if {![file exists $mash]} {
        file mkdir $mash
    }
    set f [open [file join $mash prefs-nsdr] a+ 0644]
    puts $f "nsdrName: $name"
    puts $f "nsdrPhone: $phone"
    close $f

    unset dialogDone
    pack forget $w
    destroy $w
}

NsdrApplication instproc init_ui {} {
    pack [frame .f] -fill both -expand yes
    $self set ui_ [new NsdrUI .f $self]
    wm geometry . 300x500
    wm protocol . WM_DELETE_WINDOW "$self exit"
}

NsdrApplication instproc init_sap {} {
    # instantiate SAP source
    if ![$self yesno useSAP] {
	return
    }

    # FIXME this should eventually use an automatic mechanism
    #     (e.g., MZAP - draft-ietf-mboned-mzap) to discover
    #     what scope zones are present.
    #
    # FIXME Also need a simple way to figure out SAP bandwidth
    set zones {}
    foreach zone [split [$self get_option sapZones] ","] {
	lappend zones [new ScopeZone $zone]
    }

    $self instvar ui_
    set s [eval new ProgramSource/SAP $ui_ $zones]

    # eventually this may change to use AddressAllocator/MDHCP
    # or /AAP or something else.
    $self instvar allocator_
    set allocator_ [new AddressAllocator/SAP $s]

    #
    $self instvar announcer_
    set announcer_ [new ProgramAnnouncer $s]

    $self instvar sources_
    lappend sources_ $s
}

NsdrApplication private init_proxy {} {
    if ![$self yesno useProxy] {
	return
    }

    #FIXME needs lots of work
    $self instvar sources_ ui_
    set spec [$self get_option proxyLocation]
    lappend sources_ [new ProgramSource/Proxy $ui_ $spec]
}

NsdrApplication instproc init_mega {} {
    if { [$self get_option megaAddrs] != "" } {
	$self instvar source_ ui_
	foreach addr [split [$self get_option megaAddrs] ","] {
	    # FIXME this is only the unicast case.  Need the multicast case.
	    set s [new ProgramSource/SAP/MeGa $ui_ $addr]

	    $self instvar sources_
	    lappend sources_ $s
	}
    }
}

NsdrApplication instproc gourl {url} {
    set browser [$self get_option webbrowser]
    catch {exec $browser $url &}
}

NsdrApplication instproc exit {} {
    $self instvar sources_
    foreach s $sources_ {
	$s shutdown
    }
    exit
}
