# application-rvc.tcl --
#
#       This is basically a Vic with the GUI in an SDS service.  It provides
#       the SDS GUI for the cameras, vcr, etc.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import RTPApplication
import RvcAgent
import RvcObserver
import VideoPipeline
import CSdsService
import CServiceManager
import CService

import 405Client

global g_app

#
# Code for RvcApplication
#

Class RvcApplication -superclass RTPApplication

RvcApplication instproc InitArgs { options } {
    $options register_option -t defaultTTL
    $options register_option -s serverPort
    $options register_option -f videoFormat
    $options register_option -b videoBandwidth
    $options register_option -q videoQuality
    $options register_option -p videoInputPort
    $options register_option -F frameRate
    $options register_option -n norm
    $options register_option -l loopback

    # for the service discovery service
    $options register_option -sa optServiceAddress
    $options register_option -sp optServicePort
    $options register_option -st optServiceTTL

    # for the service discovery service
    $options register_option -rp optComPort

    # for amxd 
    $options register_option -ah optAmxHost
    $options register_option -ap optAmxPort

    # for the configuration file
    $options register_option -cf optConfigFile
}

RvcApplication instproc InitResources { options } {
    $options add_default defaultTTL 16
    $options add_default serverPort 8888
    $options add_default videoFormat jpeg
    $options add_default videoBandwidth 524288
    $options add_default videoQuality 85
    $options add_default videoInputPort "Composite1"
    $options add_default frameRate 15
    $options add_default norm auto
    $options add_default loopback 0

    # for the service discovery service
    $options add_default optServiceAddress "224.4.6.8"
    $options add_default optServicePort "12344"
    $options add_default optServiceTTL "16"

    # for the service discovery service
    $options add_default optComPort "11408"

    # file name for configuration file
    $options add_default optConfigFile ".rvcconfig"
}

RvcApplication private LoadConfigFile { } {
    $self instvar m_lConfig

    set options [$self options]
    set szConfigFileName [$options get_option optConfigFile]

    set fileConfig [open $szConfigFileName "r"]

    # get the host name to pick out which configuration to get
    set hostname [exec hostname]

    # need this indirection because [ and ] are special in both tcl and regex
    set openBracket "\["
    set closeBracket "\]"

    # set the two patterns up to be used later
    set pattern "\\$openBracket"
    append pattern $hostname
    append pattern "\\$closeBracket"

    set pattern2 "\\$openBracket"
    append pattern2 "*"
    append pattern2 "\\$closeBracket"

    # go through each line of the config file
    while { [gets $fileConfig szLine] >= 0 } {
	set szLine [string trim $szLine]

	# get to the correct section
	if { ![string match $pattern $szLine] } {
	    continue
	}

	# now we're at the right section
	while { [gets $fileConfig szSectionLine] >= 0 } {
	    set szSectionLine [string trim $szSectionLine]

	    # check that it's not an empty line
	    if {$szSectionLine == ""} {
		continue
	    }

	    # if we hit the next section then just return
	    if { [string match $pattern2 $szSectionLine] } {
		break
	    }

	    # otherwise it's a section line
	    set lLine [split $szSectionLine =]
	    set index [string trim [lindex $lLine 0]]
	    set value [string trim [lindex $lLine 1]]
	    set aConfig($index) $value

	    # Since some of the parameters from config file is
	    # the same as command line arguments (e.g. FORMAT 
	    # and -videoFormat), we override the default value
	    # of command line arguments with the values from 
	    # config file.  This is to maintain consistency 
	    # as some of the routines (such as the constructor
	    # for RvcAgent) reads value from command line arguments
	    # and not from config file.

	    $self update_defaults $options $index $value
	}

	break
    }

    close $fileConfig

    set m_lConfig [array get aConfig]

#    puts "config = $m_lConfig"
}


#-----------------------------------------------------------------------------
#
# RvcApplication instproc update_defaults { options key value }
#
# Input:
#   options - the Configuration object associated with this object
#             (obtained thru [$self options]
#   key, value - a key value pair from rvc's configuration file.
#
# Description:
#   This is used to override the default values passed in via
#   command line arguments, it is to maintain consistency because
#   some routines read values from command line arguments and not
#   from the configuration file.  It is called everytime a key-value
#   pair is read from the configuration file.
#
#-----------------------------------------------------------------------------

RvcApplication instproc update_defaults { options key value } {
    switch -exact $key {
	FORMAT {
	    $options add_default videoFormat $value	
	} 
	QUALITY {
	    $options add_default videoQuality $value	
	}
	PORT {
	    $options add_default videoInputPort $value	
	}
	FPS {
	    $options add_default frameRate $value	
	}
	BPS {
	    $options add_default videoBandwidth $value	
	}
	NORM {
	    $options add_default norm $value	
	}
	AMX_PORT {
	    $options add_default amxPort $value	
	}
	AMX_HOST {
	    $options add_default amxHost $value	
	}
    }
} 


#-----------------------------------------------------------------------------
#
# RvcApplication InitAMX
#
# Input:
#   amxdHost, amxdPort - the host and port of amxd.  (This could be "" if
#      they are not initialized by command line args or configuration file)
#
# Description:
#   Create a new Dp client to talk to amxd.  Then tell amxd to switch
#   input to value specified in the configuration file.
#
#-----------------------------------------------------------------------------
RvcApplication instproc InitAMX {amxHost amxPort} {
    $self instvar m_405
    $self instvar m_lConfig

    array set aConfig $m_lConfig

    if {$amxHost == "" || $amxPort == ""} {
	puts stderr "No AMX host/port specified."
	return
    }
    set m_405 [new 405Client $amxHost $amxPort]
    set callback "$self ProcessAmxMsg"
    # FIXME - this is an ugly hack that works (for now) to get htsr or htsr2
    #   from the hostname, won't work if things ever change
    set hostname [exec hostname]
    set parts [split $hostname "."]
    set shortHostname [lindex $parts 0]
    set filter [$self GetFilter $shortHostname]
    $m_405 callback_register $callback $filter
    $m_405 callback_enable $callback

    $self AMXSwitch $aConfig(AMX_SWITCH_STATE)
}

RvcApplication instproc init { argv } {
    $self next rvc

    # set some member variables
    $self instvar m_agent
    $self instvar m_observer
    $self instvar m_lConfig
    $self instvar m_switchReal
    $self instvar m_cameraMoveButtonPressed

    set m_agent 0
    set m_observer 0
    set m_switchReal 0
    set m_cameraMoveButtonPressed 0

    # Initiailization of variables and resources.
    set options [$self options]
    $self InitArgs $options
    $self InitResources $options

    $options load_preferences "rtp rvc"
    set argv [$options parse_args $argv]

    # load the configuration file
    $self LoadConfigFile

    # create the sds session object
    set inetServiceAddr [$options get_option optServiceAddress]
    set iServicePort [$options get_option optServicePort]
    set iServiceTTL [$options get_option optServiceTTL]

    $self instvar m_sdsService
    set m_sdsService [new CSdsService $self $inetServiceAddr $iServicePort \
	    $iServicePort $iServiceTTL]

    # setup the service objects
    $self instvar m_serviceManager
    $self instvar m_service

    set iPort [$options get_option optComPort]
    set m_serviceManager [new CServiceManager "ServiceApp" "Video" $iPort]
    $m_serviceManager Attach $self

    # setup the amx stuff if we have to
    array set aConfig $m_lConfig

    # Try to initialize AMX, if successful, m_405 will be set to a new
    # 405Client.
    $self InitAMX [$options get_option amxHost] [$options get_option amxPort] 

    puts "RVC initialization done"
}

##############################################################################
#
# RvcApplication instproc NewConnection { service } {
#
# Input:
# service - the new service that got connected
#
# Output:
# 1 - if handling this service
# 0 - otherwise
#
# Description:
# A call back function for the service manager.  This function will be called
# when a new connection has just been noticed by the service manager
#
##############################################################################
RvcApplication instproc NewConnection { service } {

    $service MapMessage "SYN_START" $self "SynStart"

    return 1
}

##############################################################################
#
# RvcApplication instproc SynStart { service arguments } {
#
# Input:
# service - the service object that called this function
# arguments - the arguments that are passed to this funct from remote caller
#
# Output:
# none
#
# Description:
#
##############################################################################
RvcApplication instproc SynStart { service arguments } {
    $self instvar m_agent
    $self instvar m_observer
    $self instvar m_serviceManager
    $self instvar m_service
    $self instvar m_lConfig
    array set aConfig $m_lConfig

    set options [$self options]
    # if these are valid then we're already being used
    if { $m_agent != 0 || $m_observer != 0 } {
	delete $service 
	return
    }

    set inetMStudioAddr [lindex $arguments 0]
    set iMStudioPort [lindex $arguments 1]

    set m_observer [new RvcObserver $self]
    set m_agent [new RvcAgent $self "$inetMStudioAddr/$iMStudioPort"]

    # set the parameters then start sending the multicast stream
    $m_agent set_fmt [$options get_option videoFormat]
    if [info exists aConfig(DEVICE)] {
	set device [$m_agent get_device_by_name $aConfig(DEVICE)]
	if {$device != "" && $device != [$m_agent current_device]} {
	    $m_agent set_device $device
	}
    }
    $m_agent set_inport [$options get_option videoInputPort]
    $m_agent set_norm [$options get_option norm]
    $m_agent start
    $m_agent set_fps [$options get_option frameRate]
    $m_agent set_bps [$options get_option videoBandwidth]
    $m_agent set_quality $aConfig(QUALITY)


    # now connect to the client with the souce id
    set inetAddr [lindex $arguments 2]
    set iPort [lindex $arguments 3]
    set iSourceId [$m_agent get_local_srcid]

    set m_service [$m_serviceManager NewService $inetAddr $iPort \
	    "VideoService" $iSourceId]

    # set up the mapping for this service
    $m_service MapMessage "GET_UI_CONTROL" $self "GetUIControl"
    $m_service MapMessage "DEVICE" $self "Device"
    $m_service MapMessage "PORT" $self "Port"
    $m_service MapMessage "FORMAT" $self "Format"
    $m_service MapMessage "FPS" $self "FrameRate"
    $m_service MapMessage "AMX_SWITCH" $self "RemoteAMXSwitch"
    $m_service MapMessage "AMX_CAMERA_CONTROL" $self "AMXCameraControl"
    $m_service MapMessage "AMX_VCR_CONTROL" $self "AMXVCRControl"
    $m_service MapMessage "TOGGLE_RN" $self "ToggleRn"
    $m_service MapMessage "CLOSE_LINK" $self "CloseService"
    $m_service MapMessage "GET_INFO" $self GetInfo
    $m_service MapMessage "FIR" $self "FIR"
}

##############################################################################
#
# RvcApplication instproc CloseService { service arguments }
#
# Input:
# service - the service object that called this function
# arguments - the arguments that are passed to this funct from remote caller
#
# Output:
# none
#
# Description:
# Need to close the service - ie. stop sending out the video and shutdown
# everything and wait until the next client comes
#
##############################################################################
RvcApplication instproc CloseService { service arguments } {
    $self instvar m_agent
    $self instvar m_observer

    $m_agent stop
    $m_agent detach_observer $m_observer

    delete $m_agent
    delete $m_observer

    set m_agent 0
    set m_observer 0

    exit
}


RvcApplication instproc GetUIControl { service arguments } {
    $self instvar m_agent
    $self instvar m_405

    # for unique variable names
    set cmd "regsub -all -- {\\\.} \$winFrame {_} __name \n"

    append cmd { 
	frame $winFrame.enc 
	frame $winFrame.cam 
	set __f [$self get_option smallfont]
	label $winFrame.enc.l -text Encoder -font $__f
	pack  $winFrame.enc.l -side left -fill y
	label $winFrame.cam.l -text Camera -font $__f
	pack  $winFrame.cam.l -side left -fill y
	pack  $winFrame.enc 
	pack  $winFrame.cam 
    }
    set cmd "$cmd [$self FrameRateCommand]"
    set cmd "$cmd [$self FormatCommand]"
    set cmd "$cmd [$self InPortCommand]"
    set cmd "$cmd [$self InDeviceCommand]"

    # 
    # Only creates UI for controlling AMX if we successfully 
    # contacted amxd.
    #
    if {[$m_405 isConnectedToAMX]} {
	set cmd "$cmd [$self AMXCommand] \n"
	set cmd "$cmd [$self DeviceCommand] \n"
    }
    append cmd {
	pack  $winFrame.enc $winFrame.cam -side top
    }

    $service Send CONTROL_UI $cmd
}

##############################################################################
#
# RvcApplication private InPortCommand { } {
# RvcApplication private InDeviceCommand { } {
# RvcApplication private InSignalCommand { } {
#
# Input:
# none
#
# Output:
# the Tk commands that create GUI for choosing difference capture 
# card, input ports and signal type
#
##############################################################################
RvcApplication instproc InDeviceCommand { } {
    $self instvar m_agent
    set w \$winFrame.enc.menubInDevice

    set cmd ""
    append cmd {
    	menubutton $winFrame.enc.mdev -font $__f -text Device -menu $winFrame.enc.mdev.menu -relief raised 
        pack $winFrame.enc.mdev -expand 1 -side right -fill y
	global dev$__name
	set dev$__name } [$m_agent current_device] {
	set __menuDEV [menu $winFrame.enc.mdev.menu]
	}
    foreach d [$m_agent input_devices] {

	set cmd "$cmd \$__menuDEV add radio -label \"[$d nickname]\" \
		-variable dev\$__name -font \$__f -value \"[$d nickname]\" \
		-command \"\$service Send DEVICE [$d nickname]\" \n"

    }
    return $cmd
}

RvcApplication instproc InPortCommand { } {
    $self instvar m_agent
    set cmd ""
    append cmd {
    	set __f [$self get_option smallfont]
    	menubutton $winFrame.enc.mport -text "Port" -menu $winFrame.enc.mport.menu -relief raised -font $__f
    	pack $winFrame.enc.mport -expand 1 -side right -fill y
    	global port$__name
    	set port$__name } [$m_agent current_inport] {
	set __menuPORT [menu $winFrame.enc.mport.menu] 
	}
    foreach p [$m_agent current_inport_list] {
	set cmd "$cmd \$__menuPORT add radio -label $p \
		-variable port\$__name -font \$__f -value $p \
		-command \"\$service Send PORT $p\" \n"
    }
    return $cmd
}


RvcApplication instproc FrameRateCommand { } {
    $self instvar m_agent
    set cmd ""
    append cmd {
	global fps$__name
	frame $winFrame.enc.fps -relief ridge
    	scale $winFrame.enc.fps.s -relief raised -variable fps$__name -from 1 -to 30 \
	    -orient horiz -state active -font $__f -showvalue no -highlightthickness 1 -length 70
	label $winFrame.enc.fps.l -textvariable fps$__name -font $__f -width 2
    	set fps$__name } [$m_agent current_fps] {
    	button $winFrame.enc.fps.b -text "< set fps" -font $__f -command "$service Send FPS \$fps$__name"
    	pack $winFrame.enc.fps -side right 
    	pack $winFrame.enc.fps.s $winFrame.enc.fps.l $winFrame.enc.fps.b -side left 
    }
    return $cmd
}



RvcApplication instproc RNCommand { } {
    # create a menu button for the format
    return {
	button $winFrame.butRN -text "Real" \
	    -relief raised -font $__f -command "$service Send TOGGLE_RN"
	pack $winFrame.butRN -expand 1 -side right -fill y
    }
}

RvcApplication instproc FormatCommand { } {
    $self instvar m_agent

    set cmd ""
    # create a menu button for the format
    append cmd {
	menubutton $winFrame.enc.mfmt -text Format -menu $winFrame.enc.mfmt.menu -relief raised -font $__f
    	pack $winFrame.enc.mfmt -expand 1 -side right -fill y

    	global format$__name
    	set format$__name } [$m_agent current_fmt] {
    	set __menuFMT [menu $winFrame.enc.mfmt.menu] 
    }

    set format_list { h261 jpeg }

    foreach format $format_list {
	set cmd "$cmd \$__menuFMT add radio -label $format \
		-variable format\$__name -font \$__f -value $format \
		-command \"\$service Send FORMAT $format\" \n"
    }

    return $cmd
}

RvcApplication instproc AMXCommand { } {
    $self instvar m_lConfig
    array set aConfig $m_lConfig

    # first make the switch button
    set cmd ""
    append cmd {
	set __w $winFrame.cam.menubSwitch
    	menubutton $__w -font $__f -text "Input.."  -menu ${__w}.menu -relief raised
    	pack $__w -expand 1 -side left
    }
    # create the switch menu
    append cmd {
	global switch$__name
    	set switch$__name } $aConfig(AMX_SWITCH_STATE) {
    	set __menuSWITCH [menu ${__w}.menu]
    }

    # input_list is a list of (label,value) pair
    set input_list {
	{Wide Camera}  wideCamera
	{Speaker Camera} speakerCamera
	{Audience Camera} audienceCamera
	{Front PC} frontPC
	{Laptop} laptop
	{Elmo} elmo
	{Front VCR} frontVCR
	{Rack VCR} rackVCR
	{Liveboard} liveboard
	{MBone PC} mbonePC
	{SGI} sgiPC
    }
    foreach {label value} $input_list {
	set cmd "$cmd \$__menuSWITCH add radio -font \$__f -label \"$label\"\
		-variable switch\$__name -value $value \
		-command \"\$service Send AMX_SWITCH $value\" \n"
    }

    return $cmd
}

RvcApplication instproc DeviceCommand { } {
    $self instvar m_lConfig
    array set aConfig $m_lConfig

    set cmd ""

    switch -exact -- $aConfig(AMX_SWITCH_STATE) {
	speakerCamera -
	audienceCamera {

	    # put in a frame for the pan button
	    append cmd {
		set f [$self get_option smallfont]
		frame $winFrame.cam.framePan
		pack $winFrame.cam.framePan -side right
		}

	    # put in the up down left right zin and zout buttons

	    # the commented out sections are the old interface, where pushing
	    #   the button caused a small pulse movement in that direction
	    # now, you hold down the button to start motion, and release when
	    #   you want it to stop
#	    set cmd "$cmd button \$winFrame.cam.framePan.right -text Right \
#		    -command \"\$service Send AMX_CAMERA_CONTROL right\" \n"
	    append cmd {
		set __w $winFrame.cam.framePan.right
		button $__w -font $__f -text ">"
		pack $__w -side right -fill y
		bind $__w <Button-1> "$service Send AMX_CAMERA_CONTROL {start right}"
		bind $__w <ButtonRelease-1> "$service Send AMX_CAMERA_CONTROL stop"

		frame $winFrame.cam.framePan.updown
		pack $winFrame.cam.framePan.updown -side right

		set __w $winFrame.cam.framePan.updown.up
		button $__w -font $__f -text "^"
		pack $__w -fill x -side top
		bind $__w <Button-1> "$service Send AMX_CAMERA_CONTROL {start up}"
		bind $__w <ButtonRelease-1> "$service Send AMX_CAMERA_CONTROL stop"

		set __w $winFrame.cam.framePan.updown.down
		button $__w -font $__f -text "v"
		pack $__w -side bottom
		bind $__w <Button-1> "$service Send AMX_CAMERA_CONTROL {start down}"
		bind $__w <ButtonRelease-1> "$service Send AMX_CAMERA_CONTROL stop"

		set __w $winFrame.cam.framePan.left
		button $__w -font $__f -text "<"
		pack $__w -side right -fill y
		bind $__w <Button-1> "$service Send AMX_CAMERA_CONTROL {start left}"
		bind $__w <ButtonRelease-1> "$service Send AMX_CAMERA_CONTROL stop"
	    }

	    append cmd {
		set __w $winFrame.cam.framePan.zinout
		frame $__w
	    	pack $__w -side right 
	    	button ${__w}.in -font $__f -text "zoom in" 
	    	pack ${__w}.in -fill x -side top
	    	bind ${__w}.in <Button-1> "$service Send AMX_CAMERA_CONTROL {start in}"
	    	bind ${__w}.in <ButtonRelease-1> "$service Send AMX_CAMERA_CONTROL stop"

	    	button ${__w}.out -font $__f -text "zoom out"
	    	pack ${__w}.out -side bottom
	    	bind ${__w}.out <Button-1> "$service Send AMX_CAMERA_CONTROL {start out}"
	    	bind ${__w}.out <ButtonRelease-1> "$service Send AMX_CAMERA_CONTROL stop"
	    }

	    # create a menu button for extra effects
	    append cmd {
		set __w $winFrame.cam.framePan.finout
		frame $__w
	    	pack $__w -side right
	    	button ${__w}.in -font $__f -text "fade in" -command "$service Send AMX_CAMERA_CONTROL {fade in}"
	    	pack ${__w}.in -fill x -side top

	    	button ${__w}.out -font $__f -text "fade out" -command "$service Send AMX_CAMERA_CONTROL {fade out}"
	    	pack ${__w}.out -side bottom
	    }
	}
	frontVCR -
	rackVCR {
	    # put in the power, pause, stop, rewind, forward, play
	    # and record buttons
	    append cmd {
		frame $winFrame.cam.frameVCR
		pack $winFrame.cam.frameVCR -side right

		frame $winFrame.cam.frameVCR.0
		pack $winFrame.cam.frameVCR.0 -side right

		button $winFrame.cam.frameVCR.0.ff -font $__f -text ">>" \
			-command "$service Send AMX_VCR_CONTROL ffwd"
		pack $winFrame.cam.frameVCR.0.ff -fill x -side top

		button $winFrame.cam.frameVCR.0.rec -font $__f -text Rec \
			-command "$service Send AMX_VCR_CONTROL rec"
		pack $winFrame.cam.frameVCR.0.rec -side bottom -fill x

		frame $winFrame.cam.frameVCR.1
		pack $winFrame.cam.frameVCR.1 -side right

		button $winFrame.cam.frameVCR.1.play -font $__f -text ">" \
			-command "$service Send AMX_VCR_CONTROL play"
		pack $winFrame.cam.frameVCR.1.play -side top -fill x

		button $winFrame.cam.frameVCR.1.stop -font $__f -text Stop \
			-command "$service Send AMX_VCR_CONTROL stop"
		pack $winFrame.cam.frameVCR.1.stop -side bottom -fill x

		frame $winFrame.cam.frameVCR.2
		pack $winFrame.cam.frameVCR.2 -side right

		button $winFrame.cam.frameVCR.2.rr -font $__f -text "<<" \
			-command "$service Send AMX_VCR_CONTROL rew"
		pack $winFrame.cam.frameVCR.2.rr -side top -fill x

		button $winFrame.cam.frameVCR.2.pause -font $__f -text "||" \
			-command "$service Send AMX_VCR_CONTROL pause"
		pack $winFrame.cam.frameVCR.2.pause -side bottom -fill x

		button $winFrame.cam.frameVCR.power -font $__f -text Power \
			-command "$service Send AMX_VCR_CONTROL power"
		pack $winFrame.cam.frameVCR.power -side right -fill y
	    }
	}

	elmo {
	}

	wideCamera -
	laptop -
	liveboard -
	sgiPC -
	mbonePC -
	frontPC {
	}

    }

    return $cmd
}

RvcApplication instproc Device { service args } {
    $self instvar m_agent m_service
    set device [$m_agent get_device_by_name $args]
    if {$device != "" && $device != [$m_agent current_device]} {
	$m_agent set_device $device
	$self GetUIControl $m_service 0
    } else {
	puts stderr "Device is $device"
    }
}

RvcApplication instproc Port { service port } {
    $self instvar m_agent
    $m_agent set_inport $port
}

RvcApplication instproc Format { service arguments } {
#    puts "format changing to $arguments"

    $self instvar m_agent
    $m_agent set_fmt $arguments
}


RvcApplication instproc FrameRate { service arguments } {
#    puts "format changing to $arguments"

    $self instvar m_agent
    $m_agent set_fps $arguments
}


#RvcApplication instproc Signal { service arguments } {
#    puts "format changing to $arguments"
#
#    $self instvar m_agent
#    $m_agent set_signal $arguments
#}


RvcApplication public RemoteAMXSwitch { service target } {
    $self instvar m_service

    # do the switch
    $self AMXSwitch $target

    # now update the user interface for the client
    $self GetUIControl $m_service 0
}

# the string argument to UPDATE_INFO is put in the DC display
RvcApplication public GetInfo { service } {
    $self instvar m_switchReal

    if {$m_switchReal} {
	$service Send UPDATE_INFO "RealNetworks Following"
    } else {
	$service Send UPDATE_INFO ""
    }
}

RvcApplication public ToggleRn { service } {
    $self instvar m_service m_switchReal

    if {$m_switchReal} {
	set m_switchReal 0
    } else {
	set m_switchReal 1
    }

    # update the UI
    $self GetInfo $m_service
}

RvcApplication public AMXSwitch { target } {
    $self instvar m_405
    $self instvar m_lConfig
    $self instvar m_switchReal
    array set aConfig $m_lConfig

    set matrix $aConfig(AMX_OUTPUT_SWITCH)

    if {$m_switchReal} {
	$m_405 matrix_switchVideoStream $target realNetworks
    }

    # matrix is the output, target is the input
    # use the amxd matrix library
    $m_405 matrix_switchVideoStream $target $matrix


    set aConfig(AMX_SWITCH_STATE) $target
    set m_lConfig [array get aConfig]
}

RvcApplication public AMXCameraControl { service argument } {
    $self instvar m_405
    $self instvar m_lConfig
    $self instvar m_cameraMoveButtonPressed
    array set aConfig $m_lConfig

    # make sure that the switch is switched to the camera view
    switch -exact -- $aConfig(AMX_SWITCH_STATE) {
 	audienceCamera {
	    set cam "audience"
	}
 	speakerCamera {
	    set cam "speaker"
	}
 	default { return }
    }

    set arg [lindex $argument 0]

    switch -exact -- $arg {
	in -
	out -
	left -
	right -
	up -
	down {
	    $m_405 camera_pulseMove $cam $argument
	}
	start {
	    set arg2 [lindex $argument 1]
	    switch -exact -- $arg2 {
		up -
		down -
		left -
		right -
		in -
		out {
		    set m_cameraMoveButtonPressed 1
		    $m_405 camera_startMove $cam $arg2
		}
		default {
		    return
		}
	    }
	}
	stop {
	    set m_cameraMoveButtonPressed 0
	    $m_405 camera_stopMove $cam
	}
	fade {
	    set arg2 [lindex $argument 1]
	    switch -exact -- $arg2 {
		in {
		    $m_405 camera_fadeIn $cam
		}
		out {
		    $m_405 camera_fadeOut $cam
		}
	    }
	}
	default {
	    puts stdout "amxcameracontrol: urecognized argument is $argument"
	    return
	}
    }


}

RvcApplication public AMXVCRControl { service argument } {
    $self instvar m_405
    $self instvar m_lConfig
    array set aConfig $m_lConfig

    # make sure that the switch is switched to the vcr
    # give a vcr number if it is a vcr otherwise just return
    switch -exact -- $aConfig(AMX_SWITCH_STATE) {
 	frontVCR {
	    $m_405 vcr_doFunction front $argument
	}
 	rackVCR {
	    $m_405 vcr_doFunction rack $argument
	}
 	default {
	    return
	}
    }
}

##############################################################################
#
# RvcApplication public GetSdsServiceData { } {
#
# Input:
# none
#
# Output:
# the data that the will go out to the sds system
#
# Description:
# This is a callback function for the service discovery system.  Need to
# return the data that will go out to the sds system.  So far there are
# three fields with their values
#
##############################################################################
RvcApplication public GetSdsServiceData { } {
    $self instvar m_lConfig

    set options [$self options]
    set iComPort [$options get_option optComPort]
    set hostname [exec hostname]

    array set aConfig $m_lConfig
    set location $aConfig(LOCATION)
    set type $aConfig(TYPE)

    set data [list "UPDATE:" [concat $hostname $iComPort] \
	    [list [list "LOCATION:" $location] \
	    	  [list "TYPE:" $type]]]

    return $data
}

# this is used to keep the RVC matrix switcher state consistent with
#   the actual box
#
# when other processes change the switch configuration, the RVC checks if
#   the video it is getting is routed from a different source and updates
#   the GUI
RvcApplication public ProcessAmxMsg {amxMsg} {
    $self instvar m_cameraMoveButtonPressed
    $self instvar m_lConfig
    $self instvar m_service
    array set aConfig $m_lConfig

    #puts stdout "RvcApplication::ProcessAmxMsg: got $amxMsg"

    set hostname [exec hostname]

    if {[llength $amxMsg] != 0} {
	set eventInfo [lindex $amxMsg 0]
	set type [lindex $eventInfo 0]
	set eventData [lindex $amxMsg 1]

	if {$type == "matrixSwitch"} {
	    set input [lindex $eventData 0]
	    set output [lindex $eventData 1]
	    # FIXME - this is an ugly hack that works (for now)
	    if {"${output}.bmrc.berkeley.edu" == "$hostname"} {
		# the thing switched is what we're encoding

		if {$input != $aConfig(AMX_SWITCH_STATE)} {
		    # handle the case where the director is moving the camera
		    #    through the buttons, but it gets switched while he's
		    #    holding down a button
		    if {$m_cameraMoveButtonPressed} {
			# fake that the button was released
			$self AMXCameraControl "dummy" stop
		    }
		}

		set aConfig(AMX_SWITCH_STATE) $input
		set m_lConfig [array get aConfig]
		# update the GUI
		$self GetUIControl $m_service 0
	    }
	}
    }
}

RvcApplication instproc GetFilter {watchOutput} {
    # This filters out all events except for matrixSwitch events that have an
    #   output of $watchOutput
    set filter ""
    set filter "$filter set doCallback 0\n"
    set filter "$filter set info \[lindex \$arg 0\]\n"
    set filter "$filter set type \[lindex \$info 0\]\n"
    set filter "$filter set data \[lindex \$arg 1\]\n"
    set filter "$filter if \{\$type == \"matrixSwitch\"\} \{\n"
    set filter "$filter \tset output \[lindex \$data 1\]\n"
    set filter "$filter \tif \{\$output == \"$watchOutput\"\} \{\n"
    set filter "$filter \t\tset doCallback 1\n"
    set filter "$filter \t\}\n"
    set filter "$filter \}\n"
    return $filter
}

RvcApplication instproc FIR { service } {
    $self instvar m_agent
    $m_agent trigger_fir
}
