/*
 * vw.h --
 *
 *      Defines the video wiget that's used to display video in certain 
 *      tools, e.g., vic.  Multiple implementations are provided in order 
 *      to take advantage of various OS-specific performance features.
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/render/vw.h,v 1.14 2002/02/11 21:17:22 lim Exp $
 */

#ifndef mash_video_h
#define mash_video_h

#ifdef USE_DDRAW
#define  DIRECTDRAW_VERSION 0x0300 // run against DX3.0, since that's the minimum we need
#include <windows.h>  // depended upon by certain windows header files
#include <multimon.h> // for multiple monitor support
#include <ddraw.h>    // defines directdraw functions and objects
#include <uuids.h>    // provides uuid for the DirectDrawFactory
#include <ddrawex.h>  // functions/declarations for allocating DX3 interfaces
#endif

#include "config.h"
#include "tclcl.h"

extern "C" {
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#ifdef USE_SHM
#include <X11/extensions/XShm.h>
#endif
#include <tk.h>
}

class VideoWindow;
class RGB_Converter;

/*
 * An image to be displayed in a VideoWindow.
 */
class VideoImage {
    protected:
	VideoImage(Tk_Window, int width, int height);
    public:
	virtual ~VideoImage();
	inline int width() const { return (width_); }
	inline int height() const { return (height_); }
	virtual void putimage(Display* dpy, Window window, GC gc,
			      int sx, int sy, int x, int y,
			      int w, int h) const = 0;
    protected:
	int bpp_;		/* bits per pixel (FIXME must be 1,8,16,32) */
	int width_;
	int height_;
	Display* dpy_;/*FIXME*/
};

class StandardVideoImage : public VideoImage {
    protected:
	StandardVideoImage(Tk_Window, int width, int height);
	static int use_shm_;
	static int noXShm(ClientData, XErrorEvent*);
    public:
	virtual ~StandardVideoImage();
	static StandardVideoImage* allocate(Tk_Window, int width, int height);
	inline u_char* pixbuf() { return ((u_char*)image_->data); }
	inline XImage* ximage() { return (image_); }
    protected:
	XImage* image_;
};

#ifdef USE_DDRAW

//#define NOCLIPPER // for debugging, use to ignore overlapping of windows/widgets

#define MAX_SYSTEM_MONITORS 10 // max # of supported monitors attached to a single machine
#define MAX_WIDGET_MONITORS 4  // max # of monitors that a widget can be in simultaneously

struct deviceListType {  // information filled by addmonitor(); [0] may get set by setPrimaryMonitor
    GUID deviceGUID; // GUID of monitor
    bool haveGUID;	 // reports if GUID present (rather than NULL)

    HMONITOR driverMonitor; // information about monitor

    MONITORINFO driverMonitorInfo; // information about monitor
    bool haveMonitorInfo; // reports if driver monitor information present

    LPDIRECTDRAW3 g_pDD; // DirectDraw object for managing device
    LPDIRECTDRAWSURFACE3 lpDDSPrimarySurface; // primary surface for the monitor
    // images from 0 or more widgets are blt'ed (i.e., copied) to this surface
#ifndef NOCLIPPER
    LPDIRECTDRAWCLIPPER lpDDCPrimarySurfaceClipper; // clipper for the primary surface
#endif
    DDCAPS hardwareCaps; // capabilities of the monitor
    DDCAPS emulatedCaps; // emulated capabilities of the monitor
};

class DDrawVideoImage : public StandardVideoImage {
    public:
	DDrawVideoImage(Tk_Window, int width, int height); // constructor
	virtual ~DDrawVideoImage(); // destructor

	// function that places image
	void putimage(Display* dpy, Window window, GC gc,
		      int sx, int sy, int x, int y,
		      int w, int h) const;

	// get all monitor coordinates in a space delimited list of
	// the form { left, top, right, bottom }. The first element
	// of the list is the virtual screen coordinates, so the
	// return value should always contain at least two elements.
	static void getTclMonitorList(char* monitor_info);

	// check if directDraw is usable and can handle size.  The directdraw
	// static state is initialized the first time that this function is called
	static bool DDrawOkay(int w, int h, HWND hWnd); 

	// function is called (via callback) to add a monitor to the system
	// multimon is true if callback is for DirectDrawEnumerateEx, false for DirectDrawEnumerate
	static bool addMonitor(GUID FAR *lpGUID, LPSTR lpDriverDescription, 
				LPSTR lpDriverName, HWND hw, HMONITOR hm, bool multimon);

	// used in unimonitor systems (via callback) to store information 
	// that was learned about the primary monitor
	static void setPrimaryMonitor(HMONITOR hMonitor);

	// called by event handler to indicate that the window has moved
	void signalWindowMoved() {updateMonitorCoverage=true;}

	// examine window coordinates to see where a window has moved to
	// i.e., on what monitors in the widget visible
	void examineMonitorCoverage(HWND);

    protected:
	// allocates a off-screen surface for a given monitor
      	LPDIRECTDRAWSURFACE3 allocateSurface(int deviceNum);

	int currentMonitors[MAX_WIDGET_MONITORS]; // tracks which monitors the widget is in
	// value in array is offset in 'deviceList' (-1 means unused)

	LPDIRECTDRAWSURFACE3 currentMonitorsSurface[MAX_WIDGET_MONITORS];
	  // surfaces that correspond to the monitors in currentMonitors

	bool updateMonitorCoverage; // tracks whether window has moved (update currentMonitors)

	Tk_Window toplevelTkWindow_; // for window move event handler
	Tk_Window tk_; // for window move event handler

    public:
	static bool ddrawInitAttempted; // tracks if attempt has been made to alloc/use ddraw
	static bool ddrawInitSuccessful; // marks if ddraw attempt was successful
	static int deviceListLength; // number of monitors in system

	// list that stores attributes for monitors
	static deviceListType deviceList[MAX_SYSTEM_MONITORS]; 
};
#endif

#ifdef USE_SHM
/*
 * A class for ximages, which will be allocate in shared memory
 * if available.  The constructor takes the width and height
 * of the bitmap, plus a size which must be >= width*height.
 * The size might want to be bigger than width*height for decoders
 * that might walk off the end of an image because of bit errors,
 * but the distance they walk off the end is bounded.
 */
class SharedVideoImage : public StandardVideoImage {
    public:
	SharedVideoImage(Tk_Window, int width, int height);
	SharedVideoImage(Tk_Window, int width, int height,
			 u_char* shmaddr, int shmid);
	virtual ~SharedVideoImage();
	inline int shmid() const { return (shminfo_.shmid); }
	inline char* buffer() const { return (shminfo_.shmaddr); }
	void putimage(Display* dpy, Window window, GC gc,
		      int sx, int sy, int x, int y,
		      int w, int h) const;
	inline int valid() const { return (shminfo_.shmid >= 0); }
    protected:
	void init(Tk_Window tk);
	XShmSegmentInfo	shminfo_;
};
#endif

class BareWindow : public TclObject {
    public:
	BareWindow(const char* name, XVisualInfo* vinfo = 0);
	~BareWindow();
	virtual int command(int argc, const char*const* argv);
	inline Tk_Window tkwin() { return (tk_); }
	inline int width() { return (width_); }
	inline int height() { return (height_); }
	virtual void setsize(int w, int h);

	inline void map() { Tk_MapWindow(tk_); }
	inline void unmap() { Tk_UnmapWindow(tk_); }
	void sync();
	inline void raise() { XRaiseWindow(dpy_, Tk_WindowId(tk_)); }

	virtual void redraw() {}
	virtual void destroy();
	virtual int destroyed() { return destroyed_; }
    protected:
	void draw(int miny, int maxy);

	Display* dpy_;
	Tk_Window tk_;
	int width_;
	int height_;
	int destroyed_;
    private:
	static void handle(ClientData, XEvent*);
};

inline void BareWindow::sync()
{
#ifndef WIN32
	XSync(Tk_Display(tk_), 0);
#endif
}

class VideoWindow : public BareWindow {
    public:
	VideoWindow(const char* name, XVisualInfo* vinfo = 0);
	~VideoWindow();
	virtual int command(int argc, const char*const* argv);
	void setimage(VideoImage* v) { vi_ = v; }
	void render(const VideoImage* vi, int miny = 0, int maxy = 0,
		    int minx = 0, int maxx = 0);
	inline void complete() { sync(); }	/* complete last render call */
	void redraw();
	inline void damage(int v) { damage_ = v; }
	inline int damage() const { return (damage_); }

	inline void voff(int v) { voff_ = v; }
	inline void hoff(int v) { hoff_ = v; }

	int bpp();

	/*
	 * Return the VideoWindow object associated with
	 * the tk window identified by name.  Name has the
	 * usual tk form, e.g., ".top.video.win0".
	 */
	static inline VideoWindow* lookup(const char* name) {
		return ((VideoWindow*)TclObject::lookup(name));
	}
    protected:
	void draw(int miny, int maxy, int minx, int maxx);
	void dim();
	void clear();

	static GC gc_;
	const VideoImage* vi_;
	int callback_pending_;
	int damage_;
	int voff_;
	int hoff_;
    private:
	static void display(ClientData);
	static void doclear(ClientData);
	static void dodim(ClientData);
};

class CaptureWindow : public BareWindow {
    public:
	CaptureWindow(const char* name, XVisualInfo*);
	virtual ~CaptureWindow();
	void capture(u_int8_t* frm);
	inline int basewidth() { return (base_width_); }
	inline int baseheight() { return (base_height_); }
	void setsize(int w, int h);
	inline void converter(RGB_Converter* v) { converter_ = v; }
    protected:
	void grab_image();
	GC gc_;
	int base_width_;
	int base_height_;
	StandardVideoImage* image_;
	RGB_Converter* converter_;
};

#endif
