/*
 * net.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/net/net.cc,v 1.25 2002/02/03 04:13:45 lim Exp $";

#include <stdlib.h>
#include <math.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#ifdef WIN32
#include <winsock.h>
#else
#include <sys/socket.h>
#include <sys/uio.h>
#endif
#include "net.h"
#include "crypt.h"
#include "pktbuf.h"

Network::Network() :
	addr_(0),
	local_(0),
	lport_(0),
	sport_(0),
	rport_(0),
	ttl_(0),
	rsock_(-1),
	ssock_(-1),
	filter_own_(1),
	crypt_(0)
{
}

Network::~Network()
{
}

int Network::command(int argc, const char*const* argv)
{
	if (argc == 2) {
		Tcl& tcl = Tcl::instance();
		char* cp = tcl.buffer();
		if (strcmp(argv[1], "addr") == 0 ||
		    strcmp(argv[1], "interface") == 0 ||
		    strcmp(argv[1], "port") == 0 ||
		    strcmp(argv[1], "sport") == 0 ||
		    strcmp(argv[1], "rport") == 0 ||
		    strcmp(argv[1], "ttl") == 0 ||
		    strcmp(argv[1], "ismulticast") == 0)
			strcpy(cp, "0");
		else if (strcmp(argv[1], "flush") == 0) {
			u_int32_t from;
			int from_port;
			while (dorecv(wrkbuf_, wrkbuflen_, rsock_, from,
				      from_port) > 0)
				;
		} else
			return (TclObject::command(argc, argv));
		tcl.result(cp);
		return (TCL_OK);
	} else if (argc == 3) {
		if (strcmp(argv[1], "crypt") == 0) {
			/*
			 * 'crypt ""' will turn of encryption because
			 * lookup() will return null.
			 */
			crypt_ = (Crypt*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
	}
	return (TclObject::command(argc, argv));
}

void Network::nonblock(int fd)
{
#ifdef WIN32
	u_long flag = 1;
	if (ioctlsocket(fd, FIONBIO, &flag) == -1) {
		fprintf(stderr, "ioctlsocket: FIONBIO: %lu\n", GetLastError());
		exit(1);
	}
#else
        int flags = fcntl(fd, F_GETFL, 0);
#if defined(hpux) || defined(__hpux)
        flags |= O_NONBLOCK;
#else
        flags |= O_NONBLOCK|O_NDELAY;
#endif
        if (fcntl(fd, F_SETFL, flags) == -1) {
                perror("fcntl: F_SETFL");
                exit(1);
        }
#endif
}

u_char* Network::wrkbuf_;
int Network::wrkbuflen_;

void Network::expand_wrkbuf(int len)
{
	if (wrkbuflen_ == 0)
		wrkbuf_ = (u_char*)malloc(len);
	else
		wrkbuf_ = (u_char*)realloc((u_char*)wrkbuf_, len);
	wrkbuflen_ = len;
}

void Network::dosend(u_char* buf, int len, int fd)
{
	int cc = ::send(fd, (char*)buf, len, 0);
	if (cc < 0) {
 		/*
 		 * Due to a bug in kern/uipc_socket.c on several
 		 * systems, datagram sockets incorrectly persist
 		 * in an error state on receipt of any ICMP
 		 * error.  This causes unicast connection
 		 * rendezvous problems, and worse, multicast
 		 * transmission problems because several systems
 		 * incorrectly send port unreachables for
 		 * multicast destinations.  Our work around
 		 * is to call getsockopt(..., SO_ERROR, ...)
 		 * which resets so->so_error.
 		 *
 		 * This bug originated at CSRG in Berkeley
 		 * and was present in the BSD Reno networking
 		 * code release.  It has since been fixed
 		 * in OSF-3.x.  It is know to remain
 		 * in 4.4BSD and AIX-4.1.3.
 		 *
 		 * A fix is to change the following lines from
 		 * kern/uipc_socket.c:
 		 *
 		 *	if (so_serror)
 		 *		snderr(so->so_error);
 		 *
 		 * to:
 		 *
 		 *	if (so->so_error) {
 		 * 		error = so->so_error;
 		 *		so->so_error = 0;
 		 *		splx(s);
 		 *		goto release;
 		 *	}
 		 *
 		 */
 		int err, savederrno;
		socklen_t errlen = sizeof(err);
 		savederrno = errno;
 		getsockopt(fd, SOL_SOCKET, SO_ERROR, (char *)&err, &errlen);
 		switch (savederrno) {
		case ECONNREFUSED:
			/* no one listening at some site - ignore */
			break;

		case ENETUNREACH:
		case EHOSTUNREACH:
			/*
			 * These "errors" are totally meaningless.
			 * There is some broken host sending
			 * icmp unreachables for multicast destinations.
			 * UDP probably aborted the send because of them --
			 * try exactly once more.  E.g., the send we
			 * just did cleared the errno for the previous
			 * icmp unreachable, so we should be able to
			 * send now.
			 */
			(void)::send(fd, (char*)buf, len, 0);
			break;

		default:
			perror("send");
			return;
		}
	}
}

void Network::send(u_char* buf, int len)
{
	if (crypt_)
		buf = crypt_->Encrypt(buf, len);
	dosend(buf, len, ssock_);
}

void Network::send(const pktbuf* pb)
{
	/*FIXME*/
	send(pb->dp, pb->len);
}

int Network::dorecv(u_char* buf, int len, int fd, u_int32_t& from,
		    int &fromport)
{
	sockaddr_in sfrom;
	socklen_t fromlen = sizeof(sfrom);
	int cc = ::recvfrom(fd, (char*)buf, len, 0,
			    (sockaddr*)&sfrom, &fromlen);
	if (cc < 0) {
		if (errno != EWOULDBLOCK && errno != ECONNREFUSED)
			perror("recvfrom");
		return (-1);
	}
	from = sfrom.sin_addr.s_addr;
	fromport = ntohs(sfrom.sin_port);
	if (filter_own_ && from == local_ && fromport == lport_)
		return (0);

	return (cc);
}

int Network::recv(u_char* buf, int len, u_int32_t& from, int &fromport)
{
	if (crypt_) {
		if (len+8 > wrkbuflen_)
			expand_wrkbuf(len+8);
		int cc = dorecv(wrkbuf_, len, rsock_, from, fromport);
                if (cc) {
                    return (crypt_->Decrypt(wrkbuf_, cc, buf));
                } else {
                    return 0;
                }
	}
	return (dorecv(buf, len, rsock_, from, fromport));
}


/*int
Network::recv(u_char* buf, int len, u_int32_t& from)
{
	int fromport;
	return recv(buf, len, from, fromport);
}*/


void Network::reset()
{
}

CtrlHandler::CtrlHandler()
	: ctrl_inv_bw_(0.),
	  ctrl_avg_size_(128.)
{
}

inline void CtrlHandler::schedule_timer()
{
	msched(int(fmod(double(random()), rint_) + rint_ * .5 + .5));
}

void CtrlHandler::net(Network* n)
{
	DataHandler::net(n);
	cancel();
	if (n != 0) {
		/*
		 * schedule a timer for our first report using half the
		 * min ctrl interval.  This gives us some time before
		 * our first report to learn about other sources so our
		 * next report interval will account for them.  The avg
		 * ctrl size was initialized to 128 bytes which is
		 * conservative (it assumes everyone else is generating
		 * SRs instead of RRs).
		 */
		double rint = ctrl_avg_size_ * ctrl_inv_bw_;
		if (rint < CTRL_MIN_RPT_TIME / 2. * 1000.)
			rint = CTRL_MIN_RPT_TIME / 2. * 1000.;
		rint_ = rint;
		schedule_timer();
	}
}

void CtrlHandler::sample_size(int cc)
{
	ctrl_avg_size_ += CTRL_SIZE_GAIN * (double(cc + 28) - ctrl_avg_size_);
}

void CtrlHandler::adapt(int nsrc, int nrr, int we_sent)
{
	/*
	 * compute the time to the next report.  we do this here
	 * because we need to know if there were any active sources
	 * during the last report period (nrr above) & if we were
	 * a source.  The bandwidth limit for ctrl traffic was set
	 * on startup from the session bandwidth.  It is the inverse
	 * of bandwidth (ie., ms/byte) to avoid a divide below.
	 */
	double ibw = ctrl_inv_bw_;
	if (nrr) {
		/* there were active sources */
		if (we_sent) {
			ibw *= 1./CTRL_SENDER_BW_FRACTION;
			nsrc = nrr;
		} else {
			ibw *= 1./CTRL_RECEIVER_BW_FRACTION;
			nsrc -= nrr;
		}
	}
	double rint = ctrl_avg_size_ * double(nsrc) * ibw;
	if (rint < CTRL_MIN_RPT_TIME * 1000.)
		rint = CTRL_MIN_RPT_TIME * 1000.;
	rint_ = rint;
}

void CtrlHandler::timeout()
{
	sm_->announce(this);
	schedule_timer();
}

void DataHandler::dispatch(int)
{
	sm_->recv(this);
}

void CtrlHandler::dispatch(int)
{
	sm_->recv(this);
}
