/*
 * decoder-jpeg.h --
 *
 *      Motion Jpeg decoder header file
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/codec/decoder-jpeg.h,v 1.13 2002/02/03 03:13:33 lim Exp $ */

#include "jpeg/jpeg.h"


class JpegReassembler {
public:
    JpegReassembler();
    ~JpegReassembler();

    u_char* reassemble(const rtphdr* rh, const u_char* bp, int& len);
#ifdef ERROR_RECOVERY
    u_char* recoverReassem(const rtphdr* rh, const u_char* bp, int& len,
			   u_int32_t restartInterval, u_int32_t restartCount);
#endif

    inline int badoff() const { return (badoff_); }
    inline int hugefrm() const { return (hugefrm_); }

protected:
    int decimate_;
    int ndec_;

    /*
     * slots_ is a table of recently received packets
     * used in reassemble() to determine when we have
     * received a complete frame.
     */
#define MAX_JPEG_SLOTS 1024
    int nslots_;
    int slotmask_;
    struct slot {
	int seqno;
	int eof;        /* nbytes in last pkt of frame, of
			 * 0 if not last pkt of frame */
	u_int32_t off;
	u_int32_t ts;
    }* slots_;

    /*
     * Reassembly buffers.  We do double-buffering, which allows
     * packets to arrive out of order across frame boundaries,
     * but not across an entire frame (i.e., we don't want to see
     * any packets from frame k+2 until we're done with frame k).
     * We use RTP timestamps to keep track of which frame is
     * allocated to which buffer.  (RTP guarantees that the
     * timestamps are constant across a frame, and increase
     * between succesive frames.  FIXME is latter really true?)
     */
    struct rbuf {
	int drop;
	u_int32_t ts;
	u_char* bp;
	u_char* current;
	u_int32_t restartInterval; // Size of the restart marker intervals
	u_int32_t restartCount;    // Number of restart markers seen before
	// this packet of the frame
	int len;
    };

#ifdef ERROR_RECOVERY
    bool advancePrevPtr(rbuf* curr, rbuf * prev);
    bool recoverFromPrev(rbuf * curr, rbuf * prev, int recoverto);
#endif
	
    rbuf rb0_;
    rbuf rb1_;
    int rbsize_;
    int hugefrm_;
    int badoff_;

    /* The 2 pointers for recovery purpose.  The current_ pointer is the
     * location of the current data input.  The prev_ pointer is the
     * location of the current data input of the previous frame
     */
    rbuf *current_;
    rbuf *prev_;
    bool doneodd;  // See if a complete odd frame is done
};


class MotionJpegDecoder : public Decoder {
public:
    MotionJpegDecoder();
    virtual ~MotionJpegDecoder();
    int colorhist(u_int* hist) const;
    virtual void stats(char* wrk);

    virtual int command(int argc, const char*const* argv);
    
protected:
    virtual void recv(pktbuf*);
    virtual void redraw();

    int drop_even_;	// only used with fields, if non-zero, then
    			// drop even fields and line double odd fields
    int badtype_;

    // input quantization
    int inq_;

    // JPEG/RTP parameters type code
    int type_;

    // Length of jpeg restart interval
    u_int32_t restart_interval_;

    // Number of jpeg restart marker has past
    u_int32_t restart_count_;

    void configure();
    JpegPixelDecoder* codec_;
    JpegDecoder::config config_;
    JpegReassembler reasm_;
};
