/*
 * mb-play.cc --
 *
 *      MediaBoard Archive Playback
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @(#) $Header: /usr/mash/src/repository/mash/mash-1/archive/mb-play.cc,v 1.22 2002/02/03 03:09:26 lim Exp $
 */


#include "archive/mb-play.h"
#include "mb/mb.h"
#include "mb/mb-nethost.h"
#include "mb/mb-cmd.h"


DEFINE_OTCL_CLASS(MBPlaybackMgr, "MB_Manager/Play") {
	INSTPROC(create_srm_source);
}


DEFINE_OTCL_CLASS(MBPlaybackStream, "ArchiveStream/Play/Mediaboard") {
}


int
MBPlaybackMgr::create_srm_source(int argc, const char * const *argv)
{
	TclObject *streamObj;
	const char *uid, *addr, *userName;

	BEGIN_PARSE_ARGS(argc, argv);
	ARG(streamObj); // TclObject *
	ARG(uid);       // char *
	ARG(addr);      // char *
	ARG(userName);  // char *
	END_PARSE_ARGS;

	if (sources_==NULL) {
		sources_ = session()->source_manager()->local_sources();
	}
	// reset the indices
	nextADUIdx_ = fillSAIdx_ = NULL;


	MBPlaybackStream *stream = (MBPlaybackStream *) streamObj;
	Tcl &tcl = Tcl::instance();
	SRM_Source *source;
	MBPlaybackRcvr *rcvr;

	// create the SRM source for this stream
	if (session()->Invoke("create-local", uid, addr, userName,
				     NULL)==TCL_ERROR) {
		tcl.resultf("error creating SRM source for new stream: %s",
			    tcl.result());
		goto error;
	}

	// attach the stream and SRM rcvr objects together;
	source = (SRM_Source*)tcl.lookup(tcl.result());
	if (source!=NULL && (rcvr = (MBPlaybackRcvr*)source->handler())!=NULL){
		stream->attach(rcvr);
		rcvr->attach(stream);
	}
	else {
		tcl.resultf("invalid SRM source or packet handler object");
		goto error;
	}

	return TCL_OK;

error:
	tcl.add_errorf("\ninvoked from within MBPlaybackMgr::"
		       "NewStream()");
	return TCL_ERROR;
}


int
MBPlaybackMgr::next_ADU(u_char *pb, int len, srm_src &id, int &pkt_type,
			int &next)
{
	next=0;
	pkt_type = APP_DATA;
	// TODO: should poll all receivers or preq as well

	if (sources_==NULL) {
		MTrace(trcArchive, ("Haven't created any sources"));
		return 0;
	}

	if (nextADUIdx_==NULL) {
		nextADUIdx_ = sources_->getFirst();
	}
	else {
		nextADUIdx_ = sources_->getNext(nextADUIdx_);
		if (nextADUIdx_==NULL) nextADUIdx_ = sources_->getFirst();
	}

	int outlen=0;
	MBPlaybackRcvr *rcvr;
	while ((outlen==0 && next==0)
	       && sources_->IsDone(nextADUIdx_)==FALSE) {
		rcvr = (MBPlaybackRcvr*) (sources_->getData(nextADUIdx_)->
					  handler());
		outlen = rcvr->NextADU(pb, len, next, NULL);
		if (outlen==0 && next==0) {
			nextADUIdx_ = sources_->getNext(nextADUIdx_);
		} else {
			// set the id of this guy
			id = rcvr->getSrcId();
		}
	}
	return outlen;
}



int
MBPlaybackMgr::periodic_update(Byte *pb)
{
	if (sources_==NULL) {
		MTrace(trcArchive, ("Haven't created any sources"));
		return 0;
	}

	if (fillSAIdx_==NULL) {
		fillSAIdx_ = sources_->getFirst();
	}
	else {
		fillSAIdx_ = sources_->getNext(fillSAIdx_);
		if (fillSAIdx_==NULL) fillSAIdx_ = sources_->getFirst();
	}

	int outlen=0;
	MBPlaybackRcvr *rcvr;
	while (outlen==0 && sources_->IsDone(fillSAIdx_)==FALSE) {
		rcvr = (MBPlaybackRcvr*) (sources_->getData(fillSAIdx_)->
					  handler());
		outlen = rcvr->FillSA(pb, getMTU());
		if (outlen==0) {
			fillSAIdx_ = sources_->getNext(fillSAIdx_);
		}
	}

	if (outlen==0) {
		// I don't have anything to send; I should reset the SA timer
		// FIXME: ignoring the return code from this method
		session()->sa_timer()->Invoke("reset", NULL);
	}
	return outlen;
}


int
MBPlaybackRcvr::handleCmd(MBCmd * /*pCmd*/, MBPageObject * /*pPage*/,
			  const MBTime& /*oldTime*/, const MBTime& /*newTime*/)
{
	// ignore any packets we get from the network
	return TRUE;
}


Bool
MBPlaybackRcvr::Dispatch(MBCmd *pCmd, MBPageObject *pPage)
{
	char foo1[100], foo2[100];
	strcpy(foo1, (char*)pPage->getId().sid);
	strcpy(foo2, (char*)getSrcId());
	MTrace(trcArchive|trcExcessive, ("##^^ Page %s:%d (%p) for receiver "
					 "(%s, %p) has maxSeqno %d "
					 "before Dispatch",
					 foo1, pPage->getId().uid, pPage,
					 foo2, this,
					 pPage->getMaxSeqno()));
	/* Note: right now handlecmd just calls a dummy executecmd
	 * that does nothing, so the time paramaters does not matter,
	 * we should change it to some sensible value if that is not
	 * the case */
	if (MBBaseRcvr::handleCmd(pCmd, pPage, cMBTimeAny, cMBTimeAny)
	    != MB_EXE_OK)
		return FALSE;
	MTrace(trcArchive|trcExcessive, ("##^^ Page %s:%d (%p) for receiver "
					 "(%s, %p) has maxSeqno %d "
					 "before Dispatch for command %d",
					 foo1, pPage->getId().uid, pPage,
					 foo2, this, pPage->getMaxSeqno(),
					 pCmd->getSeqno()));
	int dataSize = pPage->RequestSendAmount()+sizeof(Pkt_DataHdr);
	getMgr()->RequestSend(dataSize);

	// notify all observers
	bytesSent_ += dataSize;
	stream_->Invokef("notify_observers bytes_sent %lu",
			 (unsigned long) bytesSent_);
	return TRUE;
}



void
MBPlaybackStream::Clip(timeval /*start*/, timeval end)
{
	endTS_ = logical2mb(end.tv_sec, end.tv_usec);
}


void
MBPlaybackStream::LTS_Speed()
{
	PlaybackStream::LTS_Speed();
}


void
MBPlaybackStream::LTS_Reference()
{
	PlaybackStream::LTS_Reference();
}


int
MBPlaybackStream::ReadRecord()
{
	u_int16_t len;
	DataFile *file;
	PageId pageId;
	u_int32_t sn;
	Tcl &tcl = Tcl::instance();

	file = DataFile_();
	if (firstTime_==TRUE) {
		if (file->Seek(file->getHeaderSize(), SEEK_SET)==TCL_ERROR)
			goto error;
		firstTime_ = FALSE;
	}

	state.bufferLen = sizeof(Pkt_PageHdr);
	if (state.buffer.alloc(state.bufferLen)==FALSE) {
		tcl.resultf("buffer allocation error");
		goto error;
	}

	if (file->Read(state.buffer.pb, sizeof(Pkt_PageHdr)) <= 0) {
		tcl.resultf("cannot read data file: %s",
			    strerror(Tcl_GetErrno()));
		goto error;
	}

	Pkt_PageHdr *pHdr;
	pHdr = (Pkt_PageHdr*) (state.buffer.pb);
	net2host(pHdr->pd_page, pageId);

	// modify the srcId field in the PageId so that it doesn't interfere
	// with a normal receiver
	pHdr->pd_page.sid = rcvr_->getSrcId();

	state.sseq   = net2host(pHdr->pd_sseq);
	state.eseq   = net2host(pHdr->pd_eseq);
	MTrace(trcArchive, ("Reading record with seqnos %d to %d (%d)",
		state.sseq, state.eseq, MYNUM(this)));

	for (sn=state.sseq; sn <= state.eseq; sn++) {
		if (state.buffer.alloc(state.bufferLen +
				       sizeof(Pkt_CmdHdr))==FALSE) {
			tcl.resultf("cannot allocate memory");
			goto error;
		}
		if (file->Read(state.buffer.pb + state.bufferLen,
			       sizeof(Pkt_CmdHdr)) < 0) {
			tcl.resultf("cannot read data file: %s",
			    strerror(Tcl_GetErrno()));
			goto error;
		}

		len = net2host(((Pkt_CmdHdr*)(state.buffer.pb +
					      state.bufferLen))->dh_len);
		if (state.buffer.alloc(state.bufferLen + len)==FALSE) {
			tcl.resultf("cannot allocate memory");
			goto error;
		}
		if (file->Read((state.buffer.pb + state.bufferLen +
				sizeof(Pkt_CmdHdr)),
			       len - sizeof(Pkt_CmdHdr)) < 0) {
			tcl.resultf("cannot read data file: %s",
			    strerror(Tcl_GetErrno()));
			goto error;
		}
		state.bufferLen += len;
	}

	state.pPage = rcvr_->DefinePage(pageId);
	state.seqno = state.sseq;
	state.pb    = state.buffer.pb + sizeof(Pkt_PageHdr);
	MTrace(trcArchive, ("Successfully read the data (%d)", MYNUM(this)));
	return TCL_OK;

error:
	tcl.add_errorf("\ninvoked from within MBPlaybackStream::ReadRecord()");
	return TCL_ERROR;
}


int
MBPlaybackStream::NextEvent(timeval &logical)
{
	timeval now;
	/*, diff_tv;
	u_int32_t nowTS;*/
	Pkt_CmdHdr *pHdr;
	u_int32_t timestamp;

	if (state.seqno > state.eseq) {
		// we are done with the previous record; fetch the next one!
		if (DataFile_()->Eof()==TRUE) {
			logical.tv_sec = logical.tv_usec = 0;
			return TCL_OK;
		}

		if (ReadRecord()==TCL_ERROR) {
			if (DataFile_()->Eof()==TRUE) {
				logical.tv_sec = logical.tv_usec = 0;
				return TCL_OK;
			}
			else goto error;
		}
	}

	pHdr = (Pkt_CmdHdr*) state.pb;
	timestamp = net2host(pHdr->dh_ts);
	if (endTS_ > 0 && timestamp > endTS_) {
		// we are done
		MTrace(trcArchive, ("Exceeded ending ts (%lu): %lu (%d)",
				    (unsigned long) endTS_,
				    (unsigned long) timestamp, MYNUM(this)));
		logical.tv_sec = logical.tv_usec = 0;
		return TCL_OK;
	}

	now = LTS_()->NowLogical();
	logical = mb2logical(timestamp, now);
	return TCL_OK;

error:
	Tcl::instance().add_errorf("\ninvoked from within "
				   "MBPlaybackStream::NextEvent()");
	return TCL_ERROR;
}


void
MBPlaybackStream::DoEvent()
{
	MBCmd* pCmd=MBCmd::Create(state.seqno, state.pb);
	state.seqno++;
	state.pb += net2host(((Pkt_CmdHdr*)state.pb)->dh_len);

	MTrace(trcArchive, ("Doing an event for page %ld, cmd %u, type %u "
			    "(%d)", state.pPage->getId().uid, state.seqno-1,
			    pCmd->getType(), MYNUM(this)));

	if (!pCmd) {
		// should treat as an error packet and ask for resend?
		SignalError(("Received corrupted packet: corrupted command"));
		return;
	}

	// modify the packet of this command to the current system time!
	timeval now = LTS_()->NowSystem();
	pCmd->SetTimeStamp(tod2mb(now));

	if (!rcvr_->Dispatch(pCmd, state.pPage)) {
		SignalError(("Received corrupted or invalid command"));
		return;
	}
}
