<?php
/* 	OpenDb - Open Media Lending Database
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

include_once("./functions/item_attribute.php");
include_once("./functions/listutils.php");

/**
	Assumes $c is single character
*/
function is_alpha($c)
{
	if( ($c >= '0' && $c <= '9') || ($c >= 'A' && $c <= 'Z') || ($c >= 'a' && $c <= 'z'))
		return TRUE;
	else
		return FALSE;
}

/*
* The escape sequences \r \n \t, must be enclosed in double quotes, otherwise PHP does
* not recognise them.
*/
function is_nonword_char($c)
{
	if ($c == '(' || $c == ')' || $c == '[' || $c == ']' || $c == '{' || $c == '}' || $c == ':' || $c == '.' || $c == '-' || $c == ' ' || $c == "\t" || $c == "\n" || $c == "\r")
		return TRUE;
	else
		return FALSE;
}

/**
	This function could be improved, but for now it should suffice!
*/
function is_roman_numeral($text)
{
	// Roman numerals from 1 to 20!
	$numerals = array('I', 'II', 'III', 'IV', 'V', 'VI', 'VII', 'VIII', 'IX', 'X', 'XI', 'XII', 'XIII', 'XIV', 'XV', 'XVI', 'XVII', 'XVIII', 'XIX', 'XX');
	if(in_array($text, $numerals))
		return TRUE;
	else
		return FALSE;
}

/**
	Small function to process word.
*/
function ucword($word)
{
	if(is_roman_numeral($word))
		return $word;//as is!
	else
		return ucfirst(strtolower($word));
}

/**
	This function will split a sentence into separate words.

	Sentences will be split on normal word boundaries, such as
	spaces,tabs,newlines, but also when a bracket (,{,[,],},) are
	encountered.  All space will be preserved however and restored
	to the final string.

	This function will also recognise Roman numerals from I to XX (1-20) 
	and if they are already UPPERCASE, will maintain case.

	This function differs from ucwords, in that it will set all 
	other characters in a word to lowercase!
*/
function initcap($text)
{
	$word='';
	$rtext='';
	for ($i=0; $i<strlen($text); $i++)
	{
		// We might be in the middle of a word.
		if(strlen($word)>0 || is_alpha($text[$i]))
		{
			// test for end of word.
			if(strlen($word)>0 && is_nonword_char($text[$i]))
			{
				$rtext .= ucword($word);
				$word='';

				$rtext .= $text[$i];
			}
			else
				$word .= $text[$i];
		}
		else//just copy it to return array.
			$rtext .= $text[$i];	
	}

	// Do final word.
	if(strlen($word)>0)
	{
		$rtext .= ucword($word);
		$word='';	
	}
	return $rtext;
}

/**
	Assumes $c is single character
*/
function is_legal_code_char($c)
{
	if( ($c >= '0' && $c <= '9') || ($c >= 'A' && $c <= 'Z') || ($c >= 'a' && $c <= 'z') || $c == '_' || $c == '-' || $c == '.')
		return TRUE;
	else
		return FALSE;
}

/**
	Ensure legal function name
*/
function is_legal_function_name($word)
{
	if(strlen($word)>0)
	{
		for($i=0; $i<strlen($word); $i++)
		{
			if(!is_legal_code_char($word[$i]))
				return FALSE;
		}
		return TRUE;
	}
	else
		return FALSE;
}

/**
	Get the function type of a specified 
	function_spec.  No other processing
	is required.
*/
function get_function_type($function_spec)
{
	$start = strpos($function_spec, '(');
    if($start !== FALSE)
    	$type = trim(substr($function_spec, 0, $start));
	else
		$type = trim($function_spec);	

	return $type;
}

/*
* Assumes the $args_r is compatible with the 'args' array
* returned from prc_function_spec(...)
* 
	Will reconstitute the function arguments from
	the array of functions.

	If a argument has any ',' in it we assume it
	needs to double quoted.

	Any double quotes in the argument, and they
	need to be escaped.
* 
* @param $function_name - If null, then will reconstitute
* 			the arguments (excluding the brackets) only.
* @param $arguments_r - The arguments.  If empty, will return
* 		the $function_name.
*/
function get_rebuilt_function($function_name, $arguments_r)
{
	if(!is_array($arguments_r))
	{
		if($function_name !== NULL)
		{
			if(strlen($arguments_r)==0)
				return $function_name;
			else
				return $function_name.'('.$arguments_r.')';
		}
		else 
			return $arguments_r;
	}
	else
	{
		$argsText = '';

		@reset($arguments_r);
		while(list(,$argument) = @each($arguments_r))
		{
			// If argument includes a comma, or is whitespace (For a delimiter for example),
			// it must be enclosed in quotes.
			if( (strlen($argument)>0 && strlen(trim($argument))==0) || strpos($argument, ',') !== FALSE)
				$argument = "\"".str_replace("\"", "\\\"", $argument)."\"";
			else if(strpos($argument, '"')!==FALSE)
				$argument = str_replace("\"", "\\\"", $argument);
				
			if(strlen($argsText)==0)
				$argsText .= $argument;
			else
				$argsText .= ', '.$argument;
		}

		if($function_name !== NULL)
		{
			if(strlen($argsText)==0)
				return $function_name;
			else
				return $function_name.'('.$argsText.')';
		}
		else 
			return $argsText;
	}
}

/**
	Will return an array of the following form:
		type=>function_name, args=>array(args)
*/
function prc_function_spec($function_spec, $require_legal_func_name=FALSE) 
{
    $start = strpos($function_spec, '(');
    if($start !== FALSE)
    {
    	// Now we have something to parse.
        $type = trim(substr($function_spec, 0, $start));

		// Now ensure the function name is valid.
		if($require_legal_func_name==FALSE || is_legal_function_name($type))
		{
	        $end = strrpos($function_spec, ')');
			if($end>$start)//Otherwise a fuckup...
        	{
        		// Now we have the args, lets tokenise them.
	        	$args = trim(substr($function_spec, $start+1, $end-($start+1)));
				$arr = prc_args($args);
        	}
	
			return array(type=>strtolower($type), args=>$arr);
		}
		else
			return NULL;
	}
    else if($require_legal_func_name==FALSE)
	{ 
		// No (), so arg[0] is the whole thing.
    	$type = trim($function_spec);

		// Empty argument list.
		return array(type=>strtolower($type), args=>array());
	}
	else
		return NULL;
}

function prc_args($args)
{
	$argument='';
	$quote=NULL;
	
	// Allows us to keep track of nested braces.
	$curly_brace=0;
	$round_brace=0;
	$square_brace=0;
	$dbl_quote=FALSE;
	$sgl_quote=FALSE;
	
	for($i=0; $i<strlen($args); $i++)
	{
		switch($args[$i])
		{
			case '"':
				if($sgl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0 || ($i>0 && $args[$i-1]=="\\"))
					$argument .= $args[$i];
				else
					$dbl_quote = !$dbl_quote;
				break;
				
			case '\'':
				if($dbl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0 || ($i>0 && $args[$i-1]=="\\"))
					$argument .= $args[$i];
				else
					$sgl_quote = !$sgl_quote;
				break;
				
			case '\\':
				// If in braces, always include escape character, so it will be seen by the recursive calls to pcr_args.
				if($curly_brace>0 || $round_brace>0 || $square_brace>0)
				{
					$argument .= $args[$i];
				}
				else if($i>0 && $args[$i-1]=="\\")// As previous argument was an escape character, we should put this one in!
					$argument .= $args[$i];
				// else ignore
				break;
				
			case '{':
				// Do not recognise nested braces if inside quotes.
				if(!$dbl_quote && !$sgl_quote)
				{
					$curly_brace++;
				}	
				$argument .= $args[$i];
				break;
				
			case '}':
				// Do not recognise nested braces if inside quotes.
				if(!$dbl_quote && !$sgl_quote && $curly_brace>0)
				{
					$curly_brace--;
				}	
				$argument .= $args[$i];
				break;
				
			case '[':
				if(!$dbl_quote && !$sgl_quote)
				{
					$square_brace++;
				}
				$argument .= $args[$i];
				break;
				
			case ']':
				if(!$dbl_quote && !$sgl_quote && $square_brace>0)
				{
					$square_brace--;
				}
				$argument .= $args[$i];
				break;
				
			case '(':
				if(!$dbl_quote && !$sgl_quote)
				{
					$round_brace++;
				}
				$argument .= $args[$i];
				break;
				
			case ')':
				if(!$dbl_quote && !$sgl_quote && $round_brace>0)
				{
					$round_brace--;
				}
				$argument .= $args[$i];
				break;
				
			case ',':
				if($i>0 && $args[$i-1]=="\\")
				{
					// Get rid of escape character.
					$argument .= $args[$i];
				}
				else if($dbl_quote || $sgl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0)
				{
					// Inside nested block, so ignore argument separator.
					$argument .= $args[$i];
				}
				else
				{
					$arguments[] = $argument;
					$argument='';
				}
				break;
			
			case " ":
			case "\t":
			case "\n":
			case "\r":
				if($i>0 && $args[$i-1]=="\\")
				{
					// Get rid of escape character.
					$argument[strlen($argument)-1] = $args[$i];
				}
				else if(strlen($argument)>0)
				{
					// If already encountered non-whitespace for this argument, we need to keep it.
					$argument .= $args[$i];
				}
				else if($dbl_quote || $sgl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0)
				{
					// Inside nested block
					$argument .= $args[$i];
				}
				
				break;
				
			default:
				$argument .= $args[$i];
		}
	}	
	
	if(strlen($argument)>0)
		$arguments[] = $argument;
	return $arguments;
}

/**
* Stub version to avoid the pass-by-reference of the $mask_configuration, which
* will cause problems if used more than once in the page, where the $parsed_title_mask_r
* is not being cached as well. (item_add,item_input,item_display,etc)
*/
function expand_item_title_mask($item_instance_r, $mask_configuration)
{
	if(strlen($mask_configuration)>0)
	{
		return expand_type_title_mask($item_instance_r, $mask_configuration, $parsed_title_mask_r);
	}
	else
	{
		return $item_instance_r['title'];
	}
}

/**
	Based on the $mask_configuration array or straight mask value, will return
	a parsed title mask for the $s_item_type.
	
	The steps:
		Check for a $mask_configuration[$s_item_type]
		Check for a $mask_configuration[$s_item_type_group]
		Check for a $mask_configuration['DEFAULT']
		Otherwise use $mask_configuration

	Will update (via pass-by-reference) the $mask_configuration	& $parsed_title_mask_r variables,
	so that they only have to be parsed once!
*/
function expand_type_title_mask($item_instance_r, &$mask_configuration, &$parsed_title_mask_r)
{
	// Display masked title if configured
	if(($config_key = get_item_type_config_key(NULL, $item_instance_r['s_item_type'], $mask_configuration)) != NULL)
	{
		if(strlen($mask_configuration[$config_key])>0)
		{
			if(is_empty_array($parsed_title_mask_r[$config_key]))
			{
				$parsed_title_mask_r[$config_key] = parse_field_mask($mask_configuration[$config_key]);
			}
			return expand_title_mask($item_instance_r, $mask_configuration[$config_key], $parsed_title_mask_r[$config_key]);
		}
		else
		{
			return $item_instance_r['title'];
		}
	}
	else if(strlen($mask_configuration)>0) // Global definition.
	{
		if(is_empty_array($parsed_title_mask_r))
		{
			$parsed_title_mask_r = parse_field_mask($mask_configuration);
		}
		return expand_title_mask($item_instance_r, $mask_configuration, $parsed_title_mask_r);
	}
	else
	{
		return $item_instance_r['title'];
	}
}

/**
	Return a database value for $item_id based on whether $mask_variable
    points to title or to an item_attribute.  For item_attribute references
	there are some further options.

	A legal mask item can include the following:
		{s_attribute_type}
		{s_attribute_type.img}
		{s_attribute_type.value}  // The .value will result in the same as if no .option was specified.
		{s_attribute_type.display}
		{title}
		{s_item_type}
		{item_id}
		{instance_no}
		{s_status_type}
*/
function get_mask_variable_value($mask_variable, $item_instance_r)
{
	global $LANG_VARS;

	$value = '';

	if(is_array($mask_variable) && isset($mask_variable['s_attribute_type']) && isset($mask_variable['option']))
	{
		$value = NULL;
		
		// the options that require a item_id context
		if(is_numeric($item_instance_r['item_id']))
		{
			if($mask_variable['option'] == 'img')
			{
				$lookup_attr_r = fetch_sattr_type_lookup_for_item_attr($item_instance_r['item_id'], $mask_variable['s_attribute_type']);
				if($lookup_attr_r!==FALSE)
				{
					if(strlen($lookup_attr_r['img'])>0 && $lookup_attr_r['img'] != 'none')
						$value = _theme_image($lookup_attr_r['img'], NULL, $lookup_attr_r['display'], 'absmiddle');
					else // If no image, then use (value).
						$value = '('.$lookup_attr_r['value'].')';						
				}
			}
			else if($mask_variable['option'] == 'display' && is_numeric($item_instance_r['item_id']))
			{
				$value = fetch_sattr_type_lookup_for_item_attr($item_instance_r['item_id'], $mask_variable['s_attribute_type'], 'display');
			}
					
			// as a last resort.
			if(strlen($value) == 0)
			{
				$value = fetch_attribute_val($item_instance_r['item_id'], $mask_variable['s_attribute_type']);
			}
		}
		else if(strlen($item_instance_r['s_item_type'])>0)// s_item_type context items.
		{
			if($mask_variable['option'] == 'prompt') // The s_attribute_type prompt
			{
				$value = fetch_s_item_type_attr_prompt($item_instance_r['s_item_type'], $mask_variable['s_attribute_type']);
			}
		}
		return $value;
	}
	// Special variable references.
	else if(!is_array($mask_variable) && $mask_variable == 'title')
		return $item_instance_r['title'];
	else if(!is_array($mask_variable) && $mask_variable == 's_item_type')
		return $item_instance_r['s_item_type'];
	else if(!is_array($mask_variable) && $mask_variable == 'item_id')
		return $item_instance_r['item_id'];
	else if(!is_array($mask_variable) && $mask_variable == 'instance_no')
		return $item_instance_r['instance_no'];
	else if(!is_array($mask_variable) && $mask_variable == 's_status_type')
		return $item_instance_r['s_status_type'];
	else // plain text
	{
		return $mask_variable;
	}
}

/**
*/
function get_title_mask_macro_element_r($macro_type)
{
	global $_OPENDB_TITLE_MASK_MACRO_RS;
	
	if(is_array($_OPENDB_TITLE_MASK_MACRO_RS) && isset($_OPENDB_TITLE_MASK_MACRO_RS[$macro_type]))
		return $_OPENDB_TITLE_MASK_MACRO_RS[$macro_type];
	else
		return NULL;
}

/**
	Modify display_mask so that all {variable} are replaced with
	references to the array indexes returned.
*/
function parse_field_mask(&$display_mask)
{
	$i = 0;
	$array_of_vars = null;
	$inside_variable=0;
	$variable='';
	$new_display_mask = '';

	for ($i=0; $i<strlen($display_mask); $i++)
	{
		if($inside_variable>0)
		{
			// If closing bracket
			if($display_mask[$i] == '}')
			{
				// Indicate close of reference.
				$inside_variable--;

				// Only if we have encountered final closing bracket!
				if($inside_variable==0)
				{
					if(strlen($variable)>0)
					{
						// Constrain to legal function names here.
						$function_r = prc_function_spec($variable,TRUE);
						
						// So we get a clean func definition for each $variable.
						unset($func);

						// Only legal function names.
						if(is_array($function_r))
						{
							// Will actually define the particular arguments here.
							switch($function_r['type'])
							{
								case 'if': // if(varname[<|<=|>=|>|==|!=]value, "title_mask")
									$func['type'] = $function_r['type']; //'if'

									// Parse the condition.
									parse_if_condition($function_r['args'][0], $func);
									
									$func['if_mask_elements'] = parse_field_mask($function_r['args'][1]);
									// Remember parse_field_mask resets the argument to {1}, {2}, etc.
									$func['if_mask'] = $function_r['args'][1];

									if(count($function_r['args'])>2)
									{
										$func['else_mask_elements'] = parse_field_mask($function_r['args'][2]);
										// Remember parse_field_mask resets the argument to {1}, {2}, etc.
										$func['else_mask'] = $function_r['args'][2];
									}
									$array_of_vars[] = $func;
									break;
								
								case 'switch': // switch(value, case, result[, case, result[,...][, default])
									$func['type'] = $function_r['type']; //'if'
									$func['varname'] = $function_r['args'][0];
									
									$j = 1;
									while( ($j + 1) < count($function_r['args']) )
									{
										$case = $function_r['args'][$j++];
										$result['mask_elements'] = parse_field_mask($function_r['args'][$j]);
										$result['mask'] = $function_r['args'][$j];
										
										$func['cases'][] = array('case'=>$case,
																'result'=>$result);
										$j++;
									}
									
									// a default element
									if($j < count($function_r['args']))
									{
										$result['mask_elements'] = parse_field_mask($function_r['args'][$j]);
										$result['mask'] = $function_r['args'][$j];
										$func['default_case'] = $result;
									}
									
									$array_of_vars[] = $func;
									break;
									
								case 'ifdef':
									$func['type'] = $function_r['type']; //'ifdef'
									$func['varname'] = $function_r['args'][0];

									// Now do the mask
									$func['if_mask_elements'] = parse_field_mask($function_r['args'][1]);
									// Remember parse_field_mask resets the argument to {1}, {2}, etc.
									$func['if_mask'] = $function_r['args'][1];
									
									if(count($function_r['args'])>2)
									{
										$func['else_mask_elements'] = parse_field_mask($function_r['args'][2]);
										// Remember parse_field_mask resets the argument to {1}, {2}, etc.
										$func['else_mask'] = $function_r['args'][2];
									}

									$array_of_vars[] = $func;
									break;
									
								case 'theme_img': // Only supports one level deep for the prompt_expression (second argument).
									$func['value'] = NULL;
									$func['type'] = $function_r['type'];
									$func['img'] = $function_r['args'][0];
									$func['title_mask_elements'] = parse_field_mask($function_r['args'][1]);
									
									// If not an array, then expand 
									if(!is_array($func['title_mask_elements']))
									{
										$func['title_mask_elements'][] = $function_r['args'][1];
										$func['title_mask'] = '{0}';
									}
									else // otherwise nested {...} block for value.
									{
										// this will cause the widget to be rendered, but no help entry
										// will be added to the listings page, because the $_OPENDB_TITLE_MASK_MACRO_RS
										// has not been updated.
										$func['title_mask'] = $function_r['args'][1];
									}									
									
									$array_of_vars[] = $func;
									break;
								
								case 'config_var_key': // config_var_keyid(name, value)
									$func['type'] = $function_r['type'];
									$func['name'] = $function_r['args'][0];
									$tmp_parsed_arg = parse_field_mask($function_r['args'][1]);
									if(is_array($tmp_parsed_arg))
									{
										$func['value'] = array('mask'=>$function_r['args'][1],
																'elements'=>$tmp_parsed_arg);
									}
									else
									{
										$func['value'] = $function_r['args'][1];
									}
									
									$array_of_vars[] = $func;
									break;
									
								case 'config_var_value': // config_var_value(name, keyid)
									$func['type'] = $function_r['type'];
									$func['name'] = $function_r['args'][0];
									$tmp_parsed_arg = parse_field_mask($function_r['args'][1]);
									if(is_array($tmp_parsed_arg))
									{
										$func['key'] = array('mask'=>$function_r['args'][1],
																'elements'=>$tmp_parsed_arg);
									}
									else
									{
										$func['key'] = $function_r['args'][1];
									}
									$array_of_vars[] = $func;
									break;
									
								default:
									// for unknown functions - copy the type, and expand any embedded
									// mask definitions.
									$func['type'] = $function_r['type'];
									for($j=0; $j<count($function_r['args']); $j++)
									{
										$tmp_parsed_arg = parse_field_mask($function_r['args'][$j]);
										if(is_array($tmp_parsed_arg))
										{
											$func['args'][] = array('mask'=>$function_r['args'][$j],
																'elements'=>$tmp_parsed_arg);
										}
										else
										{
											$func['args'][] = $function_r['args'][$j];
										}
									}

									$array_of_vars[] = $func;
									
									break;
							}
						}//if(is_array($function_r))
						else if($variable == 'title' || 
									$variable == 's_item_type' || 
									$variable == 's_status_type' || 
									$variable == 'item_id' || 
									$variable == 'instance_no')
						{
							$array_of_vars[] = $variable;
						}
						else
						{
							$index_of_sep = strrpos($variable, '.');
							if($index_of_sep !== FALSE)
							{
								$array_of_vars[] = array('s_attribute_type'=>strtoupper(substr($variable, 0, $index_of_sep)),
														'option'=>substr($variable,$index_of_sep+1));
							}
							else
							{
								$array_of_vars[] = array('s_attribute_type'=>strtoupper($variable), 'option'=>'value');
							}
						}
						
						// Add a {array_reference} to display_mask.
						$new_display_mask .= '{'.(count($array_of_vars)-1).'}';
					}
					$variable='';
				}
				else // No final closing bracket.
				{
 			    	$variable .= '}';
				}
			}
			else if($display_mask[$i] == '{')
			{
				$inside_variable++;
				$variable .= '{';
			}
			else
			{
				$variable .= $display_mask[$i];
			}
		}
		else if ($display_mask[$i] != '{')//first opening bracket.
		{
			$new_display_mask .= $display_mask[$i];
		}
		else //if($display_mask[$i] == '{')
		{
			$inside_variable++;
		}
	}

	// Return parsed mask (via pass-by-reference) back to caller.
	$display_mask = $new_display_mask;

	// Return array of parsed sections as well.
	return $array_of_vars;
}

/**
	Return TRUE if condition parsed successfully.  Assigns varname,
	op and value array elements to pass-by-reference $func array.

		Operators: >=, >, <=, <, ==, !=
*/
function parse_if_condition($if_condition, &$func)
{
	if( ($pos = strpos($if_condition, '>='))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '>=';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, '>'))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '>';
		$func['value']= trim(substr($if_condition,$pos+1));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, '<='))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '<=';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, '<'))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '<';
		$func['value']= trim(substr($if_condition,$pos+1));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, '=='))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '==';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, '!='))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '!=';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else
		return FALSE;
}

/**
	Does the comparison, with the same operators as 
	parse_if_condition.
*/
function test_if_condition($attribute_val, $op, $value)
{
	if(strlen($attribute_val)>0 && strlen($value)>0)
	{
		switch($op)
		{
			case '>=':
				return $attribute_val >= $value;
			case '>':
				return $attribute_val > $value;
			case '<=':
				return $attribute_val <= $value;
			case '<':
				return $attribute_val < $value;
			case '==':
				return $attribute_val == $value;
			case '!=':
				return $attribute_val != $value;
		}
	}
}

/**
* A simple field mask parser, which only supports 'if' and 'switch' and config_var_key(...)
* config_var_value(...)
*/
function expand_field_mask(&$values_rs, $mask, &$mask_element_rs, $config_var_rs = NULL)
{
	// If no parsed mask elements, then return $mask.
	if(is_empty_array($mask_element_rs))
	{
		// Only return mask if there is something to return.
		if(strlen($mask)>0)
			return $mask;
		else
			return $title;
	}

	for($i=0; $i<count($mask_element_rs); $i++)
	{
		// no array set, or simple attribute variable 's_attribute_type.option' not set.
		if(is_not_empty_array($mask_element_rs[$i]) && 
				!isset($mask_element_rs[$i]['s_attribute_type']) && 
				!isset($mask_element_rs[$i]['option']))
		{
			// Replace the array index.
			switch($mask_element_rs[$i]['type'])
			{
				case 'ifdef': // ifdef(s_attribute_type, "if_mask"[, "else_mask"])
					if(isset($values_rs[$mask_element_rs[$i]['varname']]))
						$value = expand_field_mask($values_rs, $mask_element_rs[$i]['if_mask'], $mask_element_rs[$i]['if_mask_elements'], $config_var_rs);
					else if(strlen($mask_element_rs[$i]['else_mask'])>0)
						$value = expand_field_mask($values_rs, $mask_element_rs[$i]['else_mask'], $mask_element_rs[$i]['else_mask_elements'], $config_var_rs);
					else
						$value = NULL;
					break;
					
				case 'if': // if(varname[<|<=|>=|>|==|!=]value, "if_mask"[, "else_mask"])
					$value = $values_rs[$mask_element_rs[$i]['varname']];

					// The attribute is defined, so now lets do the comparison.
					if(!empty($value))
					{
						if(test_if_condition($value, $mask_element_rs[$i]['op'], $mask_element_rs[$i]['value']))
							$value = expand_field_mask($values_rs, $mask_element_rs[$i]['if_mask'], $mask_element_rs[$i]['if_mask_elements'], $config_var_rs);
						else if(strlen($mask_element_rs[$i]['else_mask'])>0)
							$value = expand_field_mask($values_rs, $mask_element_rs[$i]['else_mask'], $mask_element_rs[$i]['else_mask_elements'], $config_var_rs);
						else
						{
							$value = NULL;
						}
					}
					else
					{
						$value = NULL;
					}
					break;
				
				case 'switch':
					$value = $values_rs[$mask_element_rs[$i]['varname']];

					// The attribute is defined, so now lets do the comparison.
					if(!empty($value))
					{
						if(is_not_empty_array($mask_element_rs[$i]['cases']))
						{
							for($j=0; $j<count($mask_element_rs[$i]['cases']); $j++)
							{
								// if a match.
								if(strcmp($value, $mask_element_rs[$i]['cases'][$j]['case']) === 0)
								{
									$value = expand_field_mask($values_rs, $mask_element_rs[$i]['cases'][$j]['result']['mask'], $mask_element_rs[$i]['cases'][$j]['result']['mask_elements'], $config_var_rs);
									break 2; // break out of switch
								}
							}
						}
						
						if(is_not_empty_array($mask_element_rs[$i]['default_case']))
						{
							$value = expand_field_mask($values_rs, $mask_element_rs[$i]['default_case']['mask'], $mask_element_rs[$i]['cases']['default_case']['mask_elements'], $config_var_rs);
						}
					}
					else
					{
						$value = NULL;
					}
					break;
				
				case 'config_var_key': // config_var_key(name, value)
					if(is_not_empty_array($config_var_rs) && is_array($config_var_rs[$mask_element_rs[$i]['name']]))
					{
						if(is_array($mask_element_rs[$i]['value']))
							$srchValue = expand_field_mask($values_rs, $mask_element_rs[$i]['value']['mask'], $mask_element_rs[$i]['value']['elements'], $config_var_rs);
						else
							$srchValue = $mask_element_rs[$i]['value'];
						
						$tmpValue = array_search2($srchValue, $config_var_rs[$mask_element_rs[$i]['name']]);
						if($tmpValue !== FALSE)
							$value = $tmpValue;
						else
							$value = '';
					}
					else
					{
						$value = '';
					}
					break;
					
				case 'config_var_value': // config_var_value(name, keyid)
					if(is_not_empty_array($config_var_rs) && is_array($config_var_rs[$mask_element_rs[$i]['name']]))
					{
						if(is_array($mask_element_rs[$i]['key']))
							$srchKey = expand_field_mask($values_rs, $mask_element_rs[$i]['key']['mask'], $mask_element_rs[$i]['key']['elements'], $config_var_rs);
						else
							$srchKey = $mask_element_rs[$i]['key'];
							
						if(isset($config_var_rs[$mask_element_rs[$i]['name']][$srchKey]))
							$value = $config_var_rs[$mask_element_rs[$i]['name']][$srchKey];
						else
							$value = '';
					}
					else
					{
						$value = '';
					}
					break;
					
				default: // No valid function specified, so set to empty.
					$value = '';
			}
		}
		else // standard variable (title, instance_no or item_attribute)
		{
			// in the case of this function, all {variables} are not actually s_attribute_type references, but
			// references to key's in the $values_rs array, thus we ignore the 'option' and assume 'value' in
			// every case.
			if(is_array($mask_element_rs[$i]) && isset($mask_element_rs[$i]['s_attribute_type']) && isset($mask_element_rs[$i]['option']))
			{
				$value = ifempty(
							$values_rs[$mask_element_rs[$i]['s_attribute_type']],
							$values_rs[strtolower($mask_element_rs[$i]['s_attribute_type'])]);
			}
			else
			{
				$value = $values_rs[$mask_element_rs[$i]];
			}
		}
		
		// Replace the array index.		
		$mask = str_replace('{'.$i.'}', $value, $mask);	
	}

	// Now return expanded subject.
	return $mask;
}

/**
	This takes the parsed array originally returned from parse_field_mask($display_mask) and expands it with the
	specific $item_instance_r information.
*/
function expand_title_mask($item_instance_r, $mask, &$mask_element_rs, $config_var_rs = NULL)
{
	// If no parsed mask elements, then return $mask.
	if(is_empty_or_not_array($mask_element_rs))
	{
		// Only return mask if there is something to return.
		if(strlen($mask)>0)
			return $mask;
		else
			return $item_instance_r['title'];
	}

	for($i=0; $i<count($mask_element_rs); $i++)
	{
		// no array set, or simple attribute variable 's_attribute_type.option' not set.
		if(is_not_empty_array($mask_element_rs[$i]) && 
				!isset($mask_element_rs[$i]['s_attribute_type']) && 
				!isset($mask_element_rs[$i]['option']))
		{
			// Replace the array index.
			switch($mask_element_rs[$i]['type'])
			{
				case 'ifdef': // ifdef(s_attribute_type, "if_mask"[, "else_mask"])
					if(is_item_attribute_set($item_instance_r['item_id'], strtoupper($mask_element_rs[$i]['varname'])))
						$value = expand_title_mask($item_instance_r, $mask_element_rs[$i]['if_mask'], $mask_element_rs[$i]['if_mask_elements']);
					else if(strlen($mask_element_rs[$i]['else_mask'])>0)
						$value = expand_title_mask($item_instance_r, $mask_element_rs[$i]['else_mask'], $mask_element_rs[$i]['else_mask_elements']);
					else
						$value = NULL;
					break;

				case 'if': // if(varname[<|<=|>=|>|==|!=]value, "if_mask"[, "else_mask"])
					if($mask_element_rs[$i]['varname'] == 'instance_no')
						$value = $item_instance_r['instance_no'];
					else if($mask_element_rs[$i]['varname'] == 's_status_type')
						$value = $item_instance_r['s_status_type'];
					else if($mask_element_rs[$i]['varname'] == 's_item_type')
						$value = $item_instance_r['s_item_type'];
					else
					{
						$value = fetch_attribute_val($item_instance_r['item_id'], strtoupper($mask_element_rs[$i]['varname']));
					}
					
					// The attribute is defined, so now lets do the comparison.
					if($value !== FALSE)
					{
						if(test_if_condition($value, $mask_element_rs[$i]['op'], $mask_element_rs[$i]['value']))
							$value = expand_title_mask($item_instance_r, $mask_element_rs[$i]['if_mask'], $mask_element_rs[$i]['if_mask_elements']);
						else if(strlen($mask_element_rs[$i]['else_mask'])>0)
							$value = expand_title_mask($item_instance_r, $mask_element_rs[$i]['else_mask'], $mask_element_rs[$i]['else_mask_elements']);
						else
						{
							$value = NULL;
						}
					}
					else
					{
						$value = NULL;
					}
					break; 
				
				case 'switch':
					if($mask_element_rs[$i]['varname'] == 'instance_no')
						$value = $item_instance_r['instance_no'];
					else if($mask_element_rs[$i]['varname'] == 's_status_type')
						$value = $item_instance_r['s_status_type'];
					else if($mask_element_rs[$i]['varname'] == 's_item_type')
						$value = $item_instance_r['s_item_type'];
					else
					{
						$value = fetch_attribute_val($item_instance_r['item_id'], strtoupper($mask_element_rs[$i]['varname']));
					}
					
					// The attribute is defined, so now lets do the comparison.
					if(!empty($value))
					{
						if(is_not_empty_array($mask_element_rs[$i]['cases']))
						{
							for($j=0; $j<count($mask_element_rs[$i]['cases']); $j++)
							{
								// if a match.
								if(strcmp($value, $mask_element_rs[$i]['cases'][$j]['case']) === 0)
								{
									$value = expand_title_mask($item_instance_r, $mask_element_rs[$i]['cases'][$j]['result']['mask'], $mask_element_rs[$i]['cases'][$j]['result']['mask_elements']);
									break 2; // break out of switch
								}
							}
						}
						
						if(is_not_empty_array($mask_element_rs[$i]['default_case']))
						{
							$value = expand_title_mask($item_instance_r, $mask_element_rs[$i]['default_case']['mask'], $mask_element_rs[$i]['cases']['default_case']['mask_elements']);
						}
					}
					else
					{
						$value = NULL;
					}
					break;
					
				case 'theme_img':
					if(strlen($mask_element_rs[$i]['value'])>0)
					{
						$value = $mask_element_rs[$i]['value'];
					}
					else
					{
						if($mask_element_rs[$i]['value'] === NULL && 
									strlen(($vTitle = expand_title_mask(array('s_item_type'=>$item_instance_r['s_item_type']), $mask_element_rs[$i]['title_mask'], $mask_element_rs[$i]['title_mask_elements'])))>0)
						{
							// This variable will be an array of all theme_img elements encountered
							global $_OPENDB_TITLE_MASK_MACRO_RS;
							if(strlen($_OPENDB_TITLE_MASK_MACRO_RS['theme_img'][$mask_element_rs[$i]['img']])==0)
							{
								$_OPENDB_TITLE_MASK_MACRO_RS['theme_img'][$mask_element_rs[$i]['img']] = $vTitle;
							}
							$mask_element_rs[$i]['value'] = _theme_image($mask_element_rs[$i]['img'], NULL, $vTitle, 'absmiddle');
							$value = $mask_element_rs[$i]['value'];
						}
						else // No, then leave for item_id context.
						{
							// an indicator to not try generic expand_title_mask for $s_item_type, because it does not work.
							$mask_element_rs[$i]['value'] = '';
							
							$vTitle = expand_title_mask($item_instance_r, $mask_element_rs[$i]['title_mask'], $mask_element_rs[$i]['title_mask_elements']);
							$value = _theme_image($mask_element_rs[$i]['img'], NULL, $vTitle, 'absmiddle');
						}
					}
					break;
				
				case 'config_var_key': // config_var_key(name, value)
					if(is_not_empty_array($config_var_rs) && is_array($config_var_rs[$mask_element_rs[$i]['name']]))
					{
						if(is_array($mask_element_rs[$i]['value']))
							$srchValue = expand_title_mask($item_instance_r, $mask_element_rs[$i]['value']['mask'], $mask_element_rs[$i]['value']['elements'], $config_var_rs);
						else
							$srchValue = $mask_element_rs[$i]['value'];
						
						$tmpValue = array_search2($srchValue, $config_var_rs[$mask_element_rs[$i]['name']]);
						if($tmpValue !== FALSE)
							$value = $tmpValue;
						else
							$value = '';
					}
					else
					{
						$value = '';
					}
					break;
					
				case 'config_var_value': // config_var_value(name, keyid)
					if(is_not_empty_array($config_var_rs) && is_array($config_var_rs[$mask_element_rs[$i]['name']]))
					{
						if(is_array($mask_element_rs[$i]['key']))
							$srchKey = expand_title_mask($item_instance_r, $mask_element_rs[$i]['key']['mask'], $mask_element_rs[$i]['key']['elements'], $config_var_rs);
						else
							$srchKey = $mask_element_rs[$i]['key'];
							
						if(isset($config_var_rs[$mask_element_rs[$i]['name']][$srchKey]))
							$value = $config_var_rs[$mask_element_rs[$i]['name']][$srchKey];
						else
							$value = '';
					}
					else
					{
						$value = '';
					}
					break;
					
				default: // No valid function specified, so set to empty.
					$value = '';
			}
		}//if(is_not_empty_array($mask_element_rs[$i]))
		else // standard variable (title, instance_no, item_attribute or plain text)
		{
			$value = get_mask_variable_value(
							$mask_element_rs[$i], 
							$item_instance_r);
		}
		
		// Now do the replacement.
		$mask = str_replace('{'.$i.'}', $value, $mask);	
			
	}//for($i=0; $i<count($mask_element_rs); $i++)

	// Now return expanded subject.
	return $mask;
}
?>
