<?php
/* 	OpenDb - Open Lending Database Project
	Copyright (C) 2001,2002 by Jason Pell

	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

include_once("./functions/item_attribute.php");

/**
	Assumes $c is single character
*/
function is_alpha($c)
{
	if( ($c >= '0' && $c <= '9') || ($c >= 'A' && $c <= 'Z') || ($c >= 'a' && $c <= 'z'))
		return TRUE;
	else
		return FALSE;
}

/*
* The escape sequences \r \n \t, must be enclosed in double quotes, otherwise PHP does
* not recognise them.
*/
function is_nonword_char($c)
{
	if ($c == '(' || $c == ')' || $c == '[' || $c == ']' || $c == '{' || $c == '}' || $c == ':' || $c == '.' || $c == '-' || $c == ' ' || $c == "\t" || $c == "\n" || $c == "\r")
		return TRUE;
	else
		return FALSE;
}

/**
	This function could be improved, but for now it should suffice!
*/
function is_roman_numeral($text)
{
	// Roman numerals from 1 to 20!
	$numerals = array('I', 'II', 'III', 'IV', 'V', 'VI', 'VII', 'VIII', 'IX', 'X', 'XI', 'XII', 'XIII', 'XIV', 'XV', 'XVI', 'XVII', 'XVIII', 'XIX', 'XX');
	if(in_array($text, $numerals))
		return TRUE;
	else
		return FALSE;
}

/**
	Small function to process word.
*/
function ucword($word)
{
	if(is_roman_numeral($word))
		return $word;//as is!
	else
		return ucfirst(strtolower($word));
}

/**
	This function will split a sentence into separate words.

	Sentences will be split on normal word boundaries, such as
	spaces,tabs,newlines, but also when a bracket (,{,[,],},) are
	encountered.  All space will be preserved however and restored
	to the final string.

	This function will also recognise Roman numerals from I to XX (1-20) 
	and if they are already UPPERCASE, will maintain case.

	This function differs from ucwords, in that it will set all 
	other characters in a word to lowercase!
*/
function initcap($text)
{
	$word="";
	$rtext="";
	for ($i=0; $i<strlen($text); $i++)
	{
		// We might be in the middle of a word.
		if(strlen($word)>0 || is_alpha($text[$i]))
		{
			// test for end of word.
			if(strlen($word)>0 && is_nonword_char($text[$i]))
			{
				$rtext .= ucword($word);
				$word="";

				$rtext .= $text[$i];
			}
			else
				$word .= $text[$i];
		}
		else//just copy it to return array.
			$rtext .= $text[$i];	
	}

	// Do final word.
	if(strlen($word)>0)
	{
		$rtext .= ucword($word);
		$word="";	
	}
	return $rtext;
}

/**
	Assumes $c is single character
*/
function is_legal_code_char($c)
{
	if( ($c >= '0' && $c <= '9') || ($c >= 'A' && $c <= 'Z') || ($c >= 'a' && $c <= 'z') || $c == '_' || $c == '-' || $c == '.')
		return TRUE;
	else
		return FALSE;
}

/**
	Ensure legal function name
*/
function is_legal_function_name($word)
{
	if(strlen($word)>0)
	{
		for($i=0; $i<strlen($word); $i++)
		{
			if(!is_legal_code_char($word[$i]))
				return FALSE;
		}
		return TRUE;
	}
	else
		return FALSE;
}

/**
	Get the function type of a specified 
	function_spec.  No other processing
	is required.
*/
function get_function_type($function_spec)
{
	$start = strpos($function_spec, "(");
    if($start !== FALSE)
    	$type = trim(substr($function_spec, 0, $start));
	else
		$type = trim($function_spec);	

	return $type;
}

/*
* Assumes the $args_r is compatible with the 'args' array
* returned from prc_function_spec(...)
* 
	Will reconstitute the function arguments from
	the array of functions.

	If a argument has any ',' in it we assume it
	needs to double quoted.

	Any double quotes in the argument, and they
	need to be escaped.
* 
* @param $function_name - If null, then will reconstitute
* 			the arguments (excluding the brackets) only.
* @param $arguments_r - The arguments.  If empty, will return
* 		the $function_name.
*/
function get_rebuilt_function($function_name, $arguments_r)
{
	if(!is_array($arguments_r))
	{
		if($function_name !== NULL)
		{
			if(strlen($arguments_r)==0)
				return $function_name;
			else
				return $function_name."(".$arguments_r.")";
		}
		else 
			return $arguments_r;
	}
	else
	{
		$argsText = "";

		@reset($arguments_r);
		while(list(,$argument) = @each($arguments_r))
		{
			// If argument includes a comma, or is whitespace (For a delimiter for example),
			// it must be enclosed in quotes.
			if( (strlen($argument)>0 && strlen(trim($argument))==0) || strpos($argument, ",") !== FALSE)
				$argument = "\"".str_replace("\"", "\\\"", $argument)."\"";

			if(strlen($argsText)==0)
				$argsText .= "$argument";
			else
				$argsText .= ", $argument";
		}

		if($function_name !== NULL)
		{
			if(strlen($argsText)==0)
				return $function_name;
			else
				return $function_name."(".$argsText.")";
		}
		else 
			return $argsText;
	}
}

/**
	Will return an array of the following form:
		type=>function_name, args=>array(args)
*/
function prc_function_spec($function_spec, $require_legal_func_name=FALSE) 
{
    $start = strpos($function_spec, "(");
    if($start !== FALSE)
    {
    	// Now we have something to parse.
        $type = trim(substr($function_spec, 0, $start));

		// Now ensure the function name is valid.
		if($require_legal_func_name==FALSE || is_legal_function_name($type))
		{
	        $end = strrpos($function_spec, ")");
			if($end>$start)//Otherwise a fuckup...
        	{
        		// Now we have the args, lets tokenise them.
	        	$args = trim(substr($function_spec, $start+1, $end-($start+1)));
				$arr = prc_args($args);
        	}
	
			return array(type=>strtolower($type), args=>$arr);
		}
		else
			return NULL;
	}
    else if($require_legal_func_name==FALSE)
	{ 
		// No (), so arg[0] is the whole thing.
    	$type = trim($function_spec);

		// Empty argument list.
		return array(type=>strtolower($type), args=>array());
	}
	else
		return NULL;
}

function prc_args($args)
{
	$argument="";
	$quote=NULL;
	
	// Allows us to keep track of nested braces.
	$curly_brace=0;
	$round_brace=0;
	$square_brace=0;
	$dbl_quote=FALSE;
	$sgl_quote=FALSE;
	
	for($i=0; $i<strlen($args); $i++)
	{
		switch($args[$i])
		{
			case '"':
				if($sgl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0 || ($i>0 && $args[$i-1]=="\\"))
					$argument .= $args[$i];
				else
					$dbl_quote = !$dbl_quote;
				break;
				
			case '\'':
				if($dbl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0 || ($i>0 && $args[$i-1]=="\\"))
					$argument .= $args[$i];
				else
					$sgl_quote = !$sgl_quote;
				break;
				
			case '\\':
				// If in braces, always include escape character, so it will be seen by the recursive calls to pcr_args.
				if($curly_brace>0 || $round_brace>0 || $square_brace>0)
				{
					$argument .= $args[$i];
				}
				else if($i>0 && $args[$i-1]=="\\")// As previous argument was an escape character, we should put this one in!
					$argument .= $args[$i];
				// else ignore
				break;
				
			case '{':
				// Do not recognise nested braces if inside quotes.
				if(!$dbl_quote && !$sgl_quote)
				{
					$curly_brace++;
				}	
				$argument .= $args[$i];
				break;
				
			case '}':
				// Do not recognise nested braces if inside quotes.
				if(!$dbl_quote && !$sgl_quote && $curly_brace>0)
				{
					$curly_brace--;
				}	
				$argument .= $args[$i];
				break;
				
			case '[':
				if(!$dbl_quote && !$sgl_quote)
				{
					$square_brace++;
				}
				$argument .= $args[$i];
				break;
				
			case ']':
				if(!$dbl_quote && !$sgl_quote && $square_brace>0)
				{
					$square_brace--;
				}
				$argument .= $args[$i];
				break;
				
			case '(':
				if(!$dbl_quote && !$sgl_quote)
				{
					$round_brace++;
				}
				$argument .= $args[$i];
				break;
				
			case ')':
				if(!$dbl_quote && !$sgl_quote && $round_brace>0)
				{
					$round_brace--;
				}
				$argument .= $args[$i];
				break;
				
			case ',':
				if($i>0 && $args[$i-1]=="\\")
				{
					// Get rid of escape character.
					$argument .= $args[$i];
				}
				else if($dbl_quote || $sgl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0)
				{
					// Inside nested block, so ignore argument separator.
					$argument .= $args[$i];
				}
				else
				{
					$arguments[] = $argument;
					$argument="";
				}
				break;
			
			case " ":
			case "\t":
			case "\n":
			case "\r":
				if($i>0 && $args[$i-1]=="\\")
				{
					// Get rid of escape character.
					$argument[strlen($argument)-1] = $args[$i];
				}
				else if(strlen($argument)>0)
				{
					// If already encountered non-whitespace for this argument, we need to keep it.
					$argument .= $args[$i];
				}
				else if($dbl_quote || $sgl_quote || $curly_brace>0 || $round_brace>0 || $square_brace>0)
				{
					// Inside nested block
					$argument .= $args[$i];
				}
				
				break;
				
			default:
				$argument .= $args[$i];
		}
	}	
	
	if(strlen($argument)>0)
		$arguments[] = $argument;
	return $arguments;
}

/**
* Stub version to avoid the pass-by-reference of the $mask_configuration, which
* will cause problems if used more than once in the page, where the $parsed_title_mask_r
* is not being cached as well. (item_add,item_input,item_display,etc)
*/
function expand_item_title_mask($title, $s_item_type, $item_id, $instance_no, $s_status_type, $mask_configuration)
{
	if(strlen($mask_configuration)>0)
	{
		return expand_type_title_mask($title, $s_item_type, $item_id, $instance_no, $s_status_type, $mask_configuration, $parsed_title_mask_r);
	}
	else
		return $title;		
}

/*
* This function is responsible for mapping a s_item_type_groups, group name
* to an array in the mask_configuration variable.
* 
* @param $mask_configuration - Passed-by-reference for performance reasons, will
* 								not modify here!
*/
function get_s_item_type_mask_group_name($s_item_type, &$mask_configuration)
{
	global $CONFIG_VARS;
	
	reset($CONFIG_VARS['listings.s_item_type_groups']);
	while(list($group,) = each($CONFIG_VARS['listings.s_item_type_groups']))
	{
		if(is_not_empty_array($CONFIG_VARS['listings.s_item_type_groups'][$group]) && in_array($s_item_type,$CONFIG_VARS['listings.s_item_type_groups'][$group]))
		{
			if(strlen($mask_configuration[$group])>0)
				return $group;
		}			
	}
	
	// Nothing found.
	return NULL;
}

/**
	Based on the $mask_configuration array or straight mask value, will return
	a parsed title mask for the $s_item_type.
	
	The steps:
		Check for a $mask_configuration[$s_item_type]
		Check for a $mask_configuration['DEFAULT']
		Otherwise use $mask_configuration

	Will update (via pass-by-reference) the $mask_configuration	& $parsed_title_mask_r variables,
	so that they only have to be parsed once!
*/
function expand_type_title_mask($title, $s_item_type, $item_id, $instance_no, $s_status_type, &$mask_configuration, &$parsed_title_mask_r)
{
	// Display masked title if configured
	if(strlen($mask_configuration)>0)
	{
		// Definitions for separate s_item_type's
		if(is_array($mask_configuration))
		{
			if(strlen($mask_configuration[$s_item_type])>0)
			{
				if(is_empty_array($parsed_title_mask_r[$s_item_type]))
				{
					$parsed_title_mask_r[$s_item_type] = parse_title_mask($mask_configuration[$s_item_type], $s_item_type);
				}
				return expand_title_mask($mask_configuration[$s_item_type], $parsed_title_mask_r[$s_item_type], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
			}
			else if(strlen(($s_item_type_group_name = get_s_item_type_mask_group_name($s_item_type, $mask_configuration)))>0)
			{
				if(is_empty_array($parsed_title_mask_r[$s_item_type_group_name]))
				{
					$parsed_title_mask_r[$s_item_type_group_name] = parse_title_mask($mask_configuration[$s_item_type_group_name], $s_item_type);
				}
				return expand_title_mask($mask_configuration[$s_item_type_group_name], $parsed_title_mask_r[$s_item_type_group_name], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
			}
			else if(strlen($mask_configuration['DEFAULT'])>0)
			{
				if(is_empty_array($parsed_title_mask_r['DEFAULT']))
				{
					$parsed_title_mask_r['DEFAULT'] = parse_title_mask($mask_configuration['DEFAULT'], $s_item_type);
				}
				return expand_title_mask($mask_configuration['DEFAULT'], $parsed_title_mask_r['DEFAULT'], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
			}
			else
				return $title;
		}
		else// Global definition.
		{
			if(is_empty_array($parsed_title_mask_r))
			{
				$parsed_title_mask_r = parse_title_mask($mask_configuration, $s_item_type);
			}
			return expand_title_mask($mask_configuration, $parsed_title_mask_r, $title, $s_item_type, $item_id, $instance_no, $s_status_type);
		}
	}
	else	
		return $title;
}

/**
	Return a database value for $item_id based on whether $mask_variable
    points to title or to an item_attribute.  For item_attribute references
	there are some further options.

	A legal mask item can include the following:
		{s_attribute_type}
		{s_attribute_type.img}
		{s_attribute_type.value}  // The .value will result in the same as if no .option was specified.
		{s_attribute_type.display}
		{title}
		{s_item_type}
		{item_id}
		{instance_no}
		{s_status_type}

	NOTE:
		From 0.51-dev6 onwards.  The $alt parameter is no longer $lookup_attr_r['display'] in
		call to _theme_image.
*/
function get_mask_variable_value($mask_variable, $title, $s_item_type, $item_id, $instance_no, $s_status_type)
{
	global $LANG_VARS;

	$value = "";

	// Special 'title' and 'instance_no' variable references.
	if($mask_variable == "title")
		return $title;
	else if($mask_variable == "s_item_type")
		return $s_item_type;
	else if($mask_variable == "item_id")
		return $item_id;
	else if($mask_variable == "instance_no")
		return $instance_no;
	else if($mask_variable == "s_status_type")
		return $s_status_type;
	else // a standard {variable}
	{
		$index_of_sep = strrpos($mask_variable, ".");
		if($index_of_sep !== FALSE)
		{
			$s_attribute_type = strtoupper(substr($mask_variable, 0, $index_of_sep));
			$option = substr($mask_variable,$index_of_sep+1); 
			$value = null;
			
			if(strlen($s_attribute_type)>0)
			{
				// the options that require a item_id context
				if(is_numeric($item_id))
				{
					if($option == "img")
					{
						$lookup_attr_r = fetch_sattr_type_lookup_for_item_attr($item_id, $s_attribute_type);
						if($lookup_attr_r!==FALSE)
						{
							if(strlen($lookup_attr_r['img'])>0 && $lookup_attr_r['img'] != "none")
								$value = _theme_image($lookup_attr_r['img'], NULL, $lookup_attr_r['display'], "absmiddle");
							else // If no image, then use (value).
								$value = "(".$lookup_attr_r['value'].")";						
						}
					}
					else if($option == "display" && is_numeric($item_id))
					{
						$value = fetch_sattr_type_lookup_for_item_attr($item_id, $s_attribute_type, 'display');
					}
				
					// If no records found for item_attribute/s_attribute_type combination or the 'value'
					// option specified.
					if(is_numeric($item_id) && strlen($value)==0)
						$value = fetch_attribute_val($item_id, $s_attribute_type);
				}
				else if(strlen($s_item_type)>0)// s_item_type context items.
				{
					if($option == "prompt") // The s_attribute_type prompt
					{
						$value = fetch_s_item_type_attr_prompt($s_item_type, $s_attribute_type);
					}
				}
			}
		}
		else
		{
			if(is_numeric($item_id))
				$value = fetch_attribute_val($item_id, strtoupper($mask_variable));
			else // If not being passed in in item_id context, assume plain text.
				$value = $mask_variable;
		}
	}
	
	//else
	return $value;
}

function get_title_mask_macro_element_r($macro_type)
{
	global $_OPENDB_TITLE_MASK_MACRO_RS;
	
	if(is_array($_OPENDB_TITLE_MASK_MACRO_RS) && isset($_OPENDB_TITLE_MASK_MACRO_RS[$macro_type]))
		return $_OPENDB_TITLE_MASK_MACRO_RS[$macro_type];
	else
		return NULL;
}

/**
	Modify display_mask so that all {variable} are replaced with
	references to the array indexes returned.
*/
function parse_title_mask(&$display_mask, $s_item_type)
{
	$i = 0;
	$array_of_vars = null;
	$inside_variable=0;
	$variable="";
	$new_display_mask = "";

	for ($i=0; $i<strlen($display_mask); $i++)
	{
		if($inside_variable>0)
		{
			// If closing bracket
			if($display_mask[$i] == '}')
			{
				// Indicate close of reference.
				$inside_variable--;

				// Only if we have encountered final closing bracket!
				if($inside_variable==0)
				{
					if(strlen($variable)>0)
					{
						// Constrain to legal function names here.
						$function_r = prc_function_spec($variable,TRUE);
						
						// So we get a clean func definition for each $variable.
						unset($func);

						// Only legal function names.
						if(is_array($function_r))
						{
							// Will actually define the particular arguments here.
							switch($function_r['type'])
							{
								case 'if': // if(varname[<|<=|>=|>|==|!=]value, "title_mask")
									$func['type'] = $function_r['type']; //'ifgt'

									// Parse the condition.
									parse_if_condition($function_r['args'][0], $func);
									
									$func['if_mask_elements'] = parse_title_mask($function_r['args'][1], $s_item_type);
									// Remember parse_title_mask resets the argument to {1}, {2}, etc.
									$func['if_mask'] = $function_r['args'][1];

									if(count($function_r['args'])>2)
									{
										$func['else_mask_elements'] = parse_title_mask($function_r['args'][2], $s_item_type);
										// Remember parse_title_mask resets the argument to {1}, {2}, etc.
										$func['else_mask'] = $function_r['args'][2];
									}
									$array_of_vars[] = $func;
									break;
							
								case 'ifdef':
									$func['type'] = $function_r['type']; //'ifdef'
									$func['varname'] = $function_r['args'][0];

									// Now do the mask
									$func['if_mask_elements'] = parse_title_mask($function_r['args'][1], $s_item_type);
									// Remember parse_title_mask resets the argument to {1}, {2}, etc.
									$func['if_mask'] = $function_r['args'][1];
									
									if(count($function_r['args'])>2)
									{
										$func['else_mask_elements'] = parse_title_mask($function_r['args'][2], $s_item_type);
										// Remember parse_title_mask resets the argument to {1}, {2}, etc.
										$func['else_mask'] = $function_r['args'][2];
									}

									$array_of_vars[] = $func;
									break;
									
								case 'theme_img': // Only supports one level deep for the prompt_expression (second argument).
									$func['type'] = $function_r['type'];
									$func['img'] = $function_r['args'][0];
									$func['title_mask_elements'] = parse_title_mask($function_r['args'][1], $s_item_type);
									
									// If not an array, then expand 
									if(!is_array($func['title_mask_elements']))
									{
										// Is the mask variable expandable with just $s_item_type context.
										$title = get_mask_variable_value($function_r['args'][1], NULL, $s_item_type, NULL, NULL, NULL);
										if(strlen($title)>0)
										{
											// This variable will be an array of all theme_img elements encountered
											global $_OPENDB_TITLE_MASK_MACRO_RS;
											if(!isset($_OPENDB_TITLE_MASK_MACRO_RS['theme_img'][$func['img']]))
												$_OPENDB_TITLE_MASK_MACRO_RS['theme_img'][$func['img']] = $title;
											
											$func['value'] = _theme_image($func['img'], NULL, $title, "absmiddle");
										}
										else // No, then leave for item_id context.
										{
											$func['title_mask_elements'][] = $function_r['args'][1];
											$func['title_mask'] = "{0}";
										}
									}
									else // otherwise nested {...} block for value.
									{
										$func['title_mask'] = $function_r['args'][1];
									}									
									
									$array_of_vars[] = $func;
									break;
									
								default:
									$array_of_vars[] = $variable;
									break;
							}
						}
						else
						{
							$array_of_vars[] = $variable;
						}

						// Add a {array_reference} to display_mask.
						$new_display_mask .= "{".(count($array_of_vars)-1)."}";
					}
					$variable="";
				}
				else // No final closing bracket.
 			    	$variable .= "}";
			}
			else if($display_mask[$i] == '{')
			{
				$inside_variable++;
				$variable .= "{";
			}
			else
			{
				$variable .= $display_mask[$i];
			}
		}
		else if ($display_mask[$i] != '{')//first opening bracket.
		{
			$new_display_mask .= $display_mask[$i];
		}
		else //if($display_mask[$i] == '{')
			$inside_variable++;
	}

	// Return parsed mask (via pass-by-reference) back to caller.
	$display_mask = $new_display_mask;

	// Return array of parsed sections as well.
	return $array_of_vars;
}

/**
	Return TRUE if condition parsed successfully.  Assigns varname,
	op and value array elements to pass-by-reference $func array.

		Operators: >=, >, <=, <, ==, !=
*/
function parse_if_condition($if_condition, &$func)
{
	if( ($pos = strpos($if_condition, ">="))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '>=';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, ">"))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '>';
		$func['value']= trim(substr($if_condition,$pos+1));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, "<="))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '<=';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, "<"))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '<';
		$func['value']= trim(substr($if_condition,$pos+1));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, "=="))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '==';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else if( ($pos = strpos($if_condition, "!="))!==FALSE )
	{
		$func['varname'] = trim(substr($if_condition,0,$pos));
		$func['op'] = '!=';
		$func['value']= trim(substr($if_condition,$pos+2));
		return TRUE;
	}
	else
		return FALSE;
}

/**
	Does the comparison, with the same operators as 
	parse_if_condition.
*/
function test_if_condition($attribute_val, $op, $value)
{
	if(strlen($attribute_val)>0 && strlen($value)>0)
	{
		switch($op)
		{
			case '>=':
				return $attribute_val >= $value;
			case '>':
				return $attribute_val > $value;
			case '<=':
				return $attribute_val <= $value;
			case '<':
				return $attribute_val < $value;
			case '==':
				return $attribute_val == $value;
			case '!=':
				return $attribute_val != $value;
		}
	}
}

/**
	This takes the parsed array originally returned from 
	parse_title_mask($display_mask) and expands it with the
	specific $item_id/$title information.
*/
function expand_title_mask($mask, $mask_elements_r, $title, $s_item_type, $item_id, $instance_no, $s_status_type)
{
	// If no parsed mask elements, then return $mask.
	if(is_empty_array($mask_elements_r))
	{
		// Only return mask if there is something to return.
		if(strlen($mask)>0)
			return $mask;
		else
			return $title;
	}

	for($i=0; $i<count($mask_elements_r); $i++)
	{
		if(is_not_empty_array($mask_elements_r[$i]))
		{
			// Replace the array index.
			switch($mask_elements_r[$i]['type'])
			{
				case 'ifdef': // ifdef(s_attribute_type, "if_mask"[, "else_mask"])
					if(is_item_attribute_set($item_id, strtoupper($mask_elements_r[$i]['varname'])))
						$value = expand_title_mask($mask_elements_r[$i]['if_mask'], $mask_elements_r[$i]['if_mask_elements'], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
					else if(strlen($mask_elements_r[$i]['else_mask'])>0)
						$value = expand_title_mask($mask_elements_r[$i]['else_mask'], $mask_elements_r[$i]['else_mask_elements'], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
					else
						$value = null;
					break;

				case 'if': // if(varname[<|<=|>=|>|==|!=]value, "if_mask"[, "else_mask"])
					if($mask_elements_r[$i]['varname'] == "instance_no")
						$value = $instance_no;
					else if($mask_elements_r[$i]['varname'] == "s_status_type")
						$value = $s_status_type;
					else if($mask_elements_r[$i]['varname'] == "s_item_type")
						$value = $s_item_type;
					else
						$value = fetch_attribute_val($item_id, strtoupper($mask_elements_r[$i]['varname']));

					// The attribute is defined, so now lets do the comparison.
					if($value !== FALSE)
					{
						if(test_if_condition($value, $mask_elements_r[$i]['op'], $mask_elements_r[$i]['value']))
							$value = expand_title_mask($mask_elements_r[$i]['if_mask'], $mask_elements_r[$i]['if_mask_elements'], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
						else if(strlen($mask_elements_r[$i]['else_mask'])>0)
							$value = expand_title_mask($mask_elements_r[$i]['else_mask'], $mask_elements_r[$i]['else_mask_elements'], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
						else
							$value = null;
					}
					else
						$value = null;

					break; 

				case 'theme_img':
					if(strlen($mask_elements_r[$i]['value'])>0)
						$value = $mask_elements_r[$i]['value'];
					else
					{
						$title = expand_title_mask($mask_elements_r[$i]['title_mask'], $mask_elements_r[$i]['title_mask_elements'], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
						$value = _theme_image($mask_elements_r[$i]['img'], NULL, $title, "absmiddle");
					}
				
					break;
				default: // No valid function specified, so set to empty.
					$value = "";
			}
			
			// Now do the replacement.
			$mask = str_replace("{".$i."}", $value, $mask);
		}
		else // standard variable (title, instance_no or item_attribute)
		{
			$value = get_mask_variable_value($mask_elements_r[$i], $title, $s_item_type, $item_id, $instance_no, $s_status_type);
			
			// Replace the array index.		
			$mask = str_replace("{".$i."}", $value, $mask);	
		}
	}

	// Now return expanded subject.
	return $mask;
}
?>
