#include "jdxtypes.h"
#include "jdxblock.h" // for UnitTest
#include <tjutils/tjtest.h>

JDXstring::JDXstring(const STD_string& ss,const STD_string& name,bool userParameter,
                     compatMode mode,parameterMode parameter_mode, const STD_string& parx_equivalent)
  : STD_string(ss) {
    set_label(name);
    JcampDxClass::set_compatmode(mode); // Do NOT call virtual function in constructor!
    set_userDefParameter(userParameter);
    JcampDxClass::set_parmode(parameter_mode); // Do NOT call virtual function in constructor!
    parx_equiv.name=parx_equivalent;
}


JDXstring& JDXstring::operator = (const JDXstring& ss) {
  JcampDxClass::operator = (ss);
  STD_string::operator = (ss);
  parx_equiv=ss.parx_equiv;
  return *this;
}


STD_string JDXstring::printvalstring() const {
  Log<JcampDx> odinlog(this,"printvalstring");
  if(get_filemode()==exclude) return "";
  STD_string result;
  if(get_compatmode()==bruker) {
    ndim nn(1);
    int l=length()*_BRUKER_MODE_STRING_CAP_FACTOR_;
    if(!l) l=_BRUKER_MODE_STRING_CAP_START_;
    if(l<_BRUKER_MODE_STRING_MIN_SIZE_) l=_BRUKER_MODE_STRING_MIN_SIZE_;
    nn[0]=l;
    result+=STD_string(nn)+"\n";
  }
  if(get_compatmode()==bruker) result+="<";
  result+=STD_string(*this);
  if(get_compatmode()==bruker) result+=">";
  ODINLOG(odinlog,normalDebug) << "returning >" << result << "<" << STD_endl;
  return result;
}



bool JDXstring::parsevalstring(const STD_string& str) {
  Log<JcampDx> odinlog(this,"parsevalstring");
  ODINLOG(odinlog,normalDebug) << "str=>" << str << "<" << STD_endl;
  STD_string without_size_field;
  if(get_compatmode()==bruker) without_size_field=extract(str,"\n","");
  else without_size_field=str;
  ODINLOG(odinlog,normalDebug) << "without_size_field=>" << without_size_field << "<" << STD_endl;
  STD_string shrinked=shrink(without_size_field);
  ODINLOG(odinlog,normalDebug) << "shrinked=>" << shrinked << "<" << STD_endl;
  if (shrinked.length()>=2 && shrinked[(unsigned int)0]=='<' && shrinked[shrinked.length()-1]=='>') (*this)=extract(without_size_field,"<",">",true);
  else (*this)=without_size_field;
  return true;
}



STD_string JDXstring::get_parx_code(parxCodeType type, const ParxEquiv& equiv) const {
#ifdef PARAVISION_PLUGIN
  STD_string result=JcampDxClass::get_parx_code(type,equiv);
  if(type==parx_def) result=get_parx_def_string(get_parx_equiv().type,1);
  if(type==parx_passval) result="sprintf("+equiv.name+",\"%s\","+get_label()+");\n";
  return result;
#else
  return "";
#endif
}


#ifndef NO_UNIT_TEST
class JDXstringTest : public UnitTest {

 public:
  JDXstringTest() : UnitTest("JDXstring") {}

 private:
  bool check() const {
    Log<UnitTest> odinlog(this,"check");

    JDXstring teststr1("value","teststr1",true,notBroken);
    JDXstring teststr2("value","teststr2",true,bruker);

    // testing JDXstring::print
    STD_string expected="##$teststr1=value\n";
    STD_string printed=teststr1.print();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "print() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }
    expected="##$teststr2=( "+itos(_BRUKER_MODE_STRING_MIN_SIZE_)+" )\n<value>\n";
    printed=teststr2.print();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "print() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }

    // testing JDXstring::parse
    JcampDxBlock block;
    block.append(teststr1);
    block.append(teststr2);

    int parseresult=block.parseblock("##TITLE=block\n##$teststr1=parseval\n##$teststr2=( 64 )\n<parseval>\n$$ Comment\n##$dummy=dummyval\n##END=");
    if(parseresult!=2) {
      ODINLOG(odinlog,errorLog) << "JcampDxBlock::parseblock() failed: parseresult=" << parseresult << "!=" << 2 << STD_endl;
      return false;
    }
    if(STD_string(block.get_label())!="block") {
      ODINLOG(odinlog,errorLog) << "JcampDxBlock::get_label() failed: >" << block.get_label() << "< != >block<" << STD_endl;
      return false;
    }
    if(STD_string(teststr1)!=STD_string(teststr2)) {
      ODINLOG(odinlog,errorLog) << "after block.parseblock(): >" << STD_string(teststr1) << "< != >" << STD_string(teststr2) << "<" << STD_endl;
      return false;
    }

    return true;
  }

};


void alloc_JDXstringTest() {new JDXstringTest();} // create test instance
#endif


//////////////////////////////////////////////////////////////////

JDXbool::JDXbool(bool flag, const STD_string& name,bool userParameter,
                   compatMode mode,parameterMode parameter_mode, const STD_string& parx_equivalent)
  : val(flag) {
    set_label(name);
    JcampDxClass::set_compatmode(mode); // Do NOT call virtual function in constructor!
    set_userDefParameter(userParameter);
    JcampDxClass::set_parmode(parameter_mode); // Do NOT call virtual function in constructor!
    parx_equiv.name=parx_equivalent;
}


JDXbool& JDXbool::operator = (const JDXbool& jb) {
  JcampDxClass::operator = (jb);
  val=jb.val;
  parx_equiv=jb.parx_equiv;
  return *this;
}

bool JDXbool::parsevalstring (const STD_string& parstring) {
  Log<JcampDx> odinlog(this,"parsevalstring");
  STD_string yesnostr(shrink(tolowerstr(parstring)));
  ODINLOG(odinlog,normalDebug) << "yesnostr=" << yesnostr << STD_endl;
  if(yesnostr=="yes" || yesnostr=="true") val=true;
  else val=false;
  ODINLOG(odinlog,normalDebug) << "this=" << bool(*this) << STD_endl;
  return true;
}

STD_string JDXbool::printvalstring() const {
  if(val) return "Yes";
  else return "No";
}

STD_string JDXbool::get_parx_code(parxCodeType type, const ParxEquiv& equiv) const {
#ifdef PARAVISION_PLUGIN
  STD_string result=JcampDxClass::get_parx_code(type,equiv);
  if(type==parx_def) result=get_parx_def_string(equiv.type,0); 
  return result;
#else
  return "";
#endif
}

#ifndef NO_UNIT_TEST
class JDXboolTest : public UnitTest {

 public:
  JDXboolTest() : UnitTest("JDXbool") {}

 private:
  bool check() const {
    Log<UnitTest> odinlog(this,"check");

    JDXbool testbool(false,"testbool");
    testbool=true;
    STD_string expected="##$testbool=Yes\n";
    STD_string printed=testbool.print();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "print() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }
    JcampDxBlock boolblock;
    boolblock.append(testbool);
    boolblock.parseblock("##TITLE=boolblock\n##$testbool=No\n##END=");
    if(bool(testbool)!=false) {
      ODINLOG(odinlog,errorLog) << "after boolblock.parseblock(): for bool " << bool(testbool) << "!=" << false << STD_endl;
      return false;
    }

    return true;
  }

};

void alloc_JDXboolTest() {new JDXboolTest();} // create test instance
#endif

///////////////////////////////////////////////////////////////

JDXenum::JDXenum(const STD_string& first_entry, const STD_string& name,bool userParameter,
                 compatMode mode,parameterMode parameter_mode, const STD_string& parx_equivalent) {
  add_item(first_entry);
  set_label(name);
  JcampDxClass::set_compatmode(mode); // Do NOT call virtual function in constructor!
  set_userDefParameter(userParameter);
  JcampDxClass::set_parmode(parameter_mode); // Do NOT call virtual function in constructor!
  parx_equiv.name=parx_equivalent;
}
    
JDXenum& JDXenum::operator = (const JDXenum& je) {
  JcampDxClass::operator = (je);
  entries=je.entries;
  for(STD_map<int,STD_string>::const_iterator it=entries.begin();it!=entries.end();++it) {
    if ( (it->first) == (je.actual->first) ) actual=it;
  }
  parx_equiv=je.parx_equiv;
  return *this;
}
  
JDXenum& JDXenum::add_item(const STD_string& item, int index) {
  if(item=="") return *this; 
  int n=0;
  if(index<0) {
    for(STD_map<int,STD_string>::const_iterator it=entries.begin();it!=entries.end();++it) {
      if ( it->first > n ) n=it->first;
    }
    if(entries.size()) n++;
  } else n=index;

  entries[n]=item;
  actual=entries.find(n);
  return *this;
}
  
JDXenum& JDXenum::set_actual(const STD_string& item) {
  STD_map<int,STD_string>::const_iterator it;
  for(it=entries.begin();it!=entries.end();++it) {
    if ( (it->second) == item ) actual=it;
  }
  return *this;
}

JDXenum& JDXenum::set_actual(int index) {
  STD_map<int,STD_string>::const_iterator it;
  for(it=entries.begin();it!=entries.end();++it) {
    if ( (it->first) == index ) actual=it;
  }
  return *this;
}


JDXenum& JDXenum::clear() {
  entries.clear();
  actual=entries.end();
  return *this;
}

JDXenum::operator int () const {
  if(actual!=entries.end()) return actual->first;
  else return 0;
}

JDXenum::operator STD_string () const {
  if(actual!=entries.end()) return actual->second;
  else return "";
}

static STD_string notext("none");
  
const STD_string& JDXenum::get_item(unsigned int index) const {
  STD_map<int,STD_string>::const_iterator it=entries.begin();
  for(unsigned int i=0; i<index; i++) {
    ++it;
    if(it==entries.end()) return notext;
  }
  return it->second;
}

unsigned int JDXenum::get_item_index() const {
  unsigned int result=0;
  for(STD_map<int,STD_string>::const_iterator it=entries.begin();it!=entries.end();++it) {
    if ( it==actual ) return result;
    result++;
  }
  return 0;
}

JDXenum& JDXenum::set_item_index(unsigned int index) {
  STD_map<int,STD_string>::const_iterator it=entries.begin();
  for(unsigned int i=0; i<index; i++) {
    if(it==entries.end()) return *this;
    ++it;
  }
  actual=it;
  return *this;    
}



bool JDXenum::parsevalstring (const STD_string& parstring) {
  STD_string newpar(parstring);

  // check whether parstring is already there:
  bool already_there=false;
  STD_map<int,STD_string>::const_iterator it;
  for(it=entries.begin();it!=entries.end();++it) {
    if ( (it->second) == newpar ) {
      already_there=true;
      actual=it;
    }
  }
  if(!already_there && (entries.size()==0) ) add_item(newpar);
  return true;
}


STD_string JDXenum::get_parx_code(parxCodeType type, const ParxEquiv& equiv) const {
#ifdef PARAVISION_PLUGIN
  STD_string result=JcampDxClass::get_parx_code(type,equiv);
  if(type==parx_def) result=get_parx_def_string(equiv.type,0); 
  return result;
#else
  return "";
#endif
}

ParxEquiv JDXenum::get_parx_equiv() const {
  parx_equiv.type=toupperstr(STD_string(get_label()));
  return parx_equiv;
}


svector JDXenum::get_alternatives() const {
  unsigned int n=n_items();
  svector result; result.resize(n);
  for(unsigned int i=0; i<n; i++) {
    result[i]=get_item(i);
  }
  return result;
}



STD_string JDXenum::printvalstring() const {
  if(actual!=entries.end()) return (actual->second);
  else return "emptyEnum";
}
  

#ifndef NO_UNIT_TEST
class JDXenumTest : public UnitTest {

 public:
  JDXenumTest() : UnitTest("JDXenum") {}

 private:
  bool check() const {
    Log<UnitTest> odinlog(this,"check");

    // testing JDXenum
    JDXenum testenum("","testenum");
    testenum.add_item("item7",7);
    testenum.add_item("item0",0);
    testenum.add_item("item5",5);
    testenum.add_item("item1",1);
    testenum="item5";
    STD_string expected="##$testenum=item5\n";
    STD_string printed=testenum.print();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "testenum::print() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }
    testenum=7;
    expected="##$testenum=item7\n";
    printed=testenum.print();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "testenum::print() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }
    JcampDxBlock enumblock;
    enumblock.append(testenum);
    enumblock.parseblock("##TITLE=enumblock\n##testenum=item1\n##END=");
    if((int)testenum!=1) {
      ODINLOG(odinlog,errorLog) << "after enumblock.parseblock(): for JDXenum " << (int)testenum << "!=" << 1 << STD_endl;
      return false;
    }

    return true;
  }

};

void alloc_JDXenumTest() {new JDXenumTest();} // create test instance
#endif


///////////////////////////////////////////////////////////////

JDXaction::JDXaction(bool init_state, const STD_string& name,bool userParameter, compatMode mode,parameterMode parameter_mode)
  : state(init_state) {
    JcampDxClass::set_filemode(exclude); // Do NOT call virtual function in constructor!
    set_label(name);
    JcampDxClass::set_compatmode(mode); // Do NOT call virtual function in constructor!
    set_userDefParameter(userParameter);
    JcampDxClass::set_parmode(parameter_mode); // Do NOT call virtual function in constructor!
}

JDXaction& JDXaction::operator = (const JDXaction& ja) {
  JcampDxClass::operator = (ja);
  return *this;
}

JDXaction::operator bool () const {
  bool result=state;
  state=false;
  return result;
}

bool JDXaction::parsevalstring (const STD_string& parstring) {
  STD_string actionstr(shrink(tolowerstr(parstring)));
  if(actionstr=="busy") state=true;
  else state=false;
  return true;
}

STD_string JDXaction::printvalstring() const {
  if(state) return "CLICK_HERE";
  else return "NOW";
}


///////////////////////////////////////////////////////////////

JDXfileName::JDXfileName(const STD_string& filename, const STD_string& name, bool userParameter, compatMode mode, parameterMode parameter_mode)
 :   JDXstring(filename, name, userParameter, mode, parameter_mode) {
  common_init();
  normalize(filename, dir, *this, dirname_cache, basename_cache, suffix_cache);
}


JDXfileName::JDXfileName (const JDXfileName& jf) {
  common_init();
  JDXfileName::operator = (jf);
}


JDXfileName& JDXfileName::operator = (const STD_string& filename) {
  normalize(filename, dir, *this, dirname_cache, basename_cache, suffix_cache);
  return *this;
}


JDXfileName& JDXfileName::operator = (const JDXfileName& jf) {
  JDXstring::operator = (jf);
  dir=jf.dir;
  normalize(jf, dir, *this, dirname_cache, basename_cache, suffix_cache);
  defaultdir=jf.defaultdir;
  return *this;
}


bool JDXfileName::exists() const {
  bool result=false;
#ifndef NO_FILEHANDLING
  if(is_dir()) result=checkdir(c_str());
  else         result=(filesize(c_str())>=0);
#endif
  return result;
}



STD_string JDXfileName::get_basename_nosuffix() const {
  STD_string result(get_basename());
  if(get_suffix()!="") return replaceStr(result,"."+get_suffix(),"");
  return result;
}


JDXfileName& JDXfileName::set_defaultdir(const STD_string& defdir) {
  STD_string dummystr;
  normalize(defdir, true, defaultdir, dummystr, dummystr, dummystr);
  return *this;
}

bool JDXfileName::parsevalstring (const STD_string& parstring) {
  normalize(parstring, dir, *this, dirname_cache, basename_cache, suffix_cache);
  return true;
}


void JDXfileName::normalize(const STD_string& fname, bool dir, STD_string& result, STD_string& result_dirname, STD_string& result_basename, STD_string& result_suffix) {
  Log<JcampDx> odinlog("JDXfileName","normalize");

  STD_string tmpstr(fname); // mutable copy

  // remove quotes
  tmpstr=replaceStr(tmpstr, "\"","");
  tmpstr=replaceStr(tmpstr, "\'","");

   // remove preceeding blanks
  int beginpos=textbegin(tmpstr,0);
  if(beginpos<0) beginpos=0;
  tmpstr=tmpstr.substr(beginpos,tmpstr.length()-beginpos);

  // check whether we have absolute or relative path
  bool abspath=false;

  STD_string drive;
#ifdef USING_WIN32
  // replace slashes that are left on Windows
  tmpstr=replaceStr(tmpstr,"/",SEPARATOR_STR);
  if(tmpstr.length()>=2 && (tmpstr[1]==':') ) {
    drive=STD_string(1,tmpstr[0])+":"; // store 
    tmpstr=tmpstr.substr(2,tmpstr.length()-2); // and strip off drive letter
    abspath=true; // accept drive-letter notation
  }
#endif

  if(tmpstr.length()>=1 && (tmpstr[0]==SEPARATOR_CHAR) ) abspath=true;

  ODINLOG(odinlog,normalDebug) << "tmpstr/abspath=>" << tmpstr << "</" << abspath << STD_endl;

  // Decompose path
  svector toks=tokens(tmpstr,SEPARATOR_CHAR);
  int ntoks=toks.size();
  ODINLOG(odinlog,normalDebug) << "ntoks/toks=" << ntoks << "/" << toks.printbody() << STD_endl;

  // set suffix, only change/reset suffix if we actually have a file
  if(ntoks) {
    result_suffix="";
    STD_string fname="XXX"+toks[ntoks-1]; // Quick hack in case filename consists only of '.' and the suffix
    svector sufftoks=tokens(fname,'.');
    if(sufftoks.size()>1) result_suffix=tolowerstr(sufftoks[sufftoks.size()-1]);
  }

  if(dir) result_suffix=""; // No suffix for dirs

  // set defaults
  result=drive;
  result_dirname=drive;
  result_basename="";

  if(!abspath && (ntoks==1)) result_dirname="."; // File is in PWD

  // Recompose path and store dirname/basename
  if(ntoks) {

    if(abspath) {
      result+=SEPARATOR_STR;
      result_dirname+=SEPARATOR_STR;
    }

    ODINLOG(odinlog,normalDebug) << "result/result_dirname(pre loop)=>" << result << "<>" << result_dirname << "<" << STD_endl;

    for(int i=0; i<ntoks; i++) {
      result+=toks[i];

      if(i==(ntoks-1)) {
        result_basename=toks[i]; // Store last token in basename
      } else {
        result_dirname+=toks[i];
        result+=SEPARATOR_STR;
        if(i<(ntoks-2)) result_dirname+=SEPARATOR_STR;
      }
    }

    ODINLOG(odinlog,normalDebug) << "result/result_dirname(post loop)=>" << result << "<>" << result_dirname << "<" << STD_endl;
  }

}



#ifndef NO_UNIT_TEST
class JDXfileNameTest : public UnitTest {

 public:
  JDXfileNameTest() : UnitTest("JDXfileName") {}

 private:
  bool check() const {
    Log<UnitTest> odinlog(this,"check");

    JDXfileName teststr1=STD_string("image.bmp");

    STD_string expected="image.bmp";
    STD_string printed=teststr1;
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::STD_string failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }


    expected="bmp";
    printed=teststr1.get_suffix();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::get_suffix()[from filename] failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }


    expected="cpp";
    teststr1.set_suffix(expected);
    printed=teststr1.get_suffix();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::get_suffix()[when set] failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }

    teststr1=STD_string(".bmp");
    expected="bmp";
    printed=teststr1.get_suffix();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::get_suffix()[empty basename] failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }


    teststr1=STD_string("//tmp///test");
    expected=STD_string(SEPARATOR_STR)+"tmp";
    printed=teststr1.get_dirname();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::get_dirname() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }

    teststr1=STD_string("subdir///file");

    expected="subdir"+STD_string(SEPARATOR_STR)+"file";
    printed=teststr1;
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::STD_string failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }

    expected="file";
    printed=teststr1.get_basename();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::get_basename() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }

    expected="subdir";
    printed=teststr1.get_dirname();
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName::get_dirname() failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }


#ifdef USING_WIN32
    teststr1=STD_string("C:");
    expected="C:";
    printed=STD_string(teststr1);
    if(printed!=expected) {
      ODINLOG(odinlog,errorLog) << "JDXfileName(WIN32): failed: got >" << printed << "<, but expected >" << expected << "<" << STD_endl;
      return false;
    }
#endif

    return true;
  }

};

void alloc_JDXfileNameTest() {new JDXfileNameTest();} // create test instance
#endif


///////////////////////////////////////////////////////////////

JDXformula::JDXformula(const STD_string& formula, const STD_string& name, bool userParameter, compatMode mode, parameterMode parameter_mode)
 : JDXstring(formula,name,userParameter,mode,parameter_mode) {
}

JDXformula& JDXformula::operator = (const JDXformula& jf) {
  JDXstring::operator = (jf);
  syntax = jf.syntax;
  return *this;
}

