#include "sample.h"

Sample::Sample(const STD_string& label, bool uniformFOV, bool uniformT1T2) : JcampDxBlock(label) {

  uniFOV=uniformFOV;
  uniT1T2=uniformT1T2;

  resize(1,1,1,1);

  FOVall=20.0; FOVall.set_minmaxval(0.0,ODIN_DEFAULT_FOV).set_unit(ODIN_SPAT_UNIT);

  FOV[xAxis]=20.0;
  FOV[yAxis]=20.0;
  FOV[zAxis]=20.0;
  FOV.set_unit(ODIN_SPAT_UNIT);

  offset.set_unit(ODIN_SPAT_UNIT);

  freqrange=10.0; freqrange.set_minmaxval(0.0,50.0).set_unit(ODIN_FREQ_UNIT);

  freqoffset=0.0; freqoffset.set_minmaxval(-100.0,100.0).set_unit(ODIN_FREQ_UNIT);


  T1=0.0; T1.set_minmaxval(0.0,2000.0).set_unit(ODIN_TIME_UNIT);
  T2=0.0; T2.set_minmaxval(0.0,500.0).set_unit(ODIN_TIME_UNIT);

  T1map.set_filemode(compressed);       T1map.set_parmode(hidden);
  T2map.set_filemode(compressed);       T2map.set_parmode(hidden);
  ppmMap.set_filemode(compressed);      ppmMap.set_parmode(hidden);
  spinDensity.set_filemode(compressed); spinDensity.set_parmode(hidden);
  B1map.set_filemode(compressed);       B1map.set_parmode(hidden);
  DcoeffMap.set_filemode(compressed);   DcoeffMap.set_parmode(hidden);

  Sample::append_all_members();
}




Sample::Sample(const Sample& ss) {
  Sample::operator = (ss);
}


Sample& Sample::operator = (const Sample& ss) {
  JcampDxBlock::operator = (ss);

  FOVall=ss.FOVall;
  FOV=ss.FOV;
  uniFOV=ss.uniFOV;
  offset=ss.offset;

  freqrange=ss.freqrange;
  freqoffset=ss.freqoffset;

  uniT1T2=ss.uniT1T2;
  T1=ss.T1;
  T2=ss.T2;
  T1map=ss.T1map;
  T2map=ss.T2map;
  haveT1map=ss.haveT1map;
  haveT2map=ss.haveT2map;

  ppmMap=ss.ppmMap;
  have_ppmMap=ss.have_ppmMap;

  spinDensity=ss.spinDensity;
//  have_spinDensity=ss.have_spinDensity;

  B1map=ss.B1map;
  have_B1map=ss.have_B1map;

  DcoeffMap=ss.DcoeffMap;
  have_DcoeffMap=ss.have_DcoeffMap;

  Sample::append_all_members();

  return *this;
}


Sample& Sample::set_FOV(float fov) {
  FOVall=fov;
  FOV.farray::operator = (fov);
  uniFOV=true;
  return *this;
}

Sample& Sample::set_FOV(axis direction, float fov) {
  FOV[direction]=fov;
  uniFOV=false;
  return *this;
}

float Sample::get_FOV(axis direction) const {
  if(uniFOV) return FOVall;
  return FOV[direction];
}


Sample& Sample::resize(unsigned int xsize, unsigned int ysize,  unsigned int zsize, unsigned int freqsize) {
  Log<Para> odinlog(this,"resize");

  haveT1map=false;
  haveT2map=false;
  have_ppmMap=false;
  have_B1map=false;
  have_DcoeffMap=false;


  ndim newsize(4);
  newsize[freqDim]=freqsize;
  newsize[zDim]=zsize;
  newsize[yDim]=ysize;
  newsize[xDim]=xsize;

  ODINLOG(odinlog,normalDebug) << "newsize=" << newsize << STD_endl;
  ODINLOG(odinlog,normalDebug) << "spinDensity.get_extent()=" << spinDensity.get_extent() << STD_endl;

  // leave spinDensity untouched if it has already the right size (required for load)
  if(newsize!=spinDensity.get_extent()) {
    ODINLOG(odinlog,normalDebug) << "resizing" << STD_endl;
    spinDensity.redim (newsize);
    spinDensity.farray::operator = (1.0);
  }


  ODINLOG(odinlog,normalDebug) << "extent=" << get_extent() << STD_endl;
  return *this;
}


const ndim& Sample::get_extent() const {
  return spinDensity.get_extent();
}


Sample& Sample::set_T1map(const farray& t1map) {
  if(uniT1T2) return *this;
  if(get_extent()!=t1map.get_extent()) return *this;
  T1map=t1map;
  haveT1map=true;
  return *this;
}

const farray& Sample::get_T1map() const {
  if(!haveT1map) {
    T1map.redim(get_extent());
    T1map.farray::operator = (T1);
    haveT1map=true;
  }
  return T1map;
}


Sample& Sample::set_T2map(const farray& t2map) {
  if(uniT1T2) return *this;
  if(get_extent()!=t2map.get_extent()) return *this;
  T2map=t2map;
  haveT2map=true;
  return *this;
}

const farray& Sample::get_T2map() const {
  if(!haveT2map) {
    T2map.redim(get_extent());
    T2map.farray::operator = (T2);
    haveT2map=true;
  }
  return T2map;
}

Sample& Sample::set_ppmMap(const farray& ppmmap) {
  if(get_extent()==ppmmap.get_extent()) {
    ppmMap=ppmmap;
    have_ppmMap=true;
  }
  return *this;
}

const farray& Sample::get_ppmMap() const {
  Log<Para> odinlog(this,"get_ppmMap");
  ODINLOG(odinlog,normalDebug) << "get_extent()=" << get_extent() << STD_endl;
  if(!have_ppmMap) {
    ppmMap.redim(get_extent());
    ppmMap.farray::operator = (0.0);
    have_ppmMap=true;
  }
  ODINLOG(odinlog,normalDebug) << "ppmMap.get_extent()=" << ppmMap.get_extent() << STD_endl;
  return ppmMap;
}


Sample& Sample::set_spinDensity(const farray& sd) {
  if(get_extent()!=sd.get_extent()) return *this;
  spinDensity=sd;
  return *this;
}

const farray& Sample::get_spinDensity() const {
  return spinDensity;
}


Sample& Sample::set_B1map(const carray& b1map) {
  if(get_extent()!=b1map.get_extent()) return *this;
  B1map=b1map;
  have_B1map=true;
  return *this;
}

const carray& Sample::get_B1map() const {
  if(!have_B1map) {
    B1map.redim(get_extent());
    B1map.carray::operator = (STD_complex(1.0));
    have_B1map=true;
  }
  return B1map;
}

Sample& Sample::set_DcoeffMap(const farray& dmap) {
  if(get_extent()!=dmap.get_extent()) return *this;
  DcoeffMap=dmap;
  have_DcoeffMap=true;
  return *this;
}

const farray& Sample::get_DcoeffMap() const {
  if(!have_DcoeffMap) {
    DcoeffMap.redim(get_extent());
    DcoeffMap.farray::operator = (0.0);
    have_DcoeffMap=true;
  }
  return DcoeffMap;
}


Sample& Sample::update() {

  // outdate cache in interactive mode
  if(uniT1T2) {
    haveT1map=false;
    haveT2map=false;
  }

  if(T1!=0.0 || T2!=0.0) {
    if(T1>0.0 && T2>T1) T2=double(T1);
  }
  return *this;
}


int Sample::load(const STD_string& filename) {
  Log<Para> odinlog(this,"load");
  int errval=JcampDxBlock::load(filename);

  ndim nn=spinDensity.get_extent();
  ODINLOG(odinlog,normalDebug) << "nn=" << nn << STD_endl;
  ODINLOG(odinlog,normalDebug) << "spinDensity.sum()=" << spinDensity.sum() << STD_endl;
  if(nn.dim()!=4 || nn.total()==0) {
    ODINLOG(odinlog,errorLog) << "spinDensity has invalid extent=" << STD_string(nn) << STD_endl;
    return -1;
  }  
  uniFOV=false;
  uniT1T2=false;

  ODINLOG(odinlog,normalDebug) << "resizing to " << STD_string(nn) << STD_endl;
  resize(nn[3],nn[2],nn[1],nn[0]);

  ODINLOG(odinlog,normalDebug) << "ppmMap.get_extent()=" << STD_string(ppmMap.get_extent()) << STD_endl;

  if(nn==T1map.get_extent())     haveT1map=true;      else haveT1map=false;
  if(nn==T2map.get_extent())     haveT2map=true;      else haveT2map=false;
  if(nn==ppmMap.get_extent())    have_ppmMap=true;    else have_ppmMap=false;
  if(nn==B1map.get_extent())     have_B1map=true;     else have_B1map=false;
  if(nn==DcoeffMap.get_extent()) have_DcoeffMap=true; else have_DcoeffMap=false;

  ODINLOG(odinlog,normalDebug) << "have_ppmMap=" << have_ppmMap << STD_endl;

  return errval;
}


int Sample::append_all_members() {

  if(uniFOV) {
    append_member(FOVall,"FOVall");
  } else {
    append_member(FOV,"FOV");
  }
  append_member(freqrange,"FrequencyRange");
  append_member(freqoffset,"FrequencyOffset");

  append_member(T1,"RelaxationT1");
  append_member(T2,"RelaxationT2");
  append_member(T1map,"T1map");
  append_member(T2map,"T2map");
  append_member(ppmMap,"ppmMap");
  append_member(spinDensity,"spinDensity");
  append_member(B1map,"B1map");
  append_member(DcoeffMap,"DcoeffMap");

  return 0;
}
