#ifndef OSL_MILLISECONDS_H
#define OSL_MILLISECONDS_H

#include "osl/misc/cstdint.h"
#include <string>
#include <stdexcept>
#include <cassert>
#include <limits>

namespace osl
{
namespace misc
{
  struct NoMoreTime : std::runtime_error
  {
    NoMoreTime() : std::runtime_error("time limit over")
    {
    }
  };

  class MilliSeconds
  {
    int64_t msec;
  public:
    class Interval
    {
      int64_t interval;
    public:
      explicit Interval(int64_t m=std::numeric_limits<int64_t>::max()) : interval(m) {}
      int64_t value() const { return interval; }
      double toSeconds() const { return interval/1000.0; }
      static const Interval infinity() 
      {
	return Interval(std::numeric_limits<int64_t>::max());
      }
      bool isInfinity() const { return interval == std::numeric_limits<int64_t>::max(); }
      const Interval operator+(Interval r) const { return Interval(interval + r.interval); }
      const Interval operator-(Interval r) const { return Interval(interval - r.interval); }
      const Interval operator*(int scale) const { return Interval(interval*scale); }
      const Interval operator/(int scale) const { return Interval(interval/scale); }
    };
    explicit MilliSeconds(int64_t ms=0) : msec(ms) {}
    int64_t value() const { return msec; }
    const MilliSeconds operator+(int64_t diff) const { return MilliSeconds(value()+diff); }
    const MilliSeconds operator-(int64_t diff) const { return MilliSeconds(value()-diff); }
    const MilliSeconds operator+(Interval diff) const { return operator+(diff.value()); }
    const MilliSeconds operator-(Interval diff) const { return operator-(diff.value()); }
    const Interval operator-(MilliSeconds r) const
    {
      return Interval(value() - r.value());
    }
    static const MilliSeconds now();
    double elapsedSeconds() const { return (now() - *this).toSeconds(); }
  };
  inline bool operator==(MilliSeconds l, MilliSeconds r)
  {
    return l.value() == r.value();
  }
  inline bool operator<(MilliSeconds l, MilliSeconds r)
  {
    return l.value() < r.value();
  }
  inline bool operator==(MilliSeconds::Interval l, MilliSeconds::Interval r)
  {
    return l.value() == r.value();
  }
  inline bool operator<(MilliSeconds::Interval l, MilliSeconds::Interval r)
  {
    return l.value() < r.value();
  }
} // namespace misc
  using misc::MilliSeconds;
} // namespace osl


#endif // OSL_MILLISECONDS_H
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
