(** A header object for columnar structures **)

MODULE VO:Header;

(*
    A header object .
    Copyright (C) 2000  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT A   := VO:Base:Adjustment,
       D   := VO:Base:Display,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       HM  := VO:Model:Header,

       G   := VO:Object,
       T   := VO:Text;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 headerFrame* : LONGINT;
               END;

  Objects     = POINTER TO ARRAY OF G.Object;

  Header*     = POINTER TO HeaderDesc;
  HeaderDesc* = RECORD (G.GadgetDesc)
                  frame      : F.Frame;
                  model-     : HM.HeaderModel;
                  objects    : Objects;

                  adjustment : A.Adjustment;
               END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.none;
    p.headerFrame:=F.double3DOut;
  END Init;

  PROCEDURE (h : Header) Init*;

  BEGIN
    h.Init^;

    h.SetPrefs(prefs);
    
    h.frame:=F.CreateFrame();
    h.frame.SetFrame(h.prefs(Prefs).headerFrame);

    h.model:=NIL;
    h.objects:=NIL;
    h.adjustment:=NIL;
  END Init;

  PROCEDURE (h : Header) SetAdjustment*(adjustment : A.Adjustment);

  BEGIN
    h.adjustment:=adjustment;
    h.AttachModel(h.adjustment.GetTopModel());
  END SetAdjustment;

  PROCEDURE (h : Header) HandleObjects;

  VAR
    x    : LONGINT;
    text : U.Text;

  BEGIN
    IF h.model=NIL THEN
      h.objects:=NIL;
    ELSE
      NEW(h.objects,h.model.GetEntries());

      FOR x:=0 TO h.model.GetEntries()-1 DO
        text:=h.model.GetEntryText(x);
        IF text=NIL THEN
          h.objects[x]:=T.MakeCenterText("");
        ELSE
          h.objects[x]:=T.MakeCenterText(text^);
        END;
        h.objects[x].SetParent(h);
        h.objects[x].SetBackground(D.buttonBackgroundColor);
        h.objects[x].CalcSize;
        IF h.model.GetEntrySize(x)=0 THEN
          h.model.SetEntrySize(x,h.objects[x].oWidth+h.frame.minWidth);
        END;
      END;
    END;
  END HandleObjects;

  PROCEDURE (h : Header) CalcSize*;

  VAR
    font : D.Font;
    x    : LONGINT;

  BEGIN
    IF h.model=NIL THEN
      font:=D.normalFont;

      h.width:=D.display.spaceWidth*10;
      h.height:=font.height+h.frame.minHeight+D.display.spaceHeight DIV 4;
    ELSE
      h.HandleObjects;

      h.width:=D.display.spaceWidth*10;
      h.height:=0;
      FOR x:=0 TO LEN(h.objects^)-1 DO
        h.height:=U.MaxLong(h.height,h.objects[x].oMinHeight);
      END;
      INC(h.height,h.frame.minHeight+D.display.spaceHeight DIV 2);
    END;
    h.minWidth:=h.width;
    h.minHeight:=h.height;

    h.CalcSize^;
  END CalcSize;

  (**
    Draw the column headers
  **)

  PROCEDURE (h : Header) DrawHeader(draw : D.DrawInfo);

  VAR
    x,
    count,
    size  : LONGINT;

  BEGIN
    IF h.model#NIL THEN
      (* Draw header *)
      draw.InstallClip(h.x,h.y,h.width,h.height);

      x:=h.x;
      IF h.adjustment#NIL THEN
        DEC(x,h.adjustment.GetTop()-1);
      END;
      size:=0;
      FOR count:=0 TO LEN(h.objects^)-1 DO
        IF h.objects[count]#NIL THEN
            h.objects[count].Resize(h.model.GetEntrySize(count)-h.frame.minWidth,
                                    h.height-h.frame.minHeight);
          IF ~((x>=h.x+h.width)
          OR (x+h.model.GetEntrySize(count)+h.frame.minWidth<h.x)) THEN
            h.objects[count].Move(x+h.frame.leftBorder,
                                  h.y+(h.height-h.objects[count].oHeight) DIV 2);
            h.objects[count].Draw(h.oX,h.oY,h.oWidth,h.oHeight);
            h.frame.Draw(draw,
                         x,h.y,
                         h.model.GetEntrySize(count),
                         h.height);
          END;
        END;
        INC(x,h.model.GetEntrySize(count));
        INC(size,h.model.GetEntrySize(count));
      END;

      IF h.adjustment#NIL THEN
        h.adjustment.SetDimension(h.width,size);
      END;

      IF x<h.x+h.width THEN
        (* fill space behind last header *)
        h.DrawBackground(x,h.y,h.width-(x-h.x),h.height);
      END;
      draw.FreeLastClip;
    ELSE
      h.DrawBackground(h.x,h.y,h.width,h.height);
    END;
  END DrawHeader;

  (**
    The the table model for  the table object.
  **)

  PROCEDURE (h : Header) SetModel*(model : O.Model);

  BEGIN
    IF h.model#NIL THEN
      h.UnattachModel(h.model);
    END;
    IF (model#NIL) & (model IS HM.HeaderModel) THEN
      h.model:=model(HM.HeaderModel);
      h.AttachModel(model);
    ELSE
      h.model:=NIL
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (h : Header) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=h.model
  END ModelAccepted;

  PROCEDURE (q : Header) Draw*(x,y,w,h : LONGINT);

  BEGIN
    q.Draw^(x,y,w,h);

    IF ~q.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    q.DrawHeader(q.GetDrawInfo());
  END Draw;

  PROCEDURE (h : Header) Hide*;

  BEGIN
    IF h.visible THEN
      h.DrawHide;
      h.Hide^;
    END;
  END Hide;

  PROCEDURE (h : Header) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF (h.adjustment#NIL) & (model=h.adjustment.GetTopModel()) THEN
      IF h.visible & ~h.disabled THEN
        h.DrawHeader(h.GetDrawInfo());
      END;
    END;
  END Resync;

  PROCEDURE CreateHeader*():Header;

  VAR
    header : Header;

  BEGIN
    NEW(header);
    header.Init;

    RETURN header;
  END CreateHeader;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:Header.