/*
    Copyright (C) 2000 Steve Brown 
	Copyright (C) 2000,2001 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: shadow.c,v 1.47 2002/01/15 11:27:14 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/shadow.c,v $
    $Date: 2002/01/15 11:27:14 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <nss.h>
#include <mysql/mysql.h>
#include <string.h>
#include <shadow.h>
#include <time.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <pthread.h>

#ifdef HAVE_CONFIG_H
	#include "config.h"
#endif

#ifdef INCLUDE_MY_SYS
	#include <mysql/my_pthread.h>
#endif

#include "lib.h"
#include "nss-shadow.h"
#include "parser.h"
#include "ent.h"

#if USE_SHADOW

#define ENT_TYPE 2

enum nss_status _nss_mysql_getspnam_r (const char *, struct spwd *,char *, size_t,int *);
enum nss_status _nss_mysql_setspent (void);
enum nss_status _nss_mysql_endspent (void);
enum nss_status _nss_mysql_getspent_r (struct spwd *result,char *buffer, size_t buflen, int *errnop);
static int check_connection(MYSQL ** mysql_auth,struct shadowoptions * options,pthread_mutex_t * mutex);


#define clean_quit()  \
if (sql) free(sql); \
if (result) mysql_free_result(result); \
*errnop=ENOENT

#define check_mem(x) if ( (x) == NULL ) \
{ \
	if (DEBUG) _nss_mysql_log(LOG_ERR,"not enough memory"); \
	clean_quit(); \
	*errnop = EAGAIN; \
	return NSS_STATUS_TRYAGAIN; \
}


/* _nss_mysql_shadow_fill_struct
 * Get an shadow entry by name
 * name: name of the user
 * secure_name: space to escape uname
 * spw: struct we're gonna fill
 * buffer: buffer, unused
 * buflen: buffer size
 * errnop: ptr of the application errno
 */

#define lockm() \
if (m) pthread_mutex_lock(m);

#define unlockm() \
if (m) pthread_mutex_unlock(m);

enum nss_status _nss_mysql_shadow_fill_struct(const char * name, char * secure_name, int user_id, struct spwd *spw,
				int * errnop,MYSQL * mysql_auth,struct shadowoptions * soptions,pthread_mutex_t *m) {
	char * sql = NULL;
	MYSQL_RES *result = NULL;
	MYSQL_ROW sql_row;
	int i,error;
	
	*errnop = ENOENT;
	
	if (name) {
#ifndef HAVE_MYSQL_REAL_ESCAPE_STRING
		mysql_escape_string(secure_name,name,strlen(name));
#else
		mysql_real_escape_string(mysql_auth,secure_name,name,strlen(name));
#endif
	}
	
	if (name) {
		sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s,%s from %s where %s='%s' and %s",
			soptions->passwdcolumn,
			soptions->usercolumn,
			soptions->lastchange,
			soptions->min,
			soptions->max,
			soptions->warn,
			soptions->inact,
			soptions->expire,
			soptions->table,
			soptions->usercolumn,
			secure_name,
			soptions->where[0] ? soptions->where : "1=1"
			);
	} else {
    	sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s,%s from %s "
				                   "where %s=%d and %s",
			soptions->passwdcolumn,
			soptions->usercolumn,
			soptions->lastchange,
			soptions->min,
			soptions->max,
			soptions->warn,
			soptions->inact,
			soptions->expire,
			soptions->table,
			soptions->useridcolumn,
			user_id,
			soptions->where[0] ? soptions->where : "1=1"
			);
	}
	check_mem(sql);
	if (DEBUG) _nss_mysql_log(LOG_ERR,"getspnam: SQL statement: %s",sql);
	lockm();
	if (mysql_query(mysql_auth, sql)) {
		_nss_mysql_log(LOG_ERR,"getspnam: mysql_query failed: %s",mysql_error(mysql_auth));
		clean_quit();
		unlockm();
		return NSS_STATUS_UNAVAIL;
	}

	result = mysql_store_result(mysql_auth);
	if (! result) {
		_nss_mysql_log(LOG_ERR,"getspnam: mysql_store_result failed: %s",mysql_error(mysql_auth));
		clean_quit();
		unlockm();
		return NSS_STATUS_UNAVAIL;
	}
	unlockm();
	i = mysql_num_rows(result);
	if (i == 1) {
		/* normal behavior */
		sql_row = mysql_fetch_row(result);
		if (!sql_row) {
			_nss_mysql_log(LOG_ERR,"getspnam: mysql_fetch_row failed: %s.",
					       mysql_error(mysql_auth));
			clean_quit();
			return NSS_STATUS_UNAVAIL;
		}
		if (name) 
	    	spw->sp_namp = strdup(name);
		else {
			if(_nss_mysql_isempty(sql_row[1])) {
				_nss_mysql_log(LOG_ERR,"_nss_mysql_shadow_fill_struct: empty user name field for user_id %d. "
						               "Fix your database.",
							   user_id);
				clean_quit();
				return NSS_STATUS_UNAVAIL;
			}
			spw->sp_namp = strdup(sql_row[1]);
		}
		check_mem(spw->sp_namp);
			
		if (_nss_mysql_isempty(sql_row[0])) {
			_nss_mysql_log(LOG_ERR,"getspnam: Password field for %s is empty or NULL. "
					               "Fix your database.",name);
			clean_quit();
			return NSS_STATUS_UNAVAIL;
		}
		spw->sp_pwdp = strdup(sql_row[0]);
		check_mem(spw->sp_pwdp);

		spw->sp_lstchg = _nss_mysql_strtol(sql_row[2],time(NULL)-24*3600,&error);
		if (error)
			_nss_mysql_log(LOG_ERR,"getspnam: lastchange field empty for %s. "
					               "Reverting to 'yesterday. Fix your database",
								   spw->sp_namp);
		spw->sp_min = _nss_mysql_strtol(sql_row[3],1,&error);
		if (error)
			_nss_mysql_log(LOG_ERR,"getspnam: min field empty for %s. "
					               "Reverting to 1. Fix your database",
								   spw->sp_namp);
		spw->sp_max = _nss_mysql_strtol(sql_row[4],2,&error);
		if (error)
			_nss_mysql_log(LOG_ERR,"getspnam: max field empty for %s. "
					               "Reverting to 2. Fix your database",
								   spw->sp_namp);
		spw->sp_warn = _nss_mysql_strtol(sql_row[5],7,&error);
		if (error)
			_nss_mysql_log(LOG_ERR,"getspnam: warn field empty for %s. "
					               "Reverting to 7. Fix your database",
								   spw->sp_namp);
		
		spw->sp_inact = _nss_mysql_strtol(sql_row[6],-1,&error);
		if (error)
			_nss_mysql_log(LOG_ERR,"getspnam: inact field empty for %s. "
					               "Reverting to -1. Fix your database",
								   spw->sp_namp);
    	spw->sp_expire = _nss_mysql_strtol(sql_row[7],-1,&error);
		if (error)
			_nss_mysql_log(LOG_ERR,"getspnam: expire field empty for %s. "
					               "Reverting to -1. Fix your database",
								   spw->sp_namp);
		spw->sp_flag = -1; /* unused */
		clean_quit();
		*errnop = 0;
		return NSS_STATUS_SUCCESS;
	} 
	
	if (i) {
		_nss_mysql_log(LOG_ERR,"getspnam has found %d matching rows for "
				               "user %s. Fix your database.",i,name);
	} else {
		if (DEBUG) _nss_mysql_log(LOG_ERR,"user %s not found",name);
	}
	clean_quit();
	return NSS_STATUS_NOTFOUND;
}


/* check_connection 
 * checks if a connection has been opened and if it is still alive
 * tries to open a connection if not
 */

int check_connection(MYSQL ** mysql_auth,struct shadowoptions * options,pthread_mutex_t * mutex) {
	/* Is the server still alive ? */
	pthread_mutex_lock(mutex);
	if (*mysql_auth != NULL) {
		my_thread_init();
		if (mysql_ping(*mysql_auth)) {
			if (DEBUG) _nss_mysql_log(LOG_ERR,"check_connection: can't sustain connection : %s",
					mysql_error(*mysql_auth));
			*mysql_auth = NULL;
		}
	}
	
	/* DB connection */
	if (*mysql_auth == NULL) {
		if  (! _nss_mysql_db_connect(mysql_auth,options->host,options->dbuser,options->dbpasswd,options->database)) {
			pthread_mutex_unlock(mutex);
			my_thread_end();
			return 0;
		}
	} 
	pthread_mutex_unlock(mutex);
	return 1;
}


/* getspnam
 * looks for an user by its name
 * Arguments:
 * uid: user's uid
 * spw: struct we'll fill
 * buf: buffer (not used)
 * buflen: sizeof(buffer)
 * errnop: ptr on the application errno
 */


#define getsp_quit() \
_nss_mysql_free_shadow(&soptions); 

static MYSQL * nam_auth = NULL;
static pthread_mutex_t nam_mutex = PTHREAD_MUTEX_INITIALIZER;

enum nss_status _nss_mysql_getspnam_r (const char * name, struct spwd *spw,
                char *buf, size_t buflen,int * errnop) {
	enum nss_status toreturn = NSS_STATUS_UNAVAIL;
	char * buffer;	
	struct shadowoptions soptions;

	memset(&soptions,0,sizeof(struct shadowoptions));
	
	if (DEBUG) _nss_mysql_log(LOG_ERR,"getspnam called for %s\n", name);
	*errnop = ENOENT;		

	if (!name) return NSS_STATUS_NOTFOUND;

	
	/* we parse the conf file */
	if (!_nss_mysql_read_conf_file("shadow",NULL,NULL,&soptions)) {
		_nss_mysql_free_shadow(&soptions);
		_nss_mysql_log(LOG_ERR,"getspnam: cannot parse conf file");
		return NSS_STATUS_UNAVAIL;
	}

	/* DB connection */
	if (! check_connection(&nam_auth,&soptions,&nam_mutex)) {
		_nss_mysql_free_shadow(&soptions);
		return NSS_STATUS_UNAVAIL;
	}
	
	if (spw) {
		buffer = malloc(strlen(name) * 2 + 1);
		if (buffer == NULL) {
			*errnop = EAGAIN;
			getsp_quit();
			my_thread_end();
			return NSS_STATUS_TRYAGAIN;
		}
		toreturn = _nss_mysql_shadow_fill_struct(name, buffer, -1,spw,errnop,nam_auth,&soptions,&nam_mutex);
		free(buffer);
	}
	getsp_quit();
	my_thread_end();
	return toreturn;
}

/* setspent
 * Initializes data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */

enum nss_status _nss_mysql_setspent (void) {
	return _nss_mysql_setent(ENT_TYPE);
}

/* endspent
 * Kills all data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */
enum nss_status _nss_mysql_endspent (void) {
	return _nss_mysql_endent(ENT_TYPE);
}

/* getspent
 * Gets info for every group
 * Arguments:
 * pw: array to fill
 * buffer: buffer, unused 
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

enum nss_status _nss_mysql_getspent_r (struct spwd *spw,
                char * buffer, size_t buflen,int * errnop) {
	return _nss_mysql_getent_r(ENT_TYPE,spw,buffer,buflen,errnop);
}

#endif /* #if USE_SHADOW */

