/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * $Id: MaxFunction.java 3521 2007-10-16 10:55:14Z tmorgner $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */
package org.jfree.formula.function.math;

import java.math.BigDecimal;

import org.jfree.formula.EvaluationException;
import org.jfree.formula.FormulaContext;
import org.jfree.formula.LibFormulaErrorValue;
import org.jfree.formula.util.NumberUtil;
import org.jfree.formula.function.Function;
import org.jfree.formula.function.ParameterCallback;
import org.jfree.formula.lvalues.TypeValuePair;
import org.jfree.formula.typing.Type;
import org.jfree.formula.typing.sequence.NumberSequence;
import org.jfree.formula.typing.coretypes.NumberType;

/**
 * This function returns the maximum from a set of numbers.
 *
 * @author Cedric Pronzato
 *
 */
public class MaxFunction implements Function
{
  public static final BigDecimal ZERO = new BigDecimal(0);
  
  public MaxFunction()
  {
  }

  public TypeValuePair evaluate(final FormulaContext context, final ParameterCallback parameters) throws EvaluationException
  {
    final int parameterCount = parameters.getParameterCount();

    if(parameterCount == 0)
    {
      return new TypeValuePair(NumberType.GENERIC_NUMBER, ZERO);
    }

    BigDecimal last = null;
    for (int paramIdx = 0; paramIdx < parameterCount; paramIdx++)
    {
      final Type type = parameters.getType(paramIdx);
      final Object value = parameters.getValue(paramIdx);
      final NumberSequence sequence = context.getTypeRegistry().convertToNumberSequence(type, value);

      while(sequence.hasNext())
      {
        final BigDecimal next = NumberUtil.getAsBigDecimal(sequence.nextNumber());
        if(last == null)
        {
          last = next;
        }
        else
        {
          if(last.compareTo(next) == -1)
          {
            last = next;
          }
        }
      }
    }
    if(last == null)
    {
      throw new EvaluationException(
          LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
    }

    return new TypeValuePair(NumberType.GENERIC_NUMBER, last);
  }

  public String getCanonicalName()
  {
    return "MAX";
  }

}