/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>


int tt_cmp(TT *tt0, TT *tt1)
{
  int r = 0;
  int count, max;
  byte *block0, *block1;

  if (!tt_has_data(tt0) || !tt_has_data(tt1)) return(-1);
  
  max = tt_size(tt0);
  if (max != tt_size(tt1)) return(-1);
  
  if (tt_data_is_internal(tt0) && tt_data_is_internal(tt1))
    return(memcmp(tt0->data, tt1->data, max));

  block0 = malloc(TT_GENERIC_BLOCK_SIZE);
  block1 = malloc(TT_GENERIC_BLOCK_SIZE);

  for (max = tt_size(tt0), count = 0; count + TT_GENERIC_BLOCK_SIZE <= max;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt0, block0, count, TT_GENERIC_BLOCK_SIZE);
    tt_data_get_bytes(tt1, block1, count, TT_GENERIC_BLOCK_SIZE);
    r = memcmp(block0, block1, TT_GENERIC_BLOCK_SIZE);
    if (r) break;
  }

  if (!r && count < max)
  {
    tt_data_get_bytes(tt0, block0, count, max - count);
    tt_data_get_bytes(tt1, block1, count, max - count);
    r = memcmp(block0, block1, max - count);
  }
  
  free(block0);
  free(block1);
  return(r);
}


int tt_casecmp(TT *tt0, TT *tt1)
{
  int r = 0;
  int count, max;
  byte *block0, *block1;

  if (!tt_has_data(tt0) || !tt_has_data(tt1)) return(-1);
  
  max = tt_size(tt0);
  if (max != tt_size(tt1)) return(1);

  if (tt_data_is_internal(tt0) && tt_data_is_internal(tt1))
    return(memcasecmp(tt0->data, tt1->data, max));

  block0 = malloc(TT_GENERIC_BLOCK_SIZE);
  block1 = malloc(TT_GENERIC_BLOCK_SIZE);

  for (max = tt_size(tt0), count = 0; count + TT_GENERIC_BLOCK_SIZE <= max;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt0, block0, count, TT_GENERIC_BLOCK_SIZE);
    tt_data_get_bytes(tt1, block1, count, TT_GENERIC_BLOCK_SIZE);
    r = memcasecmp(block0, block1, TT_GENERIC_BLOCK_SIZE);
    if (r) break;
  }

  if (!r && count < max)
  {
    tt_data_get_bytes(tt0, block0, count, max - count);
    tt_data_get_bytes(tt1, block1, count, max - count);
    r = memcasecmp(block0, block1, max - count);
  }
  
  free(block0);
  free(block1);
  return(r);
}


int tt_memcmp(TT *tt, void *p, u32 len)
{
  int r = 0;
  int count;
  byte *block;
  
  if (!tt_has_data(tt)) return(-1);
  if (tt_size(tt) != len) return(1);
  
  if (tt_data_is_internal(tt)) return(memcmp(tt->data, p, len));
  
  block = malloc(TT_GENERIC_BLOCK_SIZE);
  
  for (count = 0; count + TT_GENERIC_BLOCK_SIZE <= len;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block, count, TT_GENERIC_BLOCK_SIZE);
    r = memcmp(block, p + count, TT_GENERIC_BLOCK_SIZE);
    if (r) break;
  }

  if (!r && count < len)
  {
    tt_data_get_bytes(tt, block, count, len - count);
    r = memcmp(block, p + count, len - count);
  }
  
  free(block);
  return(r);
}


int tt_memcasecmp(TT *tt, void *p, u32 len)
{
  int r = 0;
  int count;
  byte *block;
  
  if (!tt_has_data(tt)) return(-1);
  if (tt_size(tt) != len) return(1);
  
  if (tt_data_is_internal(tt)) return(memcasecmp(tt->data, p, len));
  
  block = malloc(TT_GENERIC_BLOCK_SIZE);
  
  for (count = 0; count + TT_GENERIC_BLOCK_SIZE <= len;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block, count, TT_GENERIC_BLOCK_SIZE);
    r = memcasecmp(block, p + count, TT_GENERIC_BLOCK_SIZE);
    if (r) break;
  }

  if (!r && count < len)
  {
    tt_data_get_bytes(tt, block, count, len - count);
    r = memcasecmp(block, p + count, len - count);
  }
  
  free(block);
  return(r);
}


int tt_regcmp_precomp(TT *tt, regex_t *preg)
{
  int r = -1;
  int count;
  int firsthalf;
  byte *block;
  
  if (!tt_has_data(tt)) return(-1);

  if (tt_data_is_internal(tt))
  {
    block = malloc(tt_size(tt) + 1);
    memcpy(block, tt_data_get(tt), tt_size(tt));
    *(block + tt_size(tt)) = 0;
    
    r = regexec(preg, block, 0, 0, 0);
    free(block);
    
    if (r) r = -1;
    return(r);
  }

  block = malloc((TT_GENERIC_BLOCK_SIZE << 1) + 1);
  *(block + (TT_GENERIC_BLOCK_SIZE << 1)) = 0;

  firsthalf = count = tt_data_get_bytes(tt, block, 0,
                                        tt_size(tt) < TT_GENERIC_BLOCK_SIZE ?
                                        tt_size(tt) - 1 : TT_GENERIC_BLOCK_SIZE);

  for ( ; count + TT_GENERIC_BLOCK_SIZE <= tt_size(tt); count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block + TT_GENERIC_BLOCK_SIZE, count, TT_GENERIC_BLOCK_SIZE);
    r = regexec(preg, block, 0, 0, 0);
    if (!r) break;
    memcpy(block, block + TT_GENERIC_BLOCK_SIZE, TT_GENERIC_BLOCK_SIZE);
  }

  if (r && count < tt_size(tt))
  {
    firsthalf += tt_data_get_bytes(tt, block + firsthalf, count, tt_size(tt) - count);
    *(block + firsthalf) = 0;
    r = regexec(preg, block, 0, 0, 0);
  }

  free(block);
  if (r) r = -1;
  return(r);
}


int tt_regcmp(TT *tt, const char *regex)
{
  regex_t preg;
  int r;
  
  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(r);
  
  r = tt_regcmp_precomp(tt, &preg);
  regfree(&preg);
  return(r);
}


int tt_regcasecmp(TT *tt, const char *regex)
{
  regex_t preg;
  int r;
  
  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB | REG_ICASE);
  if (r) return(r);
  
  r = tt_regcmp_precomp(tt, &preg);
  regfree(&preg);
  return(r);
}


size_t tt_spn(TT *tt, const char *accept)
{
  int r = -1;
  size_t n;
  int count;
  unsigned int len = tt_size(tt);
  byte *block;

  if (!tt_has_data(tt)) return(-1);
  
  if (tt_data_is_internal(tt))
    return(memspn(tt_data_get(tt), len, accept));

  block = malloc(TT_GENERIC_BLOCK_SIZE);

  for (count = 0; count + TT_GENERIC_BLOCK_SIZE <= len;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block, count, TT_GENERIC_BLOCK_SIZE);
    n = memspn(block, TT_GENERIC_BLOCK_SIZE, accept);

    if (n < TT_GENERIC_BLOCK_SIZE)
    {
      r = count + n;
      break;
    }
  }

  if (r == -1)
  {
    if (count < len)
    {
      tt_data_get_bytes(tt, block, count, len - count);
      n = memspn(block, len - count, accept);
    
      count += n;
    }
    
    r = count;
  }
  
  free(block);
  return(r);
}


size_t tt_cspn(TT *tt, const char *reject)
{
  int r = -1;
  size_t n;
  int count;
  unsigned int len = tt_size(tt);
  byte *block;

  if (!tt_has_data(tt)) return(-1);
  
  if (tt_data_is_internal(tt))
    return(memcspn(tt_data_get(tt), len, reject));

  block = malloc(TT_GENERIC_BLOCK_SIZE);

  for (count = 0; count + TT_GENERIC_BLOCK_SIZE <= len;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block, count, TT_GENERIC_BLOCK_SIZE);
    n = memcspn(block, TT_GENERIC_BLOCK_SIZE, reject);

    if (n < TT_GENERIC_BLOCK_SIZE)
    {
      r = count + n;
      break;
    }
  }

  if (r == -1)
  {
    if (count < len)
    {
      tt_data_get_bytes(tt, block, count, len - count);
      n = memcspn(block, len - count, reject);
    
      count += n;
    }
    
    r = count;
  }
  
  free(block);
  return(r);
}


int tt_chr(TT *tt, int c)
{
  int r = -1;
  void *p;
  int count;
  unsigned int len = tt_size(tt);
  byte *block;
  
  if (!tt_has_data(tt)) return(-1);
  
  if (tt_data_is_internal(tt))
  {
    p = memchr(tt_data_get(tt), c, len);
    if (p) return((unsigned int) p - (unsigned int) tt_data_get(tt));
    else return(-1);
  }

  block = malloc(TT_GENERIC_BLOCK_SIZE);
  
  for (count = 0; count + TT_GENERIC_BLOCK_SIZE <= len;
       count += TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block, count, TT_GENERIC_BLOCK_SIZE);
    p = memchr(block, c, TT_GENERIC_BLOCK_SIZE);

    if (p)
    {
      r = count + ((unsigned int) p - (unsigned int) block);
      break;
    }
  }

  if (r == -1 && count < len)
  {
    tt_data_get_bytes(tt, block, count, len - count);
    p = memchr(block, c, len - count);

    if (p) r = count + ((unsigned int) p - (unsigned int) block);
  }
  
  free(block);
  return(r);
}


int tt_rchr(TT *tt, int c)
{
  int r = -1;
  void *p;
  int count;
  unsigned int len = tt_size(tt);
  byte *block;

  if (!tt_has_data(tt)) return(-1);
  
  if (tt_data_is_internal(tt))
  {
    p = memrchr(tt_data_get(tt), c, len);
    if (p) return((unsigned int) p - (unsigned int) tt_data_get(tt));
    else return(-1);
  }

  block = malloc(TT_GENERIC_BLOCK_SIZE);
  
  for (count = len - TT_GENERIC_BLOCK_SIZE; count >= 0;
       count -= TT_GENERIC_BLOCK_SIZE)
  {
    tt_data_get_bytes(tt, block, count, TT_GENERIC_BLOCK_SIZE);
    p = memrchr(block, c, TT_GENERIC_BLOCK_SIZE);

    if (p)
    {
      r = count + ((unsigned int) p - (unsigned int) block);
      break;
    }
  }

  if (r == -1 && count < 0)
  {
    tt_data_get_bytes(tt, block, 0, count + TT_GENERIC_BLOCK_SIZE);
    p = memrchr(block, c, count + TT_GENERIC_BLOCK_SIZE);

    if (p) r = ((unsigned int) p - (unsigned int) block);
  }

  free(block);
  return(r);
}
