/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "types.h"
#include "mem.h"

/*
#include "memory.h"
#include "util.h"
*/

#define MAGIC_NOR_BYTE 0x55
#define MAGIC_SEC_BYTE 0xcc
#define MAGIC_END_BYTE 0xaa

#define DBG_MEMORY 1
#define DBG_MEMSTAT 0

#if SIZEOF_UNSIGNED_LONG == 8
# define EXTRA_ALIGN 4
#else
# define EXTRA_ALIGN 0
#endif


#define INFO_BUCKETS 53
#define info_hash(p)  ( *(u32*)((p)) % INFO_BUCKETS )

struct mem_info_entry *mem_info_strings[INFO_BUCKETS]; /* hash table */

struct mem_table_entry *mem_table;  /* the table with the memory info */
unsigned mem_table_size; /* number of allocated entries */
unsigned mem_table_len;  /* number of used entries */
struct mem_table_entry *mem_table_unused;/* to keep track of unused entries */
void mem_table_dump_on_exit(void);
void _mem_table_dump(void);
void mem_check_all(const char *info);

void _mem_table_dump(void);



/****************
 *  * Put the new P into the debug table and return a pointer to the table entry.
 *  * mode is true for security. BY is the name of the function which called us.
 *  */
void _mem_debug_add_entry(byte *p, unsigned n, int mode, const char *info,
                                 const char *by)
{
  unsigned index;
  struct mem_table_entry *e;
  struct mem_info_entry *ie;
  
  if (mem_table_len < mem_table_size) index = mem_table_len++;
  else 
  {
    struct mem_table_entry *e;

    /* look for a used entry in the table.  We take the first one,
     * so that freed entries remain as long as possible in the table
     * (free appends a new one) */

    if ((e = mem_table_unused))
    {
      index = e - mem_table;
      mem_table_unused = e->next;
      e->next = NULL;
    }
    else 
    {
      /* no free entries in the table: extend the table */
      if( !mem_table_size ) 
      {
        /* first time */
        mem_table_size = 100;
        if( !(mem_table = calloc( mem_table_size, sizeof *mem_table )) )
          mem_bug("memory debug table malloc failed");
        index = 0;
        mem_table_len = 1;
        atexit( mem_table_dump_on_exit );
      }
      else 
      {
        /* realloc */
        unsigned n = mem_table_size / 4; /* enlarge by 25% */
        if(!(mem_table = realloc(mem_table, (mem_table_size+n)*sizeof *mem_table)))
          mem_bug("Debug table realloc failed.");
        memset(mem_table + mem_table_size, 0, n * sizeof *mem_table );
        mem_table_size += n;
        index = mem_table_len++;
      }
    }
  }

  e = mem_table + index;

  if (e->inuse) mem_bug("[Mem] Oops - entry %u is flagged as in use.", index);

  e->user_p = p + 4;
  e->user_n = n;
  e->count++;

  if (e->next) mem_bug("[Mem] Oops - entry is in free entry list.");

  /* do we already have this info string? */
  for (ie = mem_info_strings[info_hash(info)]; ie; ie = ie->next)
    if (ie->info == info)
      break;

  if (!ie)
  {
    /* no: make a new entry */
    
    if(!(ie = malloc(sizeof *ie))) mem_bug("Can't allocate info entry.\n");
    ie->next = mem_info_strings[info_hash(info)];
    mem_info_strings[info_hash(info)] = ie;
    ie->info = info;
    ie->count = 0;
  }
  
  ie->count++;
  e->info = ie;
  e->inuse = 1;
  
  /* put the index at the start of the memory */
  p[0] = index;
  p[1] = index >> 8 ;
  p[2] = index >> 16 ;
  p[3] = mode? MAGIC_SEC_BYTE : MAGIC_NOR_BYTE  ;

  /* FIXME: Remove this. */
  if (DBG_MEMORY) 
    /* log_debug( "%s allocates %u bytes using %s\n", info, e->user_n, by) */;
}


/****************
 *  * Check that the memory block is correct. The magic byte has already been
 *  * checked. Checks which are done here:
 *  *    - see whether the index points into our memory table
 *  *    - see whether P is the same as the one stored in the table
 *  *    - see whether we have already freed this block.
 *  */
struct mem_table_entry *mem_check(const void *a, const char *info)
{
  unsigned n;
  const byte *p = a;
  struct mem_table_entry *e;

  n  = p[EXTRA_ALIGN + 0];
  n |= p[EXTRA_ALIGN + 1] << 8;
  n |= p[EXTRA_ALIGN + 2] << 16;

  if (n >= mem_table_len)
  {
    mem_bug("[Mem] (%s) accessed corrupted %p.",
            info, p + EXTRA_ALIGN + 4); return(0);
  }

  e = mem_table + n;

  if (e->user_p != p + EXTRA_ALIGN + 4)
  {
    mem_bug("[Mem] (%s) accessed corrupted %p - ref mismatch.",
            info, p + EXTRA_ALIGN + 4); return(e);
  }
  
  if (!e->inuse)
  {
    mem_bug("[Mem] (%s) accessed corrupted %p - marked as free.",
            info, p + EXTRA_ALIGN + 4); return(e);
  }
  
  if(!(p[EXTRA_ALIGN + 3] == MAGIC_NOR_BYTE
        || p[EXTRA_ALIGN + 3] == MAGIC_SEC_BYTE))
  {
    mem_bug("[Mem] (%s) accessed corrupted %p - underflow by %02x.",
            info, p+EXTRA_ALIGN + 4, p[EXTRA_ALIGN + 3]);
    return(e);
  }
  
  if (p[EXTRA_ALIGN + 4 + e->user_n] != MAGIC_END_BYTE)
  {
    mem_bug("[Mem] (%s) accessed corrupted %p - overflow by %02x.",
            info, p+EXTRA_ALIGN + 4, p[EXTRA_ALIGN + 4 + e->user_n]);
    return(e);
  }

  return e;
}


/****************
 *  * free the entry and the memory (replaces free)
 *  */
void mem_entry_free( byte *p, const char *info )
{
  struct mem_table_entry *e, *e2;
  
  mem_check_all("_mem_debug_add_entry");

  e = mem_check(p, info);
  if (!e) return;

  if (DBG_MEMORY) /* log_debug("%s frees %u bytes alloced by %s\n",
                            info, e->user_n, e->info->info) */ ;
  if (!e->inuse) 
  {
    if (e->user_p == p + EXTRA_ALIGN + 4)
      mem_bug("[Mem] Freeing an already freed block at %p.", p + EXTRA_ALIGN + 4);
    else
      mem_bug("[Mem] Freeing block at %p which is flagged as freed.", p + EXTRA_ALIGN + 4);
  }
    
  e->inuse = 0;
  e->next = NULL;

  if (!mem_table_unused) mem_table_unused = e;
  else 
  {
    for (e2 = mem_table_unused; e2->next; e2 = e2->next) ;
    e2->next = e;
  }
  
  memset(p, 'f', e->user_n + 5);
}



void mem_entry_dump(struct mem_table_entry *e)
{
  unsigned n = e - mem_table;
  
  fprintf(stderr, "mem %4u%c %5u %p %5u %s (%u)\n",
          n, e->inuse?'a':'u', e->count,  e->user_p, e->user_n,
          e->info->info, e->info->count );
}



void mem_table_dump_on_exit(void)
{
  if (DBG_MEMSTAT) _mem_table_dump();
}


void _mem_table_dump(void)
{
  unsigned n;
  struct mem_table_entry *e;
  ulong sum = 0, chunks = 0;

  for (e = mem_table, n = 0; n < mem_table_len; n++, e++)
  {
    if (e->inuse) 
    {
      mem_entry_dump(e);
      sum += e->user_n;
      chunks++;
    }
  }

  fprintf(stderr, "          memory used: %8lu bytes in %ld chunks\n",
          sum, chunks );
}



void mem_check_all( const char *info )
{
  unsigned n;
  struct mem_table_entry *e;

  for (e = mem_table, n = 0; n < mem_table_len; n++, e++)
    if (e->inuse)
      mem_check(e->user_p - 4 - EXTRA_ALIGN, info);
}



