# Makefile.PL for Perl extension Curl::easy.
# Check out the file README for more info.

use ExtUtils::MakeMaker;

# See lib/ExtUtils/MakeMaker.pm for details of how to influence
# the contents of the Makefile that is written.
#
# This utility helper generates the constants function from curl.h
#
# It is normally only used by the maintainer, but if you're curl is older
# or missing some constants, you can delete curlopt-constants.c and re-run 'perl Makefile.PL'
#

#
# You may need to specify where to find curl.h on your platform
# These are guesses only
#

my @includes = qw(
    /usr/include
    /usr/local/curl/include
    /usr/local/include/curl
    ../../include
    ../curl/include
    C:\\INCLUDE\\CURL
);

#
# Get curl to tell us where it is, if we can
#
my $cflags = `curl-config --cflags`;
my $lflags = `curl-config --libs`;

my ($flag) = ($cflags =~ m/-I(\S+)/);

if (defined $flag) {
    unshift @includes, $flag;    # first guess
}

my $curl_d = "";
my $curl_h;

# otherwise try a list of common locations
foreach my $try (@includes) {
    if (-e $try . "/curl/curl.h") {
        $curl_d = $try;
        $curl_h = $try . "/curl/curl.h";
        last;
    }
}

if (!defined($curl_h)) {
    die "Cannot find curl.h - cannot build constants files  - see Makefile.PL";
} else {
    $curl_d = "-I" . $curl_d;
    print "Found curl.h in $curl_h\n";
    open(CURL_H, "<" . $curl_h) or die "Can't open curl.h\n";
    my %cinit_types;
    my %cinit_codes;
    my @curlinfo;
    while (<CURL_H>) {
        if ($_ =~ m/CINIT\(/ and $_ !~ m/#/) {
            my ($option, $type, $code) =
                m/.*CINIT\((\w*)\s*,\s*(\w+)\s*,\s*(\d+).*/;
            $cinit_types{$option} = $type;
            $cinit_codes{$option} = $code;
        } elsif ($_ =~ m/^\s*(CURLINFO_\w+)/) {
            push @curlinfo, $1;
        }

    }
    close(CURL_H);

    # some things are ifdefed out...
    foreach my $ifdef0 (qw(FLAGS PROGRESSMODE)) {
        delete $cinit_types{$ifdef0};
        delete $cinit_codes{$ifdef0};
    }

    print "Building curlopt-constants.c for your libcurl version\n";

    open(CURL_XS, ">curlopt-constants.c")
        or die "Can't write curlopt-constants.c\n";

    # boilerplate xs constant function here
    print CURL_XS <<HERE
static int
constant(char *name, int arg)
{
    errno = 0;
    if (strncmp(name, "CURLINFO_", 9) == 0) {
	name += 9;
    switch (*name) {
HERE
        ;
    foreach my $next_initial ('A' .. 'Z') {
        print CURL_XS "        case '$next_initial':\n";
        my $count = 0;
        foreach my $curlinfo_name (sort @curlinfo) {
            my $initial = substr($curlinfo_name, 9, 1);
            my $option = substr($curlinfo_name, 9);
            if ($next_initial eq $initial) {

                print CURL_XS
"            if (strEQ(name, \"$option\")) return CURLINFO_$option;\n";

                $count++;
            }
        }
        if ($count) {
            print CURL_XS "            break;\n";
        }
    }

    print CURL_XS "    }\n";
    print CURL_XS "    }\n";
    print CURL_XS <<HERE2

    if (strncmp(name, "CURLOPT_", 8) == 0) {
	name += 8;
	switch (*name) {
HERE2
        ;

    foreach my $next_initial ('A' .. 'Z') {
        print CURL_XS "        case '$next_initial':\n";
        my $count = 0;
        foreach my $option (sort keys %cinit_types) {
            my $initial = substr($option, 0, 1);
            if ($next_initial eq $initial) {

                print CURL_XS
"            if (strEQ(name, \"$option\")) return CURLOPT_$option;\n";

                $count++;
            }
        }
        if ($count) {
            print CURL_XS "            break;\n";
        }
    }
    print CURL_XS "    }\n";
    print CURL_XS "    }\n";
    print CURL_XS <<HERE

    if (strEQ(name, "USE_INTERNAL_VARS")) return USE_INTERNAL_VARS;
    errno = EINVAL;
    return 0;
}
HERE
        ;

    close(CURL_XS);

    print "Building easy.pm constants for your libcurl version\n";

    open(EASY_PM,    ">easy.pm")    or die "Can't create easy.pm\n";
    open(EASY_PM_IN, "<easy.pm.in") or die "Can't read easy.pm.in\n";
    while (my $line = <EASY_PM_IN>) {
        if ($line !~ m/^\@CURLOPT_INCLUDE\@/) {
            print EASY_PM $line;
        } else {
            foreach my $option (sort keys %cinit_types) {
                print EASY_PM "CURLOPT_$option\n";
            }
            foreach my $option (sort @curlinfo) {
                print EASY_PM $option . "\n";
            }
        }
    }
    close(EASY_PM);
    close(EASY_PM_IN);
}

#
# Now write the makefile
#
WriteMakefile(
    'NAME'         => 'Curl::easy',
    'VERSION_FROM' => 'easy.pm',      # finds $VERSION
    'LIBS'         => $lflags,        # e.g., '-lm'
    'DEFINE'       => '',             # e.g., '-DHAVE_SOMETHING'
    'INC'          => $curl_d,        # e.g., '-I/usr/include/other'
    'clean' => { FILES => "head.out body.out" }
);

