/* $Id: mod_witch.c,v 1.21 2001/07/15 21:00:30 toma Exp $
 *  mod_witch
 *  (C) Tamas SZERB <toma@rulez.org>
 */


#include "httpd.h"
#include "http_log.h"
#include "http_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>

#define WITCH_VERSION "$Id: mod_witch.c,v 1.21 2001/07/15 21:00:30 toma Exp $"
#define WITCH_MAXQLEN 2048

static enum {
	WITCH_LOG,
	WITCH_ERROR
} witch_error;

static enum {
	WITCH_IP,
	WITCH_HOSTNAME
} witch_logstyle;

//const char * delim = " :,+|";

typedef struct {
	char *ident;
	int option;									/* LOG_CONS, LOG_NDELAY, LOG_PERROR, LOG_PID */
	int facility;								/* LOG_AUTH, LOG_AUTHPRIV, LOG_CRON, LOG_DAEMON,
															   LOG_KERN, LOG_LOCAL0-LOG_LOCAL7, LOG_LPR, 
																 LOG_MAIL, LOG_NEWS, LOG_SYSLOG, LOG_USER, 
																 LOG_UUCP */
	int level;									/* LOG_EMERG, LOG_ALERT, LOG_CRIT, LOG_ERR, 
															   LOG_WARNING, LOG_NOTICE, LOG_INFO, LOG_DEBUG */
	int error;
	int logstyle;
} witch_server_cfg;

module MODULE_VAR_EXPORT witch_module;

static void witch_server_init(server_rec * c, pool * p);
static void *witch_server_config(pool * p, server_rec * c);
static int witch_log(request_rec * r);
static void witch_child_init(server_rec * c, pool * p);
static void witch_child_exit(server_rec * c, pool * p);

static const char *witch_set_ident(cmd_parms * cmd, void *dummy, char *arg);
static const char *witch_set_option(cmd_parms * cmd, void *dummy, char *arg);
static const char *witch_set_facility(cmd_parms * cmd, void *dummy, char *arg);
static const char *witch_set_level(cmd_parms * cmd, void *dummy, char *arg);
static const char *witch_set_error(cmd_parms * cmd, void *dummy, char *arg);
static const char *witch_set_logstyle(cmd_parms * cmd, void *dummy, char *arg);

/* 
 * Config fuctions
 */
static command_rec witch_cmds[] = {
	{"WitchIdent", witch_set_ident, NULL,
	RSRC_CONF, TAKE1, "ident - usually the program name"},
	{"WitchOption", witch_set_option, NULL,
	RSRC_CONF, TAKE1, "witch option"},
	{"WitchFacility", witch_set_facility, NULL,
	RSRC_CONF, TAKE1, "witch facility"},
	{"WitchLevel", witch_set_level, NULL,
	RSRC_CONF, TAKE1, "witch level"},
	{"WitchError", witch_set_error, NULL,
	RSRC_CONF, TAKE1, "witch error handling"},
	{"WitchLogStyle", witch_set_logstyle, NULL,
	RSRC_CONF, TAKE1, "witch logstyle"},
	{NULL}
};

static const char *witch_set_ident(cmd_parms * cmd, void *dummy, char *arg)
{
	server_rec *s = cmd->server;
	witch_server_cfg *cfg = (witch_server_cfg *)
		ap_get_module_config(s->module_config, &witch_module);

	cfg->ident=arg;
	return NULL;
}

static const char *witch_set_option(cmd_parms * cmd, void *dummy, char *arg)
{
	server_rec *s = cmd->server;

	witch_server_cfg *cfg = (witch_server_cfg *)
		ap_get_module_config(s->module_config, &witch_module);
//	cfg->option = atoi(arg);

	cfg->option = 0;
	if (!strcmp("LOG_CONS", arg)) cfg->option|=LOG_CONS;
	else
	if (!strcmp("LOG_NDELAY", arg)) cfg->option|=LOG_NDELAY;
	else
	if (!strcmp("LOG_PERROR", arg)) cfg->option|=LOG_PERROR;
	else
	if (!strcmp("LOG_PID", arg)) cfg->option|=LOG_PID;
	else
	return "No valid argument for WitchOption";

	return NULL;
}

static const char *witch_set_facility(cmd_parms * cmd, void *dummy, char *arg)
{
	server_rec *s = cmd->server;
	witch_server_cfg *cfg = (witch_server_cfg *)
		ap_get_module_config(s->module_config, &witch_module);
//	cfg->facility = atoi(arg);

	cfg->facility = 0;
	if (!strcmp("LOG_AUTH", arg)) cfg->facility|=LOG_AUTH;
	else
	if (!strcmp("LOG_AUTHPRIV", arg)) cfg->facility|=LOG_AUTHPRIV;
	else
	if (!strcmp("LOG_CRON", arg)) cfg->facility|=LOG_CRON;
	else
	if (!strcmp("LOG_DAEMON", arg)) cfg->facility|=LOG_DAEMON;
	else
	if (!strcmp("LOG_KERN", arg)) cfg->facility|=LOG_KERN;
	else
	if (!strcmp("LOG_LOCAL0", arg)) cfg->facility|=LOG_LOCAL0;
	else
	if (!strcmp("LOG_LOCAL1", arg)) cfg->facility|=LOG_LOCAL1;
	else
	if (!strcmp("LOG_LOCAL2", arg)) cfg->facility|=LOG_LOCAL2;
	else
	if (!strcmp("LOG_LOCAL3", arg)) cfg->facility|=LOG_LOCAL3;
	else
	if (!strcmp("LOG_LOCAL4", arg)) cfg->facility|=LOG_LOCAL4;
	else
	if (!strcmp("LOG_LOCAL5", arg)) cfg->facility|=LOG_LOCAL5;
	else
	if (!strcmp("LOG_LOCAL6", arg)) cfg->facility|=LOG_LOCAL6;
	else
	if (!strcmp("LOG_LOCAL7", arg)) cfg->facility|=LOG_LOCAL7;
	else
	if (!strcmp("LOG_LPR", arg)) cfg->facility|=LOG_LPR;
	else
	if (!strcmp("LOG_MAIL", arg)) cfg->facility|=LOG_MAIL;
	else
	if (!strcmp("LOG_NEWS", arg)) cfg->facility|=LOG_NEWS;
	else
	if (!strcmp("LOG_SYSLOG", arg)) cfg->facility|=LOG_SYSLOG;
	else
	if (!strcmp("LOG_USER", arg)) cfg->facility|=LOG_USER;
	else
	if (!strcmp("LOG_UUCP", arg)) cfg->facility|=LOG_UUCP;
	else
	return "No valid value for WitchFacility";

	return NULL;
}

static const char *witch_set_level(cmd_parms * cmd, void *dummy, char *arg)
{
	server_rec *s = cmd->server;
	witch_server_cfg *cfg = (witch_server_cfg *)
		ap_get_module_config(s->module_config, &witch_module);
//	cfg->level = atoi(arg);

	cfg->level = 0;
	if (!strcmp("LOG_EMERG", arg)) cfg->level|=LOG_EMERG;
	else
	if (!strcmp("LOG_ALERT", arg)) cfg->level|=LOG_ALERT;
	else
	if (!strcmp("LOG_CRIT", arg)) cfg->level|=LOG_CRIT;
	else
	if (!strcmp("LOG_ERR", arg)) cfg->level|=LOG_ERR;
	else
	if (!strcmp("LOG_WARNING", arg)) cfg->level|=LOG_WARNING;
	else
	if (!strcmp("LOG_NOTICE", arg)) cfg->level|=LOG_NOTICE;
	else
	if (!strcmp("LOG_INFO", arg)) cfg->level|=LOG_INFO;
	else
	if (!strcmp("LOG_DEBUG", arg)) cfg->level|=LOG_DEBUG;
	else
	return "No valid value for WitchLevel";

	return NULL;
}

static const char *witch_set_error(cmd_parms * cmd, void *dummy, char *arg)
{
	server_rec *s = cmd->server;
	witch_server_cfg *cfg = (witch_server_cfg *)
		ap_get_module_config(s->module_config, &witch_module);

	if (strcmp("log", arg) == 0) cfg->error = WITCH_LOG;
	else
	if (strcmp("error", arg) == 0) cfg->error = WITCH_ERROR;
	else 
	return "No valid argument for WitchError";

	return NULL;
}

static const char *witch_set_logstyle(cmd_parms * cmd, void *dummy, char *arg)
{
	server_rec *s = cmd->server;
	witch_server_cfg *cfg = (witch_server_cfg *)
		ap_get_module_config(s->module_config, &witch_module);
	if (strcmp("ip", arg) == 0) cfg->logstyle = WITCH_IP;
	else 
	if (strcmp("hostname", arg) == 0) cfg->logstyle = WITCH_HOSTNAME;
	else
	return "No valid argument for WitchLogStyle";

	return NULL;
}


/* fscking apache structures: */

module MODULE_VAR_EXPORT witch_module = {
	STANDARD_MODULE_STUFF,
	witch_server_init,					/* initializer */
	NULL,												/* dir config creater */
	NULL,												/* dir merger --- default is to override */
	witch_server_config,				/* server config */
	NULL,												/* merge server config */
	witch_cmds,									/* command table */
	NULL,												/* handlers */
	NULL,												/* filename translation */
	NULL,												/* check_user_id */
	NULL,												/* check auth */
	NULL,												/* check access */
	NULL,												/* type_checker */
	NULL,												/* fixups */
	witch_log,									/* logger */
	NULL,												/* header parser */
	witch_child_init,						/* child_init */
	witch_child_exit,						/* child_exit */
	NULL												/* post read-request */
};

static void witch_server_init(server_rec * c, pool * p)
{
	ap_add_version_component("Witch/" WITCH_VERSION);
}

static void *witch_server_config(pool * p, server_rec * c)
{
	witch_server_cfg *cfg =
		(witch_server_cfg *) ap_pcalloc(p, sizeof *cfg);

	/* defaults: */
	cfg->ident = "apache";
	cfg->option = LOG_PID;
	cfg->facility = LOG_DAEMON;
	cfg->level = LOG_INFO;
	cfg->error = WITCH_LOG;
	cfg->logstyle = WITCH_IP;

	return cfg;
}

static int witch_log(request_rec * r)
{
/*	char * the_time = "[08/Jul/2001:15:42:01 +0200]"; */
	#define TIME_BUFSIZE 40
	char time_buffer[TIME_BUFSIZE]; /* ^^ */
	struct tm *loctime;

	char query[WITCH_MAXQLEN];
	char *rlogname, *ruser, *rclient;

	witch_server_cfg *cfg =
		(witch_server_cfg *) ap_get_module_config(r->server->module_config,
			&witch_module);

	if ( cfg->error == WITCH_ERROR) return SERVER_ERROR;

	rlogname = (char *) ap_get_remote_logname(r);
	if (rlogname == NULL) rlogname = "-";

	ruser = r->connection->user;

	if (ruser == NULL) ruser = "-";
	else
	if (strlen(ruser) == 0) ruser = "\"\"";

	switch ( cfg->logstyle ) {
		case WITCH_IP:
			rclient = r->connection->remote_ip;
		break;
		case WITCH_HOSTNAME:
			rclient = (char *) ap_get_remote_host(r->connection, 
				r->per_dir_config, 1 );
		break;
	}

	loctime = localtime (&(r->request_time));
	strftime( time_buffer, TIME_BUFSIZE, "[%d/%b/%Y:%H:%M:%S %z]", loctime);

	snprintf(query, WITCH_MAXQLEN, "%s %s %s %s \"%s\" %d %d %s",
		rclient,					/* remote client/ip address (WitchLogStyle) */
		rlogname,					/* remote username by identd (IdentityCheck) or - */
		ruser,						/* remote user (authentication, usually basic) or - */
		time_buffer,			/* [hit's time] */
		r->the_request,		/* "the request" */
		r->status,				/* the status (200: OK, and so on) */
		r->bytes_sent,		/* the bytes was sent to the client */
											/* and the extra log stuff: */
		r->server->server_hostname); /* the virtual server's name */

	syslog(cfg->level, "%s", query);

	return OK;
}

static void witch_child_init(server_rec * c, pool * p)
{
	witch_server_cfg *cfg;

	cfg = (witch_server_cfg *) ap_get_module_config(c->module_config,
		&witch_module);

	openlog(cfg->ident, cfg->option, cfg->facility);

}

static void witch_child_exit(server_rec * c, pool * p)
{
	closelog();
}
