/*
 *  mod_musicindex.c
 *  mod_musicindex
 *
 *  $Id: mod_musicindex.c,v 1.69 2004/07/08 18:47:23 boudinr Exp $
 *
 *  Created by Regis BOUDIN on Sat Dec 28 2002.
 *  Copyright (c) 2002-2004 Regis BOUDIN
 *  Copyright (c) 2002-2004 Thibaut VARENE
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */


/**
 * @mainpage
 * @section about About
 *	mod_musicindex is an Apache module aimed at being a C implementation
 *	of the Perl module <a href="http://search.cpan.org/dist/Apache-MP3/">
 *	Apache::MP3</a>. It allows nice displaying of directories containing
 *	MP3 or Ogg Vorbis files, including sorting them on various fields,
 *	streaming/downloading them, constructing playlist, and searching.
 *
 * @section features Features
 *	- Allow/deny file downloading.
 *	- Allow/deny file streaming.
 *	- Allow/deny file searching (recursively or not).
 *	- Default sorting order in Apache configuration and dynamic sorting while browsing.
 *	- Highly configurable field displaying.
 *	- Custom folder picture (using cover.png, cover.jpg or cover.gif if found).
 *	- Cache subsystem (using flat files).
 *	- Can redirect to an icecast server running in "staticdir" mode.
 *	- Custom cross-directories playlists.
 *	- Multiple CSS support.
 *	- Per directory configuration (using .htaccess).
 *	- XHTML output.
 *
 * @section dependencies Dependencies
 * 	- <a href="http://httpd.apache.org/download.cgi">Apache</a> (1.3 or 2).
 *	- <a href="http://www.vorbis.com/files/1.0/unix/libogg-1.0.tar.gz">libogg 1.0</a>.
 *	- <a href="http://www.vorbis.com/files/1.0/unix/libvorbis-1.0.tar.gz">libvorbis 1.0</a>.
 *	- <a href="ftp://ftp.mars.org/pub/mpeg/">libmad0</a>.
 *	- <a href="ftp://ftp.mars.org/pub/mpeg/">libid3tag0</a>.
 */

/**
 * @file 
 * Core file.
 *
 * This file is the core of the module. It contains Apache's mandatory stuff.
 *
 * @author Regis Boudin
 * @version $Revision: 1.69 $
 * @date 2002-2004
 *
 * @warning Use it at your own risks!
 *
 * @todo Complete code documentation.
 * @todo (whishlist) On the fly tag rewriting.
 * @todo Considering the previous item, we might also need a handler for
 * streaming, otherwise we won't be able to restrain file downloading
 * (Apache::MP3 doesn't do anything better there fwiw).
 * @todo prepare the possibility to generate ices playlists.
 * @todo i18n (in progress).
 * @todo enforce strict type policy.
 * @todo AAC support.
 */

#include "mod_musicindex.h"
#include "playlist.h"
#include "config.h"
#include "html.h"
#include "inf.h"

#define MUSIC_HEADER_STRING "mod_musicindex/" MUSIC_VERSION_STRING

#ifdef __BUILD_FOR_APACHE2
/** mime types handled */
static const char *handlers[] = {
	"audio/mpeg",
	"application/ogg",
	"audio/flac",
	"audio/x-ogg", /* At some point, we should be able to remove this one */
	NULL
};
#endif

static int handle_musicindex(request_rec *r)
{
	/* Get the module configuration */
	mu_config *conf = (mu_config *)ap_get_module_config(r->per_dir_config, &musicindex_module);

	/* if the request is not a GET and the module is not active, let the dir
	to another module */
	if (!(conf->options & MI_ACTIVE))
		return DECLINED;
	if (r->method_number != M_GET)
		return DECLINED;

#ifdef __BUILD_FOR_APACHE2
	if(strcmp(r->handler, DIR_MAGIC_TYPE))
		return DECLINED;
#endif

	/* before doing anything (checking args, calling subfunctions), check we
	   can open the directory */
	if (access(r->filename, R_OK|X_OK) != 0) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
			"[musicindex] (%s) Can't open directory: %s",
			__func__, r->filename);
		return HTTP_FORBIDDEN;
	}

	r->allowed |= (1 << M_GET);

	/* This part mostly comes from mod_autoindex. If the requested dir does
	not end with a '/', generate a new request with it */
	if (r->uri[0] == '\0' || r->uri[strlen(r->uri) - 1] != '/') {
		char *file;
		if (r->args != NULL)
			file = ap_pstrcat(r->pool, ap_escape_uri(r->pool, r->uri),
						"/", "?", r->args, NULL);
		else
			file = ap_pstrcat(r->pool, ap_escape_uri(r->pool, r->uri),
						"/", NULL);
		
		ap_table_setn(r->headers_out, "Location", ap_construct_url(r->pool, file, r));
		return HTTP_MOVED_PERMANENTLY;
	}

	/* This function will parse the arguments given by the client to
	correctly set the flags and parameters for this request */
	treat_args(r, conf);

	/* Depending on the request type, set the HTTP headers. By default we
	return a web page. */
	if (conf->options & MI_STREAM) {
		ap_set_content_type(r, "audio/mpegurl");
		ap_table_setn(r->headers_out, "Content-Disposition",
			"filename = \"playlist.m3u\"");
	}
	else if (conf->options & MI_RSS) {
		ap_set_content_type(r, "text/xml; charset=\"utf-8\"");
	}
	else
		ap_set_content_type(r, "text/html; charset=\"utf-8\"");

	/* Call the magic function that will generate and send data depending
	depending on the request, cookie, configuration, etc... */
	return musicindex_list(r, conf);

}

/**
 * Handler for requests on music files.
 *
 * At the moment it can only forbid acces to files if neither streaming nor download is allowed.
 *
 * @param r Apache request_rec to get and send data
 *
 * @return Standard HTTP code (see httpd.h for more infos)
 */
int handle_musicfile(request_rec *r)
{
	mu_config *conf = (mu_config *)ap_get_module_config(r->per_dir_config, &musicindex_module);

#ifdef __BUILD_FOR_APACHE2
	unsigned short i;
#endif

	if ((r->method_number != M_GET) || (!(conf->options & MI_ACTIVE)))
		return DECLINED;
	
#ifdef __BUILD_FOR_APACHE2
	for (i=0; handlers[i] && strcmp(r->handler, handlers[i]); i++) {
		if (!strcmp(r->handler, handlers[i]))
			break;
	}
	
	if (handlers[i] == NULL)
		return DECLINED;
#endif

	if ((r->args == NULL) && ((conf->options&MI_ALLOWDWNLD) ||
				  ((conf->options&MI_ALLOWSTREAM) &&
				   (conf->iceserver == NULL)        )))
		return DECLINED;
	
	if ((conf->options&MI_ALLOWSTREAM) && (strcmp(r->args, "stream") == 0))
		return playlist_single(r, conf);
	
	return HTTP_FORBIDDEN;
}

const command_rec musicindex_cmds[] = {
	AP_INIT_RAW_ARGS("MusicSortOrder", sort_order, NULL, OR_INDEXES,
		"can be : title album artist track disc length bitrate filetype genre filename date uri"),
	AP_INIT_RAW_ARGS("MusicFields", set_fields, NULL, OR_INDEXES,
		"can be : title album artist track disc length bitrate filetype genre filename date"),
	AP_INIT_FLAG("MusicAllowStream", allow_stream, NULL, OR_INDEXES,
		"Allow streaming"),
	AP_INIT_FLAG("MusicAllowDownload", allow_download, NULL, OR_INDEXES,
		"Allow downloading"),
	AP_INIT_FLAG("MusicAllowSearch", allow_search, NULL, OR_INDEXES,
		"Enable searching (EXPERIMENTAL)"),
	AP_INIT_FLAG("MusicLister", music_lister, NULL, OR_INDEXES,
		"Enable the Musicindex index maker"),
	AP_INIT_TAKE1("MusicCachePath", set_cache_path, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the cache absolute path, without trailing '/'."),
	AP_INIT_RAW_ARGS("MusicPageTitle", set_page_title, NULL, OR_INDEXES,
		"Set the root title of the page."),
	AP_INIT_TAKE1("MusicIceServer", set_ice_server, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the icecast server address : [server.domain.org]:8000"),
	AP_INIT_TAKE1("MusicCssDefault", set_css_default, NULL, OR_INDEXES,
		"Set the default CSS file"),
	AP_INIT_TAKE1("MusicCookieLife", set_cookie_life, NULL, ACCESS_CONF|RSRC_CONF,
		"Set the lifetime (in seconds) of the cookie sent for custom playlists"),
	AP_INIT_FLAG("MusicAllowRss", allow_rss, NULL, OR_INDEXES,
		"Allow RSS feed"),
	{NULL}
};

#ifndef __BUILD_FOR_APACHE2 /* we build for apache 1.3 */

/**
 * Adds the mod signature into Apache's headers.
 *
 * @param s server_rec
 * @param p pool
 */
static void musicindex_init(server_rec *s, apr_pool_t *p)
{
#ifdef ENABLE_GETTEXT
	setlocale(LC_ALL, "");
	textdomain("mod_musicindex");
#endif

	ap_add_version_component(MUSIC_HEADER_STRING);
}

const handler_rec musicindex_handlers[] = {
	{DIR_MAGIC_TYPE, handle_musicindex},
	{"audio/mpeg", handle_musicfile},
	{"audio/x-ogg", handle_musicfile}, /* Backward compatibility */
	{"application/ogg", handle_musicfile},
	{"audio/flac", handle_musicfile},
	{NULL}
};

module MODULE_VAR_EXPORT musicindex_module = {
	STANDARD_MODULE_STUFF,
	musicindex_init,		/**< initializer */
	create_musicindex_config,	/**< dir config creater */
	merge_musicindex_configs,	/**< dir merger */
	NULL,				/**< server config */
	NULL,				/**< merge server config */
	musicindex_cmds,		/**< command table */
	musicindex_handlers,		/**< handlers */
	NULL,				/**< filename translation */
	NULL,				/**< check_user_id */
	NULL,				/**< check auth */
	NULL,				/**< check access */
	NULL,				/**< type_checker */
	NULL,				/**< fixups */
	NULL,				/**< logger */
	NULL,				/**< header parser */
	NULL,				/**< child_init */
	NULL,				/**< child_exit */
	NULL				/**< post read-request */
};

#else /* we build for apache 2 */

/**
 * Adds the mod signature into Apache's headers.
 *
 * @param p pool
 * @param plog pool
 * @param ptemp pool
 * @param s server_rec
 */
static int musicindex_init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp, server_rec *s)
{
#ifdef ENABLE_GETTEXT
	setlocale(LC_ALL, "");
	textdomain("mod_musicindex");
#endif

	ap_add_version_component(p, MUSIC_HEADER_STRING);

	return OK;
}

static void register_hooks(apr_pool_t *p)
{
	/* for directories, set the mod_autoindex to be called after us */
	static const char * const after[]={ "mod_autoindex.c",NULL };
	ap_hook_handler(handle_musicindex,NULL,after,APR_HOOK_MIDDLE);
	ap_hook_handler(handle_musicfile,NULL,NULL,APR_HOOK_MIDDLE);
	ap_hook_post_config(musicindex_init, NULL, NULL, APR_HOOK_LAST);
}

module AP_MODULE_DECLARE_DATA musicindex_module = {
    STANDARD20_MODULE_STUFF,
    create_musicindex_config,	/* create per-directory config structure */
    merge_musicindex_configs,	/* merge per-directory config structures */
    NULL,			/* create per-server config structure */
    NULL,			/* merge per-server config structures */
    musicindex_cmds,		/* command apr_table_t */
    register_hooks		/* register hooks */
};

#endif /* __BUILD_FOR_APACHE2 */
