/*
 * ezbounce.cpp
 *
 * (C) 1998-2000 Murat Deligonul
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "autoconf.h"

#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "server.h"
#include "general.h"
#include "conn.h"
#include "ezbounce.h"
#include "config.h"

#define MARK fprintf(stderr, "%s: %s: %d: marked\n", __FILE__, __PRETTY_FUNCTION__, __LINE__)

time_t start_time;

struct config_struct config;

static void sig_handler(int); 

static void setup_signals();
static void usage(const char *);
static bool parse_cmdline(int, char **, char **, char **, bool *, int *, pid_t *);
void redir_stdxxx(void);

/* #define NOFORK */

static void setup_signals()
{
    struct sigaction sv;
    
    sv.sa_flags = 0;
    sv.sa_handler = SIG_IGN;
    sigaction(SIGPIPE, &sv, NULL);
    sv.sa_handler = &sig_handler;
    sigaction(SIGINT, &sv, NULL);
    sigaction(SIGTERM, &sv, NULL);
    sigaction(SIGHUP, &sv, NULL);
    sigaction(SIGALRM, &sv, NULL);
}


int main(int argc, char *argv[])
{  
    char tmp[20];
    unsigned num_listening = 0;
    
#ifndef NOFORK
    pid_t pid;
#endif
    pid_t uid = 0;                /* become this user */
    bool nobg = 0;                /* go to the back ground? */
    char *iface = NULL;           /* interface to bind to for listening*/
    char *iface_conn = NULL;      /* interface to bind to for connecting */
    int logindex  = 0;            /* argv[logindex] is the log */

    printf("%s\n(c) 1998-2001 by Murat Deligonul (druglord@freelsd.org)\n\n", EZBOUNCE_VERSION);

    if (argc < 2 || !parse_cmdline(argc, argv, &iface, &iface_conn, &nobg, &logindex, &uid))
    {
        usage(argv[0]);
        return 1;
    }

    printf("Reading config file %s ...\n", argv[logindex]);
    if ((argc = load_config_file(&config, argv[logindex])) < 0)
    {
        fprintf(stderr, "Error(s) while loading config file -- exiting.\n");
        return 1;
    }
    else if (argc == 0)
    {
        perror("Unable to open config file");
        return 1;
    }
    else
        printf("Config file successfully loaded.\n");

    printf("User password is: %s\n", config.password);
    
    /*
     * Resolve iface_conn if needed and 
     * put it into config 
     */                     
    struct in_addr in;
    if ((iface_conn) && (argc = fill_in_addr(iface_conn, &in) <= 0))
    {
        fprintf(stderr, "Error: can't bind connections to %s: %s\n", iface_conn, 
                (argc == 0) ? strerror(errno) : "unknown host");
        return 1;
    }
    else if (iface_conn) {
        config.iface_conn = in;
        printf("Default connection interface: %s (%s)\n", iface_conn, inet_ntoa(in));
    }

    /*
     * Do the same for iface now 
     */
    if (iface && (argc = fill_in_addr(iface, &in)) <= 0)
    {
        fprintf(stderr, "Can't listen on host: %s: %s\n", iface,
                (argc == 0) ? strerror(errno) : "unknown host");
        return 1;
    }
    else if (iface) {
        config.iface = in;
        printf("Listening on interface: %s (%s)\n", iface, inet_ntoa(in));
    }

    start_time = time(NULL);
    
    argc = 1;
    fprintf(stderr,"Listening on ports");
    while (gettok(config.ports, tmp, 20, ',', argc++))
    {
        if (!ircproxy_listen((u_short)atoi(tmp), &config.iface))
            fprintf(stderr, " (%s:%s)", tmp, strerror(errno));
        else {
            fprintf(stderr, " %s", tmp);
            num_listening++;
        }
    }
    printf("\nListening on total of %d ports\n", num_listening);
    if (!num_listening)
    {
        fprintf(stderr, "oops, I can't run without listening on any ports\n");
        return 1;
    }

    /* change uid if needed */
    if (uid)
    {   
        if (!setuid(uid))
            printf("Now operating under userid %d\n.", uid);
        else
            perror("setuid");
    }

    
    /* start the log file */
    if (!ircproxy_startlog(config.logfile))
        perror("WARNING: Unable to open log file");
        
#ifndef NOFORK
    if ((nobg) || ((pid = fork()) == 0))
    { 
        setsid();
        /* redirect stderr to our silly little log file.
           close stdin & stdout. */
        close(STDIN_FILENO);
        /* close(STDOUT_FILENO); -- config file reloading will output to stdout */
        if (!nobg)
            ircproxy_redir_stdxxx();
#endif
        /* Write out pid-file if needed */
        if (config.flags & WRITE_PIDFILE)
        {
            int f = open(config.pidfile, O_CREAT | O_WRONLY);
            fdprintf(f, "%d", getpid());
            fchmod(f, 0644);
            close(f);
        }
        
        /* no longer needed */
        delete[] config.logfile;
        delete[] config.pidfile;
        config.logfile = NULL;
        config.pidfile = NULL;
        
        setup_signals(); 
        set_dns_timeout(config.max_dns_wait_time);
        start_proxy();

        /* Server is finished. */
        ircproxy_closelog();
        stop_proxy();

        /* Destroy everything (to keep ccmalloc and such happy) */
        rule_set * r;
        while ((r = rule_set::first()))
            delete r;
        delete config.ports;
        delete config.admins;
        delete config.vhost_table;
        delete[] config.configfile;
        delete[] config.logfile;
        delete[] config.password;
        delete[] config.pidfile;
        delete[] config.motd_file;
        delete[] config.logdir;
        delete[] config.dcc_ports;
        delete[] config.autoserver;
        return 0;
#ifndef NOFORK
    }
    else if (pid < 0)
        perror("Can't go into the background");
    else
        printf("IRC Proxy started -- pid: %d\n", pid);
#endif 
    return 0;
}

static void sig_handler(int sig)
{
    switch (sig)
    {
    case SIGHUP:
        printlog("Got SIGHUP; reloading configuration file\n");
        ircproxy_rehash();
        break;
    case SIGTERM:
        printlog("Got terminate signal, killing server now\n");
        ircproxy_die(1, "got terminate signal");
        break;
    case SIGINT:
        printlog("Interrupt signal: waiting for server to shut down\n");
        ircproxy_die(0, "keyboard interrupt");
    
    default:
        break;
    }

}

static void usage(const char *bname)
{
    fprintf(stderr, "Usage:\n%s [options] <configuration file>\n", bname);
    fprintf(stderr, "Available options:\n");
    fprintf(stderr, "  -f              don't go into the background\n");
    fprintf(stderr, "  -c <hostname>   bind connections to a specific interface (vhost)\n");
    fprintf(stderr, "  -b <hostname>   listen for connections on a different interface\n");
    fprintf(stderr, "  -u <uid>        become user uid after binding listen socket\n\n");
}


static bool parse_cmdline(int argc, char **argv, char **pIface, char **pIface_conn, 
                          bool * pNobg, int *pIndex, pid_t * uid)
{
    for (int z = 1; z < argc; z++)
    {
        switch (argv[z][0])
        {
        /* a -flag, next char must be -B or -F */
        case '-':
            switch (argv[z][1])
            {
            case 'u':
            case 'U':
                *uid = atoi(argv[++z]);
                continue;
            case 'f':
            case 'F':
                *pNobg = 1;
                break;
            case 'b':
            case 'B':              
                if (z + 1 < argc)
                    *pIface = argv[++z];
                continue;
            case 'c':
            case 'C':
                if (z + 1 < argc)
                    *pIface_conn = argv[++z];
                continue;
            case '-':
                if (strcasecmp(&argv[z][2], "help") != 0)
                    continue;
            case 'h':
                return 0;
            default:
                break;
            }
        default:
            *pIndex = z;
        }
    }
    return (*pIndex != 0);
}

