/*
    ext2_block_relocator.c -- ext2 block relocator
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_block_relocator_c[] = "$Id: ext2_block_relocator.c,v 1.20 2004/09/30 14:05:55 sct Exp $";

#define USE_EXT2_IS_DATA_BLOCK
#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"



struct ext2_block_entry
{
	blk_t		num;
	blk_t		dest;
	blk_t		refblock;
	unsigned	refoffset:16;
	unsigned	isindirectblock:16;
};

struct ext2_block_relocator_state
{
	blk_t			 newallocoffset;
	blk_t			 allocentries;
	blk_t			 usedentries;
	blk_t			 resolvedentries;
	struct ext2_block_entry *block;

	struct {
		struct ext2_block_entry *dst;
		int			 num;
	} start[4];
};



static int compare_block_entries(const void *x0, const void *x1)
{
	const struct ext2_block_entry *b0;
	const struct ext2_block_entry *b1;

	b0 = (const struct ext2_block_entry *)x0;
	b1 = (const struct ext2_block_entry *)x1;

	if (b0->num < b1->num)
		return -1;

	if (b0->num > b1->num)
		return 1;

	return 0;
}

static int compare_block_entries_ind(const void *x0, const void *x1)
{
	const struct ext2_block_entry *b0;
	const struct ext2_block_entry *b1;

	b0 = (const struct ext2_block_entry *)x0;
	b1 = (const struct ext2_block_entry *)x1;

	if (b0->isindirectblock > b1->isindirectblock)
		return -1;

	if (b0->isindirectblock < b1->isindirectblock)
		return 1;

	return 0;
}

static int compare_block_entries_ref(const void *x0, const void *x1)
{
	const struct ext2_block_entry *b0;
	const struct ext2_block_entry *b1;

	b0 = (const struct ext2_block_entry *)x0;
	b1 = (const struct ext2_block_entry *)x1;

	if (b0->refblock < b1->refblock)
		return -1;

	if (b0->refblock > b1->refblock)
		return 1;

	return 0;
}

struct ext2_block_entry *findit(struct ext2_block_relocator_state *state, blk_t block)
{
	int			 min_e;
	int			 max_e;
	struct ext2_block_entry *retv;
	int			 t;
	blk_t			 tval;

	max_e = state->usedentries - 1;
	min_e = 0;
	retv = NULL;

 repeat:
	if (min_e > max_e)
		goto out;

	t = (min_e + max_e) >> 1;
	tval = state->block[t].num;

	if (tval > block)
		max_e = t - 1;

	if (tval < block)
		min_e = t + 1;

	if (tval != block)
		goto repeat;

	retv = &state->block[t];

 out:
	return retv;
}

static int doblock(struct ext2_fs *fs, struct ext2_block_relocator_state *state,
		   blk_t blk, blk_t refblock, off_t refoffset, int indirect)
{
	struct ext2_block_entry *ent;

	if ((ent = findit(state, blk)) == NULL)
		return 1;

	if (ent->refblock) {
		fprintf(stderr, "error: cross-linked blocks found! "
			"run e2fsck first!\n");
		return 0;
	}

	ent->refblock = refblock;
	ent->refoffset = refoffset;
	ent->isindirectblock = indirect;

	state->resolvedentries++;
	state->start[indirect].num++;

	return 1;
}

static int doindblock(struct ext2_fs *fs,
		      struct ext2_block_relocator_state *state,
		      blk_t blk,
		      blk_t refblock,
		      off_t refoffset)
{
	struct ext2_buffer_head *bh;
	int			 i;
	__u32			*uptr;

	if (!doblock(fs, state, blk, refblock, refoffset, 1))
		return 0;

	bh = ext2_bread(fs, blk);
	uptr = (__u32 *)bh->data;

	for (i = 0; i < (fs->blocksize >> 2); i++)
		if (uptr[i])
			if (!doblock(fs, state, uptr[i], blk, i<<2, 0))
				return 0;

	ext2_brelse(bh, 0);

	return 1;
}

static int dodindblock(struct ext2_fs *fs,
		       struct ext2_block_relocator_state *state,
		       blk_t blk,
		       blk_t refblock,
		       off_t refoffset)
{
	struct ext2_buffer_head *bh;
	int			 i;
	__u32			*uptr;

	if (!doblock(fs, state, blk, refblock, refoffset, 2))
		return 0;

	bh = ext2_bread(fs, blk);
	uptr = (__u32 *)bh->data;

	for (i = 0; i < (fs->blocksize >> 2); i++)
		if (uptr[i])
			if (!doindblock(fs, state, uptr[i], blk, i<<2))
				return 0;

	ext2_brelse(bh, 0);

	return 1;
}

static int dotindblock(struct ext2_fs *fs,
		       struct ext2_block_relocator_state *state,
		       blk_t blk,
		       blk_t refblock,
		       off_t refoffset)
{
	struct ext2_buffer_head *bh;
	int			 i;
	__u32			*uptr;

	if (!doblock(fs, state, blk, refblock, refoffset, 3))
		return 0;

	bh = ext2_bread(fs, blk);
	uptr = (__u32 *)bh->data;

	for (i = 0; i < (fs->blocksize >> 2); i++)
		if (uptr[i])
			if (!dodindblock(fs, state, uptr[i], blk, i<<2))
				return 0;

	ext2_brelse(bh, 0);

	return 1;
}



/* Ehrm.... sorry, pedanticists! :-) */
#define offsetof(type, field) ((long)(&(((type *)0)->field)))

static int doinode(struct ext2_fs *fs, struct ext2_block_relocator_state *state,
		   ino_t ino)
{
	struct ext2_inode tmp;
	struct ext2_inode *inode = &tmp;

	ext2_read_inode(fs, ino, inode);

	if (inode->i_blocks) {
		blk_t blk;
		int   i;
		off_t inodeoffset;
		blk_t inodeblock;

		inodeoffset = ext2_get_inode_offset(fs, ino, &inodeblock);
		for (i = 0; i < EXT2_NDIR_BLOCKS; i++)
			if ((blk = inode->i_block[i]))
				if (!doblock(fs, state, blk, inodeblock,
					     inodeoffset +
					     offsetof(struct ext2_inode,
						      i_block[i]), 0))
					return 0;

		if ((blk = inode->i_block[EXT2_IND_BLOCK]))
			if (!doindblock(fs, state, blk, inodeblock,
					inodeoffset +
					offsetof(struct ext2_inode,
						 i_block[EXT2_IND_BLOCK])))
				return 0;

		if ((blk = inode->i_block[EXT2_DIND_BLOCK]))
			if (!dodindblock(fs, state, blk, inodeblock,
					 inodeoffset +
					 offsetof(struct ext2_inode,
						  i_block[EXT2_DIND_BLOCK])))
				return 0;

		if ((blk = inode->i_block[EXT2_TIND_BLOCK]))
			if (!dotindblock(fs, state, blk, inodeblock,
					 inodeoffset +
					 offsetof(struct ext2_inode,
						  i_block[EXT2_TIND_BLOCK])))
				return 0;

	}

	return 1;
}

static int inode_block_scan(struct ext2_fs *fs,
			    struct ext2_block_relocator_state *state)
{
	int i;

	state->start[0].num = 0;
	state->start[1].num = 0;
	state->start[2].num = 0;
	state->start[3].num = 0;

	for (i = 0; i < fs->numgroups; i++) {
		struct ext2_buffer_head *bh;
		int			 j;
		int			 offset;

		if (fs->flags & FL_VERBOSE) {
			printf(" scanning group %i... ", i);
			fflush(stdout);
		}

		bh = ext2_bread(fs, fs->gd[i].bg_inode_bitmap);
		offset = i * fs->sb.s_inodes_per_group + 1;

		for (j = 0; j < fs->sb.s_inodes_per_group; j++)
			if (check_bit(bh->data, j)) {
				if (!doinode(fs, state, offset + j)) {
					ext2_brelse(bh, 0);
					return 0;
				}

				if (state->resolvedentries == state->usedentries)
					break;
			}

		ext2_brelse(bh, 0);

		if (fs->flags & FL_VERBOSE) {
			printf("%i/%i blocks resolved\r",
			       state->resolvedentries,
			       state->usedentries);
			fflush(stdout);
		}

		if (state->resolvedentries == state->usedentries)
			break;
	}

	if (fs->flags & FL_VERBOSE)
		printf("\n");

	state->start[3].dst = state->block;
	state->start[2].dst = state->start[3].dst + state->start[3].num;
	state->start[1].dst = state->start[2].dst + state->start[2].num;
	state->start[0].dst = state->start[1].dst + state->start[1].num;

	return (state->resolvedentries == state->usedentries);
}


static int ext2_block_relocator_copy(struct ext2_fs *fs,
				     struct ext2_block_relocator_state *state)
{
	unsigned char *buf;
	int ret = 1;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	if ((buf = (unsigned char *)malloc(MAXCONT << fs->logsize)) != NULL) {
		int num;
		int numleft;
		struct ext2_block_entry *ptr;

		numleft = state->usedentries;
		ptr = state->block;
		while (numleft && ret) {
			num = min(numleft, MAXCONT);
			while (num != 1) {
				if (ptr[0].num + num - 1 == ptr[num - 1].num &&
				    ptr[0].dest + num - 1 == ptr[num - 1].dest)
					break;

				num >>= 1;
			}

			ext2_bcache_flush_range(fs, ptr[0].num, num);
			ext2_bcache_flush_range(fs, ptr[0].dest, num);

			ext2_read_blocks(fs, buf, ptr[0].num, num);
			ext2_write_blocks(fs, buf, ptr[0].dest, num);

			ptr += num;
			numleft -= num;

			if (fs->flags & FL_VERBOSE) {
				printf("copied %i/%i blocks\r",
				       state->usedentries - numleft,
				       state->usedentries);
				fflush(stdout);
			}
		}

		free(buf);

		if (fs->flags & FL_SAFE)
			ext2_sync(fs);

		if (fs->flags & FL_VERBOSE)
			printf("\n");
	} else {
		int i;
		int ret = 1;

		for (i = 0; i < state->usedentries && ret; i++) {
			struct ext2_block_entry *block;

			block = &state->block[i];
			ext2_copy_block(fs, block->num, block->dest);
		}
	}

	return ret;
}

static int ext2_block_relocator_ref(struct ext2_fs *fs,
				    struct ext2_block_relocator_state *state,
				    struct ext2_block_entry *block)
{
	struct ext2_buffer_head	*bh;
	static int numerrors = 0;

	if (!block->refblock && !block->refoffset) {
		fprintf(stderr,
			"block %i has no reference? weird\n",
			block->num);

		return 0;
	}

	bh = ext2_bread(fs, block->refblock);

	if ((*((__u32 *)(bh->data + block->refoffset))) != block->num) {
		fprintf(stderr,
			"block %i ref error! (->%i {%i, %i})\n",
			block->num,
			block->dest,
			block->refblock,
			block->refoffset);
		ext2_brelse(bh, 0);

		if (numerrors++ < 4)
			return 1;

		fprintf(stderr, "all is not well!\n");
		return 0;
	}

	*((__u32 *)(bh->data + block->refoffset)) = block->dest;
	bh->dirty = 1;
	ext2_brelse(bh, 0);

	ext2_set_block_state(fs, block->dest, 1, 1);
	ext2_set_block_state(fs, block->num, 0, 1);

	if (block->isindirectblock) {
		struct ext2_block_entry *dst;
		int			 i;
		int			 num;

		dst = state->start[block->isindirectblock-1].dst;
		num = state->start[block->isindirectblock-1].num;

		for (i = 0; i < num; i++)
			if (dst[i].refblock == block->num)
				dst[i].refblock = block->dest;
	}

	return 1;
}

static int ext2_block_relocator_grab_blocks(struct ext2_fs *fs,
					    struct ext2_block_relocator_state *state)
{
	int group;
	blk_t start;
	int ptr;

	ptr = 0;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	for (group = 0, start = fs->sb.s_first_data_block;
	     group < fs->numgroups;
	     group++, start += fs->sb.s_blocks_per_group) {
		if (fs->gd[group].bg_free_blocks_count) {
			struct ext2_buffer_head *bh;
			int j;
			int bpg;
			int raid_bb, raid_ib;
			int itend = state->newallocoffset;

			bpg = fs->sb.s_blocks_per_group;
			if (start + bpg > fs->newblocks)
				bpg = fs->newblocks - start;

			raid_bb = itend + (fs->stride * group % (bpg - itend));
			raid_ib = (raid_bb + 1 >= bpg) ? itend : raid_bb + 1;

			bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);

			for (j = itend; j < bpg; j++) {
				/* Don't allocate new RAID bb and ib blocks */
				if (fs->stride &&
				    (j == raid_bb || j == raid_ib))
					continue;

				if (!check_bit(bh->data, j)) {
					state->block[ptr++].dest = start + j;

					if (ptr == state->usedentries) {
						ext2_brelse(bh, 0);
						return 1;
					}
				}

			}
			ext2_brelse(bh, 0);
		}
	}

	fprintf(stderr, "%s: not enough free blocks to fill list\n", fs->prog);
	return 0;
}

static int ext2_block_relocator_flush(struct ext2_fs *fs,
				      struct ext2_block_relocator_state *state)
{
	int i;

	if (!state->usedentries)
		return 1;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	qsort(state->block, state->usedentries, sizeof(struct ext2_block_entry),
	      compare_block_entries);

	if (!inode_block_scan(fs, state))
		return 0;

	if (!ext2_block_relocator_grab_blocks(fs, state))
		return 0;

	ext2_block_relocator_copy(fs, state);

	qsort(state->block, state->usedentries, sizeof(struct ext2_block_entry),
	      compare_block_entries_ind);

	for (i = 3; i >= 0; i--) {
		struct ext2_block_entry *dst;
		int			 j;
		int			 num;

		dst = state->start[i].dst;
		num = state->start[i].num;

		if (!num)
			continue;

		if (fs->flags & FL_VERBOSE) {
			printf("relocating %s blocks\n",
			       ((char *[4]){"direct",
					    "singly indirect",
					    "doubly indirect",
					    "triply indirect"})[i]);
			fflush(stdout);
		}

		qsort(dst, num, sizeof(struct ext2_block_entry),
		      compare_block_entries_ref);

		for (j = 0; j < num; j++)
			if (!ext2_block_relocator_ref(fs, state, &dst[j]))
				return 0;

		if (fs->flags & FL_SAFE)
			ext2_sync(fs);
	}

	state->usedentries = 0;
	state->resolvedentries = 0;

	return 1;
}

static int ext2_block_relocator_mark(struct ext2_fs *fs,
				     struct ext2_block_relocator_state *state,
				     blk_t block)
{
	int i;

	if (fs->flags & FL_DEBUG)
		printf("marking block %u for relocation\r", block);

	if (!ext2_get_block_state(fs, block) || !ext2_is_data_block(fs, block)){
		fprintf(stderr,
			"BUG! block %i shouldn't have been marked!\nPlease "
			"send a filesystem dump and ext2resize version to:\n"
			"<ext2resize-devel@lists.sourceforge.net>\n\n",
			block);

		return 0;
	}

	if (state->usedentries == state->allocentries - 1)
		if (!ext2_block_relocator_flush(fs, state))
			return 0;

	i = state->usedentries;
	state->block[i].num = block;
	state->block[i].dest = 0;
	state->block[i].refblock = 0;
	state->block[i].refoffset = 0;

	state->usedentries++;
	return 1;
}

static int ext2_block_relocate_grow(struct ext2_fs *fs,
				    struct ext2_block_relocator_state *state)
{
	struct ext2_inode tmp;
	struct ext2_inode *inode = &tmp;
	int   newitoffset;
	blk_t start;
	int   group;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	newitoffset = fs->newgdblocks + 3;
	if (fs->newgdblocks <= fs->gdblocks + fs->resgdblocks)
		return 1;

	ext2_read_inode(fs, EXT2_RESIZE_INO, inode);
	inode->i_mtime = 0;
	state->newallocoffset = newitoffset + fs->inodeblocks;
	if (fs->flags & FL_DEBUG)
		printf("moving data (oldgdblocks %d+%d, newgdblocks %d, "
		       "olditoffset %d, newitoffset %d, newallocoffset %d)\n",
		       fs->gdblocks, fs->resgdblocks, fs->newgdblocks,
		       fs->itoffset, newitoffset, state->newallocoffset);

	for (group = 0, start = fs->sb.s_first_data_block;
	     group < fs->numgroups;
	     group++, start += fs->sb.s_blocks_per_group) {
		struct ext2_buffer_head *bh;
		blk_t olditoffset, olditend;
		blk_t newitend;
		blk_t bb, new_bb;
		blk_t ib, new_ib;
		blk_t j;
		int need_reloc;
		int has_sb;
		int bpg;

		need_reloc = 0;
		olditoffset = fs->gd[group].bg_inode_table - start;
		olditend = olditoffset + fs->inodeblocks;
		newitend = newitoffset + fs->inodeblocks;

		bpg = fs->sb.s_blocks_per_group;
		if (start + bpg > fs->sb.s_blocks_count)
			bpg = fs->sb.s_blocks_count - start;

		has_sb = ext2_bg_has_super(fs, group);

		bb = fs->gd[group].bg_block_bitmap - start;
		ib = fs->gd[group].bg_inode_bitmap - start;
		if (fs->stride) {
			blk_t new_bmap = newitend +
				(fs->stride * group % (bpg - newitend));
			new_bb = new_bmap >= bpg ? newitend : new_bmap;
		} else
			new_bb = has_sb ? newitoffset - 2 : 0;
		new_ib = (new_bb + 1 >= bpg) ? newitend : new_bb + 1;

		bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);

		/* Mark blocks for relocation that fall in the end of the
		 * new group descriptor table (if they are not metadata).
		 */
		if (has_sb) {
			if (fs->flags & FL_DEBUG)
				printf("new gd table: start=%d, end=%d\n",
				       start + 1, start + fs->newgdblocks);
			for (j = fs->gdblocks + 1; j <= fs->newgdblocks; j++) {
				if (j == bb || j == ib) {
					need_reloc = 1;
					if (fs->flags & FL_DEBUG)
						printf("skip bitmap %d\n",
						       start + j);
					continue;
				}
				if (j >= olditoffset && j < olditend) {
					if (fs->flags & FL_DEBUG)
						printf("skip inode table %d\n",
						       start + j);
					continue;
				}
				if (ext2_block_iterate(fs, inode, start + j,
						       EXT2_ACTION_FIND) >= 0) {
					if (fs->flags & FL_DEBUG)
						printf("skip reserved %d\n",
						       start + j);
					continue;
				}

				if (check_bit(bh->data, j) &&
				    !ext2_block_relocator_mark(fs, state,
							       start + j)) {
					ext2_brelse(bh, 0);
					return 0;
				} else if (fs->flags & FL_DEBUG)
					printf("relocating block %d for GDT\n",
					       start + j);
			}
		}
		if (newitend - olditend <= 0 && !need_reloc) {
			if (fs->flags & FL_VERBOSE)
				printf("don't need relocation for group %d\n",
				       group);
			continue;
		}
		/* Mark blocks for relocation that fall under the new
		 * inode table (don't mark old inode or block birmaps).
		 */
		if (fs->flags & FL_DEBUG)
			printf("inode table: old=%d, new=%d\n",
			       fs->gd[group].bg_inode_table,
			       start + newitoffset);
		for (j = newitoffset; j < newitend; j++) {
			if (fs->flags & FL_DEBUG)
				printf("check new inode table %d\n", j + start);
			if (j < olditend || j == bb || j == ib ||
			    j == new_bb || j == new_ib)
				continue;
			if (check_bit(bh->data, j) &&
			    !ext2_block_relocator_mark(fs, state,
						       start + j)) {
				ext2_brelse(bh, 0);
				return 0;
			} else if (fs->flags & FL_DEBUG)
				printf("relocating block %d for itable\n",
				       start + j);
		}
		/* Mark blocks for relocation that fall under the new inode
		 * or block bitmaps.
		 */
		if (fs->flags & FL_DEBUG)
			printf("checking bb: old %d, new %d\n", start + bb,
			       start + new_bb);
		if (new_bb != bb && new_bb != ib &&
		    (new_bb < olditoffset || new_bb >= olditend) &&
		    check_bit(bh->data, new_bb)) {
			if (!ext2_block_relocator_mark(fs,state,start+new_bb)) {
				ext2_brelse(bh, 0);
				return 0;
			} else if (fs->flags & FL_DEBUG)
				printf("relocating block %d for bb\n", start+j);
		}
		if (fs->flags & FL_DEBUG)
			printf("checking ib: old %d, new %d\n", start + ib,
			       start + new_ib);
		if (new_ib != ib && new_ib != bb &&
		    (new_ib < olditoffset || new_ib >= olditend) &&
		    check_bit(bh->data, new_ib)) {
			if (!ext2_block_relocator_mark(fs,state,start+new_ib)) {
				ext2_brelse(bh, 0);
				return 0;
			} else if (fs->flags & FL_DEBUG)
				printf("relocating block %d for ib\n", start+j);
		}

		ext2_brelse(bh, 0);
	}

	if (inode->i_mtime)
		ext2_write_inode(fs, EXT2_RESIZE_INO, inode);
	return ext2_block_relocator_flush(fs, state);
}

static int ext2_block_relocate_shrink(struct ext2_fs *fs,
				      struct ext2_block_relocator_state *state)
{
	struct ext2_inode tmp;
	struct ext2_inode *inode = &tmp;
	int group;
	blk_t start;

	if (fs->flags & FL_DEBUG)
		printf("%s\n", __FUNCTION__);

	state->newallocoffset = fs->itoffset + fs->inodeblocks;
	ext2_read_inode(fs, EXT2_RESIZE_INO, inode);
	inode->i_mtime = 0;

	for (group = 0, start = fs->sb.s_first_data_block;
	     group < fs->numgroups;
	     group++, start += fs->sb.s_blocks_per_group) {
		struct ext2_buffer_head *bh;
		blk_t			 begin;
		blk_t			 bb;
		blk_t			 ib;
		blk_t			 it;
		int			 itend;
		int			 bpg;
		int			 has_sb;
		int			 delgrp = 0;
		int			 j;

		if (fs->newblocks >= start + fs->sb.s_blocks_per_group)
			continue;		/* group will survive */

		bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);

		bpg = fs->sb.s_blocks_per_group;
		if (start + bpg > fs->sb.s_blocks_count)
			bpg = fs->sb.s_blocks_count - start;

		bb = fs->gd[group].bg_block_bitmap - start;
		ib = fs->gd[group].bg_inode_bitmap - start;
		it = fs->gd[group].bg_inode_table - start;
		itend = it + fs->inodeblocks;

		has_sb = ext2_bg_has_super(fs, group);

		if (start + itend + 50 >= fs->newblocks) {
			if (fs->flags & FL_DEBUG)
				printf("will remove group %d %s superblock\n",
				       group, has_sb ? "with" : "without");
			/* group is fully chopped off */
			begin = has_sb ? fs->gdblocks + 1 : 0;
			delgrp = 1;
		} else {
			/* group is partly chopped off */
			begin = fs->newblocks - start;
			if (fs->flags & FL_DEBUG)
				printf("will truncate group %d at %d blocks\n",
				       group, begin);

			/* FIXME need to handle case where RAID bitmaps are
			 *       past end of fs, and relocate them...
			 *       bb > bpg || ib > bpg
			 */
		}

		for (j = begin; j < bpg; j++) {
			if (!check_bit(bh->data, j))
				/* skip unused blocks */
				continue;

			if (j >= it && j < itend) {
				if (delgrp)
					continue;
				fprintf(stderr, 
					"%s: trying to move block %d in itable!\n",
					__FUNCTION__,
					j + start);
				return 0;
			}

			if (j == bb || j == ib) {
				int new = (j == bb ? itend : itend + 1);

				if (delgrp)
					continue;
				if (fs->flags & FL_DEBUG)
					printf("moving RAID %s bitmap at %d\n",
					       j==bb ?"block":"inode", start+j);
				if (check_bit(bh->data, new) &&
				    !ext2_block_relocator_mark(fs, state,
							       start + new)) {
					ext2_brelse(bh, 0);
					return 0;
				}
				continue;
			}

			if (has_sb && j < 1 + fs->gdblocks + fs->resgdblocks &&
			    ext2_block_iterate(fs, inode, start + j,
					       EXT2_ACTION_DELETE) >= 0)
				/* remove reserved blocks without relocation */
				continue;

			if (!ext2_block_relocator_mark(fs, state, start + j)) {
				ext2_brelse(bh, 0);
				return 0;
			}
		}

		ext2_brelse(bh, 0);
	}

	if (inode->i_mtime)
		ext2_write_inode(fs, EXT2_RESIZE_INO, inode);
	return ext2_block_relocator_flush(fs, state);
}

int ext2_block_relocate(struct ext2_fs *fs)
{
	struct ext2_block_relocator_state state;

	if (fs->flags & FL_VERBOSE)
		printf("relocating blocks....\n");

	state.newallocoffset = 0;
	state.allocentries = (ext2_relocator_pool_size << 10) /
		sizeof(struct ext2_block_entry);
	state.usedentries = 0;
	state.resolvedentries = 0;
	state.block = (struct ext2_block_entry *)fs->relocator_pool;

	if (fs->newblocks < fs->sb.s_blocks_count)
		return ext2_block_relocate_shrink(fs, &state);

	return ext2_block_relocate_grow(fs, &state);
}
