/*
    ext2_meta.c -- ext2 metadata mover
    Copyright (C) 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2_meta_c[] = "$Id: ext2_meta.c,v 1.12 2000/10/04 08:59:00 adilger Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include "ext2.h"

int ext2_inode_bitmap_push(struct ext2_fs *fs, int group, int diff)
{
	blk_t dst = fs->gd[group].bg_inode_bitmap + diff;

	if (fs->flags & FL_DEBUG)
		printf("move group %d inode bitmap by %d\n", group, diff);

	if (diff == 0)
		return 1;

	if (ext2_get_block_state(fs, dst)) {
		fprintf(stderr, "error: block relocator should have moved %i\n",
			dst);
		return 0;
	}

	ext2_set_block_state(fs, dst, 1, 0);
	ext2_set_block_state(fs, fs->gd[group].bg_inode_bitmap, 0, 0);
	ext2_copy_block(fs, fs->gd[group].bg_inode_bitmap, dst);
	fs->gd[group].bg_inode_bitmap = dst;
	fs->metadirty |= EXT2_META_GD;

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);
	return 1;
}

int ext2_block_bitmap_push(struct ext2_fs *fs, int group, int diff)
{
	blk_t dst = fs->gd[group].bg_block_bitmap + diff;

	if (fs->flags & FL_DEBUG)
		printf("move group %d block bitmap by %d\n", group, diff);

	if (diff == 0)
		return 1;

	if (ext2_get_block_state(fs, dst)) {
		fprintf(stderr, "error: block relocator should have moved %i\n",
			dst);
		return 0;
	}

	ext2_set_block_state(fs, dst, 1, 0);
	ext2_set_block_state(fs, fs->gd[group].bg_block_bitmap, 0, 0);
	ext2_copy_block(fs, fs->gd[group].bg_block_bitmap, dst);
	fs->gd[group].bg_block_bitmap = dst;
	fs->metadirty |= EXT2_META_GD;

	if (fs->flags & FL_SAFE)
		ext2_sync(fs);
	return 1;
}

int ext2_metadata_push(struct ext2_fs *fs)
{
	int new_itoffset;
	int group;
	blk_t start;

	if (fs->flags & FL_DEBUG)
		printf(__FUNCTION__ "\n");

	new_itoffset = fs->newgdblocks + 3;

	if (new_itoffset <= fs->itoffset)
		return 1;

	if (fs->flags & FL_DEBUG)
		printf("moving metadata (old offset +%d, new offset +%d)\n",
		       fs->itoffset, new_itoffset);

	for (group = 0, start = fs->sb.s_first_data_block;
	     group < fs->numgroups;
	     group++, start += fs->sb.s_blocks_per_group) {
		blk_t old_itend;
		blk_t new_itend;
		blk_t bb, new_bb;
		blk_t ib, new_ib;
		blk_t j;
		int bpg;
		int has_sb;
		int diff;

		diff = start + new_itoffset - fs->gd[group].bg_inode_table;
		old_itend =fs->gd[group].bg_inode_table +fs->inodeblocks -start;
		new_itend = new_itoffset + fs->inodeblocks;
		bpg = fs->sb.s_blocks_per_group;
		if (start + bpg > fs->sb.s_blocks_count)
			bpg = fs->sb.s_blocks_count - start;

		has_sb = ext2_bg_has_super(fs, group);

		bb = fs->gd[group].bg_block_bitmap - start;
		ib = fs->gd[group].bg_inode_bitmap - start;
		if (fs->stride) {
			blk_t new_bmap = new_itend +
				(fs->stride * group % (bpg - new_itend));
			new_bb = new_bmap >= bpg ? new_itend : new_bmap;
		} else
			new_bb = has_sb ? new_itoffset - 2 : 0;
		new_ib = (new_bb + 1 >= bpg) ? new_itend : new_bb + 1;

		if (fs->flags & FL_DEBUG)
			printf("moving group %d (bb %d/%d, ib %d/%d, "
			       "itend %d/%d)\n", group, start + bb,
			       start + new_bb, start + ib, start + new_ib,
			       start + old_itend, start + new_itend);

		if (diff <= 0 && new_bb == bb && new_ib == ib)
			continue;

		if (fs->stride) {
			if (!ext2_inode_bitmap_push(fs, group, new_ib - ib) ||
			    !ext2_block_bitmap_push(fs, group, new_bb - bb))
				return 0;
			if (fs->flags & FL_SAFE)
				ext2_sync(fs);
		}

		/* inode table */
		for (j = 0; j < diff; j++)
			if (ext2_get_block_state(fs, start + old_itend + j)) {
				fprintf(stderr, "error: block relocator "
					"should have moved %i\n",
					start + old_itend + j);
				return 0;
			} else
				ext2_set_block_state(fs, start + old_itend + j,
						     1, 0);

		if (fs->flags & FL_DEBUG)
			printf("move group %d inode table by %d\n", group,diff);
		ext2_move_blocks(fs, fs->gd[group].bg_inode_table,
				 fs->inodeblocks, start + new_itoffset);
		for (j = 0; j < diff; j++)
			ext2_set_block_state(fs, fs->gd[group].bg_inode_table+j,
					     0, 0);
		fs->gd[group].bg_inode_table = start + new_itoffset;

		fs->metadirty |= EXT2_META_GD;

		if (fs->flags & FL_SAFE)
			ext2_sync(fs);

		/* move block bitmap and inode bitmap */
		if (has_sb && !fs->stride) {
			if (!ext2_inode_bitmap_push(fs, group, new_bb - bb) ||
			    !ext2_block_bitmap_push(fs, group, new_ib - ib))
				return 0;

			ext2_zero_blocks(fs, start + fs->gdblocks + 1,
					 fs->newgdblocks - fs->gdblocks);
			if (fs->flags & FL_SAFE)
				ext2_sync(fs);
		}

		if (fs->flags & FL_VERBOSE)
			printf(" ....completed group %d/%d\r",
			       group + 1, fs->numgroups);
	}

	fs->itoffset = new_itoffset;

	if (fs->flags & FL_VERBOSE)
		printf("\n");

	return 1;
}
