/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/******************************************************************
*
*  File: modify.c
*
*  Contents:  low-level triangulation modification routines:
*
*            edge_divide() - subdivide an edge
*            cross_cut() -  subdivide a facet
*
*/

#include "include.h"

/******************************************************************
*
*  Function: face_triangulate()
*
*  Purpose:  Triangulate a n-sided face by putting in central vertex.
*/

void face_triangulate(f_id,edgecount)
facet_id f_id;
int edgecount;
{
  int i,j,k;
  vertex_id center; 
  facetedge_id fe,pre_fe;
  vertex_id rimv;
  edge_id spoke;
  facet_id fe_in,fe_out,next_fe;
  REAL *centerx,*x;
  struct boundary *bdry;
  WRAPTYPE wrap = 0;

  if ( web.modeltype == LAGRANGE ) 
     kb_error(1237,"Can't face_triangulate() in Lagrange model.\n",RECOVERABLE);


  if ( web.torus_flag )
  { /* check for REAL wrapping in torus */
     fe = get_facet_fe(f_id);
     for ( j = 0, wrap = 0 ; j < edgecount ; j++, fe = get_next_edge(fe) )
     { 
        wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
        for ( i = 0 ; i < SDIM ; i++ )
        { switch ( (wrap >> (i*TWRAPBITS)) & WRAPMASK )
          { case POSWRAP: case NEGWRAP: case 0: break;
             default: set_facet_fe(f_id,fe); break;
          }
        }
     }
  }

  /* put a new vertex in the center */
  center = new_vertex(NULL,f_id);
  if ( get_fattr(f_id) & FIXED )
     set_attr(center,FIXED);

  /* center coordinates are average of vertices */
  centerx = get_coord(center);
  for ( i = 0 ; i < SDIM ; i++ ) centerx[i] = 0.0;
  generate_facet_fe_init();
  while ( generate_facet_fe(f_id,&fe) )
  { REAL w[MAXCOORD];

    x = get_coord(get_fe_tailv(fe));
    if ( web.symmetry_flag )
      { (*sym_wrap)(x,w,wrap);
         x = w;
         wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
      }
    for ( i = 0 ; i < SDIM ; i++ ) 
      centerx[i] += x[i];
  }
  for ( i = 0 ; i < SDIM ; i++ ) centerx[i] /= edgecount;

  /* find centerpoint parameters for facet on boundary */
  if ( get_fattr(f_id) & BOUNDARY )    /* not working for torus */
     {
        REAL defaultp[MAXCOORD];
        REAL *paramb,*parammid,*xb;
        vertex_id base_v;
        REAL s[MAXCOORD];
        REAL midp[MAXCOORD];
        int no_interp = 0;
        int vercount = 0;

        set_attr(center,BOUNDARY);
        bdry = get_facet_boundary(f_id);
        set_boundary_num(center,bdry->num);
        for ( i = 0 ; i < bdry->pcount ; i++ ) midp[i] = 0.;

        /* center parameters extrapolate from a vertex */
        /* try to find a vertex on same boundary */
        base_v = NULLVERTEX;
        generate_facet_fe_init();
        while ( generate_facet_fe(f_id,&fe) )
          { 
             if ( bdry == get_boundary(get_fe_tailv(fe)) )
             { base_v = get_fe_tailv(fe);
                for ( i = 0 ; i < bdry->pcount ; i++ ) 
                    midp[i] += get_param(base_v)[i];
             }
             else no_interp = 1;
             vercount++;
          }
        if ( valid_id(base_v) )
          { paramb = get_param(base_v);
             xb = get_coord(base_v);
             for ( i = 0 ; i < SDIM ; i++ )
                s[i] = xb[i];  /* displacement vector */
          }
        else
          { paramb = defaultp;
             defaultp[0] = defaultp[1] = defaultp[2] = 0.1;
             for ( i = 0 ; i < SDIM ; i++ )
                s[i] = eval(bdry->coordf[i],defaultp,NULLID);
             sprintf(msg,"Could not find vertex on same boundary as facet %d.\n",
                 get_original(f_id));
          }

        parammid = get_param(center);
        if ( interp_bdry_param && !no_interp )
        { for ( i = 0 ; i < bdry->pcount ; i++ )
             parammid[i] = midp[i]/vercount;
          for ( k = 0 ; k < SDIM ; k++ )
            centerx[k] = eval(bdry->coordf[k],parammid,center);
        } 
        else b_extrapolate(bdry,s,centerx,centerx,paramb,parammid,center);
     }

  /* install edge from rim to center */
  fe = get_facet_fe(f_id);  /* canonical starting point */
  pre_fe = get_prev_edge(fe);
  rimv = get_fe_tailv(fe);
  spoke = new_edge(rimv,center,f_id);
  if ( get_fattr(f_id) & FIXED )
     set_attr(spoke,FIXED);
  if ( get_fattr(f_id) & NO_REFINE )
     set_attr(spoke,NO_REFINE);
  if ( get_fattr(f_id) & BOUNDARY )    
     {
        set_attr(spoke,BOUNDARY);
        bdry = get_facet_boundary(f_id);
        set_edge_boundary_num(spoke,bdry->num);
     }
  else if ( get_fattr(f_id) & CONSTRAINT )    
     {
        ATTR attr = get_fattr(f_id) & (BDRY_ENERGY | BDRY_CONTENT | CONSTRAINT );
        conmap_t * conmap = get_f_constraint_map(f_id);

        set_attr(spoke,attr);
        set_attr(center,attr);
        set_e_conmap(spoke,conmap);
        set_v_conmap(center,conmap);
        project_v_constr(center,ACTUAL_MOVE);
        if ( web.modeltype == QUADRATIC )
          {
             vertex_id mid = get_edge_midv(spoke);
             set_attr(mid,attr);
             set_v_conmap(mid,conmap);
             project_v_constr(mid,ACTUAL_MOVE);
          }
     }

  if ( web.symmetry_flag )
          set_edge_wrap(spoke,0);
  fe_in = new_facetedge(f_id,spoke);
  set_edge_fe(spoke,fe_in);
  set_prev_edge(fe_in,pre_fe);
  set_next_edge(pre_fe,fe_in);
  fe_out = new_facetedge(f_id,edge_inverse(spoke));
  set_prev_edge(fe_out,fe_in);
  set_next_edge(fe_in,fe_out);
  set_prev_edge(fe,fe_out);
  set_next_edge(fe_out,fe);
  set_next_facet(fe_in,fe_inverse(fe_out));
  set_prev_facet(fe_in,fe_inverse(fe_out));
  set_next_facet(fe_out,fe_inverse(fe_in));
  set_prev_facet(fe_out,fe_inverse(fe_in));

  /* now go around cutting off triangles */
  while ( !equal_id(get_next_edge(fe),fe_in) )
     { next_fe = get_next_edge(fe); 
        cross_cut(get_prev_edge(fe),fe);
        fe = next_fe;
     }    

 if ( web.symmetry_flag ) /* check for axial vertices */
 {
    fe = get_facet_fe(f_id);
    for ( i = 0 ; i < FACET_VERTS ; i++ )
    { if ( get_vattr(get_fe_tailv(fe)) & AXIAL_POINT )
      { set_facet_fe(f_id,fe); break; }
      fe = get_next_edge(fe);
    }
 }

  top_timestamp = ++global_timestamp;
}


/*****************************************************************
*
* function: unstar()
*
* purpose: remove center vertex from starred triangle
*    necessary for edge removal of one side
*
* return: -1 failure, 0 unremovable star, 1 success
*/

int unstar(fe_a)
facetedge_id fe_a;  /* removable vertex is at tail */
{ vertex_id v_id = get_fe_tailv(fe_a);
  facet_id fkeep = get_fe_facet(fe_a);
  facetedge_id fe_b = get_next_edge(fe_a);
  facetedge_id fe_c = get_next_edge(fe_b);
  facetedge_id fe_d = get_next_facet(fe_c);
  facetedge_id fe_e = get_next_edge(fe_d);
  facetedge_id fe_f = get_next_edge(fe_e);
  facetedge_id fe_g = get_next_facet(fe_e);
  facetedge_id fe_h = get_next_edge(fe_g);
  facetedge_id fe_i = get_next_edge(fe_h);

  if ( verbose_flag )
  { sprintf(msg,"Unstarring vertex %d\n",ordinal(v_id)+1);
     outstring(msg);
  }

  /* some checks on legality */
  if ( get_vertex_evalence(v_id) != 3 )  /* not lonesome star */
        { if ( verbose_flag )
          { sprintf(msg,"Unstarring failed, vertex %d has valence %d\n",
                ordinal(v_id)+1,get_vertex_evalence(v_id));
             outstring(msg);
          }
          return 0;
        }
  if ( !equal_id(fe_c,get_next_facet(fe_d)) ) 
        { if ( verbose_flag )
          { sprintf(msg,"Unstarring failed, triple valence edge %d\n",
                ordinal(get_fe_edge(fe_c))+1);
             outstring(msg);
          }
          return -1;
        }
  if ( !equal_id(fe_e,get_next_facet(fe_g))  )
        { if ( verbose_flag )
          { sprintf(msg,"Unstarring failed, triple valence edge %d\n",
                ordinal(get_fe_edge(fe_e))+1);
             outstring(msg);
          }
          return -1;
        }
  if ( !equal_id(inverse_id(fe_a),get_next_facet(fe_i)) ) 
        { if ( verbose_flag )
          { sprintf(msg,"Unstarring failed, triple valence edge %d\n",
                ordinal(get_fe_edge(fe_a))+1);
             outstring(msg);
          }
          return -1;
        }
  if ( !equal_id(inverse_id(fe_a),get_prev_facet(fe_i)) )
        { if ( verbose_flag )
          { sprintf(msg,"Unstarring failed, triple valence edge %d\n",
                ordinal(get_fe_edge(fe_a))+1);
             outstring(msg);
          }
          return -1;
        }

  /* fix up fe's around edges of new big facet */
  set_fe_facet(fe_f,inverse_id(fkeep));
  set_fe_facet(fe_h,fkeep);
  set_facet_fe(fkeep,fe_b);
  set_next_edge(fe_b,inverse_id(fe_f));
  set_next_edge(fe_f,inverse_id(fe_b));
  set_prev_edge(fe_f,inverse_id(fe_h));
  set_prev_edge(fe_h,inverse_id(fe_f));
  set_next_edge(fe_h,fe_b);
  set_prev_edge(fe_b,fe_h);

  /* discard */
  free_element(get_fe_facet(fe_e));
  free_element(get_fe_facet(fe_i));
  free_element(get_fe_edge(fe_a));
  free_element(get_fe_edge(fe_d));
  free_element(get_fe_edge(fe_e));
  free_element(fe_a);
  free_element(fe_c);
  free_element(fe_d);
  free_element(fe_e);
  free_element(fe_g);
  free_element(fe_i);
  free_element(v_id);

  top_timestamp = ++global_timestamp;
  return 1;
}

/*********************************************************
*
*  edge_divide()
*
*  Purpose: Subdivide an edge in the first stage of 
*              refinement.  Marks the new vertex with
*              the NEWVERTEX attribute, and both new edges
*              with the NEWEDGE attribute, so they can be
*              properly skipped during refinement.
*              Also sets FIXED attribute bit of new vertex and
*              edge if the old edge is FIXED.
* 
*  Input:    edge_id e_id - the edge to subdivide
*
*  Output:  new legal configuration with new vertex
*
*
*/
 
void edge_divide(e_id)
edge_id e_id;
{
  REAL s[MAXCOORD],*t,*mu,*mv,*h,m[MAXCOORD],q1[MAXCOORD],q3[MAXCOORD];
  edge_id  new_e;
  vertex_id divider,old_mid=0,new_mid=0,headv,tailv;
  int i,j,k,n;
  facetedge_id new_fe,old_fe;
  int wrap = 0,wrap1=0,wrap2=0;
  REAL w[MAXCOORD];
  vertex_id *oldv,*newv;
  REAL *oldx[20];
  REAL  newx[20][MAXCOORD];
  REAL prod;
  REAL oldxx[20][MAXCOORD];
  REAL *newxx;

  if ( !valid_element(e_id) ) return;

  if ( web.representation == SIMPLEX )
     kb_error(1239,"Edge divide not implemented for simplex representation.\n",
        COMMAND_ERROR);

  if ( verbose_flag )
  { sprintf(msg,"Refining edge %d\n",ordinal(e_id)+1);
     outstring(msg);
  }

  headv = get_edge_headv(e_id);
  tailv = get_edge_tailv(e_id);
  t = get_coord(tailv);
  h = get_coord(headv);
  if ( web.symmetry_flag )
  { wrap = get_edge_wrap(e_id);
    /* always use tail as base, for predictability */
    if ( wrap ) /* see which endpoint closer to origin, to use as base */
    { /* use tail as base */
      (*sym_wrap)(get_coord(headv),w,wrap);
      h = w;
      wrap1 = 0; wrap2 = wrap;
    }
    else wrap1 = wrap2 = 0;
  }
  if ( web.modeltype == LINEAR )
    { 
      for ( k = 0 ; k < SDIM ; k++ ) m[k] = (t[k] + h[k])/2;
      divider = new_vertex(m,e_id);
      set_attr(divider,get_eattr(e_id) & (FIXED|BARE_NAKED));
     }
  else if ( web.modeltype == QUADRATIC )
     { 
        divider = get_edge_midv(e_id);  /* use old midpoint */
        unset_attr(divider,Q_MIDPOINT);
        set_vertex_edge(divider,NULLID); /* so set_vertex_edge works */
        /* get coordinates of new midpoint(s) */
        mu = get_coord(divider);
        for ( k = 0 ; k < SDIM ; k++ )
          {
             q1[k] = 0.375*t[k] + 0.75*mu[k] - 0.125*h[k];
             m[k] = mu[k];
             q3[k] = 0.375*h[k] + 0.75*mu[k] - 0.125*t[k];
          }
        if ( wrap1 )
          { (*sym_wrap)(q1,w,wrap1);
             for ( k = 0 ; k < SDIM ; k++ ) q1[k] = w[k];
          }
     }
  else if ( web.modeltype == LAGRANGE )
    {
      for ( k = 0 ; k < SDIM ; k++ ) m[k] = (t[k] + h[k])/2;
      divider = new_vertex(m,e_id);
      set_attr(divider,get_eattr(e_id) & (FIXED|BARE_NAKED));

      /* get old vertex coordinates */
      oldv = get_edge_vertices(e_id);
      for ( i = 0 ; i <= web.lagrange_order ; i++ )
      { if ( i < web.lagrange_order ) oldx[i] = get_coord(oldv[i]);
         else oldx[i] = h;  /* in case of unwrapping */
         for ( j = 0, prod = 1.0 ; j <= web.lagrange_order ; j++ )
            if ( i != j )  prod *= i-j;
         for ( k = 0 ; k < SDIM ; k++ )
            oldxx[i][k] = oldx[i][k]/prod;
      }
    }

  /* make refine() work properly */
  set_attr(divider,NEWVERTEX);

  remove_vertex_edge(headv,inverse_id(e_id));
  new_e = dup_edge(e_id);
  set_edge_tailv(new_e,divider);
  set_edge_headv(e_id,divider);
  insert_vertex_edge(headv,inverse_id(new_e));
  set_attr(new_e,NEWEDGE | get_eattr(e_id));
  if ( web.symmetry_flag )
     { set_edge_wrap(e_id,wrap1);  /* new vertex in same unit cell as tail */
        set_edge_wrap(new_e,wrap2); 
     }
  if ( (web.modeltype == LINEAR) && web.metric_flag )
     { /* get midpoint in metric middle */
        REAL front,rear;  /* edge lengths */
        REAL tt[MAXCOORD],hh[MAXCOORD];
        REAL *xm = get_coord(divider);
        for(;;)  /* binary search for reasonable endpoint */ 
        {
          calc_edge(e_id);
          rear = get_edge_length(e_id);
          calc_edge(new_e);
          front = get_edge_length(new_e);
          if ( rear < 0.8*front )
          { /* bisect high end */
             t = tt;
             for ( i = 0 ; i < SDIM ; i++ )
             { t[i] = xm[i];
                xm[i] = (t[i] + h[i])/2;
             }
          }
          else if ( rear > 1.2*front )
          { /* bisect low end */
             h = hh;
             for ( i = 0 ; i < SDIM ; i++ )
             { h[i] = xm[i];
                xm[i] = (t[i] + h[i])/2;
             }
          }
          else break;
        } 
     }

     
  if ( web.modeltype == QUADRATIC )
     {
        /* new midpoint for old edge */
        old_mid = new_vertex(q1,e_id);
        set_edge_midv(e_id,old_mid);
        set_attr(old_mid,get_eattr(e_id) & (FIXED|BARE_NAKED));

        /* set up midpoint of new edge */
        new_mid = get_edge_midv(new_e);
        set_attr(new_mid,get_eattr(e_id) & (FIXED|BARE_NAKED));
        mv = get_coord(new_mid);
        for ( k = 0 ; k < SDIM ; k++ ) mv[k] = q3[k];
     }
  else if ( web.modeltype == LAGRANGE )
    { 
      /* calculate new vertex coordinates */
      for ( n = 0 ; n < web.lagrange_order ; n++ )
      { REAL coeff;
         REAL x;
         x = n + 0.5;
         for ( j = 0, prod = 1.0 ; j <= web.lagrange_order ; j++ )
            prod *= x - j;
         for ( k = 0 ; k < SDIM ; k++ ) newx[n][k] = 0.0;
         for ( i = 0 ; i <= web.lagrange_order ; i++ )
         { coeff = prod/(x-i);
            for ( k = 0 ; k < SDIM ; k++ )
                newx[n][k] += coeff*oldxx[i][k];
         }
      }
      /* install in vertices */
      newv = get_edge_vertices(new_e);
      for ( n = 1 ; n < web.lagrange_order ; n++ )
         set_vertex_edge(newv[n],new_e);
      for ( i = web.lagrange_order-2 ; i >= 0 ; i-=2 )
      { newxx = get_coord(newv[i]);
         for ( k = 0 ; k < SDIM ; k++ )
             newxx[k] = oldx[(i+web.lagrange_order)/2][k];
      }
      for ( i = web.lagrange_order-1 ; i >= 0 ; i-=2 )
      { newxx = get_coord(newv[i]);
         for ( k = 0 ; k < SDIM ; k++ )
             newxx[k] = newx[(i+web.lagrange_order-1)/2][k];
      }
      for ( i = 2*((web.lagrange_order-1)/2) ; i > 0 ; i-=2 )
      {
         for ( k = 0 ; k < SDIM ; k++ )
             oldx[i][k] = oldx[i/2][k];
      }
      for ( i = 2*(web.lagrange_order/2)-1 ; i > 0 ; i-=2 )
      {
         for ( k = 0 ; k < SDIM ; k++ )
             oldx[i][k] = newx[i/2][k];
      }
      /* change vertices in neighboring facets */
      if ( web.representation == SOAPFILM )
      { facetedge_id fe = get_edge_fe(e_id); 
         facetedge_id start_fe = fe;
         if ( valid_id(fe) )
          do
          { facet_id f_id = get_fe_facet(fe);
             vertex_id *v = get_facet_vertices(f_id);
             for ( i = web.lagrange_order-1; i > 0 ; i-- )
             { for ( k = 0; k < web.skel[FACET].ctrlpts; k++ )
                  if ( v[k] == oldv[i] )
                     v[k] = (i <= (web.lagrange_order-1)/2) ?
                                oldv[2*i] : newv[2*i-web.lagrange_order];
             }
             fe = get_next_facet(fe);
          } while ( !equal_id(start_fe,fe) ); 
        }
     }

  /* for free boundary edges, cannot just interpolate parameters
      due to wrap-around of periodic parameters. So tangent extrapolate
      from one endpoint.
    */
  if ( get_eattr(e_id) & BOUNDARY )
     { 
        struct boundary *bdry;
        REAL *paramb,*parammid;
        REAL  defaultp[MAXCOORD];

        bdry = get_edge_boundary(e_id);
        set_edge_boundary_num(new_e,bdry->num);
        if ( web.modeltype == LINEAR )
          { REAL *parama;
             set_attr(divider,BOUNDARY);
             set_boundary_num(divider,bdry->num);

             /* find parameters of new midpoint */
             mv = get_coord(divider);
             parammid = get_param(divider);
             if ( interp_bdry_param &&
                  (get_boundary(headv) == bdry) && (get_boundary(tailv) == bdry) )
              {
                 parama = get_param(tailv);
                 paramb = get_param(headv);
                 for (  i = 0 ; i < bdry->pcount ; i++ )
                      parammid[i] = (parama[i] + paramb[i])/2;
              }
             else
             if ( (get_boundary(headv) != bdry) && (get_boundary(tailv) != bdry) )
                { sprintf(errmsg,"Vertices %d and %d are on different boundaries from edge %d .\n",
                     ordinal(headv)+1,ordinal(tailv)+1,ordinal(e_id)+1);
                  kb_error(1240,errmsg,WARNING);

                  paramb = defaultp;
                  defaultp[0] = defaultp[1] = defaultp[2] = 0.1;
                  for ( i = 0 ; i < SDIM ; i++ )
                     s[i] = eval(bdry->coordf[i],defaultp,NULLID);
                  mu = s;
                  /* projecting on tangent */
                  b_extrapolate(bdry,mu,mv,mv,paramb,parammid,NULLID);
                }
             else
#ifdef PARAMAVG
             if ( (get_boundary(headv) == bdry) && (get_boundary(tailv) == bdry) )
              {
                 mu = get_coord(tailv);
                 parama = get_param(tailv);
                 paramb = get_param(headv);
                 /* projecting on tangent */
                 b_extrapolate(bdry,mu,mv,mv,parama,parammid,tailv);
                 /* if not wrapped, take average parameter */
                 for (  i = 0 ; i < bdry->pcount ; i++ )
                    { if ( ((parama[i] < parammid[i]) && (parammid[i] < paramb[i]))
                      || ((parama[i] > parammid[i]) && (parammid[i] > paramb[i])))
                      parammid[i] = (parama[i] + paramb[i])/2;
                    }

              }
             else
#endif
             if ( (get_boundary(headv) == bdry) && !wrap2 )
              {
                 mu = get_coord(headv);
                 paramb = get_param(headv);
                 /* projecting on tangent */
                 b_extrapolate(bdry,mu,mv,mv,paramb,parammid,headv);
              }
             else
              {
                 mu = get_coord(tailv);
                 paramb = get_param(tailv);
                 /* projecting on tangent */
                 b_extrapolate(bdry,mu,mv,mv,paramb,parammid,tailv);
              }
          }
        else if ( web.modeltype == QUADRATIC )
          { REAL *parama;
             set_attr(old_mid,BOUNDARY);
             set_boundary_num(old_mid,bdry->num);
             set_attr(new_mid,BOUNDARY);
             set_boundary_num(new_mid,bdry->num);

             paramb = get_param(divider);

             if ( interp_bdry_param )
             { parama = get_param(tailv);
                parammid = get_param(old_mid);             
                for (  i = 0 ; i < bdry->pcount ; i++ )
                      parammid[i] = (parama[i] + paramb[i])/2;
                parama = get_param(headv);
                parammid = get_param(new_mid);             
                for (  i = 0 ; i < bdry->pcount ; i++ )
                      parammid[i] = (parama[i] + paramb[i])/2;
             }
             else
             {
             mu = get_coord(divider);

             /* find new parameters of old edge midpoint */
             /* projecting on tangent */
             parammid = get_param(old_mid);             
             mv = get_coord(old_mid);
             b_extrapolate(bdry,mu,mv,mv,paramb,parammid,old_mid);

             /* find parameters of new edge midpoint */
             /* projecting on tangent */
             parammid = get_param(new_mid);             
             mv = get_coord(new_mid);
             b_extrapolate(bdry,mu,mv,mv,paramb,parammid,new_mid);
             }
          }
        else if ( web.modeltype == LAGRANGE )
          {
             set_attr(divider,BOUNDARY);
             set_boundary_num(divider,bdry->num);
             oldv = get_edge_vertices(e_id);
             newv = get_edge_vertices(new_e);

             /* transfer old parameter values */
             for ( i = web.lagrange_order-2 ; i >= 0 ; i-=2 )
             { REAL *p = get_param(newv[i]);
                REAL *pa = get_param(oldv[(i+web.lagrange_order)/2]);
                for ( k = 0 ; k < bdry->pcount ; k++ )
                  p[k] = pa[k];
             }
             for ( i = 2*(web.lagrange_order/2) ; i > 0 ; i-=2 )
             { REAL *p = get_param(oldv[i]);
                REAL *pa = get_param(oldv[i/2]);
                for ( k = 0 ; k < bdry->pcount ; k++ )
                  p[k] = pa[k];
             }
             /* calculate new parameter values */
             if ( interp_bdry_param )
             { 
                for ( i = 1; i <= web.lagrange_order ; i += 2 )
                { REAL *pa = get_param(oldv[i-1]);
                  REAL *pb = (i==web.lagrange_order)?get_param(newv[1])
                                     : get_param(oldv[i+1]);
                  REAL *p  = get_param(oldv[i]);
                  for (  k = 0 ; k < bdry->pcount ; k++ )
                      p[k] = (pa[k] + pb[k])/2;
                }
                for ( i = web.lagrange_order-1; i > 0 ; i -= 2 )
                { REAL *pa = get_param(newv[i-1]);
                  REAL *pb = get_param(newv[i+1]);
                  REAL *p  = get_param(newv[i]);
                  for (  k = 0 ; k < bdry->pcount ; k++ )
                      p[k] = (pa[k] + pb[k])/2;
                }
             }
             else /* interpolate */
             { oldv = get_edge_vertices(e_id);
                for ( i = 1; i <= web.lagrange_order ; i += 2 )
                { REAL *pa = get_param(oldv[i-1]);
                  REAL *p  = get_param(oldv[i]);
                  mu = get_coord(oldv[i-1]);             
                  mv = get_coord(oldv[i]);
                  b_extrapolate(bdry,mu,mv,mv,pa,p,oldv[i]);
                }
                newv = get_edge_vertices(new_e);
                for ( i = web.lagrange_order-1; i > 0 ; i -= 2 )
                { REAL *pa = get_param(newv[i-1]);
                  REAL *p  = get_param(newv[i]);
                  mu = get_coord(newv[i-1]);             
                  mv = get_coord(newv[i]);
                  b_extrapolate(bdry,mu,mv,mv,pa,p,newv[i]);
                }
            }
        }
     }
  else
     { 
        ATTR attr = get_eattr(e_id) & (BDRY_ENERGY | BDRY_CONTENT | CONSTRAINT );
        conmap_t * conmap = get_e_constraint_map(e_id);

        set_attr(new_e,attr);
        set_attr(divider,attr);
        if ( conmap || attr )
          {
             set_e_conmap(new_e,conmap);
             set_v_conmap(divider,conmap);
             project_v_constr(divider,ACTUAL_MOVE);
             if ( web.modeltype == QUADRATIC )
                {
                  set_attr(old_mid,attr);
                  set_attr(new_mid,attr);
                  clear_v_conmap(old_mid);
                  clear_v_conmap(new_mid);
                  set_v_conmap(old_mid,conmap);
                  set_v_conmap(new_mid,conmap);
                  project_v_constr(old_mid,ACTUAL_MOVE);
                  project_v_constr(new_mid,ACTUAL_MOVE);
                }
             /* this should have been taken care of for Lagrange in dup_edge */
          }
     }

  generate_edge_fe_init();  
  new_fe = NULLID;
  while ( generate_edge_fe(e_id,&old_fe) )
     { /* create new facetedge and splice into edge net */
        facetedge_id next;
        new_fe = new_facetedge(get_fe_facet(old_fe),new_e);
        set_next_edge(new_fe,get_next_edge(old_fe));
        next = get_next_edge(old_fe);
        if ( valid_id(next) )
             set_prev_edge(get_next_edge(old_fe),new_fe);
        set_next_edge(old_fe,new_fe);
        set_prev_edge(new_fe,old_fe);
     }
  set_edge_fe(new_e,new_fe);

  generate_edge_fe_init();  
  while ( generate_edge_fe(e_id,&old_fe) )
    { /* copy over facet chain */
      facetedge_id fe,nfe;

      fe = get_next_edge(old_fe);
      nfe = get_next_facet(old_fe);
      set_next_facet(fe,get_next_edge(nfe));
      nfe = get_prev_facet(old_fe);
		set_prev_facet(fe,get_next_edge(nfe));
    }

  /* check for dangling ends with loopback fe */
  generate_edge_fe_init();  
  new_fe = NULLID;
  while ( generate_edge_fe(new_e,&new_fe) )
     { if ( valid_id(get_next_edge(new_fe)) )
          if ( !equal_id(get_prev_edge(get_next_edge(new_fe)),new_fe ) )
            set_next_edge(new_fe,get_prev_edge(get_next_edge(new_fe)));
     }

 top_timestamp = ++global_timestamp;
}

/************************************************************
*
*    cross_cut()
*
*    Purpose: subdivides a facet into two facets by introducing
*                a new edge and a new facet.
*
*    Inputs:  facetedge_id first_fe  - first in chain defining new facet
*                facetidge_id last_fe    - last in chain;
*
*    Output:  new facet created with boundary first_fe,chain,last_fe,newedge
*                also both facets marked with NEWFACET attribute, so they
*                can be left alone during a refinement.
*
*/

void cross_cut(first_fe,last_fe)
facetedge_id first_fe,last_fe;
{
  facet_id old_f,new_f;
  edge_id  new_e;
  facetedge_id fe,new_fe_new,new_fe_old;
  int wrap;
  ATTR attr;
  vertex_id headv,tailv;
  int i,k;

  old_f = get_fe_facet(first_fe);
  attr = get_fattr(old_f);

  tailv = get_fe_headv(last_fe);
  headv = get_fe_tailv(first_fe);
  if ( get_vattr(headv) & AXIAL_POINT )
     new_e = inverse_id(new_edge(headv,tailv,old_f));
  else new_e = new_edge(tailv,headv,old_f);
  if ( attr & FIXED )
  { set_attr(new_e,FIXED);
     if ( web.modeltype == QUADRATIC ) 
        set_attr(get_edge_midv(new_e),FIXED);
     else if ( web.modeltype == LAGRANGE ) 
     { vertex_id *v = get_edge_vertices(new_e);
        for ( i = 1 ; i < web.lagrange_order ; i++ )
          set_attr(v[i],FIXED);
     }
  }
  if ( attr & NO_REFINE ) set_attr(new_e,NO_REFINE);

  /* for QUADRATIC model, midpoint of new edge is left as the 
      linear interpolation given by new_edge() since triangular
      facets may not yet be established yet */

  new_f = dup_facet(old_f); /* copy facet data */
  if ( inverted(old_f) ) invert(new_f);    /* same orientation */
  set_attr(old_f,NEWFACET);
  if ( attr & BOUNDARY )
     { struct boundary *bdry = get_facet_boundary(old_f);
        REAL *paramb,*parammid;
        REAL *mu,*mv;

        set_attr(new_e,BOUNDARY);
        set_edge_boundary_num(new_e,bdry->num);
        if ( web.modeltype == QUADRATIC )
          { vertex_id divider = get_edge_midv(new_e); 
             set_attr(divider,BOUNDARY);
             set_boundary_num(divider,bdry->num);

             /* find parameters of new midpoint */
             mv = get_coord(divider);
             parammid = get_param(divider);
             if ( (get_boundary(headv) != bdry) && (get_boundary(tailv) != bdry) )
                { sprintf(errmsg,"Vertices %d and %d are on different boundaries from edge %d.\n",
                     ordinal(headv)+1,ordinal(tailv)+1,ordinal(new_e)+1);
                  kb_error(1242,errmsg,RECOVERABLE);

                }
             else
#ifdef PARAMAVG
             if ( (get_boundary(headv) == bdry) && (get_boundary(tailv) == bdry) )
              {
                 mu = get_coord(tailv);
                 parama = get_param(tailv);
                 paramb = get_param(headv);
                 /* projecting on tangent */
                 b_extrapolate(bdry,mu,mv,mv,parama,parammid,tailv);
                 /* if not wrapped, take average parameter */
                 for (  i = 0 ; i < bdry->pcount ; i++ )
                    { if ( ((parama[i] < parammid[i]) && (parammid[i] < paramb[i]))
                      || ((parama[i] > parammid[i]) && (parammid[i] > paramb[i])))
                      parammid[i] = (parama[i] + paramb[i])/2;
                    }

              }
             else
#endif
             if ( get_boundary(headv) == bdry )
              {
                 mu = get_coord(headv);
                 paramb = get_param(headv);
                 /* projecting on tangent */
                 b_extrapolate(bdry,mu,mv,mv,paramb,parammid,headv);
              }
             else
              {
                 mu = get_coord(tailv);
                 paramb = get_param(tailv);
                 /* projecting on tangent */
                 b_extrapolate(bdry,mu,mv,mv,paramb,parammid,tailv);
              }
         }
     }
  if ( attr & CONSTRAINT )    
     {
        ATTR cattr = attr & (BDRY_ENERGY | BDRY_CONTENT | CONSTRAINT );
        conmap_t * conmap = get_f_constraint_map(old_f);

        set_attr(new_e,cattr);
        set_e_conmap(new_e,conmap);
        if ( web.modeltype == QUADRATIC )
          {
             vertex_id mid = get_edge_midv(new_e);
             set_attr(mid,cattr);
             set_v_conmap(mid,conmap);
             project_v_constr(mid,ACTUAL_MOVE);
          }
     }

  new_fe_new = new_facetedge(new_f,new_e);
  new_fe_old = new_facetedge(old_f,edge_inverse(new_e));
  set_edge_fe(new_e,new_fe_new);
  set_facet_body(new_f,get_facet_body(old_f));
  set_facet_body(facet_inverse(new_f),get_facet_body(facet_inverse(old_f)));
  if ( phase_flag ) set_f_phase_density(new_f);
  set_facet_fe(new_f,last_fe); /* preserves starting point of a facet */

  set_facet_fe(old_f,new_fe_old);

  /* install new facet into its facet-edges */
  /* and set torus wrap flags if needed */
  fe = first_fe;
  wrap = 0;
  while ( 1 )
  {
     set_fe_facet(fe,new_f);
     if ( web.symmetry_flag )
        wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
     if ( equal_id(fe,last_fe) ) break;
     fe = get_next_edge(fe);
  }
  if ( wrap )
  { set_edge_wrap(new_e,(*sym_inverse)(wrap));
     if ( web.modeltype == QUADRATIC )
     { /* have to adjust coordinates of midpoint */
        REAL temp[MAXCOORD];
        REAL *mv,*tv,*hv;
        tv = get_coord(tailv);
        mv = get_coord(get_edge_midv(new_e));
        hv = get_coord(headv);
        (*sym_wrap)(hv,temp,(*sym_inverse)(wrap));
        for ( k = 0 ; k < SDIM ; k++ )
          mv[k] = (tv[k] + temp[k])/2;
     }
  }

  /* link up facet-edges */
  set_next_edge(get_prev_edge(first_fe),new_fe_old);
  set_prev_edge(new_fe_old,get_prev_edge(first_fe));
  set_prev_edge(get_next_edge(last_fe),new_fe_old);
  set_next_edge(new_fe_old,get_next_edge(last_fe));
  set_prev_edge(first_fe,new_fe_new);
  set_next_edge(new_fe_new,first_fe);
  set_next_edge(last_fe,new_fe_new);
  set_prev_edge(new_fe_new,last_fe);
  set_next_facet(new_fe_new,fe_inverse(new_fe_old));
  set_prev_facet(new_fe_new,fe_inverse(new_fe_old));
  set_next_facet(new_fe_old,fe_inverse(new_fe_new));
  set_prev_facet(new_fe_old,fe_inverse(new_fe_new));


 top_timestamp = ++global_timestamp;
}

/************************************************************************
*
* function: rebody()
*
* purpose:  redivide bodies into connected pieces. Useful after
*                neck pinches.
*
* method:  Starting from body facet, labels all adjacent facets 
*             for body.  An unlabelled body facet is assigned new body
*             and propagated to neighbors.
*
* return:    Number of new bodies.
*/

int rebody()
{ body_id b_id,old_b;
  int faces_left;
  int new_bodies = 0;
  int i;
  struct bodyface *flist, *bf;
  facet_id *stack;  /* of known faces */
  int stacktop;
  int facetop;  /* length of flist */
  facet_id f_id,ff_id;
  facetedge_id fe;

  if ( web.representation == STRING ) return string_rebody();

  /* working list of facets */
  flist = (struct bodyface *)temp_calloc(2*web.skel[FACET].count,
                sizeof(struct bodyface));
  stack = (facet_id *)temp_calloc(2*web.skel[FACET].count,sizeof(facet_id*));
  faces_left = 0; 
  FOR_ALL_FACETS(f_id)
     { if ( valid_id(get_facet_body(f_id)))
             flist[faces_left++].f_id = f_id;
        ff_id = facet_inverse(f_id);
        if ( valid_id(get_facet_body(ff_id)) )
             flist[faces_left++].f_id = ff_id;
      }
  if ( faces_left == 0 ) return 0;
  facetop = faces_left;

  /* sort in facet order, so can find facets quickly */
  qsort((char *)flist,faces_left,sizeof(struct bodyface),FCAST bfcomp);

  /* initialize stack with body facets */
  stacktop = 0;
  FOR_ALL_BODIES(b_id)
    { 
      fe = get_body_fe(b_id);
      f_id = get_fe_facet(fe);
      stack[stacktop++] = f_id;
      bf = (struct bodyface *)bsearch((char *)&f_id,(char *)flist,
                        facetop, sizeof(struct bodyface),FCAST bfcomp);
      if ( bf == NULL ) 
         { kb_error(1243,"Internal error: rebody() cannot find facet on stack.\n",WARNING); continue; }

      bf->wrapflag = 1;
      faces_left--;
    }

  /* pull stuff off stack till empty */
  while (stacktop > 0 )
    { f_id = stack[--stacktop];
      b_id = get_facet_body(f_id);
      for ( i = 0, fe = get_facet_fe(f_id); i < 3 ; i++,fe = get_next_edge(fe))
         { 
            ff_id = get_fe_facet(get_prev_facet(fe));
            if ( !valid_id(ff_id) ) continue;
            ff_id = facet_inverse(ff_id);
            if ( !equal_id(get_facet_body(ff_id),b_id) ) continue;
            bf = (struct bodyface *)bsearch((char *)&ff_id,(char *)flist,
                        facetop, sizeof(struct bodyface),FCAST bfcomp);
            if ( bf == NULL ) 
              { kb_error(1244,"Internal error: rebody() cannot find facet on stack (old body).\n",WARNING);

                 continue;
              }
            if ( bf->wrapflag ) continue;
            bf->wrapflag = 1; /* mark as part of original body */
            stack[stacktop++] = ff_id;
            faces_left--;
         }
    }

  /* now have to create new bodies */
  while ( faces_left > 0 )
    { /* find undone face */
      for ( i  =  0 ; i < facetop ; i++ )
         if ( flist[i].wrapflag == 0 ) break;
      if ( i == facetop ) break;
      flist[i].wrapflag = 1;
      f_id = flist[i].f_id;
      old_b = get_facet_body(f_id);
      b_id = dup_body(old_b);
      new_bodies++;
      set_body_fe(b_id,get_facet_fe(f_id));
      set_facet_body(f_id,b_id);
      stack[stacktop++] = f_id;
      faces_left--;
      /* pull stuff off stack till empty */
      while (stacktop > 0 )
        { f_id = stack[--stacktop];
          for ( i=0, fe = get_facet_fe(f_id); i < 3 ; i++,fe = get_next_edge(fe))
             { ff_id = facet_inverse(get_fe_facet(get_prev_facet(fe)));
                if (!valid_id(ff_id) || !valid_id(get_facet_body(ff_id))) continue;
                if ( !equal_id(get_facet_body(ff_id),old_b) ) continue;
                bf = (struct bodyface *)bsearch((char *)&ff_id,(char *)flist,
                        facetop, sizeof(struct bodyface),FCAST bfcomp);
                if ( bf == NULL )
                  { kb_error(1245,"Internal error: rebody() cannot find facet on stack (new body).\n",WARNING);
                     continue;
                  }
                if ( bf->wrapflag ) continue;
                bf->wrapflag = 1; /* mark as part of original body */
                set_facet_body(ff_id,b_id);
                stack[stacktop++] = ff_id;
                faces_left--;
             }
          }
     }

  temp_free((char*)flist);
  temp_free((char*)stack);
  top_timestamp = ++global_timestamp;

  FOR_ALL_BODIES(b_id)
    { if ( get_body_volconst(b_id) != 0.0 )
         { sprintf(msg,"Nonzero VOLCONST on body %d.  May need adjusting due to rebody.\n",ordinal(b_id)+1);
            kb_error(1977,msg,WARNING);
         }
    }
  return new_bodies;
}


/************************************************************************
*
* function: string_rebody()
*
* purpose:  redivide bodies into connected pieces. Useful after
*                neck pinches.  String model.
*
* method:  Starting from body facet edge, labels all adjacent edges 
*             for body.  An ruUnlabelled body edges is assigned new body
*             and propagated to neighbors.
*
* return:    Number of new bodies.
*/

int string_rebody()
{ body_id b_id,old_b;
  int edges_left;
  int new_bodies = 0;
  int i;
  struct bodyface *felist, *be;
  facetedge_id *stack;  /* of known faces */
  int stacktop;
  int edgetop;  /* length of flist */
  facet_id f_id,ff_id,new_f;
  edge_id e_id;
  facetedge_id fe,ffe;

  /* working list of edges */
  felist = (struct bodyface *)temp_calloc(2*web.skel[EDGE].count,
                sizeof(struct bodyface));
  stack = (facet_id *)temp_calloc(2*web.skel[EDGE].count,sizeof(edge_id*));
  edges_left = 0; 
  FOR_ALL_EDGES(e_id)
     { fe = get_edge_fe(e_id);
        if ( !valid_id(fe) ) continue;
        f_id = get_fe_facet(fe);
        if ( !valid_id(f_id) ) continue;

        if ( valid_id(get_facet_body(f_id)))
             felist[edges_left++].f_id = fe;
        ff_id = facet_inverse(f_id);
        if ( valid_id(get_facet_body(ff_id)) )
             felist[edges_left++].f_id = inverse_id(fe);

        fe = get_next_facet(fe);
        ff_id = get_fe_facet(fe);
        if ( equal_element(f_id,ff_id) ) continue;

        if ( valid_id(get_facet_body(ff_id)))
             felist[edges_left++].f_id = fe;
        ff_id = facet_inverse(ff_id);
        if ( valid_id(get_facet_body(ff_id)) )
             felist[edges_left++].f_id = inverse_id(fe);

      }
  if ( edges_left == 0 ) return 0;
  edgetop = edges_left;

  /* sort in edge order, so can find edges quickly */
  qsort((char *)felist,edges_left,sizeof(struct bodyface),FCAST bfcomp);

  /* initialize stack with body edges */
  stacktop = 0;
  FOR_ALL_BODIES(b_id)
    { fe = get_body_fe(b_id);
      stack[stacktop++] = fe;
      be = (struct bodyface *)bsearch((char *)&fe,(char *)felist,
                        edgetop, sizeof(struct bodyface),FCAST bfcomp);
      if ( be == NULL ) 
         { kb_error(1246,"Internal error: string_rebody() cannot find edge on stack.\n",WARNING); continue; }

      be->wrapflag = 1;
      edges_left--;
    }

  /* pull stuff off stack till empty */
  while (stacktop > 0 )
    { fe = stack[--stacktop];
      f_id = get_fe_facet(fe);
      b_id = get_facet_body(f_id);

      ffe = get_prev_edge(fe);
      if ( !valid_id(ffe) ) goto otherend;
      ff_id = get_fe_facet(ffe);
      if ( !equal_id(ff_id,f_id) ) goto otherend;
      be = (struct bodyface *)bsearch((char *)&ffe,(char *)felist,
                  edgetop, sizeof(struct bodyface),FCAST bfcomp);
      if ( be == NULL ) 
        { kb_error(1247,"Internal error: string_rebody() cannot find edge on stack (old body).\n",WARNING);

           goto otherend;
        }
      if ( !be->wrapflag )
      { be->wrapflag = 1; /* mark as part of original body */
        stack[stacktop++] = ffe;
        edges_left--;
      }
otherend:
      ffe = get_next_edge(fe);
      if ( !valid_id(ffe) ) continue;
      ff_id = get_fe_facet(ffe);
      if ( !equal_id(ff_id,f_id) ) continue;
      be = (struct bodyface *)bsearch((char *)&ffe,(char *)felist,
                  edgetop, sizeof(struct bodyface),FCAST bfcomp);
      if ( be == NULL ) 
        { kb_error(1248,"Internal error: string_rebody() cannot find edge on stack (old body).\n",WARNING);
          continue;
        }
      if ( !be->wrapflag )
      { be->wrapflag = 1; /* mark as part of original body */
        stack[stacktop++] = ffe;
        edges_left--;
      }
    }

  /* now have to create new bodies */
  while ( edges_left > 0 )
    { /* find undone edge */
      for ( i  =  0 ; i < edgetop ; i++ )
         if ( felist[i].wrapflag == 0 ) break;
      if ( i == edgetop ) break;
      felist[i].wrapflag = 1;
      fe = felist[i].f_id;
      f_id = get_fe_facet(fe);
      old_b = get_facet_body(f_id);
      b_id = dup_body(old_b);
      new_f = dup_facet(f_id);
      new_bodies++;
      set_body_fe(b_id,fe);
      set_facet_body(new_f,b_id);
      set_facet_fe(new_f,fe);
      set_fe_facet(fe,new_f);
      stack[stacktop++] = fe;
      edges_left--;
      /* pull stuff off stack till empty */
      while (stacktop > 0 )
      { fe = stack[--stacktop];

        ffe = get_prev_edge(fe);
        if ( !valid_id(ffe) ) goto anotherend;
        ff_id = get_fe_facet(ffe);
        if ( !equal_id(ff_id,f_id) ) goto otherend;
        be = (struct bodyface *)bsearch((char *)&ffe,(char *)felist,
                    edgetop, sizeof(struct bodyface),FCAST bfcomp);
        if ( be == NULL ) 
          { kb_error(1249,"Internal error: string_rebody() cannot find edge on stack (old body).\n",WARNING);
            goto otherend;
          }
        if ( !be->wrapflag )
        { be->wrapflag = 1; /* mark as part of original body */
          stack[stacktop++] = ffe;
          edges_left--;
          set_fe_facet(ffe,new_f);
        }
anotherend:
        ffe = get_next_edge(fe);
        if ( !valid_id(ffe) ) continue;
        ff_id = get_fe_facet(ffe);
        if ( !equal_id(ff_id,f_id) ) continue;
        be = (struct bodyface *)bsearch((char *)&ffe,(char *)felist,
                    edgetop, sizeof(struct bodyface),FCAST bfcomp);
        if ( be == NULL ) 
          { kb_error(1250,"Internal error: string_rebody() cannot find edge on stack (old body).\n",WARNING);
            continue;
          }
        if ( !be->wrapflag )
        { be->wrapflag = 1; /* mark as part of original body */
          stack[stacktop++] = ffe;
          edges_left--;
          set_fe_facet(ffe,new_f);
        }
     }
  }

  temp_free((char*)felist);
  temp_free((char*)stack);
  top_timestamp = ++global_timestamp;
  return new_bodies;
}

/****************************************************************************
*
* function: dissolve_vertex()
*
* purpose:  Remove unattached vertex from surface.
*
* return: 1 if vertex removed, 0 if not because attached.
*/

int dissolve_vertex(v_id)
vertex_id v_id;
{ edge_id e_id;

  if ( !valid_element(v_id) ) return 0;
  e_id = get_vertex_edge(v_id);
  if ( valid_id(e_id) ) return 0;
  free_element(v_id);
  top_timestamp = ++global_timestamp;
  if ( verbose_flag )
  { sprintf(msg,"Dissolving vertex %d\n",ordinal(v_id)+1);
    outstring(msg);
  }

  return 1;
}

/****************************************************************************
*
* function: dissolve_edge()
*
* purpose:  Remove edge from surface if not on a facet.
*
* return: 1 if edge removed, 0 if not because attached.
*/

int dissolve_edge(e_id)
edge_id e_id;
{ facetedge_id fe,fe_id,start_fe;
  vertex_id head,tail;
  int evalence;
  facet_id f_id;

  if ( !valid_element(e_id) ) return 0;

  evalence = get_edge_valence(e_id);

  if ( web.representation == STRING )
  {
#ifdef CODDLING
    /* check for at most one facet */
    if ( evalence > 1 )
    { if ( verbose_flag )
      { sprintf(msg,"Not dissolving edge %d since still on two facets.\n",ordinal(e_id)+1);
        outstring(msg);
      }
      return 0;
    }
#endif
  
    /* check for being at beginning or end of edge arc for all facets */
    if ( evalence > 0 )
    { start_fe = fe_id = get_edge_fe(e_id);
      do
      {   
        fe = fe_id;
        if ( valid_id(get_next_edge(fe)) && valid_id(get_prev_edge(fe)) )
        { /* check for full loop */
          do { fe = get_next_edge(fe); }
          while ( valid_id(fe) && !equal_id(fe,fe_id) );
          if ( !valid_id(fe) )
          { if ( verbose_flag )
            { sprintf(msg,"Not dissolving edge %d since would make facet two arcs.\n",ordinal(e_id)+1);
              outstring(msg);
            }
            return 0;
          }
        }
        fe_id = get_next_facet(fe_id);
      } while ( fe_id != start_fe );
      if ( evalence > 1 )
      { sprintf(msg,"Dissolving edge %d between two facets.\n",ordinal(e_id)+1);
        kb_error(3838,msg,WARNING);
      }
    }
  }
  else if ( evalence > 0 ) 
  { if ( verbose_flag )
    { sprintf(msg,"Not dissolving edge %d since still on a facet.\n",ordinal(e_id)+1);
      outstring(msg);
    }
    return 0;
  }

  if ( verbose_flag )
  { sprintf(msg,"Dissolving edge %d\n",ordinal(e_id)+1);
    outstring(msg);
  }

  fe = get_edge_fe(e_id);
  if ( !valid_id(fe) )
  { free_element(e_id); return 1; }

  head = get_edge_headv(e_id);
  tail = get_edge_tailv(e_id);

  /* reset vertex-edge links */
  remove_vertex_edge(tail,e_id);
  remove_vertex_edge(head,inverse_id(e_id));

  /* connect facetedges of stumps */
  f_id = get_fe_facet(fe);  /* in case of STRING */
  if ( valid_id(f_id) )
  {
    if ( inverted(f_id) )
    { if ( valid_id(get_prev_edge(fe)) )
       set_facet_fe(f_id,get_prev_edge(fe));
      else if ( !valid_id(get_next_edge(fe)) )
       set_facet_fe(f_id,NULLID); 
    } 
    else
    { if ( valid_id(get_next_edge(fe)) )
       set_facet_fe(f_id,get_next_edge(fe));
      else if ( !valid_id(get_prev_edge(fe)) )
       set_facet_fe(f_id,NULLID); 
    }
  }

  for ( fe_id = fe ; ; )
  { facetedge_id  prev,next;
    prev = get_prev_edge(fe_id);
    if ( valid_id(prev) )
     { if ( valid_id(f_id) ) set_next_edge(prev,NULLID);
       else set_next_edge(prev,inverse_id(prev));
     }
    next = get_next_edge(fe_id);
    if ( valid_id(next) )
     { if ( valid_id(f_id) ) set_prev_edge(next,NULLID);
       else set_prev_edge(next,inverse_id(next));
     }

    free_element(fe_id);
    fe_id = get_next_facet(fe_id);
    if ( equal_id(fe_id,fe) ) break;
  }

  free_element(e_id);
  top_timestamp = ++global_timestamp;
  return 1;
}

/****************************************************************************
*
* function: dissolve_facet()
*
* purpose:  Remove facet from surface if not on a body.
*
* return: 1 if facet removed, 0 if not because attached.
*/

int dissolve_facet(f_id)
facet_id f_id;
{ facetedge_id fe,fe_id;
  edge_id e_id;
  int fvalence;
  facet_id ff_id;
  body_id b_id;

  if ( !valid_element(f_id) ) return 0;

  fvalence = valid_id(get_facet_body(f_id)) ? 1 : 0; 
  fvalence += valid_id(get_facet_body(inverse_id(f_id))) ? 1 : 0;

  if ( fvalence > 1 )
  { if ( verbose_flag )
    { sprintf(msg,"Not dissolving facet %d since on two bodies\n",ordinal(f_id)+1);
      outstring(msg);
    }
    return 0;
  }
  if ( (fvalence == 1) && (top_timestamp > bfacet_timestamp) )
    make_bfacet_lists();

  if ( verbose_flag )
  { sprintf(msg,"Dissolving facet %d\n",ordinal(f_id)+1);
    outstring(msg);
  }

  if ( web.representation != SIMPLEX )
  { fe = get_facet_fe(f_id);
    for ( fe_id = fe ; ; )
    { facetedge_id prev,next,next_fe;
      prev = get_prev_facet(fe_id);
      next = get_next_facet(fe_id);
      e_id = get_fe_edge(fe_id);
      next_fe = get_next_edge(fe_id);
      if ( !equal_id(prev,fe_id) )
      { set_next_facet(prev,next);
        set_prev_facet(next,prev);
        set_edge_fe(e_id,next);
        free_element(fe_id);
      }
      else
      {
        set_attr(e_id,BARE_NAKED);
        set_edge_fe(e_id,NULLID);
        free_element(fe_id);
      }
      fe_id = next_fe;
      if ( equal_id(fe_id,fe) ) break;
    }
  }

  b_id = get_facet_body(f_id);
  if ( valid_id(b_id) )
  { ff_id = get_next_body_facet(f_id);
    set_body_fe(b_id,equal_element(f_id,ff_id)? NULLID : get_facet_fe(ff_id));
  }
  b_id = get_facet_body(inverse_id(f_id));
  if ( valid_id(b_id) )
  { ff_id = get_next_body_facet(inverse_id(f_id));
    set_body_fe(b_id,equal_element(inverse_id(f_id),ff_id)? NULLID : 
       get_facet_fe(ff_id));
  }

  free_element(f_id);
  top_timestamp = ++global_timestamp;
  return 1;
}

/****************************************************************************
*
* function: dissolve_body()
*
* purpose:  Remove body from surface.
*
* return: 1 if body removed.
*/

int dissolve_body(b_id)
body_id b_id;
{ facet_id f_id,ff_id;;
 
  if ( !valid_element(b_id) ) return 0;

  if ( verbose_flag )
  { sprintf(msg,"Dissolving body %d\n",ordinal(b_id)+1);
     outstring(msg);
  }

  FOR_ALL_FACETS(f_id)
     { if ( equal_id(b_id,get_facet_body(f_id)) ) 
          set_facet_body(f_id,NULLBODY);
       ff_id = inverse_id(f_id);
       if ( equal_id(b_id,get_facet_body(ff_id)) ) 
          set_facet_body(ff_id,NULLBODY);
     }
  free_element(b_id);
  top_timestamp = ++global_timestamp;
  return 1;
}

