/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: Error Plugin
 * File: evms2/engine/plugins/error/error.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "error.h"

engine_functions_t *EngFncs;

/**
 * error_setup_evms_plugin
 **/
static int error_setup_evms_plugin(engine_functions_t *functions)
{
	int rc;
	EngFncs	= functions;

	LOG_ENTRY();

	rc = EngFncs->register_name(ERROR_NAME);
	if (rc) {
		LOG_ERROR("Error registering directory name.\n");
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * error_cleanup_evms_plugin
 **/
static void error_cleanup_evms_plugin(void)
{
	LOG_ENTRY();
	EngFncs->unregister_name(ERROR_NAME);
	LOG_EXIT_VOID();
}

/**
 * error_can_delete
 *
 * Error objects can always be deleted.
 **/
static int error_can_delete(storage_object_t *object)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_discover
 *
 * Nothing to discover, since error objects aren't stored anywhere.
 *
 * FIXME: Search for DM devices with "Error" names and create objects for them?
 *        Then if someone tries to create an object with the same name and
 *        size, we can just return the existing object. This should help
 *        prevent leaving stale devices around in DM.
 *
 **/
static int error_discover(list_anchor_t input_objects,
			  list_anchor_t output_objects,
			  boolean final_call)
{
	LOG_ENTRY();
	EngFncs->merge_lists(output_objects, input_objects, NULL, NULL);
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * create_object
 **/
static storage_object_t *create_error_object(char *error_name,
					     u_int64_t size,
					     object_type_t type)
{
	storage_object_t *object = NULL;
	int rc;

	LOG_ENTRY();

	switch (type) {
	case DISK:
		rc = EngFncs->allocate_logical_disk(error_name, &object);
		break;
	case SEGMENT:
		rc = EngFncs->allocate_segment(error_name, &object);
		break;
	case REGION:
		rc = EngFncs->allocate_region(error_name, &object);
		break;
	case EVMS_OBJECT:
		rc = EngFncs->allocate_evms_object(error_name, &object);
		break;
	default:
		rc = EINVAL;
		break;
	}

	if (rc) {
		goto out;
	}

	object->size = size;
	object->data_type = DATA_TYPE;
	object->plugin = my_plugin_record;

out:
	LOG_EXIT_PTR(object);
	return object;
}

/**
 * error_create
 *
 * Create a new error object. Available options are:
 * - name: A string value specifying the name of the new error object.
 * - size: A ui64 value specifying the size (in sectors) of the new object.
 * - type: A string value specifying the type of object. The string must
 *         be "disk", "segment", "region", or "feature_object".
 * No objects are consumed when creating an error object, so the
 * input_objects list is ignored.
 **/
static int error_create(list_anchor_t input_objects,
			option_array_t *options,
			list_anchor_t output_objects)
{
	storage_object_t *object;
	char error_name[EVMS_NAME_SIZE+1];
	char *input_name;
	object_type_t type;
	u_int64_t size;
	int rc;

	LOG_ENTRY();

	parse_options(options, &input_name, &size, &type);
	rc = verify_options(input_name, type);
	if (rc) {
		goto out;
	}

	generate_error_name(input_name, error_name);
	object = create_error_object(error_name, size, type);
	if (!object) {
		rc = EINVAL;
		goto out;
	}

	EngFncs->dm_update_status(object);
	EngFncs->insert_thing(output_objects, object, INSERT_AFTER, NULL);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * error_discard
 **/
static int error_discard(list_anchor_t objects)
{
	storage_object_t *object;
	list_element_t iter;

	LOG_ENTRY();

	LIST_FOR_EACH(objects, iter, object) {
		EngFncs->free_storage_object(object);
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_delete
 **/
static int error_delete(storage_object_t *object,
			list_anchor_t child_objects)
{
	LOG_ENTRY();

	EngFncs->free_storage_object(object);

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_commit_changes
 *
 * No metadata to commit.
 **/
static int error_commit_changes(storage_object_t *object,
				commit_phase_t phase)
{
	LOG_ENTRY();
	object->flags &= ~(SOFLAG_NEW | SOFLAG_DIRTY);
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_can_activate
 **/
static int error_can_activate(storage_object_t *object)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_activate
 **/
static int error_activate(storage_object_t *object)
{
	dm_target_t target;
	int rc;

	LOG_ENTRY();

	target.start = 0;
	target.length = object->size;
	/* FIXME: Make a config-file or creation option so users
	 *        can select between "error" and "zero" targets.
	 */
	target.type = DM_TARGET_ERROR;
	target.data.linear = NULL;
	target.params = NULL;
	target.next = NULL;

	rc = EngFncs->dm_activate(object, &target);
	if (!rc) {
		object->flags &= ~SOFLAG_NEEDS_ACTIVATE;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * error_can_deactivate
 **/
static int error_can_deactivate(storage_object_t *object)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_deactivate
 **/
static int error_deactivate(storage_object_t *object)
{
	int rc;

	LOG_ENTRY();

	rc = EngFncs->dm_deactivate(object);
	if (!rc) {
		object->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
	}

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_get_option_count
 *
 * Get the total number of supported options for the specified task.
 **/
static int error_get_option_count(task_context_t *context)
{
	int count;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		count = ERROR_OPTION_COUNT;
		break;

	default:
		count = -1;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}

/**
 * error_init_task
 *
 * Initialize a user-task with the appropriate information for your plugin.
 * Set up all initial values in the option_descriptors in the context record
 * for the given task.
 **/
static int error_init_task(task_context_t *context)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		rc = init_create_task(context);
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * error_set_object
 **/
static int error_set_objects(task_context_t *context,
			     list_anchor_t declined_objects,
			     task_effect_t *effect)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		rc = 0;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * error_set_option
 *
 * Examine the specified value, and determine if it is valid for the task and
 * option_descriptor index. If it is acceptable, set that value in the
 * appropriate entry in the option_descriptor. The value may be adjusted if
 * necessary/allowed. If so, set the effect return value accordingly.
 **/
static int error_set_option(task_context_t *context,
			    u_int32_t index,
			    value_t *value,
			    task_effect_t *effect)
{
	int rc;

	LOG_ENTRY();

	switch (context->action) {
	case EVMS_Task_Create:
		rc = set_create_option(context, index, value, effect);
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * error_read
 *
 * Silently ignore reads. Return a zero-filled data buffer.
 **/
static int error_read(storage_object_t *object,
		      lsn_t lsn,
		      sector_count_t count,
		      void *buffer)
{
	LOG_ENTRY();
	memset(buffer, 0, count << EVMS_VSECTOR_SIZE_SHIFT);
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_write
 *
 * Silently ignore writes.
 **/
static int error_write(storage_object_t *object,
		       lsn_t lsn,
		       sector_count_t count,
		       void *buffer)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_add_sectors_to_kill_list
 *
 * Silently ignore.
 **/
static int error_add_sectors_to_kill_list(storage_object_t *object,
					  lsn_t lsn,
					  sector_count_t count)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/**
 * error_backup_metadata
 *
 * No metadata to save to the backup database.
 **/
static int error_backup_metadata(storage_object_t *object)
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/*
 * Table of standard plugin APIs for the Error plugin.
 */
static plugin_functions_t error_functions = {
	.setup_evms_plugin		= error_setup_evms_plugin,
	.cleanup_evms_plugin		= error_cleanup_evms_plugin,
	.can_delete			= error_can_delete,
	.discover			= error_discover,
	.create				= error_create,
	.discard			= error_discard,
	.delete				= error_delete,
	.commit_changes			= error_commit_changes,
	.can_activate			= error_can_activate,
	.activate			= error_activate,
	.can_deactivate			= error_can_deactivate,
	.deactivate			= error_deactivate,
	.get_option_count		= error_get_option_count,
	.init_task			= error_init_task,
	.set_objects			= error_set_objects,
	.set_option			= error_set_option,
	.read				= error_read,
	.write				= error_write,
	.add_sectors_to_kill_list	= error_add_sectors_to_kill_list,
	.backup_metadata		= error_backup_metadata,
};

/*
 * Plugin record for the Error plugin.
 */
plugin_record_t error_plugin = {
	.id = EVMS_ERROR_PLUGIN_ID,
	.version = {
		.major		= MAJOR_VERSION,
		.minor		= MINOR_VERSION,
		.patchlevel	= PATCH_LEVEL
	},
	.required_engine_api_version = {
		.major		= 15,
		.minor		= 0,
		.patchlevel	= 0
	},
	.required_plugin_api_version = {
		.plugin	= {
			.major		= 13,
			.minor		= 1,
			.patchlevel	= 0
		}
	},
	.short_name = EVMS_ERROR_PLUGIN_SHORT_NAME,
	.long_name = EVMS_ERROR_PLUGIN_LONG_NAME,
	.oem_name = EVMS_IBM_OEM_NAME,
	.functions = {
		.plugin = &error_functions
	},
};

plugin_record_t *evms_plugin_records[] = {
	&error_plugin,
	NULL
};

