/*
 *
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: main_window_cb.c
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>

#include <frontend.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "support.h"
#include "about.h"
#include "weblinks.h"
#include "resize.h"
#include "move.h"
#include "commit.h"
#include "delete.h"
#include "disk_cb.h"
#include "segment_cb.h"
#include "container_cb.h"
#include "region_cb.h"
#include "object_cb.h"
#include "volume_cb.h"
#include "plugin_cb.h"
#include "views.h"
#include "setinfo.h"
#include "transfer.h"
#include "unassign.h"
#include "message.h"
#include "fsutils.h"
#include "reopen.h"
#include "help.h"
#include "main_window_cb.h"
#include "logging.h"
#include "activation.h"
#include "backup.h"

#define EVMS_HOWTO_URL "http://evms.sourceforge.net/users_guide/"
#define EVMS_FAQ_URL "http://evms.sourceforge.net/faq.html"

/*
 *
 *   void on_create_evms_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog,
 *      modifies the dialog slightly and connects the
 *      appropriate signal handlers that will handle the
 *      actions for creating an EVMS volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Modified generic selection dialog is displayed and signal
 *      handlers have been connected for the corresponding actions.
 *
 */
void on_create_evms_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_create_evms_volume_window(on_create_evms_volume_clist_realize, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_create_compatibility_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for creating a compatibility volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_compatibility_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Create Compatibility Volume"),
						  _("Create"),
						  get_create_compat_volume_help_text(),
						  on_create_compat_volume_clist_realize,
						  on_create_compat_volume_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_create_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine starts the process of creating a feature object
 *      by first creating the selection window for the plugin desired
 *      to supply the feature. This window will in turn create the task
 *      window once a plugin has been selected and the "Next" button
 *      has been clicked.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_object_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Create EVMS Feature Object"),
						  NULL,
						  get_create_feature_help_text(),
						  on_feature_plugin_clist_realize,
						  on_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Create));

	gtk_widget_show(window);
}

/*
 *
 *   void on_create_region_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine starts the process of creating a region by first
 *      creating the selection window for the plugin desired to generate
 *      the region. This window will in turn create the task window
 *      once a plugin has been selected and the "Next" button has been
 *      clicked.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_region_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Create Storage Region"),
						  NULL,
						  get_create_region_help_text(),
						  on_region_manager_clist_realize,
						  on_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Create));

	gtk_widget_show(window);
}

/*
 *
 *   void on_create_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine starts the process of creating a container
 *      by first creating the selection window for the plugin desired
 *      to generate the container. This window will in turn create the task
 *      window once a plugin has been selected and the "Next" button
 *      has been clicked.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_container_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Create Storage Container"),
						  NULL,
						  get_create_container_help_text(),
						  on_container_plugin_clist_realize,
						  on_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Create_Container));

	gtk_widget_show(window);
}

/*
 *
 *   void on_create_segment_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine starts the process of creating a segment
 *      by first creating the selection window for the plugin desired
 *      to manage a segment. This window will in turn create the task
 *      window once a plugin has been selected and the "Next" button
 *      has been clicked.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_create_segment_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Create Disk Segment"),
						  NULL,
						  get_create_segment_help_text(),
						  on_segment_manager_clist_realize,
						  on_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Create));

	gtk_widget_show(window);
}

/*
 *
 *   void on_assign_segment_manager_to_storage_object_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates starts the process of assigning a segment
 *      manager to a storage object by first creating the selection window
 *      for the segment manager. This window will in turn create the object
 *      selection window once a plugin has been selected and the "Next"
 *      button has been clicked. The last panel will be for options (if there
 *      are any).
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugin selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_assign_segment_manager_to_storage_object_activate(GtkMenuItem * menuitem,
							  gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Add Segment Manager to Storage Object"),
						  NULL,
						  get_assign_segment_manager_help_text(),
						  on_segment_manager_clist_realize,
						  on_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Assign_Plugin));

	gtk_widget_show(window);
}

/*
 *
 *   void on_expand_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates a selection window with volumes that
 *      can be expanded. The user then selects one to be walked through
 *      the process of actually expanding a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Expandable volumes selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_expand_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Expand Logical Volume"),
						  NULL,
						  get_expand_volume_help_text(),
						  on_resize_volume_clist_realize,
						  on_resize_volume_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Expand));

	gtk_widget_show(window);
}

/*
 *
 *   void on_expand_object_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates a selection window with topmost objects that
 *      can be expanded. The user then selects one to be walked through
 *      the process of actually expanding the object.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Expandable objects selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_expand_object_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Expand Storage Object"),
						  NULL,
						  get_expand_object_help_text(),
						  on_resize_object_clist_realize,
						  on_resize_object_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Expand));

	gtk_widget_show(window);
}

/*
 *
 *   void on_expand_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates a selection window with containers that
 *      can be expanded. The user then selects one to be walked through
 *      the process of actually expanding the container.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Expandable containers selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_expand_container_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Expand Storage Container"),
						  NULL,
						  get_expand_container_help_text(),
						  on_expand_container_clist_realize,
						  on_resize_container_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Expand));

	gtk_widget_show(window);
}

/*
 *
 *   void on_shrink_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates a selection window with volumes that
 *      can be shrunk. The user then selects one to be walked through
 *      the process of actually shrinking a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Shrinkable volumes selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_shrink_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Shrink Logical Volume"),
						  NULL,
						  get_shrink_volume_help_text(),
						  on_resize_volume_clist_realize,
						  on_resize_volume_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Shrink));

	gtk_widget_show(window);
}

/*
 *
 *   void on_shrink_object_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates a selection window with topmost objects that
 *      can be shrunk. The user then selects one to be walked through
 *      the process of actually shrinking the object.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Shrinkable objects selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_shrink_object_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Shrink Storage Object"),
						  NULL,
						  get_shrink_object_help_text(),
						  on_resize_object_clist_realize,
						  on_resize_object_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Shrink));

	gtk_widget_show(window);
}

/*
 *
 *   void on_shrink_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates a selection window with containers that
 *      can be shrunk. The user then selects one to be walked through
 *      the process of actually shrinking the container.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Shrinkable containers selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_shrink_container_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Shrink Storage Container"),
						  NULL,
						  get_shrink_container_help_text(),
						  on_shrink_container_clist_realize,
						  on_resize_container_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_Shrink));

	gtk_widget_show(window);
}

/*
 *
 *   void on_delete_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for deleting a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_delete_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_deletion_window(_("Delete Logical Volume"),
					get_delete_volume_help_text(),
					on_delete_thing_clist_realize,
					on_delete_thing_button_clicked,
					on_recursive_delete_thing_button_clicked,
					GUINT_TO_POINTER(VOLUME));

	gtk_widget_show(window);
}

/*
 *
 *   void on_delete_object_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for deleting a storage object from
 *      the top of a storage object stack.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_delete_object_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_deletion_window(_("Delete Storage Object"),
					get_delete_storage_object_help_text(),
					on_delete_thing_clist_realize,
					on_delete_thing_button_clicked,
					on_recursive_delete_thing_button_clicked,
					GUINT_TO_POINTER(EVMS_OBJECT));

	gtk_widget_show(window);
}

/*
 *
 *   void on_delete_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for deleting a storage container.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_delete_container_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_deletion_window(_("Delete Storage Container"),
					get_delete_container_help_text(),
					on_delete_thing_clist_realize,
					on_delete_thing_button_clicked,
					on_recursive_delete_thing_button_clicked,
					GUINT_TO_POINTER(CONTAINER));

	gtk_widget_show(window);
}

/*
 *
 *   void on_remove_object_from_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for transferring a storage object
 *      out from the container consuming it.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_object_from_container_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window =
	    create_standard_selection_window(_("Remove Storage Object from Consuming Container"),
					     _("Remove"),
					     get_remove_object_from_container_help_text(),
					     on_remove_thing_from_container_clist_realize,
					     on_remove_thing_from_container_button_clicked, NULL,
					     NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_remove_segment_manager_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for removing a segment manager
 *      (that is, its meta-data and objects) from a storage
 *      object.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_remove_segment_manager_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Remove Segment Manager from Storage Object"),
						  _("Remove"),
						  NULL,
						  on_unassign_segment_manager_clist_realize,
						  on_unassign_parent_plugin_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_convert_to_compatibility_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for converting an evms volume to
 *      a compatibility volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_convert_to_compatibility_volume_menu_item_activate(GtkMenuItem * menuitem,
							   gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Convert EVMS Volume to Compatibility Volume"),
						  _("Convert"),
						  NULL,
						  on_convert_to_compatibility_volume_clist_realize,
						  on_convert_to_compatibility_volume_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_convert_to_evms_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for converting a compatibility volume
 *      to a native EVMS volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_convert_to_evms_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window =
	    create_convert_to_evms_volume_window(on_convert_to_evms_volume_clist_realize,
						 user_data);

	gtk_widget_show(window);
}

/*
 *
 *   void on_activatee_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for activating a volume or object.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_activate_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_activate_window(on_activate_clist_realize,
					user_data);

	gtk_widget_show(window);
}

/*
 *
 *   void on_deactivate_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for deactivating a volume or object.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_deactivate_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_deactivate_window(on_deactivate_clist_realize,
					  user_data);

	gtk_widget_show(window);
}

/*
 *
 *   void on_add_feature_to_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for adding a feature to a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_add_feature_to_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Add Feature to Volume (Volume Selection)"),
						  NULL,
						  NULL,
						  on_add_feature_to_volume_clist_realize,
						  on_volume_to_add_feature_to_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_modify_volume_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a volume's properties.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_volume_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Modify Logical Volume Properties"),
						  NULL,
						  get_modify_volume_properties_help_text(),
						  on_modify_volume_clist_realize,
						  on_modify_volume_selection_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_modify_object_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a storage object's properties.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_object_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Modify Storage Object Properties"),
						  NULL,
						  get_modify_storage_object_properties_help_text(),
						  on_set_info_clist_realize,
						  on_set_info_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_OBJECT));

	gtk_widget_show(window);
}

/*
 *
 *   void on_modify_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for modifying a container's properties.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_modify_container_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Modify Storage Container Properties"),
						  NULL,
						  get_modify_container_properties_help_text(),
						  on_set_info_clist_realize,
						  on_set_info_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(CONTAINER));

	gtk_widget_show(window);
}

/*
 *
 *   void on_view_messages_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine calls the routine that attempts to redisplay
 *      the informational message window if it exists.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Informational window is redisplayed if it exists.
 *
 */
void on_view_messages_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	redisplay_info_messages_window();
}

/*
 *
 *   void on_view_log_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the display of the log entries
 *      in a new window.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      A window containing the current log entries is displayed.
 *
 */
void on_view_log_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	display_not_implemented_yet_popup();
}

/*
 *
 *   void on_mkfs_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for making a file system on a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_mkfs_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Make File System"),
						  NULL,
						  get_mkfs_help_text(),
						  on_fsim_clist_realize,
						  on_plugin_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_Task_mkfs));

	gtk_widget_show(window);
}

/*
 *
 *   void on_fsck_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for checking/repairing a file system
 *      on a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_fsck_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Check/Repair File System"),
						  NULL,
						  get_fsck_help_text(),
						  on_filesys_operation_clist_realize,
						  on_initiate_filesys_operation_button_clicked,
						  NULL, NULL, NULL, NULL, GUINT_TO_POINTER(FSCK));

	gtk_widget_show(window);
}

/*
 *
 *   void on_obliterate_filesystem_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for remove a file system on volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_obliterate_filesystem_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Remove File System"),
						  _("Remove"),
						  get_unmkfs_help_text(),
						  on_filesys_operation_clist_realize,
						  on_remove_filesys_button_clicked,
						  NULL, NULL, NULL, NULL, GUINT_TO_POINTER(UNMKFS));

	gtk_widget_show(window);
}

/*
 *
 *   void on_mount_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for mounting a file system
 *      on a volume.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_mount_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_mount_operation_window(_("Mount File System"),
					       _("Mount"),
					       on_mount_volume_clist_realize,
					       on_mount_volume_button_clicked, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_unmount_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for unmounting a file system.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_unmount_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Unmount File System"),
						  _("Unmount"),
						  NULL,
						  on_unmount_volume_clist_realize,
						  on_unmount_volume_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void process_request_to_commit_changes (void)
 *
 *   Description:
 *      This routine call the function that displays a dialog
 *      confirming that the user wants to commit changes or
 *      not. If so, the dialog will initiate the commit process.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      If changes are pending, a confirmation dialog is displayed
 *      otherwise a popup is displayed indicating nothing to be done.
 *
 */
void process_request_to_commit_changes(void)
{
	int rc;
	boolean changes_pending;

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending) {
			display_commit_changes_confirmation_dialog();
		} else {
			display_nothing_to_commit_popup();
		}
	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
		display_results_window(rc, NULL, _("Unable to determine if changes are pending!"),
				       NULL, FALSE, get_main_window_id());
	}
}

/*
 *
 *   void process_request_to_backup_changes (void)
 *
 *   Description:
 *      This routine call the function that displays a dialog
 *      confirming that the user wants to backup changes or
 *      not. If so, the dialog will initiate the backup process.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      If non-activateion changes are pending, a dialog is shown saying that
 *      the metadata cannot be backed up.  Otherwise a confirmation dialog is
 *      displayed.
 *
 */
void process_request_to_backup_changes(void)
{
	int rc;
	int i;
	boolean changes_pending;
	change_record_array_t * change_records = NULL;

	rc = evms_changes_pending(&changes_pending, &change_records);
	if (rc == 0) {
		if (changes_pending) {
			/*
			 * The Engine can backup metadata if all that needs
			 * to be done is activation.
			 */
			changes_pending = FALSE;
			for (i = 0; !changes_pending && (i < change_records->count); i++) {
				if (change_records->changes_pending[i].changes & ~(CHANGE_ACTIVATE | CHANGE_REACTIVATE | CHANGE_DEACTIVATE)) {
					changes_pending = TRUE;
				}
			}
		}
		if (changes_pending) {
			display_cannot_backup_metadata_dialog();
		} else {
			display_backup_metadata_confirmation_dialog();
		}

		evms_free(change_records);
	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
		display_results_window(rc, NULL, _("Unable to determine if changes are pending!"),
				       NULL, FALSE, get_main_window_id());
	}
}

/*
 *
 *   void on_commit_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine call the function that displays a dialog
 *      confirming that the user wants to commit changes or
 *      not. If so, the dialog will initiate the commit process.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      If changes are pending, a confirmation dialog is displayed
 *      otherwise a popup is displayed indicating nothing to be done.
 *
 */
void on_commit_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	process_request_to_commit_changes();
}

/*
 *
 *   void on_backup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine call the function that displays a dialog
 *      confirming that the user wants to backup changes or
 *      not. If so, the dialog will initiate the backup process.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      If changes are pending, a confirmation dialog is displayed
 *      otherwise a popup is displayed indicating nothing to be done.
 *
 */
void on_backup_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	process_request_to_backup_changes();
}

/*
 *
 *   void process_request_to_exit (void)
 *
 *   Description:
 *      This routine checks for changes pending prior to
 *      closing the main window. If changes are pending
 *      then a popup is presented to the user to ask
 *      whether they want to continue exiting or commit
 *      changes then exit.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      We present a popup if changes are pending or simply
 *      destroy the main window if none.
 *
 */
void process_request_to_exit(void)
{
	int rc;
	boolean changes_pending;

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending)
			display_commit_changes_popup();
		else {
			gtk_widget_destroy(get_main_window_id());
		}
	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
		display_results_window(rc, NULL, _("Unable to determine if changes are pending!"),
				       NULL, FALSE, get_main_window_id());
	}
}

/*
 *
 *   void on_exit_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine checks for changes pending prior to
 *      closing the main window. If changes are pending
 *      then a popup is presented to the user to ask
 *      whether they want to continue exiting or commit
 *      changes then exit.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      We present a popup if changes are pending or simply
 *      destroy the main window if none.
 *
 */
void on_exit_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	process_request_to_exit();
}

/*
 *
 *   gboolean on_main_window_delete_event (GtkWidget *, GdkEvent *, gpointer)
 *
 *   Description:
 *      This routine checks for changes pending prior to
 *      closing the main window. If changes are pending
 *      and a commit is currently not already in progress
 *      then a popup is presented to the user to ask
 *      whether they want to continue exiting or commit
 *      changes then exit.
 *
 *   Entry:
 *      widget    - widget that caused delete-event to occur
 *      event     - everything we would like to know about event
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      We present a popup if changes are pending or simply
 *      allow closure of the main window if none.
 *
 */
gboolean on_main_window_delete_event(GtkWidget * widget, GdkEvent * event, gpointer user_data)
{
	int rc;
	boolean changes_pending;
	gboolean prevent_closure = FALSE;

	rc = evms_changes_pending(&changes_pending, NULL);
	if (rc == 0) {
		if (changes_pending) {
			if (is_commit_in_progress() == FALSE)
				display_commit_changes_popup();

			prevent_closure = TRUE;
		}
	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
		display_results_window(rc, NULL, _("Unable to determine if changes are pending!"),
				       NULL, FALSE, get_main_window_id());
	}

	return prevent_closure;
}

/* BUGBUG (not implemented yet) */
void on_beginner_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	display_not_implemented_yet_popup();
}

/* BUGBUG (not implemented yet) */
void on_experienced_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	display_not_implemented_yet_popup();
}

/*
 *
 *   void on_web_links_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates and displays the web links box.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      About box is displayed and waiting for dismissal.
 *
 */
void on_web_links_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *web_links_window;

	web_links_window = create_web_links_window();

	gtk_widget_show(web_links_window);
}

/*
 *
 *   void on_about_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates and displays the about box.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      About box is displayed and waiting for dismissal.
 *
 */
void on_about_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *about_window;

	about_window = create_about_window();

	gtk_widget_show(about_window);
}

/*
 *
 *   void on_browser_notebook_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine makes some modifications to the browser
 *      notebook before it gets displayed. These modifications
 *      are minor and are things not easily done through Glade.
 *
 *   Entry:
 *      widget    - address of the GtkNotebook widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Notebook tab icons added, tooltips added, etc.
 *
 */
void on_browser_notebook_realize(GtkWidget * widget, gpointer user_data)
{
	/*
	 * Given the GtkWidget of the GtkNotebook, do child
	 * widget lookups to find the GtkHBox in each tab
	 * to create the corresponding GtkPixmap from the
	 * GdkPixmap already allocated at init time.
	 */

	set_browser_notebook_tab_pixmap(widget, "volumes_view_tab_hbox", VOLUME, FALSE);
	set_browser_notebook_tab_pixmap(widget, "topmost_view_tab_hbox", EVMS_OBJECT, TRUE);
	set_browser_notebook_tab_pixmap(widget, "feature_objects_view_tab_hbox", EVMS_OBJECT,
					FALSE);
	set_browser_notebook_tab_pixmap(widget, "regions_view_tab_hbox", REGION, FALSE);
	set_browser_notebook_tab_pixmap(widget, "containers_view_tab_hbox", CONTAINER, FALSE);
	set_browser_notebook_tab_pixmap(widget, "segments_view_tab_hbox", SEGMENT, FALSE);
	set_browser_notebook_tab_pixmap(widget, "disks_view_tab_hbox", DISK, FALSE);
	set_browser_notebook_tab_pixmap(widget, "plugins_view_tab_hbox", PLUGIN, FALSE);

	/*
	 * Make an initial determination of which views should be realized now
	 * if they are empty. Realizing activates code that will hide the view
	 * altogether if it is truly empty.
	 */
	realize_empty_views_in_notebook(GTK_NOTEBOOK(widget));
}

/*
 *
 *   void on_plugin_columned_list_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugins installed in the system.
 *
 *   Entry:
 *      widget    - address of the plugins GtkCList widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Plugins list is populated
 *
 */
void on_plugin_columned_list_realize(GtkWidget * widget, gpointer user_data)
{
	populate_plugin_view_with_plugins(GTK_CLIST(widget));
	gtk_clist_column_titles_passive(GTK_CLIST(widget));
}

/*
 *
 *   void on_view_volumes_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the volumes view.
 *
 *      Below each volume will be the tree of objects that comprise
 *      it.
 *
 *   Entry:
 *      widget    - the address of the volumes view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_volumes_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, VOLUME_VIEW);
	populate_volume_view_ctree(ctree, NULL);
}

/*
 *
 *   void on_view_objects_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the objects view. The topmost objects
 *      are at the root of the hiearchy. They in turn contain
 *      the tree of storage objects that comprise them below them.
 *
 *   Entry:
 *      widget    - the address of the topmost objects view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_objects_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, OBJECT_VIEW);
	populate_object_view_ctree(ctree, NULL);
}

/*
 *
 *   void on_view_feature_objects_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the feature objects view. The feature objects
 *      are at the root of the hiearchy. They in turn contain
 *      the tree of storage objects that comprise them below them.
 *
 *   Entry:
 *      widget    - the address of the feature objects view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_feature_objects_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, FEATURE_VIEW);
	populate_feature_object_view_ctree(ctree, NULL);
}

/*
 *
 *   void on_view_regions_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the regions view. The region managers
 *      are at the root of the hierachy. If the region manager
 *      supports containers, they are showed underneath with
 *      the regions below them. If the region manager does not
 *      support containers then all its regions are displayed
 *      together underneath the plugin.
 *
 *   Entry:
 *      widget    - the address of the regions view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_regions_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, REGION_VIEW);
	gtk_clist_set_column_justification(GTK_CLIST(ctree), RV_CORRUPT_COLUMN, GTK_JUSTIFY_CENTER);
	populate_region_view_ctree(ctree, NULL);
}

/*
 *
 *   void on_view_containers_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the containers view. The plugins that support
 *      containers are at the root of the hierachy. They contain
 *      storage containers that they export objects from. Underneath
 *      containers is the list of objects they produce and objects
 *      they consume.
 *
 *   Entry:
 *      widget    - the address of the containers view GtkCTree widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_containers_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, CONTAINER_VIEW);
	populate_container_view_ctree(ctree, NULL);
}

/*
 *
 *   void on_view_segments_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the segments view. The segment managers
 *      are at the root of the hierachy. They contain the
 *      segments they manage and under each segment are the
 *      objects that comprise them.
 *
 *   Entry:
 *      widget    - the address of the segment view GtkCTree widget
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_segments_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, SEGMENT_VIEW);
	populate_segment_view_ctree(ctree, NULL);
}

/*
 *
 *   void on_view_disks_ctree_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine initializes the hiearchy found in the
 *      GtkCTree for the logical disk view. The device managers
 *      are at the root of the hierachy. They contain logical
 *      disks that they export. Under the disks is a list of
 *      parent objects such as a segment or volume.
 *
 *   Entry:
 *      widget    - the address of the disk view GtkCTree widget
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_disks_ctree_realize(GtkWidget * widget, gpointer user_data)
{
	GtkCTree *ctree = GTK_CTREE(widget);

	initialize_view(ctree, DISK_VIEW);
	populate_disk_view_ctree(ctree, NULL);
}

/*
 *
 *   gboolean on_view_button_press_event (GtkWidget *, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine gets invoked when there is a "button-press-event" in any
 *      of the information viewer GtkCTree or GtkCList views. We then determine
 *      if it was a right mouse button that was pressed and if over a valid row
 *      of data. If so, we call the routine that displays a context popup of
 *      actions that can be done with the row data.
 *
 *   Entry:
 *      widget    - the GtkCTree or GtkCList that generated the button pressed event
 *      event     - everything we want to know about the event
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Creates a context sensitive popup menu populated with
 *      actions suitable for the row data the user clicked on.
 *
 */
gboolean on_view_button_press_event(GtkWidget * widget, GdkEventButton * event, gpointer user_data)
{
	gint row;
	gint col;
	gint selection_in_range;
	GtkCList *clist = GTK_CLIST(widget);

	selection_in_range = gtk_clist_get_selection_info(clist, (gint) event->x,
							  (gint) event->y, &row, &col);

	if (selection_in_range && event->button == RIGHT_MOUSE_BUTTON) {
		object_handle_t handle;

		handle = GPOINTER_TO_UINT(gtk_clist_get_row_data(clist, row));

		gtk_clist_select_row(clist, row, col);

		display_action_popup_menu_for_handle(handle, event->button, event->time, NULL,
						     NULL);
	}

	return FALSE;
}

/*
 *
 *   gboolean on_view_key_press_event (GtkWidget *, GdkEventKey *, gpointer)
 *
 *   Description:
 *      This routine gets invoked when there is a "key-press-event" in any
 *      of the information viewer GtkCTree or GtkCList views. We then determine
 *      the row that has the focus and invoke the routine to display a context
 *      popup menu over that row. The context popup is only displayed if the
 *      either one of the Enter keys was pressed of Shift-F10.
 *
 *   Entry:
 *      widget    - the GtkCTree or GtkCList that generated the key press event
 *      event     - everything we want to know about the event
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Creates a context sensitive popup menu populated with
 *      actions suitable for the row data for a valid key press.
 *
 */
gboolean on_view_key_press_event(GtkWidget * widget, GdkEventKey * event, gpointer user_data)
{
	GtkCList *clist = GTK_CLIST(widget);

	if (event->keyval == GDK_Return ||
	    event->keyval == GDK_ISO_Enter ||
	    event->keyval == GDK_KP_Enter ||
	    (event->keyval == GDK_F10 && (event->state | GDK_SHIFT_MASK))) {
		object_handle_t handle;

		handle = GPOINTER_TO_UINT(gtk_clist_get_row_data(clist, clist->focus_row));

		gtk_clist_select_row(clist, clist->focus_row, 0);

		display_action_popup_menu_for_handle(handle, 0, 0,
						     (GtkMenuPositionFunc)
						     position_context_menu_over_clist_focus_row,
						     clist);
	}

	return FALSE;
}

/*
 *
 *   gboolean on_refresh_pixmap_press_event (GtkWidget *, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine gets invoked when there is a "button-press-event" in the
 *      eventbox for the refresh view pixmap. If the left mouse button was pressed
 *      we refresh the current view. If the right mouse button was pressed, we
 *      present a popup menu with refresh view choices.
 *
 *   Entry:
 *      widget    - the widget that generated the button pressed event
 *      event     - everything we want to know about the event
 *      user_data - address of user data bound to the widget
 *
 *   Exit:
 *      Either refreshes the current view or offers a popup menu
 *      with refresh view choices.
 *
 */
gboolean on_refresh_pixmap_eventbox_button_press_event(GtkWidget * widget, GdkEventButton * event,
						       gpointer user_data)
{
	if (event->button == LEFT_MOUSE_BUTTON) {
		refresh_current_view();
	} else if (event->button == RIGHT_MOUSE_BUTTON) {
		display_refresh_view_popup_menu(event);
	}

	return TRUE;
}

/*
 *
 *   void set_engine_log_level (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine is called when a set engine log level
 *      menu item is activated. The new log level is supplied
 *      in the user_data field.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - contains a debug_level_t value
 *
 *   Exit:
 *      Invokes the API to set the engine debug level appropriately
 *
 */
void set_engine_log_level(GtkMenuItem * menuitem, gpointer user_data)
{
	gint rc;

	rc = evms_set_debug_level(GPOINTER_TO_UINT(user_data));

	if (rc != SUCCESS) {
		log_error("%s: evms_set_debug_level() returned error code %d.\n", __FUNCTION__, rc);

		display_popup_window(_("Error Changing EVMS Engine Log Level"),
				     _
				     ("An error was encountered attempting to change the EVMS Engine log level."));
	}
}

/*
 *
 *   void on_engine_log_level_submenu_menu_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine sets the menu items' user data to the debug_level_t
 *      each represents. This makes it easier to select the menu item
 *      that should be marked active when the menu is displayed.
 *
 *   Entry:
 *      widget    - the address of the GtkMenu widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_engine_log_level_submenu_menu_realize(GtkWidget * widget, gpointer user_data)
{
	GtkObject *object;
	GtkObject *main_window;

	main_window = GTK_OBJECT(get_main_window_id());

	object = gtk_object_get_data(main_window, "critical_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(CRITICAL));

	object = gtk_object_get_data(main_window, "serious_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(SERIOUS));

	object = gtk_object_get_data(main_window, "error_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(ERROR));

	object = gtk_object_get_data(main_window, "warning_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(WARNING));

	object = gtk_object_get_data(main_window, "default_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(DEFAULT));

	object = gtk_object_get_data(main_window, "details_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(DETAILS));

	object = gtk_object_get_data(main_window, "debug_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(DEBUG));

	object = gtk_object_get_data(main_window, "extra_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(EXTRA));

	object = gtk_object_get_data(main_window, "entry_exit_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(ENTRY_EXIT));

	object = gtk_object_get_data(main_window, "everything_engine_level_menuitem");
	gtk_object_set_user_data(object, GUINT_TO_POINTER(EVERYTHING));
}


/*
 *
 *   void activate_check_menu_item_matching_debug_level (GtkMenu *, debug_level_t)
 *
 *   Description:
 *      This routine ensures that a menu item corresponding to a debug
 *      level is active, if not it makes the corresponding one active.
 *
 *   Entry:
 *      menu      - the menu containing the debug level menu items
 *      new_level - the level to ensure the active menu item is at
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void activate_check_menu_item_matching_debug_level(GtkMenu * menu, debug_level_t new_level)
{
	GtkMenuItem *menuitem;
	debug_level_t menuitem_level = 0;

	/*
	 * First, check to see if current active menuitem corresponds to the
	 * level we are to set. If so, cool, nothing to do. Otherwise, walk
	 * the menu's menu items looking to activate the one matching the
	 * level we were given.
	 */

	menuitem = GTK_MENU_ITEM(gtk_menu_get_active(menu));

	if (menuitem)
		menuitem_level = GPOINTER_TO_UINT(gtk_object_get_user_data(GTK_OBJECT(menuitem)));

	if (menuitem_level != new_level) {
		GList *children;

		children = gtk_container_children(GTK_CONTAINER(menu));

		if (children) {
			gint count;

			count = g_list_length(children);

			if (count > 0) {
				gint i;

				for (i = 0; i < count; i++) {
					menuitem = g_list_nth_data(children, i);

					if (GTK_IS_CHECK_MENU_ITEM(menuitem)) {
						menuitem_level =
						    GPOINTER_TO_UINT(gtk_object_get_user_data
								     (GTK_OBJECT(menuitem)));

						if (menuitem_level == new_level) {
							log_debug
							    ("%s: Making menu item corresponding to debug level %u active.\n",
							     __FUNCTION__, new_level);

							gtk_check_menu_item_set_active
							    (GTK_CHECK_MENU_ITEM(menuitem), TRUE);
							break;
						}
					}
				}
			}
		}
	}
}

/*
 *
 *   void on_engine_log_level_submenu_menu_map (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine ensures that the active menu item for the engine
 *      log level matches the current log level each time the menu
 *      gets displayed.
 *
 *   Entry:
 *      widget    - the address of the GtkMenu widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_engine_log_level_submenu_menu_map(GtkWidget * widget, gpointer user_data)
{
	debug_level_t level;

	if (evms_get_debug_level(&level) == 0)
		activate_check_menu_item_matching_debug_level(GTK_MENU(widget), level);
}

/*
 *
 *   void disable_filesys_action_menu_items_if_no_fsims (void)
 *
 *   Description:
 *      This routine ensures (as its name implies) disables the
 *      filesystem related action menu items if there are no
 *      FSIM plug-ins.
 *
 *   Entry:
 *      Nothing
 *
 *   Exit:
 *      Filesystem action menu items are made insensitive if no
 *      FSIM plug-ins were found.
 *
 */
void disable_filesys_action_menu_items_if_no_fsims(void)
{
	gint rc;
	handle_array_t *fsims;

	rc = evms_get_plugin_list(EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &fsims);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		if (fsims->count == 0) {
			GtkWidget *widget;
			GtkObject *main_window;

			main_window = GTK_OBJECT(get_main_window_id());

			widget = GTK_WIDGET(gtk_object_get_data(main_window, "mkfs_menu_item"));
			gtk_widget_set_sensitive(widget, FALSE);

			widget = GTK_WIDGET(gtk_object_get_data(main_window, "fsck_menu_item"));
			gtk_widget_set_sensitive(widget, FALSE);

			widget =
			    GTK_WIDGET(gtk_object_get_data
				       (main_window, "obliterate_filesystem_menu_item"));
			gtk_widget_set_sensitive(widget, FALSE);
		}

		evms_free(fsims);
	}
}

/*
 *
 *   void on_actions_menu_item_menu_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine disables certain menu items that are known to
 *      be unavailable throughout the GUI session.
 *
 *   Entry:
 *      widget    - the address of the GtkMenu widget
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_actions_menu_item_menu_realize(GtkWidget * widget, gpointer user_data)
{
	disable_filesys_action_menu_items_if_no_fsims();
}

/*
 *
 *   void on_toolbar_commit_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine is invoked when the user clicks on
 *      the commit button in the toolbar in order to save
 *      any pending changes.
 *
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Commits any unsaved data
 *
 */
void on_toolbar_commit_button_clicked(GtkButton * button, gpointer user_data)
{
	process_request_to_commit_changes();
}

/*
 *
 *   void on_refresh_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine is invoked when the user clicks on
 *      the refresh button in the toolbar in order to
 *      refresh the contents of all the views.
 *
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Refreshes those views that have previously been realized/displayed
 *
 */
void on_refresh_button_clicked(GtkButton * button, gpointer user_data)
{
	refresh_main_window_views();
}

/*
 *
 *   void on_exit_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine is invoked when the user clicks on
 *      the exit button in the toolbar in order to exit
 *      the application.
 *
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Exits the application
 *
 */
void on_exit_button_clicked(GtkButton * button, gpointer user_data)
{
	process_request_to_exit();
}

/*
 *
 *   void on_view_ctree_tree_expand (GtkCTree *, GList *, gpointer)
 *
 *   Description:
 *      This routine handles is invoked when a node in a ctree view
 *      is expanded. Currently, all it does is re-color the node
 *      backgrounds.
 *
 *   Entry:
 *      ctree     - the GtkCTree that generated the signal
 *      node      - the affected node
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_ctree_tree_expand(GtkCTree * ctree, GList * node, gpointer user_data)
{
	set_all_ctree_node_colors(ctree);
}

/*
 *
 *   void on_view_ctree_tree_collapse (GtkCTree *, GList *, gpointer)
 *
 *   Description:
 *      This routine handles is invoked when a node in a ctree view
 *      is collapsed. Currently, all it does is re-color the node
 *      backgrounds.
 *
 *   Entry:
 *      ctree     - the GtkCTree that generated the signal
 *      node      - the affected node
 *      user_data - address of user data bound to the widget (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_view_ctree_tree_collapse(GtkCTree * ctree, GList * node, gpointer user_data)
{
	set_all_ctree_node_colors(ctree);
}

/*
 *
 *   void on_volume_tasks_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      allow selecting a volume and plugin-specific tasks
 *      to invoke on the selection.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_volume_tasks_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Select Volume for Plug-in Tasks"),
						  NULL,
						  NULL,
						  on_thing_with_tasks_clist_realize,
						  on_plugin_function_context_selection_button_clicked,
						  NULL, NULL, NULL, NULL, GUINT_TO_POINTER(VOLUME));

	gtk_widget_show(window);
}

/*
 *
 *   void on_object_tasks_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      allow selecting a storage object and plugin-specific tasks
 *      to invoke on the selection.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_object_tasks_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Select Storage Object for Plug-in Tasks"),
						  NULL,
						  NULL,
						  on_thing_with_tasks_clist_realize,
						  on_plugin_function_context_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(EVMS_OBJECT));

	gtk_widget_show(window);
}

/*
 *
 *   void on_container_tasks_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      allow selecting a container and plugin-specific tasks
 *      to invoke on the selection.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_container_tasks_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Select Storage Container for Plug-in Tasks"),
						  NULL,
						  NULL,
						  on_thing_with_tasks_clist_realize,
						  on_plugin_function_context_selection_button_clicked,
						  NULL, NULL, NULL, NULL,
						  GUINT_TO_POINTER(CONTAINER));

	gtk_widget_show(window);
}

/*
 *
 *   void on_plugin_tasks_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      allow selecting a plugin and plugin-specific tasks
 *      to invoke on the selection.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_plugin_tasks_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Select Plug-in for Plug-in Tasks"),
						  NULL,
						  NULL,
						  on_plugin_with_tasks_clist_realize,
						  on_plugin_function_context_selection_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}

/*
 *
 *   void on_move_object_data_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for moving the data located on a
 *      storage object to an available target object.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_move_object_data_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_move_source_selection_window(0);

	gtk_widget_show(window);
}

/*
 *
 *   void on_delete_container_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine creates the generic selection dialog
 *      and connects the appropriate signal handlers that will
 *      handle the actions for deleting a storage container.
 *
 *   Entry:
 *      menuitem  - address of the GtkMenuItem widget
 *      user_data - address of user data bound with signal (not used)
 *
 *   Exit:
 *      Generic selection dialog is displayed and signal handlers
 *      have been connected for the corresponding actions.
 *
 */
void on_administer_node_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_standard_selection_window(_("Administer Remote Node"),
						  _("Administer"),
						  NULL,
						  on_remote_node_clist_realize,
						  on_remote_node_button_clicked,
						  NULL, NULL, NULL, NULL, NULL);

	gtk_widget_show(window);
}
