/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libcsm.so
 *
 *   File: segments.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <uuid/uuid.h>

#include <plugin.h>

#include "csm_plugin.h"

static boolean quorum_check_completed=FALSE;


int csm_can_set_volume( DISKSEG * seg, boolean flag )
{
        LOG_ENTRY();
        REQUIRE( isa_cluster_segment(seg) == TRUE );
        LOG_EXIT_INT(0);
        return 0;
}

int csm_can_delete( DISKSEG * seg )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int csm_can_expand( DISKSEG         *seg,
                    sector_count_t   expand_limit,
                    list_anchor_t    expansion_points )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}

int csm_can_expand_by( DISKSEG * seg, sector_count_t *size)
{
        LOG_ENTRY();
        *size = 0;
        LOG_EXIT_INT(0);
        return 0;
}


int csm_can_shrink( DISKSEG          * seg,
                    sector_count_t     shrink_limit,
                    list_anchor_t      shrink_points )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int csm_can_shrink_by( DISKSEG * seg, sector_count_t * size )
{
        LOG_ENTRY();
        *size=0;
        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function:  csm_discover
 *
 *  Called to run discovery on a list of evms objects that
 *  the engine has found thus far.  We walk the list of objects,
 *  probing each object to see if it has a CSM header.  If so,
 *  we consume the object by removing it from the list, and place
 *  all new segment storage objects on the output object list.
 *  Any object we dont like in the input_object list must be
 *  copied to the output_object list.
 *
 */
int csm_discover( list_anchor_t input_objects, list_anchor_t output_objects, boolean final_call)
{
        int  rc=0;
        int  count = 0;
        list_element_t iter;
        storage_object_t *object;

        LOG_ENTRY();

        if ( (input_objects != NULL) &&
             (output_objects != NULL) &&
             (EngFncs->list_count(input_objects) > 0)) {


                if (quorum_check_completed==FALSE) {

                        rc = EngFncs->get_clusterid( &csm_clusterid);

                        if (!rc) {
                                rc = EngFncs->get_nodeid( &csm_nodeid);
                        }

                        csm_has_quorum = EngFncs->have_quorum();

                        if (csm_has_quorum==TRUE) {
                                LOG_DEBUG("I have quorum.\n");
                        }
                        else {
                                LOG_DEBUG("I do not have quorum.\n");
                        }

                        quorum_check_completed=TRUE;
                }

                LIST_FOR_EACH( input_objects, iter, object ) {
                        discover_cluster_segments( object, output_objects, &count);
                }

        }

        LOG_EXIT_INT(count);
        return count;
}


/*
 *  Function:  assign_cluster_segment_manager_to_disk
 *
 *  This is the low level implementation of assigning the Cluster Segment
 *  Manager (CSM) to a logical disk.  In theory this means that we layout
 *  our metadata on the logical disk and produce segment storage objects
 *  that correspond to: metadata, freespace and data segments.  In practice,
 *  this means we will be creating 3 segments:
 *
 *  (1) Metadata segment: corresponds to 1st copy of the CSM metadata which
 *                        is always the very first sector of the logical disk.
 *
 *  (2) Data segment    : maps the useable area of the shared object
 *
 *  (3) Metadata segment: corresponds to 2nd copy of the CSM metadata which is
 *                        usually the very last sector of the logical disk.
 *
 *  In addition, since all shared objects belong to a disk group, we will
 *  be adding the logical disk to an EVMS container that represents the
 *  disk group and the data segment will appear on the container's produced_objects
 *  list.  Using a container means that by modifying properties of the
 *  container we can effect changes that impact a collection of shared storage
 *  objects, e.g. we can change ownership of the disk group.
 */
int assign_cluster_segment_manager_to_disk( LOGICALDISK         *ld,
					    storage_container_t *container )
{
        int                  rc=0;
        DISKSEG             *md1=NULL;
        DISKSEG             *md2=NULL;
        DISKSEG             *dataseg=NULL;
        csm_header_t        *hdr1;
        csm_header_t        *hdr2;
        disk_private_data_t *disk_pdata=NULL;
        lba_t                hdr1_lba;
        lba_t                hdr2_lba;
        lba_t                start_useable;
        lba_t                end_useable;
        guid_t               disk_id;
        char                *disk_guid_name;
        int                  vsects_per_hardsect;
        container_private_data_t *c_pdata;

        LOG_ENTRY();

        c_pdata = (container_private_data_t *) container->private_data;

        // we should not have a private data area for this disk yet because we were just
        // told to install on it.  we must have been passed a logical disk we are already
        // using and this is very wrong.
        if ( get_csm_disk_private_data(ld) ) {
                LOG_ERROR("attempting to reinstall on disk (%s) but we already seem to know about the disk.\n", ld->name);
                rc = EINVAL;
                LOG_EXIT_INT(rc);
                return rc;
        }

        hdr1= (csm_header_t *) calloc(1, EVMS_VSECTOR_SIZE);
        hdr2= (csm_header_t *) calloc(1, EVMS_VSECTOR_SIZE);

        if (hdr1 == NULL || hdr2 == NULL) {
                LOG_ERROR("unable to malloc csm headers\n");
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }

        // create disk private data area
        create_csm_disk_private_data( ld );
        disk_pdata = get_csm_disk_private_data(ld);
        if (disk_pdata == NULL) {
                LOG_ERROR("error, attempted to create disk private data and failed\n");
                if (hdr1) free(hdr1);
                if (hdr2) free(hdr2);
                rc = ENOMEM;
                LOG_EXIT_INT(rc);
                return rc;
        }

        // need the device hard sector size for allignment
        vsects_per_hardsect = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;

        // Locate the primary header in the 1st sector
        hdr1_lba = 0;

        // Locate the start of useable area immediately after the header
        // but allign with a hard sector boundary.
        start_useable = hdr1_lba + vsects_per_hardsect;

        // Locate alternate header in the last sector
        hdr2_lba = ld->start + ld->size - 1;

        // BUG Fix ...
        //
        // Lop off an extra sector at the end of the disk!!  This prevents the engine
        // from discovering what it thinks is ... the alternate feature header at the
        // end of the disk and exporting an evms volume when the CSM has lost quorum
        // and is not producing segment objects from the disk.  In this case, the engine
        // will run disocvery on the disk object and look at the last 2 sectors and find
        // a feature header in the next to last sector.  This is actually the 1st copy
        // of the feature header when the CSM does export the data segment but when the
        // CSM doesnt export the segment ... the engine thinks it found the alternate!!
        end_useable = hdr2_lba - 1;
        --end_useable;

        // generate a unique disk id
        uuid_generate_random( (char *) &disk_id );

        // REGISTER DISK - convert DISK uuid to ascii string and register it.
        disk_guid_name = guid_to_string( &disk_id );
        if (disk_guid_name) {
                rc = EngFncs->register_name( disk_guid_name );
        }
        else {
                rc = ENOMEM;
        }

        if (rc) {
                free(hdr1);
                free(hdr2);
                if (disk_guid_name) free(disk_guid_name);
                LOG_ERROR("error, unable to convert DISK uuid identifier to ascii string and register it.\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        // create 1st copy of metadata
        hdr1->signature         = CSM_DISK_MAGIC;
        hdr1->size              = sizeof(csm_header_t);
        hdr1->alternate_lba     = hdr2_lba;
        hdr1->start_useable     = start_useable;
        hdr1->end_useable       = end_useable;

        // create CSM header flag bits
        if (container->flags & SCFLAG_CLUSTER_PRIVATE) {
                hdr1->flags |= CSM_CLUSTER_PRIVATE;
        }
        else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                hdr1->flags |= CSM_CLUSTER_SHARED;
        }
        else if (container->flags & SCFLAG_CLUSTER_DEPORTED) {
                hdr1->flags |= CSM_CLUSTER_DEPORTED;
        }
        else {
                rc = EINVAL;
                free(hdr1);
                free(hdr2);
                if (disk_guid_name) free(disk_guid_name);
                LOG_ERROR("error, the container does not have its cluster storage bits set, i.e. SCFLAG_CLUSTER_XXXX\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        memcpy(&hdr1->disk_id, &disk_id, sizeof(guid_t));
        memcpy(&hdr1->nodeid, &c_pdata->nodeid, sizeof(ece_nodeid_t));
        memcpy(&hdr1->clusterid, &csm_clusterid, sizeof(ece_clusterid_t));
        strncpy(hdr1->container_name, container->name, EVMS_NAME_SIZE);

        // create 2nd copy of metadata
        memcpy(hdr2, hdr1, sizeof(csm_header_t));
        hdr2->alternate_lba     = hdr1_lba;


        md1 = create_csm_metadata_segment( ld,
                                           container,
                                           hdr1_lba,
                                           1,
                                           "metadata1",
                                           SOFLAG_DIRTY );

        md2 = create_csm_metadata_segment( ld,
                                           container,
                                           hdr2_lba,
                                           1,
                                           "metadata2",
                                           SOFLAG_DIRTY );

        if (md1 && md2) {
                rc = 0;
        }

        if (!rc) {

                if (isa_accessible_container(container)==TRUE) {

                        dataseg = create_csm_data_segment( ld,
                                                           container,
                                                           start_useable,
                                                           end_useable - start_useable + 1 );

                        if (dataseg==NULL) {
                                rc = ENOMEM;
                        }
                }

        }

        if (!rc) {

                disk_pdata->md1 = md1;
                ((seg_private_data_t *)md1->private_data)->hdr = hdr1;

                disk_pdata->md2 = md2;
                ((seg_private_data_t *)md2->private_data)->hdr = hdr2;

		rc = insert_csm_segment_into_list( ld->parent_objects, md1);

		if (!rc) {
			rc = insert_csm_segment_into_list( ld->parent_objects, md2);
		}

		if (!rc) {
			if (dataseg != NULL) {
				rc = insert_csm_segment_into_list( ld->parent_objects, dataseg);
			}
		}

		if (!rc) {
			rc = add_disk_to_container( ld, container );
		}

        }
        else {
                rc = ENOMEM;
        }

        if (rc) {
                csm_remove_object(ld);
                if (EngFncs->list_count(ld->parent_objects))  EngFncs->delete_all_elements(ld->parent_objects);
                delete_csm_disk_private_data( ld );
                if (md1) free_csm_segment(md1);
                if (md2) free_csm_segment(md2);
                if (dataseg) free_csm_segment(dataseg);
                EngFncs->unregister_name( disk_guid_name );
                free(hdr1);
                free(hdr2);
        }

        if (disk_guid_name) free(disk_guid_name);
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  get_assign_options
 *
 *  Called to parse the option descriptors and pull values needed for
 *  an assign operation.
 *
 *  The current options are:
 *
 *  Node Id:            So we can tell who owns the shared storage object
 *  Container Name:     So we know which disk group to add the shared object to
 *
 */
static int get_assign_options( option_array_t  *options,
                               ece_nodeid_t    *nodeid,
                               char            *storage_type,
                               char            *container_name )
{
        int rc=0;
        int i;
        boolean  got_nodeid=FALSE;
        boolean  got_container_name=FALSE;
        boolean  got_storage_type=FALSE;

        LOG_ENTRY();

        LOG_DEBUG("options count = %d\n", options->count);

        for (i = 0; i < options->count && rc==0; i++) {

                if (options->option[i].is_number_based) {

                        switch (options->option[i].number) {

                        case CSM_ASSIGN_OPTION_NODE_ID_INDEX:

                                rc = EngFncs->string_to_nodeid( options->option[i].value.s, nodeid);

                                if (!rc) {
                                        got_nodeid = TRUE;
                                }
                                break;

                        case CSM_ASSIGN_OPTION_TYPE_INDEX:

                                strncpy(storage_type, options->option[i].value.s, EVMS_NAME_SIZE);
                                got_storage_type = TRUE;
                                break;


                        case CSM_ASSIGN_OPTION_CONTAINER_NAME_INDEX:

                                strncpy(container_name, options->option[i].value.s, EVMS_NAME_SIZE);
                                got_container_name = TRUE;
                                break;


                        default:
                                rc = EINVAL;
                                break;
                        }
                }
                else {

                        if (strcmp(options->option[i].name,CSM_ASSIGN_OPTION_NODE_ID_STRING) == 0) {

                                rc = EngFncs->string_to_nodeid( options->option[i].value.s,nodeid);

                                if (!rc) {
                                        got_nodeid = TRUE;
                                }
                        }
                        else if (strcmp(options->option[i].name,CSM_ASSIGN_OPTION_TYPE_STRING) == 0) {
                                strncpy(storage_type, options->option[i].value.s, EVMS_NAME_SIZE);
                                got_storage_type = TRUE;
                        }
                        else if (strcmp(options->option[i].name,CSM_ASSIGN_OPTION_CONTAINER_NAME_STRING) == 0) {
                                strncpy(container_name, options->option[i].value.s, EVMS_NAME_SIZE);
                                got_container_name = TRUE;
                        }
                        else {
                                rc = EINVAL;
                        }

                }

        }


        // require node id and container name
        if (!rc) {
                if ( got_nodeid != TRUE || strlen(container_name)==0 || strlen(storage_type)==0) {
                        rc = EINVAL;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  csm_assign
 *
 *  Called to add the Cluster Segment Manager to a logical disk. Normally
 *  this means that we will lay down metadata that allows the disk to be
 *  partitioned.  However, the CSM is assigned to a storage object as a
 *  means of:
 *
 *      - identifying a storage object as being shared storage
 *      - identifying the owner of the shared storage object
 *      - adding the shared storage object to a cluster disk group
 *      - producing useable space on the share object by creating a data segment
 *        which can then be consumed by other plug-ins.
 *
 */
int csm_assign( storage_object_t * input_object, option_array_t * options )
{
        int rc;
        ece_nodeid_t  nodeid;
        char   storage_type[EVMS_NAME_SIZE+1];
        char   container_name[EVMS_NAME_SIZE+1];
        storage_container_t *container=NULL;
        boolean  created_new_container=FALSE;
        u_int32_t  flags = SCFLAG_DIRTY;

        LOG_ENTRY();

        REQUIRE( input_object != NULL );
        REQUIRE( options != NULL);
        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);

        rc = get_assign_options(options, &nodeid, storage_type, container_name);

        if (!rc) {

                rc = csm_find_container( container_name, &container );
                if (rc) {

                        if (strcmp(storage_type,_("private"))==0) {
                                flags |= SCFLAG_CLUSTER_PRIVATE;
                        }
                        else if (strcmp(storage_type,_("shared"))==0) {
                                flags |= SCFLAG_CLUSTER_SHARED;
                        }
                        else if (strcmp(storage_type,_("deported"))==0) {
                                flags |= SCFLAG_CLUSTER_DEPORTED;
                        }
                        else {
                                flags |= SCFLAG_CLUSTER_PRIVATE;
                        }

                        rc = csm_create_container_storage_object( container_name,
                                                                  &container,
                                                                  &nodeid,
                                                                  &csm_clusterid,
                                                                  flags );
                        created_new_container = TRUE;
                }

                if (!rc) {

                        rc = assign_cluster_segment_manager_to_disk( input_object,
                                                                     container );
                        if (!rc) {
				if (!rc) {
					// Tell other nodes in the cluster to rediscover everything.
					// We don't know how this disk appears on the other nodes,
					// so everything has to be rediscovered.
					EngFncs->rediscover_objects(NULL);
				}

                        } else {
                                if ( created_new_container == TRUE ) {
                                        free_csm_container(container);
                                }

			}

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  csm_unassign
 *
 *  We can unassign from any logical disk storage object that
 *  has CSM metadata on it as long as the data segment is a
 *  top-most storage object.
 *
 */
int csm_can_unassign( LOGICALDISK *ld )
{
        DISKSEG *seg;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE( isa_cluster_logical_disk(ld) == TRUE );
        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);

	if ( !csm_admin_mode ) {

		// We can't unassign from a disk if the container is deported or
		// if the container is private and ownded by another node.  If
		// either of those cases is true, we cannot check if the data
		// segment is in use, because it is not produced, and therefore
		// cannot safely allow the CSM to be unassigned from the disk.
		REQUIRE( (ld->consuming_container->flags & SCFLAG_CLUSTER_DEPORTED) == 0 );

		if ( ld->consuming_container->flags & SCFLAG_CLUSTER_PRIVATE ) {
			container_private_data_t *c_pdata;
			ece_nodeid_t nodeid;

			c_pdata = (container_private_data_t *) ld->consuming_container->private_data;
			EngFncs->get_nodeid(&nodeid);
			REQUIRE( memcmp(&c_pdata->nodeid, &nodeid, sizeof(ece_nodeid_t)) == 0 );
		}
	}

        LIST_FOR_EACH( ld->parent_objects, iter, seg ) {
                REQUIRE( EngFncs->list_count(seg->parent_objects) == 0 );
		REQUIRE( seg->consuming_container == NULL );
        }

        LOG_EXIT_INT(0);
        return 0;
}


void remove_csm_from_disk( storage_object_t * ld )
{
        disk_private_data_t *disk_pdata=NULL;
        char *str;
        csm_header_t *hdr;
        DISKSEG *seg;
        seg_private_data_t *pdata;
        list_element_t iter;

	LOG_ENTRY();

        disk_pdata = get_csm_disk_private_data(ld);

        if ( disk_pdata ) {

                // unregister the disk uuid identifier
                hdr = ((seg_private_data_t *)disk_pdata->md1->private_data)->hdr;
                if (hdr) {
                        str = guid_to_string( &hdr->disk_id );
                        if (str) {
                                EngFncs->unregister_name( str );
                                free(str);
                        }
                }

                // remove the logical disk from the cluster disk group
                EngFncs->remove_thing( ld->consuming_container->objects_consumed, ld );

                // remove the segments from the cluster disk group
                LIST_FOR_EACH(ld->parent_objects, iter, seg ){
                        pdata = (seg_private_data_t *) seg->private_data;
                        if (seg->data_type == DATA_TYPE ) {
                                EngFncs->remove_thing( ld->consuming_container->objects_produced, seg );
                        }
                }

                // toss segment storage objects
                prune_csm_seg_objects_from_list(ld->parent_objects);


		if (EngFncs->list_empty(ld->consuming_container->objects_consumed)) {
			free_csm_container(ld->consuming_container);

		} else {
			// update container info for this logical disk
			ld->consuming_container->flags |= SCFLAG_DIRTY;
		}
                ld->consuming_container = NULL;

                // toss our logical disk private data area
                delete_csm_disk_private_data( ld );
        }

	LOG_EXIT_VOID();
	return;
}

/*
 *  Function:  csm_unassign
 *
 *  This routine is called to remove the Cluster Segment Manager
 *  from the specified storage object (logical disk).  It will do
 *  so by:
 *
 *      - removing CSM metadata from the storage object by writing
 *        kill sectors (zero filled sectors) to appropriate locations
 *        on the logical disk ... erasing the metadata found at
 *        those locations.
 *
 *      - removing the logical disk from the consuming container, i.e.
 *        the cluster disk group.
 *
 *      - removing the csm data segment that maps "useable space" on
 *        the share object from the producing container, i.e. the
 *        cluster disk group.
 *
 */
int csm_unassign( LOGICALDISK *ld )
{
        disk_private_data_t *disk_pdata=NULL;


        LOG_ENTRY();

        REQUIRE( isa_cluster_logical_disk(ld) == TRUE );
        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);

        disk_pdata = get_csm_disk_private_data(ld);

	if ( disk_pdata ) {
		// blast 1st copy of metadata
		KILL_SECTORS(ld, disk_pdata->md1->start, disk_pdata->md1->size);

		// blast 2nd copy of metadata
		KILL_SECTORS(ld, disk_pdata->md2->start, disk_pdata->md2->size);

		remove_csm_from_disk( ld );
	}

        LOG_EXIT_INT(0);
        return 0;
}


int csm_create( list_anchor_t          input_objects,
                option_array_t * options,
                list_anchor_t          new_objects)
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int csm_discard( list_anchor_t segments )
{
	storage_object_t *seg;
	storage_object_t *ld;
	list_element_t    iter;

	LOG_ENTRY();

        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);
	LIST_FOR_EACH(segments, iter, seg) {
		REQUIRE( isa_cluster_segment(seg) == TRUE );
	}

	LIST_FOR_EACH(segments, iter, seg) {
		ld = EngFncs->first_thing( seg->child_objects, NULL );

		EngFncs->remove_thing(ld->parent_objects, seg);
		if (seg->data_type == DATA_TYPE) {
			EngFncs->remove_thing( seg->producing_container->objects_produced, seg );
		}

		free_csm_segment( seg );
	}

	LOG_EXIT_INT(0);
	return 0;
}


int csm_destroy( DISKSEG * seg, list_anchor_t child_objects )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int csm_expand( DISKSEG          *seg,
                storage_object_t *expand_object,
                list_anchor_t           objects,
                option_array_t   *options )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int csm_shrink( DISKSEG          * seg,
                storage_object_t * shrink_object,
                list_anchor_t            objects,
                option_array_t   * options )
{
        LOG_ENTRY();
        LOG_EXIT_INT(ENOSYS);
        return ENOSYS;
}


int csm_add_sectors_to_kill_list( DISKSEG       *seg,
                                  lsn_t          lsn,
                                  sector_count_t count)
{
        int rc = EINVAL;
        LOGICALDISK *ld=NULL;
        struct plugin_functions_s *funcs=NULL;

        LOG_ENTRY();

        REQUIRE( isa_cluster_segment( seg ) == TRUE );
        REQUIRE( (lsn + count) <= seg->size );

        ld = get_logical_disk( seg );
        if (ld) {

                funcs = (struct plugin_functions_s *)ld->plugin->functions.plugin;
                rc   = funcs->add_sectors_to_kill_list( ld, seg->start+lsn, count);

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  csm_commit_changes
 *
 *  The engine will call this plug-in API to save CSM metadata
 *  changes to disk.  The two arguments tell us which segment
 *  storage object is marked dirty and which commit phase is
 *  being processed.  We have no work for commit phase Zero. So,
 *  we only call the low level CSM commit routine for subsequent
 *  commit phases.
 */
int csm_commit_changes( DISKSEG *seg, uint engine_commit_phase )
{
        int  rc=0;

        LOG_ENTRY();

        if (isa_cluster_segment(seg) == TRUE) {

                if (engine_commit_phase > 0) {
                        rc = commit_csm_metadata( seg, get_logical_disk(seg), engine_commit_phase );
                }

        }

        LOG_EXIT_INT(0);
        return 0;
}


int csm_backup_metadata( DISKSEG *seg )
{
	LOGICALDISK *ld = get_logical_disk(seg);
	list_element_t iter;
	DISKSEG *md_seg;
	int rc = 0;

	LOG_ENTRY();

	LIST_FOR_EACH(ld->parent_objects, iter, md_seg) {
		if (md_seg->data_type == META_DATA_TYPE) {
			rc = write_metadata(md_seg, ld, seg, TRUE);
			if (rc) {
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int csm_read( DISKSEG           *seg,
              lsn_t              lsn,
              sector_count_t     count,
              void              *buffer )
{
        int rc = EINVAL;
        LOGICALDISK *ld=NULL;

        LOG_ENTRY();

        REQUIRE( isa_cluster_segment(seg) == TRUE );
        REQUIRE( lsn + count <= seg->size );

        ld = get_logical_disk( seg );
        if (ld) {
                rc = READ( ld, lsn + seg->start, count, buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}

int csm_write( DISKSEG          *seg,
               lsn_t             lsn,
               sector_count_t    count,
               void             *buffer )
{
        int rc = EINVAL;
        LOGICALDISK *ld=NULL;

        LOG_ENTRY();

        REQUIRE( isa_cluster_segment(seg) == TRUE );
        REQUIRE( lsn + count <= seg->size );

        ld = get_logical_disk( seg );
        if (ld) {
                rc = WRITE( ld, lsn + seg->start, count, buffer);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


void csm_set_volume(storage_object_t * object, boolean flag)
{
        return;
}


int csm_get_info( DISKSEG *seg, char * name, extended_info_array_t  ** info_array )
{
        int rc=0;

        LOG_ENTRY();

        REQUIRE( seg!=NULL );
        REQUIRE( seg->object_type == SEGMENT );
        REQUIRE( info_array != NULL );

        if ( name==NULL ) {
                rc = csm_get_segment_info(seg, info_array);
        }
        else if ( ( strcmp(name, "Type")==0 ) && (seg->data_type==META_DATA_TYPE) ) {
                rc = csm_get_metadata_info(seg, info_array);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  delete_all_csm_segment_private_data
 *
 *  Only called by our plug-in cleanup API.  This routine will
 *  know how to properly free up all csm segment private data areas.
 */
void delete_all_csm_segment_private_data(void)
{
        list_anchor_t list;
        int rc;
        storage_object_t *object;
        list_element_t iter;


        LOG_ENTRY();

        list = EngFncs->allocate_list();
        if (list != NULL) {

                rc = EngFncs->get_object_list( SEGMENT,
                                               0,
                                               csm_plugin_record_ptr,
                                               NULL,
                                               0,
                                               &list );
                if (!rc) {
                        LIST_FOR_EACH(list, iter, object ){
                                if (object->private_data) free(object->private_data);
                        }
                }

                EngFncs->destroy_list( list );
        }

        LOG_EXIT_VOID();
}

