/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: faulthdlr.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include "fullengine.h"
#include "faulthdlr.h"
#include "engine.h"


typedef void (*sighandler_t)(int);
static sighandler_t original_signal_handler[NSIG];
static char sig_msg[256];


/*
 * Simple signal handler that logs the type of signal into the Engine log and
 * then unregisters the signal handler for that signal and passes the signal on
 * to the previously installed signal handler.
 */
static void signal_handler(int sig_no) {

	int timestamp_len;

	timestamp(sig_msg, sizeof(sig_msg), CRITICAL);
	timestamp_len = strlen(sig_msg);
	if (engine_mode & ENGINE_DAEMON) {
		sprintf(sig_msg + timestamp_len, "Daemon: %s: ", __FUNCTION__);
	} else {
		sprintf(sig_msg + timestamp_len, "Engine: %s: ", __FUNCTION__);
	}
	timestamp_len = strlen(sig_msg);

	/*
	 * Start a new line in the log in case this signal is interrupting
	 * a log message in progress.
	 */
	write(log_file_fd, "\n", 1);

	strcpy(sig_msg + timestamp_len,"***\n");
	write(log_file_fd, sig_msg, strlen(sig_msg));

	sprintf(sig_msg + timestamp_len, "*** Signal: %s\n", sys_siglist[sig_no]);
	write(log_file_fd, sig_msg, strlen(sig_msg));

	strcpy(sig_msg + timestamp_len,"***\n");
	write(log_file_fd, sig_msg, strlen(sig_msg));

	if (original_signal_handler[sig_no] != SIG_ERR) {
		/* Reset the signal handler to its previous function. */
		signal (sig_no, original_signal_handler[sig_no]);

	} else {
		/*
		 * We don't have an original handler, which shouldn't happen.
		 * Set the signal handler to the default handler.
		 */
		signal (sig_no, SIG_DFL);
	}

	original_signal_handler[sig_no] = SIG_ERR;

	/* Pass the signal on to the next signal handler. */
	raise(sig_no);
}


void install_signal_handlers(void) {

	int i;

	LOG_PROC_ENTRY();

	/* Initialize the array of original signal handlers. */
	for (i = 0; i < NSIG; i++) {
		original_signal_handler[i] = SIG_ERR;
	}

	/* Register for all the signals that can kill a process. */

	original_signal_handler[SIGINT]    = signal(SIGINT,    signal_handler);
	original_signal_handler[SIGQUIT]   = signal(SIGQUIT,   signal_handler);
	original_signal_handler[SIGILL]    = signal(SIGILL,    signal_handler);
	original_signal_handler[SIGABRT]   = signal(SIGABRT,   signal_handler);
	original_signal_handler[SIGBUS]    = signal(SIGBUS,    signal_handler);
	original_signal_handler[SIGFPE]    = signal(SIGFPE,    signal_handler);
	original_signal_handler[SIGSEGV]   = signal(SIGSEGV,   signal_handler);
	original_signal_handler[SIGPIPE]   = signal(SIGPIPE,   signal_handler);
	original_signal_handler[SIGTERM]   = signal(SIGTERM,   signal_handler);
#ifdef SIGSTKFLT
	original_signal_handler[SIGSTKFLT] = signal(SIGSTKFLT, signal_handler);
#endif
	original_signal_handler[SIGXCPU]   = signal(SIGXCPU,   signal_handler);
	original_signal_handler[SIGXFSZ]   = signal(SIGXFSZ,   signal_handler);
	original_signal_handler[SIGVTALRM] = signal(SIGVTALRM, signal_handler);

	LOG_PROC_EXIT_VOID();
}


void remove_signal_handlers(void) {

	int i;

	LOG_PROC_ENTRY();

	for (i = 1; i < NSIG; i++) {
		if (original_signal_handler[i] != SIG_ERR) {
			signal(i, original_signal_handler[i]);
			original_signal_handler[i] = SIG_ERR;
		}
	}

	LOG_PROC_EXIT_VOID();
}
