/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: container.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

#include "fullengine.h"
#include "container.h"
#include "lists.h"
#include "engine.h"
#include "discover.h"
#include "handlemgr.h"
#include "common.h"
#include "internalAPI.h"
#include "object.h"
#include "message.h"
#include "memman.h"
#include "remote.h"


int evms_create_container(plugin_handle_t   plugin_handle,
			  handle_array_t  * input_objects,
			  option_array_t  * options,
			  object_handle_t * new_container_handle) {
	int rc = 0;
	void * object;
	object_type_t type;

	LOG_PROC_ENTRY();

	rc = check_engine_write_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_create_container(plugin_handle, input_objects, options, new_container_handle);
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	rc = translate_handle(plugin_handle,
			      &object,
			      &type);

	if (rc != HANDLE_MANAGER_NO_ERROR) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (type == PLUGIN) {
		plugin_record_t * pPlugRec = (plugin_record_t *) object;

		/* Make sure the plug-in does containers. */
		if (pPlugRec->container_functions != NULL) {

			STATIC_LIST_DECL(object_list);

			rc = make_list(input_objects, &object_list);

			if (rc == 0) {
				list_element_t iter;
				storage_object_t * obj;
				storage_container_t * disk_group = NULL;

				obj = first_thing(&object_list, NULL);
				if (obj != NULL) {
					disk_group = obj->disk_group;
				}

				/*
				 * Make sure that the input object list contains
				 * top level objects and none of them insist on
				 * being a top object.
				 */
				LIST_FOR_EACH(&object_list, iter, obj) {
					rc = isa_valid_input_object(obj, disk_group);
				}

				if (rc == 0) {
					storage_container_t * new_container;

					rc = pPlugRec->container_functions->create_container(&object_list, options, &new_container);

					if (rc == 0) {
						storage_object_t * child_obj = NULL;

						/*
						 * Now that the objects are consumed
						 * any stop data on them can be
						 * overwritten.
						 */
						LIST_FOR_EACH(new_container->objects_consumed, iter, child_obj) {
							child_obj->flags &= ~SOFLAG_HAS_STOP_DATA;
						}

						if (new_container->disk_group != NULL) {
							propigate_cluster_info(new_container->objects_produced);
						}

						sort_list(&containers_list, compare_containers, NULL);

						switch (GetPluginType(pPlugRec->id)) {
							case EVMS_DEVICE_MANAGER:
								sort_list(&disks_list, compare_objects, NULL);
								break;
							case EVMS_SEGMENT_MANAGER:
								sort_list(&segments_list, compare_objects, NULL);
								break;
							case EVMS_REGION_MANAGER:
								sort_list(&regions_list, compare_objects, NULL);
								break;
							case EVMS_FEATURE:
							case EVMS_ASSOCIATIVE_FEATURE:
								sort_list(&EVMS_objects_list, compare_objects, NULL);
								break;
							default:
								break;
						}

						rc = create_handle(new_container,
								   CONTAINER,
								   &new_container->app_handle);
						if (rc == 0) {
							*new_container_handle = new_container->app_handle;

						} else {
							LOG_WARNING("Error %d creating a handle for container %s.\n", rc, new_container->name);
						}
					}
				}
			}
		}

	} else {
		LOG_ERROR("handle is not for a plug-in.\n");
		rc = EINVAL;
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}


/*
 * evms_get_container_list returns a pointer to a handle_array_t with handles for
 * storage containers, optionally filtering on the region manager that manages
 * the container.  If the object handle for the region manager is 0, handles for
 * all of the storage containers will be returned.
 */
int evms_get_container_list(object_handle_t          plugin_handle,
			    object_handle_t          disk_group_handle,
			    container_search_flags_t flags,
			    handle_array_t       * * container_handle_list) {
	int rc = 0;
	void * object = NULL;
	object_type_t type;
	plugin_record_t * pPlugRec = NULL;
	storage_container_t * disk_group = NULL;
	handle_array_t * container_handles;

	LOG_PROC_ENTRY();

	rc = check_engine_read_access();
	if (rc != 0) {
		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (!local_focus) {
		rc = remote_get_container_list(plugin_handle, disk_group_handle, flags, container_handle_list);

		LOG_PROC_EXIT_INT(rc);
		return rc;
	}

	if (plugin_handle != 0) {
		/* Translate the handle for the region manager to make sure it is valid */
		/* and to get the plugin_record_t for the region manager. */
		rc = translate_handle(plugin_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if (type == PLUGIN) {
				pPlugRec = (plugin_record_t *) object;
			} else {
				rc = EINVAL;
			}
		}
	}

	if (disk_group_handle != 0) {
		/*
		 * Translate the handle for the disk group to make sure
		 * it is valid and to get the disk group
		 * storage_container_t.
		 */
		rc = translate_handle(disk_group_handle,
				      &object,
				      &type);

		if (rc == HANDLE_MANAGER_NO_ERROR) {
			if (type == CONTAINER) {
				disk_group = (storage_container_t *) object;
			} else {
				rc = EINVAL;
			}
		}
	}

	if (rc == 0) {
		list_anchor_t container_list;

		rc = engine_get_container_list(pPlugRec, disk_group, flags, &container_list);

		if (rc == 0) {
			rc = make_handle_array(container_list, &container_handles);

		}
		if (rc == 0) {
			uint size = sizeof(handle_array_t) + (sizeof(object_handle_t) * container_handles->count);

			*container_handle_list = alloc_app_struct(size, NULL);

			if (*container_handle_list != NULL) {
				/* Fill in the structure returned to the app. */
				memcpy(*container_handle_list, container_handles, size);

				/*
				 * Free the internal handle array.
				 */
				engine_free(container_handles);

			} else {
				rc = ENOMEM;
			}
		}

		/* We are finished with the list that was returned by */
		/* engine_get_container_list(). */
		destroy_list(container_list);
	}

	LOG_PROC_EXIT_INT(rc);
	return rc;
}

