/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: appstructs.h
 */

#ifndef EVMS_APP_STRUCTS_H_INCLUDED
#define EVMS_APP_STRUCTS_H_INCLUDED 1

#include <linux/evms/evms_user.h>

#include <dlist.h>
#include <common.h>


typedef struct engine_message_functions_s {
    /*
     * This function can be used in two ways.
     *
     * 1) Display a notification message to the user expecting no response.
     *
     * user_message(message, NULL, NULL);
     *
     * The UI must return immediately from this function.  That is, the UI must
     * not wait for user input, such a pressing an OK button.  The UI can queue
     * the message to be handled on another thread if necessary, but must then
     * return from this call.  If the UI defers the handling of the message to
     * another time (e.g., via queuing or threading), the UI must make a copy of
     * the message since the message may be freed when the call returns to the
     * Engine or plug-in that initiated the request.
     *
     * 2) Ask a question and select one item from a list of two or more items.
     *
     * user_message(message, &answer, choices);
     *
     * The "choices" parameter is a NULL terminated array of strings that
     * describe each of the choices.
     * Display the message and the choices.  Set *answer to the index of the
     * choice string that the user selects.  *answer will be initialized to the
     * default response.
     */
    int (*user_message)(char   * message_text,
                        int    * answer,
                        char * * choices);

    /*
     * Given a task handle, use the task based APIs, get_option_descriptor(),
     * set_option_value(), etc., to set the values for the option descriptors
     * contained in the task.  The task action is set to EVMS_Task_Message.
     * message_text is a title that describes the options and/or why they need
     * to be set.  Display the option descriptors as you normally would for a
     * task.  When finished setting options, simply return from the call to
     * user_communication.  Do not call evms_invoke_task() on the task.
     */
    int (*user_communication)(char        * message_text,
                              task_handle_t task);
} engine_message_functions_t;


/*
 * typedef for a callback function that is passed on evms_commit_changes().
 * The Engine will call this function with a status message at various
 * points during the commit process.
 */
typedef void (*commit_status_func_t)(char * message);

typedef struct handle_array_s {
    uint            count;
    object_handle_t handle[1];
} handle_array_t;


typedef struct plugin_info_s {
    object_handle_t handle;                         /* Handle for this plug-in record */
    plugin_id_t     id;                             /* Plug-in's ID */
    evms_version_t  version;                        /* Plug-in's version */
    evms_version_t  required_api_version;           /* Version of the Engine plug-in API */
                                                    /* that the plug-in requires */
    char          * short_name;
    char          * long_name;
    char          * oem_name;
    BOOLEAN         supports_containers;
} plugin_info_t;


typedef struct storage_object_info_s {
    object_handle_t         handle;                 /* Handle of this object */
    object_type_t           object_type;            /* SEGMENT, REGION, DISK ,... */
    data_type_t             data_type;              /* DATA_TYPE, META_DATA_TYPE, FREE_SPACE_TYPE */
    plugin_handle_t         plugin;                 /* Handle of the plug-in that manages */
                                                    /* this storage object */
    object_handle_t         producing_container;    /* storage_container that produced this object */
    object_handle_t         consuming_container;    /* storage_container that consumed this object */
    handle_array_t        * parent_objects;         /* List of parent objects, filled in by parent   */
    handle_array_t        * child_objects;          /* List of objects that comprise this object */
    u_int32_t               flags;                  /* Defined by SOFLAG_???? defines */
    char                    name[EVMS_NAME_SIZE+1]; /* Object's name */
    lba_t                   start;                  /* Relative starting block of this object */
    sector_count_t          size;                   /* Size of the storage object in sectors */
    geometry_t              geometry;               /* Optional object geometry */
    object_handle_t         volume;                 /* Volume to which this object belongs */
} storage_object_info_t;


typedef struct storage_container_info_s {
    object_handle_t         handle;                 /* Handle of this container */
    object_handle_t         plugin;                 /* Handle of the plug-in that */
                                                    /* manages this container */
    uint                    flags;                  /* Defined by SCFLAG_???? */
    char                    name[EVMS_NAME_SIZE+1]; /* Container's name */
    handle_array_t        * objects_consumed;       /* Handles of objects consumed */
                                                    /* by this container */
    handle_array_t        * objects_produced;       /* Handles of objects produced */
                                                    /* by this container */
    sector_count_t          size;                   /* Size of the container */
} storage_container_info_t;


typedef struct logical_volume_info_s {
    object_handle_t         handle;                 /* Handle of this volume */
    object_handle_t         file_system_manager;    /* Handle of the File System Interface */
                                                    /* Module that handles this volume */
    char *                  mount_point;            /* Dir where the volume is mounted, NULL if not mounted */
    sector_count_t          fs_size;                /* Size of the file system */
    sector_count_t          min_fs_size;            /* Minimum size of the file system */
    sector_count_t          max_fs_size;            /* Maximum size of the file system */
    sector_count_t          vol_size;               /* Size of the volume */
    sector_count_t          max_vol_size;           /* Maximum size for the volume */
    object_handle_t         associated_volume;      /* Handle of the volume to which this volume is */
                                                    /* associated by an associative feature */
    object_handle_t         object;                 /* Handle of the top level object for */
                                                    /* this volume */
    uint                    minor_number;           /* Volume's minor number */
    u_int64_t               serial_number;          /* Volume's serial number */
    u_int32_t               flags;                  /* Defined by VOLFLAG_???? defines */
    char                    name[EVMS_VOLUME_NAME_SIZE+1];
} logical_volume_info_t;


/*
 * Structures for returning object information
 */
typedef union {
    plugin_info_t            plugin;
    storage_object_info_t    disk;
    storage_object_info_t    segment;
    storage_container_info_t container;
    storage_object_info_t    region;
    storage_object_info_t    object;
    logical_volume_info_t    volume;
} object_info_t;


typedef struct handle_object_info_s {
    object_type_t   type;
    object_info_t   info;
} handle_object_info_t;

/*
 * Data for an expand point: handle of the object that can be expanded and
 * the maximum delta size by which it can be expanded.
 */
typedef struct expand_handle_s {
    object_handle_t     object;
    sector_count_t      max_expand_size;
} expand_handle_t;

/*
 * An array of expand points.
 */
typedef struct expand_handle_array_s {
    uint                count;
    expand_handle_t     expand_point[1];
} expand_handle_array_t;

/*
 * Data for an shrink point: handle of the object that can be shrunk and
 * the maximum delta size by which it can be shrunk.
 */
typedef struct shrink_handle_s {
    object_handle_t     object;
    sector_count_t      max_shrink_size;
} shrink_handle_t;

/*
 * An array of shrink points.
 */
typedef struct shrink_handle_array_s {
    uint                count;
    shrink_handle_t     shrink_point[1];
} shrink_handle_array_t;

#endif
